package com.vaadin.spring.roo.addon.entitymanagerview;

import java.lang.reflect.Modifier;
import java.util.ArrayList;
import java.util.List;
import java.util.logging.Logger;

import org.springframework.roo.classpath.PhysicalTypeIdentifierNamingUtils;
import org.springframework.roo.classpath.PhysicalTypeMetadata;
import org.springframework.roo.classpath.details.MethodMetadata;
import org.springframework.roo.classpath.details.MethodMetadataBuilder;
import org.springframework.roo.classpath.itd.AbstractItdTypeDetailsProvidingMetadataItem;
import org.springframework.roo.classpath.itd.InvocableMemberBodyBuilder;
import org.springframework.roo.classpath.itd.ItdSourceFileComposer;
import org.springframework.roo.metadata.MetadataIdentificationUtils;
import org.springframework.roo.metadata.MetadataItem;
import org.springframework.roo.metadata.MetadataService;
import org.springframework.roo.model.DataType;
import org.springframework.roo.model.JavaSymbolName;
import org.springframework.roo.model.JavaType;
import org.springframework.roo.project.Path;
import org.springframework.roo.support.style.ToStringCreator;
import org.springframework.roo.support.util.Assert;

import com.vaadin.spring.roo.addon.VaadinRooUtils;
import com.vaadin.spring.roo.addon.annotations.RooVaadinEntityManagerView;
import com.vaadin.spring.roo.addon.entityview.VaadinEntityViewMetadata;

/**
 * Metadata for {@link RooVaadinEntityManagerView}.
 */
public class VaadinEntityManagerViewMetadata extends
		AbstractItdTypeDetailsProvidingMetadataItem {

	private static Logger logger = Logger
			.getLogger(VaadinEntityManagerViewMetadata.class.getName());

	public static final String LIST_ENTITY_VIEWS_METHOD = "listEntityViews";

	private static final String PROVIDES_TYPE_STRING = VaadinEntityManagerViewMetadata.class
			.getName();
	private static final String PROVIDES_TYPE = MetadataIdentificationUtils
			.create(PROVIDES_TYPE_STRING);

	private MetadataService metadataService;
	private List<String> entityViewMids;

	public VaadinEntityManagerViewMetadata(String identifier,
			JavaType aspectName,
			PhysicalTypeMetadata governorPhysicalTypeMetadata,
			MetadataService metadataService,
			List<String> entityViewMids) {
		super(identifier, aspectName, governorPhysicalTypeMetadata);
		Assert.isTrue(isValid(identifier), "Metadata identification string '"
				+ identifier + "' does not appear to be a valid");
		Assert.notNull(metadataService, "Metadata service required");

		if (!isValid()) {
			return;
		}

		this.metadataService = metadataService;
		this.entityViewMids = entityViewMids;

		builder.addMethod(getListEntityViewsMethod(entityViewMids));
		
		itdTypeDetails = builder.build();

		new ItdSourceFileComposer(itdTypeDetails);
	}

	@Override
	public String toString() {
		ToStringCreator tsc = new ToStringCreator(this);
		tsc.append("identifier", getId());
		tsc.append("valid", valid);
		tsc.append("aspectName", aspectName);
		tsc.append("destinationType", destination);
		tsc.append("governor", governorPhysicalTypeMetadata.getId());
		tsc.append("itdTypeDetails", itdTypeDetails);
		return tsc.toString();
	}

	public static final String getMetadataIdentiferType() {
		return PROVIDES_TYPE;
	}

	public static final String getMetadataIdentiferString() {
		return PROVIDES_TYPE_STRING;
	}

	public static final String createIdentifier(JavaType javaType, Path path) {
		return PhysicalTypeIdentifierNamingUtils.createIdentifier(
				PROVIDES_TYPE_STRING, javaType, path);
	}

	public static final JavaType getJavaType(String metadataIdentificationString) {
		return PhysicalTypeIdentifierNamingUtils.getJavaType(
				PROVIDES_TYPE_STRING, metadataIdentificationString);
	}

	public static final Path getPath(String metadataIdentificationString) {
		return PhysicalTypeIdentifierNamingUtils.getPath(PROVIDES_TYPE_STRING,
				metadataIdentificationString);
	}

	public static boolean isValid(String metadataIdentificationString) {
		return PhysicalTypeIdentifierNamingUtils.isValid(PROVIDES_TYPE_STRING,
				metadataIdentificationString);
	}

	public List<String> getEntityViewMids() {
		return entityViewMids;
	}

	private MethodMetadata getListEntityViewsMethod(List<String> entityViewMids) {
		JavaSymbolName methodName = new JavaSymbolName(LIST_ENTITY_VIEWS_METHOD);

		MethodMetadata method = VaadinRooUtils.methodExists(methodName,
				governorTypeDetails);
		if (method != null) {
			return method;
		}

		JavaType stringClass = new JavaType("java.lang.String");
		JavaType componentClassClass = new JavaType("java.lang.Class");
		List<JavaType> typeParams = new ArrayList<JavaType>();
		typeParams.add(stringClass);
		typeParams.add(componentClassClass);

		JavaType returnType = new JavaType("java.util.Map", 0, DataType.TYPE,
				null, typeParams);
		String listClassName = returnType.getNameIncludingTypeParameters(false,
				builder.getImportRegistrationResolver());

		// result is sorted alphabetically by key (entity class display name)
		JavaType concreteResultType = new JavaType("java.util.TreeMap", 0,
				DataType.TYPE, null, typeParams);
		String arrayListClassName = concreteResultType
				.getNameIncludingTypeParameters(false,
						builder.getImportRegistrationResolver());

		InvocableMemberBodyBuilder bodyBuilder = new InvocableMemberBodyBuilder();
		bodyBuilder.appendFormalLine(listClassName + " result = new "
				+ arrayListClassName + "();");
		// add entity views to the list
		for (String mid : entityViewMids) {
			MetadataItem metadata = metadataService.get(mid);
			if (!(metadata instanceof VaadinEntityViewMetadata)) {
				// logger.info("Not an entity view: " + mid);
				continue;
			}
			VaadinEntityViewMetadata entityViewMetadata = (VaadinEntityViewMetadata) metadata;
			JavaType entityViewType = VaadinEntityViewMetadata.getJavaType(mid);
			// entity class display name, derived from the entity class name
			String key = entityViewMetadata.getEntityDisplayName();
			String entityViewClassName = entityViewType
					.getNameIncludingTypeParameters(false,
							builder.getImportRegistrationResolver());
			bodyBuilder
					.appendFormalLine("result.put("+"\""+key+"\""+ "," + entityViewClassName + ".class);");
		}
		bodyBuilder
				.appendFormalLine("return result;");

		MethodMetadataBuilder methodBuilder = new MethodMetadataBuilder(
				getId(), Modifier.PUBLIC, methodName, returnType, bodyBuilder);
		return methodBuilder.build();
	}

}
