package com.vaadin.spring.roo.addon.entitymanagerview;

import java.util.ArrayList;
import java.util.List;
import java.util.logging.Logger;

import org.apache.felix.scr.annotations.Component;
import org.apache.felix.scr.annotations.Reference;
import org.apache.felix.scr.annotations.Service;
import org.osgi.service.component.ComponentContext;
import org.springframework.roo.classpath.PhysicalTypeCategory;
import org.springframework.roo.classpath.PhysicalTypeIdentifier;
import org.springframework.roo.classpath.PhysicalTypeMetadata;
import org.springframework.roo.classpath.details.ClassOrInterfaceTypeDetails;
import org.springframework.roo.classpath.details.MemberFindingUtils;
import org.springframework.roo.classpath.itd.AbstractItdMetadataProvider;
import org.springframework.roo.classpath.itd.ItdMetadataProvider;
import org.springframework.roo.classpath.itd.ItdSourceFileComposer;
import org.springframework.roo.classpath.itd.ItdTypeDetailsProvidingMetadataItem;
import org.springframework.roo.metadata.MetadataDependencyRegistry;
import org.springframework.roo.metadata.MetadataIdentificationUtils;
import org.springframework.roo.metadata.MetadataItem;
import org.springframework.roo.metadata.MetadataNotificationListener;
import org.springframework.roo.metadata.MetadataService;
import org.springframework.roo.model.JavaType;
import org.springframework.roo.process.manager.FileManager;
import org.springframework.roo.project.Path;
import org.springframework.roo.support.util.Assert;

import com.vaadin.spring.roo.addon.VaadinRooUtils;
import com.vaadin.spring.roo.addon.annotations.RooVaadinEntityManagerView;
import com.vaadin.spring.roo.addon.entityview.VaadinEntityViewMetadata;

/**
 * Provides {@link VaadinEntityManagerViewMetadata}, creating ITD that permits
 * listing all entity views.
 *
 * This requires listening to both class and instance notifications for both
 * {@link PhysicalTypeMetadata} on the entity manager view and
 * {@link VaadinEntityViewMetadata} on the entity views. This is not possible
 * using {@link AbstractItdMetadataProvider}, so need to replicate large parts
 * of its logic.
 */
@Component(immediate = true)
@Service
public class VaadinEntityManagerViewMetadataProvider implements
		ItdMetadataProvider, MetadataNotificationListener {

	@Reference
	protected MetadataDependencyRegistry metadataDependencyRegistry;
	@Reference
	protected FileManager fileManager;
	@Reference
	protected MetadataService metadataService;

	private static Logger logger = Logger
			.getLogger(VaadinEntityManagerViewMetadataProvider.class.getName());

	protected void activate(ComponentContext context) {
		metadataDependencyRegistry.registerDependency(
				PhysicalTypeIdentifier.getMetadataIdentiferType(),
				getProvidesType());

		// get notifications also for entity view changes

		metadataDependencyRegistry.registerDependency(
				VaadinEntityViewMetadata.getMetadataIdentiferType(),
				getProvidesType());
	}

	public void notify(String upstreamDependency,
			String downstreamDependency) {

		if (MetadataIdentificationUtils
				.isIdentifyingClass(downstreamDependency)
				&& !VaadinRooUtils
						.isNotificationForJavaType(upstreamDependency)) {

			// This notification is NOT for a JavaType, and did NOT identify a
			// specific downstream instance, so we must compute
			// all possible downstream dependencies by a disk scan.
			notifyAllEntityManagerInstances();

			return;
		}

		if (MetadataIdentificationUtils
				.isIdentifyingClass(downstreamDependency)) {
			Assert.isTrue(VaadinRooUtils
					.isNotificationForJavaType(upstreamDependency),
					"Expected class-level notifications only for physical Java types (not '"
							+ upstreamDependency + "')");

			// A physical Java type has changed, and determine what the
			// corresponding local metadata identification string would have
			// been
			JavaType javaType = PhysicalTypeIdentifier
					.getJavaType(upstreamDependency);
			Path path = PhysicalTypeIdentifier.getPath(upstreamDependency);
			downstreamDependency = createLocalIdentifier(javaType, path);

			// We only need to proceed if the downstream dependency relationship
			// is not already registered
			// (if it's already registered, the event will be delivered directly
			// later on)
			if (metadataDependencyRegistry.getDownstream(upstreamDependency)
					.contains(downstreamDependency)) {
				return;
			}
		}

		// We should now have an instance-specific "downstream dependency" that
		// can be processed by this class
		Assert.isTrue(
				MetadataIdentificationUtils.getMetadataClass(
						downstreamDependency).equals(
						MetadataIdentificationUtils
								.getMetadataClass(getProvidesType())),
				"Unexpected downstream notification for '"
						+ downstreamDependency
						+ "' to this provider (which uses '"
						+ getProvidesType() + "'");

		metadataService.evict(downstreamDependency);
		if (get(downstreamDependency) != null) {
			metadataDependencyRegistry.notifyDownstream(downstreamDependency);
		}
	}

	private void notifyAllEntityManagerInstances() {
		String providesType = VaadinEntityManagerViewMetadata
				.getMetadataIdentiferString();
		List<String> allMids = VaadinRooUtils.scanPotentialTypeMids(
				metadataService, fileManager, providesType);

		for (String mid : allMids) {
			metadataService.evict(mid);
			if (get(mid) != null) {
				metadataDependencyRegistry.notifyDownstream(mid);
			}
		}
	}

	private List<String> findAllEntityViewMids() {
		String providesType = VaadinEntityViewMetadata
				.getMetadataIdentiferString();
		List<String> allMids = VaadinRooUtils.scanPotentialTypeMids(
				metadataService, fileManager, providesType);

		List<String> entityViewMids = new ArrayList<String>();
		for (String mid : allMids) {
			if (VaadinEntityViewMetadata.isValid(mid)) {
				entityViewMids.add(mid);
			}
		}

		return entityViewMids;
	}

	protected VaadinEntityManagerViewMetadata getMetadata(
			String metadataIdentificationString, JavaType aspectName,
			PhysicalTypeMetadata governorPhysicalTypeMetadata,
			String itdFilename) {
		// We know governor type details are non-null and can be safely cast

		// find all entity views
		List<String> entityViews = findAllEntityViewMids();

		return new VaadinEntityManagerViewMetadata(
				metadataIdentificationString, aspectName,
				governorPhysicalTypeMetadata, metadataService, entityViews);
	}

	public final MetadataItem get(String metadataIdentificationString) {
		Assert.isTrue(
				MetadataIdentificationUtils.getMetadataClass(
						metadataIdentificationString).equals(
						MetadataIdentificationUtils
								.getMetadataClass(getProvidesType())),
				"Unexpected request for '" + metadataIdentificationString
						+ "' to this provider (which uses '"
						+ getProvidesType() + "'");

		// Remove the upstream dependencies for this instance (we'll be
		// recreating them later, if needed)
		metadataDependencyRegistry
				.deregisterDependencies(metadataIdentificationString);

		// Compute the identifier for the Physical Type Metadata we're
		// correlated with
		String governorPhysicalTypeIdentifier = getGovernorPhysicalTypeIdentifier(metadataIdentificationString);

		// Obtain the physical type
		PhysicalTypeMetadata governorPhysicalTypeMetadata = (PhysicalTypeMetadata) metadataService
				.get(governorPhysicalTypeIdentifier);

		if (governorPhysicalTypeMetadata == null
				|| !governorPhysicalTypeMetadata.isValid()) {
			// We can't get even basic information about the physical type, so
			// abort (the ITD will be deleted by ItdFileDeletionService)
			return null;
		}

		// Determine ITD details
		String itdFilename = governorPhysicalTypeMetadata
				.getItdCanoncialPath(this);
		JavaType aspectName = governorPhysicalTypeMetadata.getItdJavaType(this);

		// Flag to indicate whether we'll even try to create this metadata
		boolean produceMetadata = false;

		// Determine if we should generate the metadata on the basis of it
		// containing a trigger annotation
		ClassOrInterfaceTypeDetails cid = null;
		if (governorPhysicalTypeMetadata.getMemberHoldingTypeDetails() != null
				&& governorPhysicalTypeMetadata.getMemberHoldingTypeDetails() instanceof ClassOrInterfaceTypeDetails) {
			cid = (ClassOrInterfaceTypeDetails) governorPhysicalTypeMetadata
					.getMemberHoldingTypeDetails();

			// Only create metadata if the type is annotated with one of the
			// metadata triggers
			JavaType trigger = new JavaType(
					RooVaadinEntityManagerView.class.getName());
			if (MemberFindingUtils.getDeclaredTypeAnnotation(cid, trigger) != null) {
				boolean isClass = cid.getPhysicalTypeCategory() == PhysicalTypeCategory.CLASS;
				produceMetadata = isClass;
			}
		}

		if (produceMetadata) {
			// This type contains an annotation we were configured to detect, or
			// there is an ITD (which may need deletion), so we need to produce
			// the metadata
			VaadinEntityManagerViewMetadata metadata;
			metadata = getMetadata(metadataIdentificationString, aspectName,
					governorPhysicalTypeMetadata, itdFilename);

			// Register a direct connection between the physical type and this
			// metadata
			// (this is needed so changes to the inheritance hierarchies are
			// eventually notified to us)
			metadataDependencyRegistry.registerDependency(
					governorPhysicalTypeMetadata.getId(),
					metadataIdentificationString);

			// Quit if no metadata created
			if (metadata == null) {
				return null;
			}

			// register upstream dependency on the physical types of all
			// found entity views
			for (String entityViewMid : metadata.getEntityViewMids()) {
				JavaType javaType = VaadinEntityViewMetadata
						.getJavaType(entityViewMid);
				Path path = VaadinEntityViewMetadata.getPath(entityViewMid);
				String entityViewPhysicalType = PhysicalTypeIdentifier
						.createIdentifier(javaType, path);
				metadataDependencyRegistry.registerDependency(
						entityViewPhysicalType, metadataIdentificationString);
			}

			// Handle the management of the ITD file
			if (writeItd(itdFilename, metadata)) {
				// A valid Itd has been written, don't proceed onto the
				// delete operation below
				return metadata;
			}

			// Delete the ITD if we determine deletion is appropriate
			if (metadata.isValid() && fileManager.exists(itdFilename)) {
				fileManager.delete(itdFilename);
			}

			return metadata;
		} else if (fileManager.exists(itdFilename)) {
			// We don't seem to want metadata anymore, yet the ITD physically
			// exists, so get rid of it
			// This might be because the trigger annotation has been removed,
			// the governor is missing a class declaration etc
			fileManager.delete(itdFilename);
		}

		return null;
	}

	private boolean writeItd(String itdFilename,
			ItdTypeDetailsProvidingMetadataItem metadata) {
		if (metadata.getMemberHoldingTypeDetails() != null) {
			// Construct the source file composer
			ItdSourceFileComposer itdSourceFileComposer = new ItdSourceFileComposer(
					metadata.getMemberHoldingTypeDetails());

			// Output the ITD if there is actual content involved
			// (if there is no content, we continue on to the deletion phase
			// at the bottom of this conditional block)
			if (itdSourceFileComposer.isContent()) {
				String itd = itdSourceFileComposer.getOutput();

				fileManager.createOrUpdateTextFileIfRequired(itdFilename, itd,
						false);
				// a valid ITD has been written out
				return true;
			}
		}
		return false;
	}

	public String getIdForPhysicalJavaType(String physicalJavaTypeIdentifier) {
		Assert.isTrue(
				MetadataIdentificationUtils.getMetadataClass(
						physicalJavaTypeIdentifier).equals(
						MetadataIdentificationUtils
								.getMetadataClass(PhysicalTypeIdentifier
										.getMetadataIdentiferType())),
				"Expected a valid physical Java type instance identifier (not '"
						+ physicalJavaTypeIdentifier + "')");
		JavaType javaType = PhysicalTypeIdentifier
				.getJavaType(physicalJavaTypeIdentifier);
		Path path = PhysicalTypeIdentifier.getPath(physicalJavaTypeIdentifier);
		return createLocalIdentifier(javaType, path);
	}

	public String getItdUniquenessFilenameSuffix() {
		return "VaadinEntityManagerView";
	}

	protected String getGovernorPhysicalTypeIdentifier(
			String metadataIdentificationString) {
		JavaType javaType = VaadinEntityManagerViewMetadata
				.getJavaType(metadataIdentificationString);
		Path path = VaadinEntityManagerViewMetadata
				.getPath(metadataIdentificationString);
		String physicalTypeIdentifier = PhysicalTypeIdentifier
				.createIdentifier(javaType, path);
		return physicalTypeIdentifier;
	}

	protected String createLocalIdentifier(JavaType javaType, Path path) {
		return VaadinEntityManagerViewMetadata.createIdentifier(javaType, path);
	}

	public String getProvidesType() {
		return VaadinEntityManagerViewMetadata.getMetadataIdentiferType();
	}

}
