package com.vaadin.spring.roo.addon.entityview;

import java.lang.reflect.Modifier;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;

import org.springframework.roo.addon.entity.EntityMetadata;
import org.springframework.roo.addon.plural.PluralMetadata;
import org.springframework.roo.classpath.details.ClassOrInterfaceTypeDetails;
import org.springframework.roo.classpath.details.FieldMetadataBuilder;
import org.springframework.roo.classpath.details.ItdTypeDetailsBuilder;
import org.springframework.roo.classpath.details.MethodMetadata;
import org.springframework.roo.classpath.details.MethodMetadataBuilder;
import org.springframework.roo.classpath.itd.InvocableMemberBodyBuilder;
import org.springframework.roo.classpath.scanner.MemberDetailsScanner;
import org.springframework.roo.metadata.MetadataService;
import org.springframework.roo.model.DataType;
import org.springframework.roo.model.ImportRegistrationResolver;
import org.springframework.roo.model.JavaPackage;
import org.springframework.roo.model.JavaSymbolName;
import org.springframework.roo.model.JavaType;
import org.springframework.roo.project.Path;

import com.vaadin.spring.roo.addon.VaadinOperationsImpl;
import com.vaadin.spring.roo.addon.VaadinRooUtils;
import com.vaadin.spring.roo.addon.entityform.VaadinClassNames;

public class ContainerMethodBuilder extends
		AbstractVaadinEntityViewMethodBuilder {

	private static final String IN_MEMORY_CONTAINER_CLASS = VaadinClassNames.BEAN_CONTAINER_CLASS;
	private static final String ENUM_CONTAINER_CLASS = "com.vaadin.data.util.BeanItemContainer";

	private boolean useBeanContainer = true;

	private final ItdTypeDetailsBuilder builder;
	
	private final JavaPackage utilPackage;

	/**
	 * Parameters used to configure a JpaContainer when creating a container.
	 */
	public static class JpaContainerParameters {
		/**
		 * true if the container contents should be editable; only applies to
		 * JPAContainer
		 */
		private final boolean mutable;

		public JpaContainerParameters(boolean mutable) {
			this.mutable = mutable;
		}

		public boolean isMutable() {
			return mutable;
		}
	}

	public ContainerMethodBuilder(
			ClassOrInterfaceTypeDetails governorTypeDetails, String metadataId,
			VaadinEntityMetadataDetails vaadinEntityDetails,
			JavaPackage utilPackage, ImportRegistrationResolver importResolver,
			MetadataService metadataService,
			MemberDetailsScanner memberDetailsScanner,
			ItdTypeDetailsBuilder builder) {
		super(governorTypeDetails, metadataId, vaadinEntityDetails,
				importResolver, metadataService, memberDetailsScanner);
		this.utilPackage = utilPackage;
		this.builder = builder;
	}

	/**
	 * Constructs a method that creates and returns a container.
	 *
	 * Depending on the data type and whether fieldName has been given, the
	 * container may or may not be cached for subsequent calls.
	 *
	 * @param type
	 * @param populateMethodName
	 * @param fieldName
	 *            name of the field in which to cache the container, null to
	 *            deduce automatically (or automatically disable caching) based
	 *            on data type and container type.
	 * @param jpaParameters
	 *            parameters used to configure JpaContainer, null not to use
	 *            JpaContainer
	 * @return
	 */
	public MethodMetadata buildGetContainerMethod(JavaType type,
			JavaSymbolName populateMethodName, JavaSymbolName fieldName,
			JpaContainerParameters jpaParameters) {
		// caller makes sure the method does not exist

		EntityMetadata typeEntityMetadata = (EntityMetadata) getMetadataService()
				.get(EntityMetadata.createIdentifier(type, Path.SRC_MAIN_JAVA));
		boolean isEnum = VaadinRooUtils.isEnumType(getMetadataService(), type);

		List<JavaType> typeParameters = Collections.singletonList(type);

		JavaType returnType;
		if (isEnum) {
			if (fieldName == null) {
				fieldName = getContainerFieldName(type);
			}
			returnType = new JavaType(ENUM_CONTAINER_CLASS, 0, DataType.TYPE,
					null, typeParameters);
		} else if (typeEntityMetadata != null && jpaParameters != null) {
			if (fieldName == null) {
				fieldName = getContainerFieldName(type);
			}
			returnType = new JavaType(VaadinClassNames.JPA_CONTAINER_CLASS, 0, DataType.TYPE,
					null, typeParameters);
		} else if (typeEntityMetadata != null) {
			JavaType idType = typeEntityMetadata.getIdentifierField()
					.getFieldType();
			typeParameters = Arrays.asList(idType, type);
			returnType = new JavaType(IN_MEMORY_CONTAINER_CLASS, 0,
					DataType.TYPE, null, typeParameters);
		} else {
			// not a supported type, do not generate a method
			return null;
		}

		InvocableMemberBodyBuilder bodyBuilder = new InvocableMemberBodyBuilder();

		if (fieldName != null) {
			// create field (side-effect of this method)
			FieldMetadataBuilder fieldBuilder = new FieldMetadataBuilder(
					getId(), Modifier.PRIVATE, fieldName, returnType, null);
			builder.addField(fieldBuilder.build());

			// do not recreate the container if have initialized field
			bodyBuilder.appendFormalLine("if (" + fieldName.getSymbolName()
					+ " == null) {");
			bodyBuilder.indent();
		}

		// build case specific parts
		String typeClassName = getClassName(type);
		String containerClassName = getClassName(returnType);
		if (isEnum) {
			// container for an enum
			buildCreateEnumContainerBody(typeParameters, bodyBuilder,
					typeClassName, containerClassName);
		} else if (jpaParameters != null) {
			// JPAContainer
			buildCreateJpaContainerBody(typeParameters, bodyBuilder,
					typeClassName, containerClassName, jpaParameters);
		} else {
			// in-memory container
			buildCreateInMemoryContainerBody(typeEntityMetadata, bodyBuilder,
					typeClassName, containerClassName);
		}

		if (fieldName != null) {
			bodyBuilder.appendFormalLine(fieldName.getSymbolName()
					+ " = container;");
			bodyBuilder.indentRemove();
			bodyBuilder.appendFormalLine("}");
			bodyBuilder.appendFormalLine("return " + fieldName.getSymbolName()
					+ ";");
		} else {
			bodyBuilder.appendFormalLine("return container;");
		}

		MethodMetadataBuilder methodBuilder = new MethodMetadataBuilder(
				getId(), Modifier.PUBLIC, populateMethodName, returnType,
				bodyBuilder);
		return methodBuilder.build();
	}

	private void buildCreateEnumContainerBody(List<JavaType> typeParameters,
			InvocableMemberBodyBuilder bodyBuilder, String typeClassName,
			String containerClassName) {
		JavaType collectionType = new JavaType("java.util.Collection", 0,
				DataType.TYPE, null, typeParameters);

		bodyBuilder.appendFormalLine(getClassName(collectionType) + " items = "
				+ getClassName("java.util.Arrays") + ".asList(" + typeClassName
				+ ".class.getEnumConstants());");
		bodyBuilder.appendFormalLine(containerClassName + " container = new "
				+ containerClassName + "(items);");
	}

	private void buildCreateInMemoryContainerBody(
			EntityMetadata typeEntityMetadata,
			InvocableMemberBodyBuilder bodyBuilder, String typeClassName,
			String containerClassName) {
		bodyBuilder.appendFormalLine(containerClassName + " container = new "
				+ containerClassName + "(" + typeClassName + ".class);");
		if (useBeanContainer) {
			String idProperty = typeEntityMetadata.getIdentifierField()
					.getFieldName().toString();
			bodyBuilder.appendFormalLine("container.setBeanIdProperty(\""
					+ idProperty + "\");");
		}
		String listMethodName = typeClassName + "."
				+ typeEntityMetadata.getFindAllMethod().getMethodName();
		bodyBuilder.appendFormalLine("for (" + typeClassName + " entity : "
				+ listMethodName + "()) {");
		bodyBuilder.indent();
		if (useBeanContainer) {
			bodyBuilder.appendFormalLine("container.addBean(entity);");
		} else {
			bodyBuilder
					.appendFormalLine("container.addItem(getIdForEntity(entity), entity);");
		}
		bodyBuilder.indentRemove();
		bodyBuilder.appendFormalLine("}");
	}

	private void buildCreateJpaContainerBody(List<JavaType> typeParameters,
			InvocableMemberBodyBuilder bodyBuilder, String typeClassName,
			String containerClassName, JpaContainerParameters jpaParameters) {
		String entityProviderUtilClassName = getClassName(utilPackage
				.getFullyQualifiedPackageName()
				+ "."
				+ VaadinOperationsImpl.ENTITY_PROVIDER_UTIL_CLASS);

		bodyBuilder.appendFormalLine(containerClassName + " container = new "
				+ containerClassName + "(" + typeClassName + ".class);");
		bodyBuilder
				.appendFormalLine("container.setEntityProvider("+entityProviderUtilClassName + ".get().getEntityProvider("
						+ typeClassName + ".class));");
	}

	private JavaSymbolName getContainerFieldName(JavaType type) {
		PluralMetadata pluralMetadata = (PluralMetadata) getMetadataService()
				.get(PluralMetadata.createIdentifier(type, Path.SRC_MAIN_JAVA));
		return new JavaSymbolName("containerFor" + pluralMetadata.getPlural());
	}

}
