package com.vaadin.spring.roo.addon.entityview;

import java.lang.reflect.Modifier;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.logging.Logger;

import org.springframework.roo.addon.entity.EntityMetadata;
import org.springframework.roo.classpath.PhysicalTypeIdentifierNamingUtils;
import org.springframework.roo.classpath.PhysicalTypeMetadata;
import org.springframework.roo.classpath.details.MethodMetadata;
import org.springframework.roo.classpath.details.MethodMetadataBuilder;
import org.springframework.roo.classpath.details.annotations.AnnotatedJavaType;
import org.springframework.roo.classpath.itd.AbstractItdTypeDetailsProvidingMetadataItem;
import org.springframework.roo.classpath.itd.InvocableMemberBodyBuilder;
import org.springframework.roo.classpath.itd.ItdSourceFileComposer;
import org.springframework.roo.classpath.scanner.MemberDetails;
import org.springframework.roo.classpath.scanner.MemberDetailsScanner;
import org.springframework.roo.metadata.MetadataIdentificationUtils;
import org.springframework.roo.metadata.MetadataService;
import org.springframework.roo.model.DataType;
import org.springframework.roo.model.JavaPackage;
import org.springframework.roo.model.JavaSymbolName;
import org.springframework.roo.model.JavaType;
import org.springframework.roo.project.Path;
import org.springframework.roo.support.style.ToStringCreator;
import org.springframework.roo.support.util.Assert;
import org.springframework.roo.support.util.StringUtils;

import com.vaadin.spring.roo.addon.VaadinRooUtils;
import com.vaadin.spring.roo.addon.abstractentityview.VaadinAbstractEntityViewMetadata;
import com.vaadin.spring.roo.addon.annotations.RooVaadinEntityView;
import com.vaadin.spring.roo.addon.entityform.VaadinClassNames;
import com.vaadin.spring.roo.addon.entityview.ContainerMethodBuilder.JpaContainerParameters;

/**
 * Metadata for {@link RooVaadinEntityView}.
 */
public class VaadinEntityViewMetadata extends
		AbstractItdTypeDetailsProvidingMetadataItem {

	// for JPAContainer
	private static final String ENTITY_ITEM_CLASS = "com.vaadin.addon.jpacontainer.EntityItem";

	private static final String TABLE_CONTAINER_FIELD_NAME = "tableContainer";

	private static final String ENTITY_TABLE_COLUMN_GENERATOR_SUFFIX = "EntityTableColumnGenerator";

	public static final String IS_CREATE_ALLOWED_METHOD = "isCreateAllowed";
	public static final String IS_UPDATE_ALLOWED_METHOD = "isUpdateAllowed";
	public static final String IS_DELETE_ALLOWED_METHOD = "isDeleteAllowed";

	public static final String GET_TABLE_CONTAINER_METHOD = "getTableContainer";
	public static final String GET_TABLE_FIELD_FACTORY_METHOD = "getTableFieldFactory";
	public static final String SETUP_GENERATED_COLUMNS_METHOD = "setupGeneratedColumns";

	public static final String GET_ENTITY_NAME_METHOD = "getEntityName";
	public static final String GET_ENTITY_CLASS_METHOD = "getEntityClass";

	public static final String GET_ITEM_FOR_ENTITY_METHOD = "getItemForEntity";
	public static final String GET_ENTITY_FOR_ITEM_METHOD = "getEntityForItem";
	public static final String GET_ID_FOR_ENTITY_METHOD = "getIdForEntity";

	public static final String CREATE_ENTITY_INSTANCE_METHOD = "createEntityInstance";

	private static final String PROVIDES_TYPE_STRING = VaadinEntityViewMetadata.class
			.getName();
	private static final String PROVIDES_TYPE = MetadataIdentificationUtils
			.create(PROVIDES_TYPE_STRING);

	private static Logger logger = Logger
			.getLogger(VaadinEntityViewMetadata.class.getName());

	private final VaadinEntityViewAnnotationValues annotationValues;
	private final VaadinEntityMetadataDetails vaadinEntityDetails;
	private final VaadinAbstractEntityViewMetadata abstractViewMetadata;
	private final MetadataService metadataService;

	// map from property name to the underlying (usually entity) type
	private Map<JavaSymbolName, JavaType> specialDomainTypes;
	private String entityDisplayName;

	private final MemberDetailsScanner memberDetailsScanner;

	public VaadinEntityViewMetadata(String identifier, JavaType aspectName,
			PhysicalTypeMetadata governorPhysicalTypeMetadata,
			MetadataService metadataService,
			MemberDetailsScanner memberDetailsScanner,
			VaadinEntityViewAnnotationValues annotationValues,
			VaadinEntityMetadataDetails vaadinEntityDetails,
			VaadinAbstractEntityViewMetadata abstractViewMetadata) {

		super(identifier, aspectName, governorPhysicalTypeMetadata);
		Assert.isTrue(isValid(identifier), "Metadata identification string '"
				+ identifier + "' does not appear to be a valid");
		Assert.notNull(annotationValues, "Annotation values required");
		Assert.notNull(metadataService, "Metadata service required");
		Assert.notNull(memberDetailsScanner, "Member details scanner required");
		Assert.notNull(vaadinEntityDetails, "Entity type details required");
		Assert.notNull(abstractViewMetadata,
				"Abstract entity view metadata required");
		
		this.annotationValues = annotationValues;
		this.vaadinEntityDetails = vaadinEntityDetails;
		this.abstractViewMetadata = abstractViewMetadata;
		this.metadataService = metadataService;
		this.memberDetailsScanner = memberDetailsScanner;

		if (!isValid()) {
			return;
		}

		// getEntityType() can be used after super() constructor call
		
		entityDisplayName = getEntityType().getSimpleTypeName()
				.replaceAll("([A-Z])", " $1").trim();
		
		specialDomainTypes = VaadinRooUtils.getSpecialDomainTypes(
				metadataService, getEntityType(), getMemberDetails(), false);

		// add various methods to the ITD
		addMethods();

		itdTypeDetails = builder.build();

		new ItdSourceFileComposer(itdTypeDetails);
	}

	protected MemberDetails getMemberDetails() {
		return VaadinRooUtils.getMemberDetails(
				vaadinEntityDetails.getJavaType(), metadataService,
				memberDetailsScanner, this.getClass().getName());
	}

	public String getIdentifierForEntityMetadata() {
		return getEntityMetadata().getId();
	}

	public VaadinEntityViewAnnotationValues getAnnotationValues() {
		return annotationValues;
	}

	@Override
	public String toString() {
		ToStringCreator tsc = new ToStringCreator(this);
		tsc.append("identifier", getId());
		tsc.append("valid", valid);
		tsc.append("aspectName", aspectName);
		tsc.append("destinationType", destination);
		tsc.append("governor", governorPhysicalTypeMetadata.getId());
		tsc.append("itdTypeDetails", itdTypeDetails);
		return tsc.toString();
	}

	public String getEntityDisplayName() {
		return entityDisplayName;
	}

	public static final String getMetadataIdentiferType() {
		return PROVIDES_TYPE;
	}

	public static final String getMetadataIdentiferString() {
		return PROVIDES_TYPE_STRING;
	}

	public static final String createIdentifier(JavaType javaType, Path path) {
		return PhysicalTypeIdentifierNamingUtils.createIdentifier(
				PROVIDES_TYPE_STRING, javaType, path);
	}

	public static final JavaType getJavaType(String metadataIdentificationString) {
		return PhysicalTypeIdentifierNamingUtils.getJavaType(
				PROVIDES_TYPE_STRING, metadataIdentificationString);
	}

	public static final Path getPath(String metadataIdentificationString) {
		return PhysicalTypeIdentifierNamingUtils.getPath(PROVIDES_TYPE_STRING,
				metadataIdentificationString);
	}

	public static boolean isValid(String metadataIdentificationString) {
		return PhysicalTypeIdentifierNamingUtils.isValid(PROVIDES_TYPE_STRING,
				metadataIdentificationString);
	}

	public boolean useJpaContainer() {
		// check superclass annotation for a @RooVaadinAbstractEntityView
		// parameter (use JPAContainer or not)
		return abstractViewMetadata.isUseJpaContainer();
	}

	private void addMethods() {
		builder.addMethod(getEntityNameMethod());
		builder.addMethod(getEntityClassMethod());

		for (MethodMetadata md : getIsAllowedMethods()) {
			builder.addMethod(md);
		}

		for (MethodMetadata md : getPersistenceMethods()) {
			builder.addMethod(md);
		}

		builder.addMethod(getCreateEntityInstanceMethod());

		// this can also add fields as a side effect
		for (MethodMetadata md : getMainContainerMethods()) {
			builder.addMethod(md);
		}

		builder.addMethod(getGeneratedTableColumnsMethod());

		for (MethodMetadata md : getDomainTypeCaptionMethods()) {
			builder.addMethod(md);
		}
	}

	private List<MethodMetadata> getDomainTypeCaptionMethods() {
		List<MethodMetadata> methods = new ArrayList<MethodMetadata>();

		DomainTypeCaptionMethodBuilder captionBuilder = new DomainTypeCaptionMethodBuilder(
				governorTypeDetails, getId(), vaadinEntityDetails,
				builder.getImportRegistrationResolver(), specialDomainTypes,
				metadataService, memberDetailsScanner);

		methods.addAll(captionBuilder.getDomainTypeCaptionMethods());

		return methods;
	}

	/**
	 * Imports a type if necessary and returns the properly qualified type name
	 * including type parameters.
	 *
	 * @param type
	 * @return
	 */
	private String getClassName(JavaType type) {
		return type.getNameIncludingTypeParameters(false,
				builder.getImportRegistrationResolver());
	}

	/**
	 * Imports a type if necessary and returns the properly qualified type name
	 * including type parameters.
	 *
	 * @param typeName
	 * @return
	 */
	private String getClassName(String typeName) {
		JavaType type = new JavaType(typeName);
		return type.getNameIncludingTypeParameters(false,
				builder.getImportRegistrationResolver());
	}

	private List<MethodMetadata> getPersistenceMethods() {
		List<MethodMetadata> methods = new ArrayList<MethodMetadata>();

		PersistenceMethodBuilder persistenceBuilder = new PersistenceMethodBuilder(
				governorTypeDetails, getId(), vaadinEntityDetails,
				builder.getImportRegistrationResolver(), metadataService,
				memberDetailsScanner);

		methods.addAll(persistenceBuilder.getBasicPersistenceMethods());

		methods.add(persistenceBuilder.getIsNewEntityMethod());

		methods.addAll(persistenceBuilder.getEntityMetadataMethods());

		return methods;
	}

	private MethodMetadata getCreateEntityInstanceMethod() {
		return createReturnLiteralMethod(CREATE_ENTITY_INSTANCE_METHOD,
				getEntityType(), "new " + getClassName(getEntityType()) + "()");
	}

	private MethodMetadata getEntityNameMethod() {
		return createReturnLiteralMethod(GET_ENTITY_NAME_METHOD,
				JavaType.STRING_OBJECT, "\"" + getEntityDisplayName() + "\"");
	}

	private MethodMetadata getEntityClassMethod() {
		String entityClassName = getClassName(getEntityType());
		JavaType returnType = new JavaType("java.lang.Class", 0, DataType.TYPE,
				null, Collections.singletonList(getEntityType()));
		return createReturnLiteralMethod(GET_ENTITY_CLASS_METHOD, returnType,
				entityClassName + ".class");
	}

	private List<MethodMetadata> getIsAllowedMethods() {
		List<MethodMetadata> methods = new ArrayList<MethodMetadata>();

		methods.add(createReturnLiteralMethod(IS_CREATE_ALLOWED_METHOD,
				JavaType.BOOLEAN_PRIMITIVE, ""
						+ getAnnotationValues().isCreate()));
		methods.add(createReturnLiteralMethod(IS_UPDATE_ALLOWED_METHOD,
				JavaType.BOOLEAN_PRIMITIVE, ""
						+ getAnnotationValues().isUpdate()));
		methods.add(createReturnLiteralMethod(IS_DELETE_ALLOWED_METHOD,
				JavaType.BOOLEAN_PRIMITIVE, ""
						+ getAnnotationValues().isDelete()));

		return methods;
	}

	private MethodMetadata getGeneratedTableColumnsMethod() {
		JavaSymbolName methodName = new JavaSymbolName(
				SETUP_GENERATED_COLUMNS_METHOD);

		MethodMetadata method = VaadinRooUtils.methodExists(methodName,
				governorTypeDetails);
		if (method != null) {
			return method;
		}

		JavaType parentType = VaadinAbstractEntityViewMetadata
				.getJavaType(abstractViewMetadata.getId());
		JavaPackage abstractViewPackage = parentType.getPackage();

		boolean generatedColumns = false;

		List<JavaType> parameterTypes = Collections.singletonList(new JavaType(
				VaadinClassNames.TABLE_CLASS));
		List<JavaSymbolName> parameterNames = Collections
				.singletonList(new JavaSymbolName("table"));

		JavaType columnGeneratorClass = AbstractVaadinEntityViewMethodBuilder
				.getJavaTypeInWebPackage(abstractViewPackage,
						ENTITY_TABLE_COLUMN_GENERATOR_SUFFIX, null);
		String columnGeneratorClassName = getClassName(columnGeneratorClass);

		InvocableMemberBodyBuilder bodyBuilder = new InvocableMemberBodyBuilder();
		
		Map<JavaSymbolName, MethodMetadata> accessors = VaadinRooUtils
				.getAccessors(getEntityType(), getEntityMetadata(),
						getMemberDetails(), false);
		
		// iterate over properties, get their types
		for (JavaSymbolName propertyName : accessors.keySet()) {
			JavaType propertyType = specialDomainTypes.get(propertyName);
			// check that an entity type
			if (propertyType != null
					&& metadataService.get(EntityMetadata.createIdentifier(
							propertyType, Path.SRC_MAIN_JAVA)) != null) {
				String propertyString = StringUtils.uncapitalize(propertyName
						.getSymbolName());
				bodyBuilder.appendFormalLine("table.removeGeneratedColumn(\""
						+ propertyString + "\");");
				bodyBuilder.appendFormalLine("table.addGeneratedColumn(\""
						+ propertyString + "\", new "
						+ columnGeneratorClassName + "((String) "
						+ AbstractVaadinEntityViewMethodBuilder
								.getItemCaptionIdMethodName(propertyType)
						+ "()));");
				generatedColumns = true;
			}
		}

		if (!generatedColumns) {
			// have to provide something in the body
			bodyBuilder.appendFormalLine("// Add generated columns here");
		}

		MethodMetadataBuilder methodBuilder = new MethodMetadataBuilder(
				getId(), Modifier.PUBLIC, methodName, JavaType.VOID_PRIMITIVE,
				AnnotatedJavaType.convertFromJavaTypes(parameterTypes),
				parameterNames, bodyBuilder);
		return methodBuilder.build();
	}

	/**
	 * Returns methods for creating the main container and any related
	 * commit/delete methods.
	 *
	 * @return
	 */
	private Collection<MethodMetadata> getMainContainerMethods() {
		// use JPAContainer for table
		boolean useJpaContainer = useJpaContainer();

		List<MethodMetadata> methods = new ArrayList<MethodMetadata>();
		methods.add(getTableContainerMethod(useJpaContainer));
		methods.add(getItemForEntityMethod(useJpaContainer));
		methods.add(getEntityForItemMethod(useJpaContainer));
		methods.add(getIdForEntityMethod());
		return methods;
	}

	/**
	 * Creates the method constructing and returning the main table container.
	 *
	 * This method may create a field as a side effect.
	 *
	 * @return
	 */
	private MethodMetadata getTableContainerMethod(boolean useJpaContainer) {
		JavaSymbolName methodName = new JavaSymbolName(
				GET_TABLE_CONTAINER_METHOD);
		MethodMetadata method = VaadinRooUtils.methodExists(methodName,
				governorTypeDetails);
		if (method != null) {
			return method;
		}

		JpaContainerParameters jpaParameters = null;
		JavaSymbolName fieldName = null;
		if (useJpaContainer) {
			jpaParameters = new JpaContainerParameters(true);
			fieldName = new JavaSymbolName(TABLE_CONTAINER_FIELD_NAME);
		}

		JavaType parentType = VaadinAbstractEntityViewMetadata
				.getJavaType(abstractViewMetadata.getId());
		JavaPackage abstractViewPackage = parentType.getPackage();

		ContainerMethodBuilder containerBuilder = new ContainerMethodBuilder(
				governorTypeDetails, getId(), vaadinEntityDetails,
				abstractViewPackage, builder.getImportRegistrationResolver(),
				metadataService, memberDetailsScanner, builder);
		return containerBuilder.buildGetContainerMethod(getEntityType(), methodName,
				fieldName, jpaParameters);
	}

	private MethodMetadata getItemForEntityMethod(boolean useJpaContainer) {
		JavaSymbolName methodName = new JavaSymbolName(
				GET_ITEM_FOR_ENTITY_METHOD);

		MethodMetadata method = VaadinRooUtils.methodExists(methodName,
				governorTypeDetails);
		if (method != null) {
			return method;
		}

		JavaType returnType = new JavaType(VaadinClassNames.ITEM_CLASS);

		InvocableMemberBodyBuilder bodyBuilder = new InvocableMemberBodyBuilder();

		bodyBuilder.appendFormalLine(getClassName(returnType)
				+ " item = getTable().getItem(entity."
				+ getEntityMetadata().getIdentifierAccessor().getMethodName()
				+ "());");
		bodyBuilder.appendFormalLine("if (item == null) {");
		bodyBuilder.indent();
		if (useJpaContainer) {
			bodyBuilder
					.appendFormalLine("item = tableContainer.createEntityItem(entity);");
		} else {
			bodyBuilder.appendFormalLine("item = new "
					+ getClassName(VaadinClassNames.BEAN_ITEM_CLASS) + "<"
					+ getClassName(getEntityType())
					+ ">(entity);");
		}
		bodyBuilder.indentRemove();
		bodyBuilder.appendFormalLine("}");
		bodyBuilder.appendFormalLine("return item;");

		List<JavaType> parameterTypes = Collections
				.singletonList(getEntityType());
		List<JavaSymbolName> parameterNames = Collections
				.singletonList(new JavaSymbolName("entity"));

		MethodMetadataBuilder methodBuilder = new MethodMetadataBuilder(
				getId(), Modifier.PUBLIC, methodName, returnType,
				AnnotatedJavaType.convertFromJavaTypes(parameterTypes),
				parameterNames, bodyBuilder);
		return methodBuilder.build();
	}

	private MethodMetadata getEntityForItemMethod(boolean useJpaContainer) {
		JavaSymbolName methodName = new JavaSymbolName(
				GET_ENTITY_FOR_ITEM_METHOD);

		MethodMetadata method = VaadinRooUtils.methodExists(methodName,
				governorTypeDetails);
		if (method != null) {
			return method;
		}

		JavaType returnType = getEntityType();
		String entityClassName = getClassName(getEntityType());

		InvocableMemberBodyBuilder bodyBuilder = new InvocableMemberBodyBuilder();

		if (useJpaContainer) {
			String entityItemClassName = getClassName(ENTITY_ITEM_CLASS);
			bodyBuilder.appendFormalLine("if (item instanceof "
					+ entityItemClassName + " && ((" + entityItemClassName
					+ ") item).getEntity() instanceof " + entityClassName
					+ ") {");
			bodyBuilder.indent();
			bodyBuilder.appendFormalLine("return (" + entityClassName + ") (("
					+ entityItemClassName + ") item).getEntity();");
			bodyBuilder.indentRemove();
			bodyBuilder.appendFormalLine("} else {");
			bodyBuilder.indent();
			bodyBuilder.appendFormalLine("return null;");
			bodyBuilder.indentRemove();
			bodyBuilder.appendFormalLine("}");
		} else {
			bodyBuilder.appendFormalLine("if (item != null) {");
			bodyBuilder.indent();
			bodyBuilder.appendFormalLine("return (("
					+ getClassName(VaadinClassNames.BEAN_ITEM_CLASS) + "<"
					+ entityClassName + ">) item).getBean();");
			bodyBuilder.indentRemove();
			bodyBuilder.appendFormalLine("} else {");
			bodyBuilder.indent();
			bodyBuilder.appendFormalLine("return null;");
			bodyBuilder.indentRemove();
			bodyBuilder.appendFormalLine("}");
		}

		List<JavaType> parameterTypes = Collections.singletonList(new JavaType(
				VaadinClassNames.ITEM_CLASS));
		List<JavaSymbolName> parameterNames = Collections
				.singletonList(new JavaSymbolName("item"));

		MethodMetadataBuilder methodBuilder = new MethodMetadataBuilder(
				getId(), Modifier.PUBLIC, methodName, returnType,
				AnnotatedJavaType.convertFromJavaTypes(parameterTypes),
				parameterNames, bodyBuilder);
		return methodBuilder.build();
	}

	private MethodMetadata getIdForEntityMethod() {
		JavaSymbolName methodName = new JavaSymbolName(GET_ID_FOR_ENTITY_METHOD);

		MethodMetadata method = VaadinRooUtils.methodExists(methodName,
				governorTypeDetails);
		if (method != null) {
			return method;
		}

		JavaType returnType = new JavaType("Object");

		InvocableMemberBodyBuilder bodyBuilder = new InvocableMemberBodyBuilder();

		bodyBuilder.appendFormalLine("return entity != null ? entity."
				+ getEntityMetadata().getIdentifierAccessor().getMethodName()
				+ "() : null;");

		List<JavaType> parameterTypes = Collections
				.singletonList(getEntityType());
		List<JavaSymbolName> parameterNames = Collections
				.singletonList(new JavaSymbolName("entity"));
		MethodMetadataBuilder methodBuilder = new MethodMetadataBuilder(
				getId(), Modifier.PUBLIC, methodName, returnType,
				AnnotatedJavaType.convertFromJavaTypes(parameterTypes),
				parameterNames, bodyBuilder);
		return methodBuilder.build();
	}

	private MethodMetadata createReturnLiteralMethod(String methodNameString,
			JavaType returnType, String literalValue) {
		return AbstractVaadinEntityViewMethodBuilder.createReturnLiteralMethod(
				governorTypeDetails, getId(), methodNameString, returnType,
				literalValue);
	}

	private EntityMetadata getEntityMetadata() {
		return vaadinEntityDetails.getEntityMetadata();
	}

	private JavaType getEntityType() {
		return vaadinEntityDetails.getJavaType();
	}

}
