package com.vaadin.spring.roo.addon.entityview;

import java.util.ArrayList;
import java.util.Collection;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.logging.Logger;

import org.apache.felix.scr.annotations.Component;
import org.apache.felix.scr.annotations.Service;
import org.osgi.service.component.ComponentContext;
import org.springframework.roo.addon.entity.EntityMetadata;
import org.springframework.roo.addon.javabean.JavaBeanMetadata;
import org.springframework.roo.classpath.PhysicalTypeIdentifier;
import org.springframework.roo.classpath.PhysicalTypeMetadata;
import org.springframework.roo.classpath.details.ClassOrInterfaceTypeDetails;
import org.springframework.roo.classpath.details.MemberFindingUtils;
import org.springframework.roo.classpath.itd.AbstractItdMetadataProvider;
import org.springframework.roo.classpath.itd.ItdTypeDetailsProvidingMetadataItem;
import org.springframework.roo.classpath.scanner.MemberDetails;
import org.springframework.roo.model.JavaType;
import org.springframework.roo.project.Path;

import com.vaadin.spring.roo.addon.VaadinRooUtils;
import com.vaadin.spring.roo.addon.abstractentityview.VaadinAbstractEntityViewMetadata;
import com.vaadin.spring.roo.addon.annotations.RooVaadinAbstractEntityView;
import com.vaadin.spring.roo.addon.annotations.RooVaadinEntityView;


/**
 * Provides {@link VaadinEntityViewMetadata}.
 */
@Component(immediate = true)
@Service
public class VaadinEntityViewMetadataProvider extends
		AbstractItdMetadataProvider {

	private static Logger logger = Logger
			.getLogger(VaadinEntityViewMetadataProvider.class.getName());

	protected void activate(ComponentContext context) {
		metadataDependencyRegistry.registerDependency(
				PhysicalTypeIdentifier.getMetadataIdentiferType(),
				getProvidesType());
		addMetadataTrigger(new JavaType(RooVaadinEntityView.class.getName()));
	}

	@Override
	protected ItdTypeDetailsProvidingMetadataItem getMetadata(
			String metadataIdentificationString, JavaType aspectName,
			PhysicalTypeMetadata governorPhysicalTypeMetadata,
			String itdFilename) {
		// We know governor type details are non-null and can be safely cast

		// We need to parse the annotation, which we expect to be present
		VaadinEntityViewAnnotationValues annotationValues = new VaadinEntityViewAnnotationValues(
				governorPhysicalTypeMetadata);
		if (!annotationValues.isAnnotationFound()
				|| annotationValues.formBackingObject == null) {
			return null;
		}

		// Lookup the form backing object's metadata
		JavaType javaType = annotationValues.formBackingObject;
		Path path = Path.SRC_MAIN_JAVA;

		String physicalTypeMetadataKey = PhysicalTypeIdentifier.createIdentifier(
				javaType, path);
		String entityMetadataKey = EntityMetadata.createIdentifier(javaType,
				path);

		// We need to lookup the metadata we depend on
		EntityMetadata entityMetadata = (EntityMetadata) metadataService
				.get(entityMetadataKey);

		// We need to abort if we couldn't find dependent metadata
		if (entityMetadata == null || !entityMetadata.isValid()) {
			return null;
		}

		// We need to be informed if our dependent metadata changes
		metadataDependencyRegistry.registerDependency(physicalTypeMetadataKey,
				metadataIdentificationString);
		metadataDependencyRegistry.registerDependency(entityMetadataKey,
				metadataIdentificationString);
		metadataDependencyRegistry.registerDependency(
				JavaBeanMetadata.createIdentifier(javaType, path),
				metadataIdentificationString);

		// We also need to be informed if a referenced type is changed
		MemberDetails memberDetails = VaadinRooUtils.getMemberDetails(javaType,
				metadataService, memberDetailsScanner, this.getClass()
						.getName());
		Collection<JavaType> specialDomainTypes = VaadinRooUtils
				.getSpecialDomainTypes(metadataService, javaType, memberDetails, true).values();
		// unique types only
		for (JavaType type : new LinkedHashSet<JavaType>(specialDomainTypes)) {
			metadataDependencyRegistry.registerDependency(
					PhysicalTypeIdentifier.createIdentifier(type, path),
					metadataIdentificationString);
			metadataDependencyRegistry.registerDependency(
					JavaBeanMetadata.createIdentifier(type, path),
					metadataIdentificationString);
		}

		// find superclasses up to one that has the annotation
		// @RooVaadinAbstractEntityView

		// this is a list of physical type metadata IDs for the superclasses of
		// the governor type and ending with that of the class with the
		// @RooVaadinAbstractEntityView annotation, or null if no such
		// superclass is found
		List<String> superclassPhysicalTypeMetadataIds = getSuperclassPhysicalTypeMetadataIds(metadataIdentificationString);
		if (superclassPhysicalTypeMetadataIds == null) {
			return null;
		}

		// register metadata dependencies on all the superclasses to monitor
		// changes to them
		for (String mid : superclassPhysicalTypeMetadataIds) {
			metadataDependencyRegistry.registerDependency(mid,
					metadataIdentificationString);
		}

		// get VaadinAbstractEntityViewMetadata for the "last" superclass
		String abstractViewPhysicalTypeId = superclassPhysicalTypeMetadataIds
				.get(superclassPhysicalTypeMetadataIds.size() - 1);
		String abstractViewId = VaadinAbstractEntityViewMetadata
				.createIdentifier(PhysicalTypeIdentifier
						.getJavaType(abstractViewPhysicalTypeId),
						PhysicalTypeIdentifier
								.getPath(abstractViewPhysicalTypeId));
		VaadinAbstractEntityViewMetadata abstractViewMetadata = (VaadinAbstractEntityViewMetadata) metadataService
				.get(abstractViewId);
		if (abstractViewMetadata == null) {
			return null;
		}

		// We do not need to monitor the parent, as any changes to the java type
		// associated with the parent will trickle down to
		// the governing java type
		return new VaadinEntityViewMetadata(metadataIdentificationString,
				aspectName, governorPhysicalTypeMetadata, metadataService,
				memberDetailsScanner, annotationValues,
				new VaadinEntityMetadataDetails(javaType, entityMetadata),
				abstractViewMetadata);
	}

	private List<String> getSuperclassPhysicalTypeMetadataIds(
			String metadataIdentificationString) {
		List<String> superclassPhysicalTypeMetadataIds = new ArrayList<String>();

		PhysicalTypeMetadata ptm = (PhysicalTypeMetadata) metadataService
				.get(getGovernorPhysicalTypeIdentifier(metadataIdentificationString));
		if (ptm == null
				|| !(ptm.getMemberHoldingTypeDetails() instanceof ClassOrInterfaceTypeDetails)) {
			return null;
		}

		JavaType abstractEntityViewAnnotationType = new JavaType(
				RooVaadinAbstractEntityView.class.getName());

		ClassOrInterfaceTypeDetails cid = (ClassOrInterfaceTypeDetails) ptm
				.getMemberHoldingTypeDetails();
		ClassOrInterfaceTypeDetails superclass = cid;
		while (true) {
			superclass = superclass.getSuperclass();
			if (superclass == null) {
				return null;
			}
			String superclassPhysicalMetadataId = superclass
					.getDeclaredByMetadataId();
			superclassPhysicalTypeMetadataIds.add(superclassPhysicalMetadataId);
			// break if superclass has @RooVaadinAbstractEntityView
			if (MemberFindingUtils.getDeclaredTypeAnnotation(superclass,
					abstractEntityViewAnnotationType) != null) {
				break;
			}
		}
		return superclassPhysicalTypeMetadataIds;
	}

	public String getItdUniquenessFilenameSuffix() {
		return "VaadinEntityView";
	}

	@Override
	protected String getGovernorPhysicalTypeIdentifier(
			String metadataIdentificationString) {
		JavaType javaType = VaadinEntityViewMetadata
				.getJavaType(metadataIdentificationString);
		Path path = VaadinEntityViewMetadata
				.getPath(metadataIdentificationString);
		String physicalTypeIdentifier = PhysicalTypeIdentifier
				.createIdentifier(javaType, path);
		return physicalTypeIdentifier;
	}

	@Override
	protected String createLocalIdentifier(JavaType javaType, Path path) {
		return VaadinEntityViewMetadata.createIdentifier(javaType, path);
	}

	public String getProvidesType() {
		return VaadinEntityViewMetadata.getMetadataIdentiferType();
	}

}
