/*
 * Decompiled with CFR 0.152.
 */
package com.vaadin.kubernetes.starter.sessiontracker;

import com.vaadin.flow.component.UI;
import com.vaadin.flow.server.VaadinSession;
import com.vaadin.flow.server.WrappedHttpSession;
import com.vaadin.flow.server.WrappedSession;
import com.vaadin.kubernetes.starter.ProductUtils;
import com.vaadin.kubernetes.starter.sessiontracker.backend.BackendConnector;
import com.vaadin.kubernetes.starter.sessiontracker.backend.SessionInfo;
import com.vaadin.kubernetes.starter.sessiontracker.serialization.TransientHandler;
import com.vaadin.kubernetes.starter.sessiontracker.serialization.TransientInjectableObjectInputStream;
import com.vaadin.kubernetes.starter.sessiontracker.serialization.TransientInjectableObjectOutputStream;
import jakarta.servlet.http.HttpSession;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.NotSerializableException;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.locks.ReentrantLock;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.context.ApplicationListener;
import org.springframework.context.event.ContextClosedEvent;

public class SessionSerializer
implements ApplicationListener<ContextClosedEvent> {
    private static final long OPTIMISTIC_SERIALIZATION_TIMEOUT_MS = 30000L;
    private final ExecutorService executorService = Executors.newFixedThreadPool(4, new SerializationThreadFactory());
    private final ConcurrentHashMap<String, Boolean> pending = new ConcurrentHashMap();
    private final BackendConnector backendConnector;
    private final TransientHandler handler;
    private final long optimisticSerializationTimeoutMs;
    private Predicate<Class<?>> injectableFilter = type -> true;

    public SessionSerializer(BackendConnector backendConnector, TransientHandler transientHandler) {
        this.backendConnector = backendConnector;
        this.handler = transientHandler;
        this.optimisticSerializationTimeoutMs = 30000L;
    }

    SessionSerializer(BackendConnector backendConnector, TransientHandler transientHandler, long optimisticSerializationTimeoutMs) {
        this.backendConnector = backendConnector;
        this.optimisticSerializationTimeoutMs = optimisticSerializationTimeoutMs;
        this.handler = transientHandler;
    }

    public void setInjectableFilter(Predicate<Class<?>> injectableFilter) {
        this.injectableFilter = injectableFilter;
    }

    public void serialize(HttpSession session) {
        this.serialize((WrappedSession)new WrappedHttpSession(session));
    }

    public void serialize(WrappedSession session) {
        Map<String, Object> values = session.getAttributeNames().stream().collect(Collectors.toMap(Function.identity(), arg_0 -> ((WrappedSession)session).getAttribute(arg_0)));
        this.queueSerialization(session.getId(), values);
    }

    public void deserialize(SessionInfo sessionInfo, HttpSession session) throws ClassNotFoundException, IOException {
        Map<String, Object> values = this.doDeserialize(sessionInfo, session.getId());
        for (Map.Entry<String, Object> entry : values.entrySet()) {
            session.setAttribute(entry.getKey(), entry.getValue());
        }
    }

    private void queueSerialization(String sessionId, Map<String, Object> attributes) {
        if (this.pending.containsKey(sessionId)) {
            SessionSerializer.getLogger().debug("Ignoring serialization request for session {} as the session is already being serialized", (Object)sessionId);
            return;
        }
        String clusterKey = this.getClusterKey(attributes);
        SessionSerializer.getLogger().debug("Starting asynchronous serialization of session {} with distributed key {}", (Object)sessionId, (Object)clusterKey);
        this.backendConnector.markSerializationStarted(clusterKey);
        this.pending.put(sessionId, true);
        this.executorService.submit(() -> {
            Consumer<SessionInfo> whenSerialized = sessionInfo -> {
                this.backendConnector.sendSession((SessionInfo)sessionInfo);
                this.backendConnector.markSerializationComplete(clusterKey);
            };
            this.handleSessionSerialization(sessionId, attributes, whenSerialized);
        });
    }

    private void handleSessionSerialization(String sessionId, Map<String, Object> attributes, Consumer<SessionInfo> whenSerialized) {
        long start = System.currentTimeMillis();
        long timeout = start + this.optimisticSerializationTimeoutMs;
        String clusterKey = this.getClusterKey(attributes);
        boolean unrecoverableError = false;
        try {
            SessionSerializer.getLogger().debug("Optimistic serialization of session {} with distributed key {} started", (Object)sessionId, (Object)clusterKey);
            while (System.currentTimeMillis() < timeout) {
                SessionInfo info = this.serializeOptimisticLocking(sessionId, attributes);
                if (info == null) continue;
                this.pending.remove(sessionId);
                SessionSerializer.getLogger().debug("Optimistic serialization of session {} with distributed key {} completed", (Object)sessionId, (Object)clusterKey);
                whenSerialized.accept(info);
                return;
            }
        }
        catch (NotSerializableException e) {
            SessionSerializer.getLogger().error("Optimistic serialization of session {} with distributed key {} failed, some attribute is not serializable. Giving up immediately since the error is not recoverable", new Object[]{sessionId, clusterKey, e});
            unrecoverableError = true;
        }
        catch (IOException e) {
            SessionSerializer.getLogger().warn("Optimistic serialization of session {} with distributed key {} failed", new Object[]{sessionId, clusterKey, e});
        }
        this.pending.remove(sessionId);
        SessionInfo sessionInfo = null;
        if (!unrecoverableError) {
            sessionInfo = this.serializePessimisticLocking(sessionId, attributes);
        }
        whenSerialized.accept(sessionInfo);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    private SessionInfo serializePessimisticLocking(String sessionId, Map<String, Object> attributes) {
        SessionInfo sessionInfo;
        long start = System.currentTimeMillis();
        String clusterKey = this.getClusterKey(attributes);
        Set<ReentrantLock> locks = this.getLocks(attributes);
        for (ReentrantLock reentrantLock : locks) {
            reentrantLock.lock();
        }
        try {
            sessionInfo = this.doSerialize(sessionId, attributes);
        }
        catch (Exception e) {
            try {
                SessionSerializer.getLogger().error("An error occurred during pessimistic serialization of session {} with distributed key {} ", new Object[]{sessionId, clusterKey, e});
            }
            catch (Throwable throwable) {
                for (ReentrantLock lock : locks) {
                    lock.unlock();
                }
                SessionSerializer.getLogger().debug("Pessimistic serialization of session {} with distributed key {} completed in {}ms", new Object[]{sessionId, clusterKey, System.currentTimeMillis() - start});
                throw throwable;
            }
            for (ReentrantLock reentrantLock : locks) {
                reentrantLock.unlock();
            }
            SessionSerializer.getLogger().debug("Pessimistic serialization of session {} with distributed key {} completed in {}ms", new Object[]{sessionId, clusterKey, System.currentTimeMillis() - start});
            return null;
        }
        for (ReentrantLock lock : locks) {
            lock.unlock();
        }
        SessionSerializer.getLogger().debug("Pessimistic serialization of session {} with distributed key {} completed in {}ms", new Object[]{sessionId, clusterKey, System.currentTimeMillis() - start});
        return sessionInfo;
    }

    private Set<ReentrantLock> getLocks(Map<String, Object> attributes) {
        HashSet<ReentrantLock> locks = new HashSet<ReentrantLock>();
        for (String key : attributes.keySet()) {
            String serviceName;
            String lockKey;
            Object lockAttribute;
            if (!key.startsWith("com.vaadin.flow.server.VaadinSession") || !((lockAttribute = attributes.get(lockKey = (serviceName = key.substring("com.vaadin.flow.server.VaadinSession".length() + 1)) + ".lock")) instanceof ReentrantLock)) continue;
            ReentrantLock lock = (ReentrantLock)lockAttribute;
            locks.add(lock);
        }
        return locks;
    }

    private SessionInfo serializeOptimisticLocking(String sessionId, Map<String, Object> attributes) throws IOException {
        String clusterKey = this.getClusterKey(attributes);
        try {
            long latestLockTime = this.findNewestLockTime(attributes);
            long latestUnlockTime = this.findNewestUnlockTime(attributes);
            if (latestLockTime > latestUnlockTime) {
                SessionSerializer.getLogger().trace("Optimistic serialization of session {} with distributed key {} failed, session is locked. Will retry", (Object)sessionId, (Object)clusterKey);
                return null;
            }
            SessionInfo info = this.doSerialize(sessionId, attributes);
            long latestUnlockTimeCheck = this.findNewestUnlockTime(attributes);
            if (latestUnlockTime != latestUnlockTimeCheck) {
                SessionSerializer.getLogger().trace("Optimistic serialization of session {} with distributed key {} failed, somebody modified the session during serialization ({} != {}). Will retry", new Object[]{sessionId, clusterKey, latestUnlockTime, latestUnlockTimeCheck});
                return null;
            }
            this.logSessionDebugInfo("Serialized session " + sessionId + " with distributed key " + clusterKey, attributes);
            return info;
        }
        catch (NotSerializableException e) {
            throw e;
        }
        catch (Exception e) {
            SessionSerializer.getLogger().trace("Optimistic serialization of session {} with distributed key {} failed, a problem occurred during serialization. Will retry", new Object[]{sessionId, clusterKey, e});
            return null;
        }
    }

    private void logSessionDebugInfo(String prefix, Map<String, Object> attributes) {
        StringBuilder info = new StringBuilder();
        for (String key : attributes.keySet()) {
            Object value = attributes.get(key);
            if (!(value instanceof VaadinSession)) continue;
            VaadinSession s = (VaadinSession)value;
            try {
                for (UI ui : s.getUIs()) {
                    info.append("[UI ").append(ui.getUIId()).append(", last client message: ").append(ui.getInternals().getLastProcessedClientToServerId()).append(", server sync id: ").append(ui.getInternals().getServerSyncId()).append("]");
                }
            }
            catch (Exception ex) {
                info.append("[ VaadinSession not accessible without locking ]");
            }
        }
        SessionSerializer.getLogger().trace("{} UIs: {}", (Object)prefix, (Object)info);
    }

    private long findNewestLockTime(Map<String, Object> attributes) {
        long latestLock = 0L;
        for (Map.Entry<String, Object> entry : attributes.entrySet()) {
            if (!(entry.getValue() instanceof VaadinSession)) continue;
            VaadinSession session = (VaadinSession)entry.getValue();
            latestLock = Math.max(latestLock, session.getLastLocked());
        }
        return latestLock;
    }

    private long findNewestUnlockTime(Map<String, Object> attributes) {
        long latestUnlock = 0L;
        for (Map.Entry<String, Object> entry : attributes.entrySet()) {
            if (!(entry.getValue() instanceof VaadinSession)) continue;
            VaadinSession session = (VaadinSession)entry.getValue();
            latestUnlock = Math.max(latestUnlock, session.getLastUnlocked());
        }
        return latestUnlock;
    }

    private SessionInfo doSerialize(String sessionId, Map<String, Object> attributes) throws Exception {
        long start = System.currentTimeMillis();
        ByteArrayOutputStream out = new ByteArrayOutputStream();
        try (TransientInjectableObjectOutputStream outStream = TransientInjectableObjectOutputStream.newInstance(out, this.handler, this.injectableFilter);){
            outStream.writeWithTransients(attributes);
        }
        SessionInfo info = new SessionInfo(this.getClusterKey(attributes), out.toByteArray());
        SessionSerializer.getLogger().debug("Serialization of attributes {} for session {} with distributed key {} completed in {}ms", new Object[]{attributes.keySet(), sessionId, info.getClusterKey(), System.currentTimeMillis() - start});
        return info;
    }

    private String getClusterKey(Map<String, Object> attributes) {
        return (String)attributes.get("clusterKey");
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private Map<String, Object> doDeserialize(SessionInfo sessionInfo, String sessionId) throws IOException, ClassNotFoundException {
        Map attributes;
        byte[] data = sessionInfo.getData();
        long start = System.currentTimeMillis();
        ClassLoader contextLoader = Thread.currentThread().getContextClassLoader();
        ByteArrayInputStream in = new ByteArrayInputStream(data);
        try (TransientInjectableObjectInputStream inStream = new TransientInjectableObjectInputStream(in, this.handler);){
            attributes = (Map)inStream.readWithTransients();
        }
        finally {
            Thread.currentThread().setContextClassLoader(contextLoader);
        }
        this.logSessionDebugInfo("Deserialized session", attributes);
        SessionSerializer.getLogger().debug("Deserialization of attributes {} for session {} with distributed key {} completed in {}ms", new Object[]{attributes.keySet(), sessionId, sessionInfo.getClusterKey(), System.currentTimeMillis() - start});
        return attributes;
    }

    private static Logger getLogger() {
        return LoggerFactory.getLogger(SessionSerializer.class);
    }

    void waitForSerialization() {
        while (!this.pending.isEmpty()) {
            SessionSerializer.getLogger().info("Waiting for {} sessions to be serialized: {}", (Object)this.pending.size(), (Object)this.pending.keySet());
            try {
                Thread.sleep(100L);
            }
            catch (InterruptedException interruptedException) {}
        }
    }

    public void onApplicationEvent(ContextClosedEvent event) {
        this.waitForSerialization();
    }

    static {
        ProductUtils.markAsUsed(SessionSerializer.class.getSimpleName());
    }

    private static class SerializationThreadFactory
    implements ThreadFactory {
        private final AtomicInteger threadNumber = new AtomicInteger(1);

        private SerializationThreadFactory() {
        }

        @Override
        public Thread newThread(Runnable runnable) {
            return new Thread(runnable, "sessionSerializer-worker-" + this.threadNumber.getAndIncrement());
        }
    }
}

