// Copyright 2017, Yahoo Holdings Inc.
// Licensed under the terms of the Apache License 2.0. Please see LICENSE file in project root for terms.
package com.yahoo.maha.parrequest2.future;

import com.yahoo.maha.parrequest2.EitherUtils;
import com.yahoo.maha.parrequest2.GeneralError;
import com.yahoo.maha.parrequest2.Nothing;
import com.yahoo.maha.parrequest2.ParCallable;

import org.apache.commons.lang3.exception.ExceptionUtils;
import org.testng.ITestContext;
import org.testng.ITestNGMethod;
import org.testng.annotations.AfterClass;
import org.testng.annotations.BeforeClass;
import org.testng.annotations.BeforeSuite;
import org.testng.annotations.Test;
import scala.Option;
import scala.util.Either;
import scala.util.Right;
import com.yahoo.maha.parrequest2.RetryAnalyzerImpl;

import java.util.List;
import java.util.stream.Collectors;

import static org.testng.Assert.assertTrue;

/**
 * Created by jians on 4/1/15.
 */
public class TestParRequestListOption {

    @BeforeSuite(alwaysRun = true)
    public void beforeSuite(ITestContext context) {
        for (ITestNGMethod method : context.getAllTestMethods()) {
            method.setRetryAnalyzer(new RetryAnalyzerImpl());
        }
    }

    private String printError(Either<GeneralError, Integer> result) {
        return result.left().get().message +
               "\nThrowable stacktrace:\n" +
               EitherUtils.map(ParFunction.from(ExceptionUtils::getStackTrace), result.left().get().throwableOption);
    }

    private ParallelServiceExecutor executor;

    private ParFunction<List<Option<String>>, Integer> stringAssert =
            ParFunction.from((input) -> {
                for (Option<String> stringOption : input) {
                    assertTrue(stringOption.get().equals("100"));
                }
                return 200;
            });

    private ParFunction<List<Option<String>>, Integer> stringAssertOnlyFirstRequestEmpty =
            ParFunction.from((input) -> {
                for (int i = 0; i < 8; i++) {
                    assertTrue(
                            (i == 0 && input.get(i).isEmpty()) ||
                                    (i > 0 && input.get(i).get().equals("100")),
                            "first request should fail and the rest should succeed"
                    );
                }
                return 200;
            });

    @BeforeClass
    public void setUp() throws Exception {
        executor = new ParallelServiceExecutor();
        executor.setDefaultTimeoutMillis(20000);
        executor.setPoolName("test-par-request-list");
        executor.setQueueSize(20);
        executor.setThreadPoolSize(10);
        executor.init();
    }

    @AfterClass
    public void tearDown() throws Exception {
        executor.destroy();
    }

    @Test
    public void testParRequestListOptionResultMapSuccess() {
        ParRequestListOption.Builder<String> builder = executor.parRequestListOptionBuilder();
        for (int i = 0; i < 8; i++) {
            builder.addParCallable(ParRequestTestCommon.stringSleep1000);
        }
        ParRequestListOption<String> request = builder.build();
        Either<GeneralError, Integer> result = request.resultMap(stringAssert);
        assertTrue(result.isRight());
        assertTrue(result.right().get() == 200);
    }

    @Test(expectedExceptions = IllegalArgumentException.class)
    public void testParRequestListOptionResultMapFailureAllMustSucceed() {
        ParRequestListOption.Builder<String> builder = executor.parRequestListOptionBuilder();
        for (int i = 0; i < 8; i++) {
            builder.addParCallable(ParRequestTestCommon.stringSleep1000);
        }
        builder.allMustSucceed(true);

        ParRequestListOption<String> request = builder.build();
        Either<GeneralError, Integer> result = request.resultMap(ParFunction.from((input) -> {
            for (Option<String> stringOption : input) {
                assertTrue(stringOption.get().equals("100"));
            }
            throw new IllegalArgumentException("failed");
        }));
    }

    @Test(expectedExceptions = IllegalArgumentException.class)
    public void testParRequestListOptionResultMapFailure() {
        ParRequestListOption.Builder<Long> builder = executor.parRequestListOptionBuilder();
        for (int i = 0; i < 8; i++) {
            builder.addParCallable(ParRequestTestCommon.longSleep500);
        }

        ParRequestListOption<Long> request = builder.build();
        Either<GeneralError, Integer> result = request.resultMap(ParFunction.from((input) -> {
            for (Option<Long> longOption : input) {
                assertTrue(longOption.get() == 100);
            }
            throw new IllegalArgumentException("failed");
        }));
    }

    @Test
    public void testParRequestListOptionResultMapFailureInFirstParCallableAllMustSucceed() {
        ParRequestListOption.Builder<String> builder = executor.parRequestListOptionBuilder();
        builder.addParCallable(ParRequestTestCommon.stringSleep1000Failure);
        for (int i = 1; i < 8; i++) {
            builder.addParCallable(ParRequestTestCommon.stringSleep1000);
        }
        builder.allMustSucceed(true);

        ParRequestListOption<String> request = builder.build();
        Either<GeneralError, Integer> result = request.resultMap(stringAssert);
        assertTrue(result.isLeft());
        assertTrue(result.left().get().message.equals("failed"), printError(result));
    }

    @Test
    public void testParRequestListOptionResultMapFailureInFirstParCallable() {
        ParRequestListOption.Builder<String> builder = executor.parRequestListOptionBuilder();
        builder.addParCallable(ParRequestTestCommon.stringSleep1000Failure);
        for (int i = 1; i < 8; i++) {
            builder.addParCallable(ParRequestTestCommon.stringSleep1000);
        }

        ParRequestListOption<String> request = builder.build();
        Either<GeneralError, Integer> result = request.resultMap(stringAssertOnlyFirstRequestEmpty);
        assertTrue(result.isRight());
        assertTrue(result.right().get() == 200);
    }

    @Test
    public void testParRequestListOptionResultMapExceptionInFirstParCallableAllMustSucceed() {
        ParRequestListOption.Builder<String> builder = executor.parRequestListOptionBuilder();
        builder.addParCallable(ParRequestTestCommon.stringSleep1000Exception);
        for (int i = 1; i < 8; i++) {
            builder.addParCallable(ParRequestTestCommon.stringSleep1000);
        }
        builder.allMustSucceed(true);

        ParRequestListOption<String> request = builder.build();
        Either<GeneralError, Integer> result = request.resultMap(stringAssertOnlyFirstRequestEmpty);
        assertTrue(result.isLeft());
        assertTrue(result.left().get().throwableOption.get().getMessage().contains("blah1"));
    }

    @Test
    public void testParRequestListOptionResultMapExceptionInFirstParCallable() {
        ParRequestListOption.Builder<String> builder = executor.parRequestListOptionBuilder();
        builder.addParCallable(ParRequestTestCommon.stringSleep1000Exception);
        for (int i = 1; i < 8; i++) {
            builder.addParCallable(ParRequestTestCommon.stringSleep1000);
        }

        ParRequestListOption<String> request = builder.build();
        Either<GeneralError, Integer> result = request.resultMap(stringAssertOnlyFirstRequestEmpty);
        assertTrue(result.isRight());
        assertTrue(result.right().get() == 200);
    }

    @Test
    public void testParRequestListOptionMapSuccess() {
        ParRequestListOption.Builder<String> builder = executor.parRequestListOptionBuilder();
        for (int i = 0; i < 8; i++) {
            builder.addParCallable(ParRequestTestCommon.stringSleep1000);
        }

        ParRequestListOption<String> request = builder.build();
        ParRequest<Integer>
                composedRequest =
                request.map("testParRequestListOptionMapSuccess",
                        ParFunction.from((input) -> {
                            for (Option<String> stringOption : input) {
                                assertTrue(stringOption.get().equals("100"));
                            }
                            return new Right<>(200);
                        }));
        Either<GeneralError, Integer> result = composedRequest.get();
        assertTrue(result.isRight());
        assertTrue(result.right().get() == 200);
    }

    @Test
    public void testParRequestListOptionMapFailure() {
        ParRequestListOption.Builder<String> builder = executor.parRequestListOptionBuilder();
        for (int i = 0; i < 8; i++) {
            builder.addParCallable(ParRequestTestCommon.stringSleep1000);
        }

        ParRequestListOption<String> request = builder.build();
        ParRequest<Integer>
                composedRequest =
                request.map("testParRequestListOptionMapFailure",
                        ParFunction.from((input) -> {
                            for (Option<String> stringOption : input) {
                                assertTrue(stringOption.get().equals("100"));
                            }
                            throw new IllegalArgumentException("failed");
                        }));
        Either<GeneralError, Integer> result = composedRequest.get();
        assertTrue(result.isLeft());
        assertTrue(result.left().get().throwableOption.get().getMessage().contains("failed"), printError(result));
    }

    @Test
    public void testParRequestListOptionMapFailureInFirstParCallableAllMustSucceed() {
        ParRequestListOption.Builder<String> builder = executor.parRequestListOptionBuilder();
        builder.addParCallable(ParRequestTestCommon.stringSleep1000Failure);
        for (int i = 1; i < 8; i++) {
            builder.addParCallable(ParRequestTestCommon.stringSleep1000);
        }
        builder.allMustSucceed(true);

        ParRequestListOption<String> request = builder.build();
        ParRequest<Integer>
                composedRequest =
                request.map("testParRequestListOptionMapFailureInFirstParCallableAllMustSucceed",
                        ParFunction.from((input) -> {
                            for (int i = 0; i < 8; i++) {
                                assertTrue(
                                        (i == 0 && input.get(i).isEmpty()) ||
                                                (i > 0 && input.get(i).get().equals("100")),
                                        "first request should fail and the rest should succeed"
                                );
                            }
                            return new Right<>(200);
                        }));
        Either<GeneralError, Integer> result = composedRequest.get();
        assertTrue(result.isLeft());
        assertTrue(result.left().get().message.equals("failed"), printError(result));
    }

    @Test
    public void testParRequestListOptionMapFailureInFirstParCallable() {
        ParRequestListOption.Builder<String> builder = executor.parRequestListOptionBuilder();
        builder.addParCallable(ParRequestTestCommon.stringSleep1000Failure);
        for (int i = 1; i < 8; i++) {
            builder.addParCallable(ParRequestTestCommon.stringSleep1000);
        }

        ParRequestListOption<String> request = builder.build();
        ParRequest<Integer>
                composedRequest =
                request.map("testParRequestListOptionMapFailureInFirstParCallableAllMustSucceed",
                        ParFunction.from((input) -> {
                            for (int i = 0; i < 8; i++) {
                                assertTrue(
                                        (i == 0 && input.get(i).isEmpty()) ||
                                                (i > 0 && input.get(i).get().equals("100")),
                                        "first request should fail and the rest should succeed"
                                );
                            }
                            return new Right<>(200);
                        }));
        Either<GeneralError, Integer> result = composedRequest.get();
        assertTrue(result.isRight());
        assertTrue(result.right().get() == 200);
    }

    @Test
    public void testParRequestListOptionMapExceptionInFirstParCallableAllMustSucceed() {
        ParRequestListOption.Builder<String> builder = executor.parRequestListOptionBuilder();
        builder.addParCallable(ParRequestTestCommon.stringSleep1000Exception);
        for (int i = 1; i < 8; i++) {
            builder.addParCallable(ParRequestTestCommon.stringSleep1000);
        }
        builder.allMustSucceed(true);

        ParRequestListOption<String> request = builder.build();
        ParRequest<Integer>
                composedRequest =
                request.map("testParRequestListOptionMapExceptionInSecondParCallableAllMustSucceed",
                        ParFunction.from((input) -> {
                            for (int i = 0; i < 8; i++) {
                                assertTrue(
                                        (i == 0 && input.get(i).isEmpty()) ||
                                                (i > 0 && input.get(i).get().equals("100")),
                                        "first request should fail and the rest should succeed"
                                );
                            }
                            return new Right<>(200);
                        }));
        Either<GeneralError, Integer> result = composedRequest.get();
        assertTrue(result.isLeft());
        assertTrue(result.left().get().throwableOption.get().getMessage().contains("blah1"));
    }

    @Test
    public void testParRequestListOptionMapExceptionInFirstParCallable() {
        ParRequestListOption.Builder<String> builder = executor.parRequestListOptionBuilder();
        builder.addParCallable(ParRequestTestCommon.stringSleep1000Exception);
        for (int i = 1; i < 8; i++) {
            builder.addParCallable(ParRequestTestCommon.stringSleep1000);
        }

        ParRequestListOption<String> request = builder.build();
        ParRequest<Integer>
                composedRequest =
                request.map("testParRequestListOptionMapExceptionInSecondParCallableAllMustSucceed",
                        ParFunction.from((input) -> {
                            for (int i = 0; i < 8; i++) {
                                assertTrue(
                                        (i == 0 && input.get(i).isEmpty()) ||
                                                (i > 0 && input.get(i).get().equals("100")),
                                        "first request should fail and the rest should succeed"
                                );
                            }
                            return new Right<>(200);
                        }));
        Either<GeneralError, Integer> result = composedRequest.get();
        assertTrue(result.isRight());
        assertTrue(result.right().get() == 200);
    }

    @Test
    public void testParRequestListOptionFlatMapSuccess() {
        ParRequestListOption.Builder<String> builder = executor.parRequestListOptionBuilder();
        for (int i = 0; i < 8; i++) {
            builder.addParCallable(ParRequestTestCommon.stringSleep1000);
        }

        ParRequestListOption<String> request = builder.build();
        ParRequest<Integer>
                composedRequest =
                request.flatMap("testParRequestListOptionFlatMapSuccess",
                        ParFunction.from((input) -> {
                            ParRequest.Builder<Integer> builder2 = new ParRequest.Builder<>(executor);
                            builder2.setParCallable(ParCallable.from(() -> {
                                for (Option<String> stringOption : input) {
                                    assertTrue(stringOption.get().equals("100"));
                                }
                                return new Right<>(200);
                            }));
                            return builder2.build();
                        }));
        Either<GeneralError, Integer> result = composedRequest.get();
        assertTrue(result.isRight());
        assertTrue(result.right().get() == 200);
    }

    @Test
    public void testParRequestListOptionFlatMapFailure() {
        ParRequestListOption.Builder<String> builder = executor.parRequestListOptionBuilder();
        for (int i = 0; i < 8; i++) {
            builder.addParCallable(ParRequestTestCommon.stringSleep1000);
        }

        ParRequestListOption<String> request = builder.build();
        ParRequest<Integer>
                composedRequest =
                request.flatMap("testParRequestListOptionFlatMapSuccess",
                        ParFunction.from((input) -> {
                            ParRequest.Builder<Integer> builder2 = new ParRequest.Builder<>(executor);
                            builder2.setParCallable(ParCallable.from(() -> {
                                for (Option<String> stringOption : input) {
                                    assertTrue(stringOption.get().equals("100"));
                                }
                                throw new IllegalArgumentException("failed");
                            }));
                            return builder2.build();
                        }));
        Either<GeneralError, Integer> result = composedRequest.get();
        assertTrue(result.isLeft());
        assertTrue(result.left().get().throwableOption.get().getMessage().contains("failed"), printError(result));
    }

    class TestResponse {

        public String err = "";
        public String success = "";
    }

    @Test
    public void testParRequestListOptionFoldSuccess() {
        ParRequestListOption.Builder<String> builder = executor.parRequestListOptionBuilder();
        for (int i = 0; i < 8; i++) {
            builder.addParCallable(ParRequestTestCommon.stringSleep1000);
        }

        final TestResponse response = new TestResponse();
        ParRequestListOption<String> request = builder.build();
        NoopRequest<Nothing> composedRequest = request.fold(ParFunction.from((input) -> {
            response.err = input.toString();
            return Nothing.get();
        }), ParFunction.from((input) -> {
            response.success = String.join("-", input.stream().map((opt) -> {
                return opt.get();
            }).collect(Collectors.toList()));
            return null;
        }));
        Either<GeneralError, Nothing> result = composedRequest.get();
        assertTrue(result.isRight());
        assertTrue(response.success.equals("100-100-100-100-100-100-100-100"));
    }

    @Test
    public void testParRequestListOptionFoldFailureInEvenNumberParCallable() {
        ParRequestListOption.Builder<String> builder = executor.parRequestListOptionBuilder();
        for (int i = 0; i < 8; i++) {
            if ((i & 1) == 1) {
                builder.addParCallable(ParRequestTestCommon.stringSleep1000);
            } else {
                builder.addParCallable(ParRequestTestCommon.stringSleep1000Exception);
            }

        }
        final TestResponse response = new TestResponse();
        ParRequestListOption<String> request = builder.build();
        NoopRequest<Nothing> composedRequest = request.fold(ParFunction.from((input) -> {
            response.err = input.toString();
            return Nothing.get();
        }), ParFunction.from((input) -> {
            response.success = String.join("-", input.stream().map((opt) -> {
                if (opt.isEmpty()) {
                    return "Empty";
                } else {
                    return opt.get();
                }
            }).collect(Collectors.toList()));
            return null;
        }));
        Either<GeneralError, Nothing> result = composedRequest.get();
        assertTrue(result.isRight());
        assertTrue(response.success.equals("Empty-100-Empty-100-Empty-100-Empty-100"));

    }

    @Test
    public void testParRequestListOptionFoldExceptionInEvenNumberParCallable() {
        ParRequestListOption.Builder<String> builder = executor.parRequestListOptionBuilder();
        for (int i = 0; i < 8; i++) {
            if ((i & 1) == 1) {
                builder.addParCallable(ParRequestTestCommon.stringSleep1000);
            } else {
                builder.addParCallable(ParRequestTestCommon.stringSleep1000Exception);
            }
        }

        final TestResponse response = new TestResponse();
        ParRequestListOption<String> request = builder.build();
        NoopRequest<Nothing> composedRequest = request.fold(ParFunction.from((input) -> {
            response.err = input.toString();
            return Nothing.get();
        }), ParFunction.from((input) -> {
            response.success = String.join("-", input.stream().map((opt) -> {
                if (opt.isEmpty()) {
                    return "Empty";
                } else {
                    return opt.get();
                }
            }).collect(Collectors.toList()));
            return null;
        }));
        Either<GeneralError, Nothing> result = composedRequest.get();
        assertTrue(result.isRight());
        assertTrue(response.success.equals("Empty-100-Empty-100-Empty-100-Empty-100"));
    }
}
