/*
 * Decompiled with CFR 0.152.
 */
package com.google.auth.oauth2;

import com.google.api.client.http.HttpTransport;
import com.google.api.client.json.GenericJson;
import com.google.api.client.json.JsonFactory;
import com.google.api.client.json.JsonGenerator;
import com.google.api.client.json.gson.GsonFactory;
import com.google.api.client.json.webtoken.JsonWebToken;
import com.google.api.client.testing.http.MockLowLevelHttpRequest;
import com.google.api.client.util.Clock;
import com.google.auth.ServiceAccountSigner;
import com.google.auth.TestUtils;
import com.google.auth.http.HttpTransportFactory;
import com.google.auth.oauth2.AccessToken;
import com.google.auth.oauth2.BaseSerializationTest;
import com.google.auth.oauth2.CredentialFormatException;
import com.google.auth.oauth2.GoogleCredentials;
import com.google.auth.oauth2.GoogleCredentialsTest;
import com.google.auth.oauth2.IdTokenCredentials;
import com.google.auth.oauth2.IdTokenProvider;
import com.google.auth.oauth2.ImpersonatedCredentials;
import com.google.auth.oauth2.MockIAMCredentialsServiceTransport;
import com.google.auth.oauth2.ServiceAccountCredentials;
import com.google.auth.oauth2.UserCredentials;
import cz.o2.proxima.internal.shaded.com.google.common.collect.ImmutableList;
import cz.o2.proxima.internal.shaded.com.google.common.collect.ImmutableSet;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.charset.Charset;
import java.security.PrivateKey;
import java.text.SimpleDateFormat;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Calendar;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.TimeZone;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.JUnit4;

@RunWith(value=JUnit4.class)
public class ImpersonatedCredentialsTest
extends BaseSerializationTest {
    public static final String SA_CLIENT_EMAIL = "36680232662-vrd7ji19qe3nelgchd0ah2csanun6bnr@developer.gserviceaccount.com";
    private static final String SA_PRIVATE_KEY_ID = "d84a4fefcf50791d4a90f2d7af17469d6282df9d";
    static final String SA_PRIVATE_KEY_PKCS8 = "-----BEGIN PRIVATE KEY-----\nMIICdgIBADANBgkqhkiG9w0BAQEFAASCAmAwggJcAgEAAoGBALX0PQoe1igW12ikv1bN/r9lN749y2ijmbc/mFHPyS3hNTyOCjDvBbXYbDhQJzWVUikh4mvGBA07qTj79Xc3yBDfKP2IeyYQIFe0t0zkd7R9Zdn98Y2rIQC47aAbDfubtkU1U72t4zL11kHvoa0/RuFZjncvlr42X7be7lYh4p3NAgMBAAECgYASk5wDw4Az2ZkmeuN6Fk/y9H+Lcb2pskJIXjrL533vrDWGOC48LrsThMQPv8cxBky8HFSEklPpkfTF95tpD43iVwJRB/GrCtGTw65IfJ4/tI09h6zGc4yqvIo1cHX/LQ+SxKLGyir/dQM925rGt/VojxY5ryJR7GLbCzxPnJm/oQJBANwOCO6D2hy1LQYJhXh7O+RLtA/tSnT1xyMQsGT+uUCMiKS2bSKx2wxo9k7h3OegNJIu1q6nZ6AbxDK8H3+d0dUCQQDTrPSXagBxzp8PecbaCHjzNRSQE2in81qYnrAFNB4o3DpHyMMY6s5ALLeHKscEWnqP8Ur6X4PvzZecCWU9BKAZAkAutLPknAuxSCsUOvUfS1i87ex77Ot+w6POp34pEX+UWb+u5iFn2cQacDTHLV1LtE80L8jVLSbrbrlH43H0DjU5AkEAgidhycxS86dxpEljnOMCw8CKoUBd5I880IUahEiUltk7OLJYS/Ts1wbn3kPOVX3wyJs8WBDtBkFrDHW2ezth2QJADj3e1YhMVdjJW5jqwlD/VNddGjgzyunmiZg0uOXsHXbytYmsA545S8KRQFaJKFXYYFo2kOjqOiC1T2cAzMDjCQ==\n-----END PRIVATE KEY-----\n";
    public static final String STANDARD_ID_TOKEN = "eyJhbGciOiJSUzI1NiIsImtpZCI6ImRmMzc1ODkwOGI3OTIyOTNhZDk3N2EwYjk5MWQ5OGE3N2Y0ZWVlY2QiLCJ0eXAiOiJKV1QifQ.eyJhdWQiOiJodHRwczovL2Zvby5iYXIiLCJhenAiOiIxMDIxMDE1NTA4MzQyMDA3MDg1NjgiLCJleHAiOjE1NjQ1MzI5NzIsImlhdCI6MTU2NDUyOTM3MiwiaXNzIjoiaHR0cHM6Ly9hY2NvdW50cy5nb29nbGUuY29tIiwic3ViIjoiMTAyMTAxNTUwODM0MjAwNzA4NTY4In0.redacted";
    public static final String TOKEN_WITH_EMAIL = "eyJhbGciOiJSUzI1NiIsImtpZCI6ImRmMzc1ODkwOGI3OTIyOTNhZDk3N2EwYjk5MWQ5OGE3N2Y0ZWVlY2QiLCJ0eXAiOiJKV1QifQ.eyJhdWQiOiJodHRwczovL2Zvby5iYXIiLCJhenAiOiIxMDIxMDE1NTA4MzQyMDA3MDg1NjgiLCJlbWFpbCI6ImltcGVyc29uYXRlZC1hY2NvdW50QGZhYmxlZC1yYXktMTA0MTE3LmlhbS5nc2VydmljZWFjY291bnQuY29tIiwiZW1haWxfdmVyaWZpZWQiOnRydWUsImV4cCI6MTU2NDUzMzA0MiwiaWF0IjoxNTY0NTI5NDQyLCJpc3MiOiJodHRwczovL2FjY291bnRzLmdvb2dsZS5jb20iLCJzdWIiOiIxMDIxMDE1NTA4MzQyMDA3MDg1NjgifQ.redacted";
    public static final String ACCESS_TOKEN = "1/MkSJoj1xsli0AccessToken_NKPY2";
    private static final ImmutableSet<String> IMMUTABLE_SCOPES_SET = ImmutableSet.of((Object)"scope1", (Object)"scope2");
    private static final String PROJECT_ID = "project-id";
    public static final String IMPERSONATED_CLIENT_EMAIL = "impersonated-account@iam.gserviceaccount.com";
    private static final List<String> IMMUTABLE_SCOPES_LIST = ImmutableList.of((Object)"scope1", (Object)"scope2");
    private static final int VALID_LIFETIME = 300;
    private static final int INVALID_LIFETIME = 43210;
    private static JsonFactory JSON_FACTORY = GsonFactory.getDefaultInstance();
    private static final String RFC3339 = "yyyy-MM-dd'T'HH:mm:ssX";
    public static final String DEFAULT_IMPERSONATION_URL = "https://iamcredentials.googleapis.com/v1/projects/-/serviceAccounts/impersonated-account@iam.gserviceaccount.com:generateAccessToken";
    public static final String IMPERSONATION_URL = "https://us-east1-iamcredentials.googleapis.com/v1/projects/-/serviceAccounts/impersonated-account@iam.gserviceaccount.com:generateAccessToken";
    private static final String USER_ACCOUNT_CLIENT_ID = "76408650-6qr441hur.apps.googleusercontent.com";
    private static final String USER_ACCOUNT_CLIENT_SECRET = "d-F499q74hFpdHD0T5";
    public static final String QUOTA_PROJECT_ID = "quota-project-id";
    private static final String REFRESH_TOKEN = "dasdfasdffa4ffdfadgyjirasdfadsft";
    public static final List<String> DELEGATES = Arrays.asList("sa1@developer.gserviceaccount.com", "sa2@developer.gserviceaccount.com");
    private GoogleCredentials sourceCredentials;
    private MockIAMCredentialsServiceTransportFactory mockTransportFactory;

    @Before
    public void setup() throws IOException {
        this.sourceCredentials = this.getSourceCredentials();
        this.mockTransportFactory = new MockIAMCredentialsServiceTransportFactory();
    }

    private GoogleCredentials getSourceCredentials() throws IOException {
        GoogleCredentialsTest.MockTokenServerTransportFactory transportFactory = new GoogleCredentialsTest.MockTokenServerTransportFactory();
        PrivateKey privateKey = ServiceAccountCredentials.privateKeyFromPkcs8((String)SA_PRIVATE_KEY_PKCS8);
        ServiceAccountCredentials sourceCredentials = ServiceAccountCredentials.newBuilder().setClientEmail(SA_CLIENT_EMAIL).setPrivateKey(privateKey).setPrivateKeyId(SA_PRIVATE_KEY_ID).setScopes(IMMUTABLE_SCOPES_LIST).setProjectId(PROJECT_ID).setHttpTransportFactory((HttpTransportFactory)transportFactory).build();
        transportFactory.transport.addServiceAccount(SA_CLIENT_EMAIL, ACCESS_TOKEN);
        return sourceCredentials;
    }

    @Test
    public void fromJson_userAsSource_WithQuotaProjectId() throws IOException {
        GenericJson json = ImpersonatedCredentialsTest.buildImpersonationCredentialsJson(IMPERSONATION_URL, DELEGATES, QUOTA_PROJECT_ID, USER_ACCOUNT_CLIENT_ID, USER_ACCOUNT_CLIENT_SECRET, REFRESH_TOKEN);
        ImpersonatedCredentials credentials = ImpersonatedCredentials.fromJson((Map)json, (HttpTransportFactory)this.mockTransportFactory);
        Assert.assertEquals((Object)IMPERSONATED_CLIENT_EMAIL, (Object)credentials.getAccount());
        Assert.assertEquals((Object)IMPERSONATION_URL, (Object)credentials.getIamEndpointOverride());
        Assert.assertEquals((Object)QUOTA_PROJECT_ID, (Object)credentials.getQuotaProjectId());
        Assert.assertEquals(DELEGATES, (Object)credentials.getDelegates());
        Assert.assertEquals(new ArrayList(), (Object)credentials.getScopes());
        Assert.assertEquals((long)3600L, (long)credentials.getLifetime());
        GoogleCredentials sourceCredentials = credentials.getSourceCredentials();
        Assert.assertTrue((boolean)(sourceCredentials instanceof UserCredentials));
    }

    @Test
    public void fromJson_userAsSource_WithoutQuotaProjectId() throws IOException {
        GenericJson json = ImpersonatedCredentialsTest.buildImpersonationCredentialsJson(IMPERSONATION_URL, DELEGATES, null, USER_ACCOUNT_CLIENT_ID, USER_ACCOUNT_CLIENT_SECRET, REFRESH_TOKEN);
        ImpersonatedCredentials credentials = ImpersonatedCredentials.fromJson((Map)json, (HttpTransportFactory)this.mockTransportFactory);
        Assert.assertEquals((Object)IMPERSONATED_CLIENT_EMAIL, (Object)credentials.getAccount());
        Assert.assertEquals((Object)IMPERSONATION_URL, (Object)credentials.getIamEndpointOverride());
        Assert.assertNull((Object)credentials.getQuotaProjectId());
        Assert.assertEquals(DELEGATES, (Object)credentials.getDelegates());
        Assert.assertEquals(new ArrayList(), (Object)credentials.getScopes());
        Assert.assertEquals((long)3600L, (long)credentials.getLifetime());
        GoogleCredentials sourceCredentials = credentials.getSourceCredentials();
        Assert.assertTrue((boolean)(sourceCredentials instanceof UserCredentials));
    }

    @Test
    public void fromJson_userAsSource_MissingDelegatesField() throws IOException {
        GenericJson json = ImpersonatedCredentialsTest.buildImpersonationCredentialsJson(IMPERSONATION_URL, DELEGATES, null, USER_ACCOUNT_CLIENT_ID, USER_ACCOUNT_CLIENT_SECRET, REFRESH_TOKEN);
        json.remove((Object)"delegates");
        ImpersonatedCredentials credentials = ImpersonatedCredentials.fromJson((Map)json, (HttpTransportFactory)this.mockTransportFactory);
        Assert.assertEquals((Object)IMPERSONATED_CLIENT_EMAIL, (Object)credentials.getAccount());
        Assert.assertEquals((Object)IMPERSONATION_URL, (Object)credentials.getIamEndpointOverride());
        Assert.assertNull((Object)credentials.getQuotaProjectId());
        Assert.assertEquals(new ArrayList(), (Object)credentials.getDelegates());
        Assert.assertEquals(new ArrayList(), (Object)credentials.getScopes());
        Assert.assertEquals((long)3600L, (long)credentials.getLifetime());
        GoogleCredentials sourceCredentials = credentials.getSourceCredentials();
        Assert.assertTrue((boolean)(sourceCredentials instanceof UserCredentials));
    }

    @Test
    public void fromJson_ServiceAccountAsSource() throws IOException {
        GenericJson json = ImpersonatedCredentialsTest.buildImpersonationCredentialsJson(IMPERSONATION_URL, DELEGATES, QUOTA_PROJECT_ID);
        ImpersonatedCredentials credentials = ImpersonatedCredentials.fromJson((Map)json, (HttpTransportFactory)this.mockTransportFactory);
        Assert.assertEquals((Object)IMPERSONATED_CLIENT_EMAIL, (Object)credentials.getAccount());
        Assert.assertEquals((Object)IMPERSONATION_URL, (Object)credentials.getIamEndpointOverride());
        Assert.assertEquals((Object)QUOTA_PROJECT_ID, (Object)credentials.getQuotaProjectId());
        Assert.assertEquals(DELEGATES, (Object)credentials.getDelegates());
        Assert.assertEquals(new ArrayList(), (Object)credentials.getScopes());
        Assert.assertEquals((long)3600L, (long)credentials.getLifetime());
        GoogleCredentials sourceCredentials = credentials.getSourceCredentials();
        Assert.assertTrue((boolean)(sourceCredentials instanceof ServiceAccountCredentials));
    }

    @Test
    public void fromJson_InvalidFormat() throws IOException {
        GenericJson json = ImpersonatedCredentialsTest.buildInvalidCredentialsJson();
        try {
            ImpersonatedCredentials.fromJson((Map)json, (HttpTransportFactory)this.mockTransportFactory);
            Assert.fail((String)"An exception should be thrown.");
        }
        catch (CredentialFormatException e) {
            Assert.assertEquals((Object)"An invalid input stream was provided.", (Object)e.getMessage());
        }
    }

    @Test
    public void createScopedRequired_True() {
        ImpersonatedCredentials targetCredentials = ImpersonatedCredentials.create((GoogleCredentials)this.sourceCredentials, (String)IMPERSONATED_CLIENT_EMAIL, null, new ArrayList(), (int)300, (HttpTransportFactory)this.mockTransportFactory);
        Assert.assertTrue((boolean)targetCredentials.createScopedRequired());
    }

    @Test
    public void createScopedRequired_False() {
        ImpersonatedCredentials targetCredentials = ImpersonatedCredentials.create((GoogleCredentials)this.sourceCredentials, (String)IMPERSONATED_CLIENT_EMAIL, null, IMMUTABLE_SCOPES_LIST, (int)300, (HttpTransportFactory)this.mockTransportFactory);
        Assert.assertFalse((boolean)targetCredentials.createScopedRequired());
    }

    @Test
    public void createScoped() {
        ImpersonatedCredentials targetCredentials = ImpersonatedCredentials.create((GoogleCredentials)this.sourceCredentials, (String)IMPERSONATED_CLIENT_EMAIL, DELEGATES, IMMUTABLE_SCOPES_LIST, (int)300, (HttpTransportFactory)this.mockTransportFactory, (String)QUOTA_PROJECT_ID);
        ImpersonatedCredentials scoped_credentials = (ImpersonatedCredentials)targetCredentials.createScoped(IMMUTABLE_SCOPES_LIST);
        Assert.assertEquals((Object)targetCredentials.getAccount(), (Object)scoped_credentials.getAccount());
        Assert.assertEquals((Object)targetCredentials.getDelegates(), (Object)scoped_credentials.getDelegates());
        Assert.assertEquals((long)targetCredentials.getLifetime(), (long)scoped_credentials.getLifetime());
        Assert.assertEquals((Object)targetCredentials.getSourceCredentials(), (Object)scoped_credentials.getSourceCredentials());
        Assert.assertEquals((Object)targetCredentials.getQuotaProjectId(), (Object)scoped_credentials.getQuotaProjectId());
        Assert.assertEquals(Arrays.asList("scope1", "scope2"), (Object)scoped_credentials.getScopes());
    }

    @Test
    public void createScopedWithImmutableScopes() {
        ImpersonatedCredentials targetCredentials = ImpersonatedCredentials.create((GoogleCredentials)this.sourceCredentials, (String)IMPERSONATED_CLIENT_EMAIL, DELEGATES, IMMUTABLE_SCOPES_LIST, (int)300, (HttpTransportFactory)this.mockTransportFactory, (String)QUOTA_PROJECT_ID);
        ImpersonatedCredentials scoped_credentials = (ImpersonatedCredentials)targetCredentials.createScoped(IMMUTABLE_SCOPES_SET);
        Assert.assertEquals((Object)targetCredentials.getAccount(), (Object)scoped_credentials.getAccount());
        Assert.assertEquals((Object)targetCredentials.getDelegates(), (Object)scoped_credentials.getDelegates());
        Assert.assertEquals((long)targetCredentials.getLifetime(), (long)scoped_credentials.getLifetime());
        Assert.assertEquals((Object)targetCredentials.getSourceCredentials(), (Object)scoped_credentials.getSourceCredentials());
        Assert.assertEquals((Object)targetCredentials.getQuotaProjectId(), (Object)scoped_credentials.getQuotaProjectId());
        Assert.assertEquals(Arrays.asList("scope1", "scope2"), (Object)scoped_credentials.getScopes());
    }

    @Test
    public void createScopedWithIamEndpointOverride() {
        ImpersonatedCredentials targetCredentials = ImpersonatedCredentials.create((GoogleCredentials)this.sourceCredentials, (String)IMPERSONATED_CLIENT_EMAIL, DELEGATES, IMMUTABLE_SCOPES_LIST, (int)300, (HttpTransportFactory)this.mockTransportFactory, (String)QUOTA_PROJECT_ID, (String)IMPERSONATION_URL);
        ImpersonatedCredentials scoped_credentials = (ImpersonatedCredentials)targetCredentials.createScoped(IMMUTABLE_SCOPES_SET);
        Assert.assertEquals((Object)targetCredentials.getIamEndpointOverride(), (Object)scoped_credentials.getIamEndpointOverride());
    }

    @Test
    public void refreshAccessToken_unauthorized() throws IOException {
        String expectedMessage = "The caller does not have permission";
        this.mockTransportFactory.transport.setTargetPrincipal(IMPERSONATED_CLIENT_EMAIL);
        this.mockTransportFactory.transport.setTokenResponseErrorCode(401);
        this.mockTransportFactory.transport.setTokenResponseErrorContent(this.generateErrorJson(401, expectedMessage, "global", "forbidden"));
        ImpersonatedCredentials targetCredentials = ImpersonatedCredentials.create((GoogleCredentials)this.sourceCredentials, (String)IMPERSONATED_CLIENT_EMAIL, null, IMMUTABLE_SCOPES_LIST, (int)300, (HttpTransportFactory)this.mockTransportFactory);
        try {
            targetCredentials.refreshAccessToken().getTokenValue();
            Assert.fail((String)String.format("Should throw exception with message containing '%s'", expectedMessage));
        }
        catch (IOException expected) {
            Assert.assertEquals((Object)"Error requesting access token", (Object)expected.getMessage());
            Assert.assertTrue((boolean)expected.getCause().getMessage().contains(expectedMessage));
        }
    }

    @Test
    public void refreshAccessToken_malformedTarget() throws IOException {
        String invalidTargetEmail = "foo";
        String expectedMessage = "Request contains an invalid argument";
        this.mockTransportFactory.transport.setTargetPrincipal(invalidTargetEmail);
        this.mockTransportFactory.transport.setTokenResponseErrorCode(400);
        this.mockTransportFactory.transport.setTokenResponseErrorContent(this.generateErrorJson(400, expectedMessage, "global", "badRequest"));
        ImpersonatedCredentials targetCredentials = ImpersonatedCredentials.create((GoogleCredentials)this.sourceCredentials, (String)invalidTargetEmail, null, IMMUTABLE_SCOPES_LIST, (int)300, (HttpTransportFactory)this.mockTransportFactory);
        try {
            targetCredentials.refreshAccessToken().getTokenValue();
            Assert.fail((String)String.format("Should throw exception with message containing '%s'", expectedMessage));
        }
        catch (IOException expected) {
            Assert.assertEquals((Object)"Error requesting access token", (Object)expected.getMessage());
            Assert.assertTrue((boolean)expected.getCause().getMessage().contains(expectedMessage));
        }
    }

    @Test
    public void credential_with_zero_lifetime() throws IllegalStateException {
        ImpersonatedCredentials targetCredentials = ImpersonatedCredentials.create((GoogleCredentials)this.sourceCredentials, (String)IMPERSONATED_CLIENT_EMAIL, null, IMMUTABLE_SCOPES_LIST, (int)0);
        Assert.assertEquals((long)3600L, (long)targetCredentials.getLifetime());
    }

    @Test
    public void credential_with_invalid_lifetime() throws IOException, IllegalStateException {
        try {
            ImpersonatedCredentials targetCredentials = ImpersonatedCredentials.create((GoogleCredentials)this.sourceCredentials, (String)IMPERSONATED_CLIENT_EMAIL, null, IMMUTABLE_SCOPES_LIST, (int)43210);
            targetCredentials.refreshAccessToken().getTokenValue();
            Assert.fail((String)String.format("Should throw exception with message containing '%s'", "lifetime must be less than or equal to 43200"));
        }
        catch (IllegalStateException expected) {
            Assert.assertTrue((boolean)expected.getMessage().contains("lifetime must be less than or equal to 43200"));
        }
    }

    @Test
    public void credential_with_invalid_scope() throws IOException, IllegalStateException {
        try {
            ImpersonatedCredentials targetCredentials = ImpersonatedCredentials.create((GoogleCredentials)this.sourceCredentials, (String)IMPERSONATED_CLIENT_EMAIL, null, null, (int)300);
            targetCredentials.refreshAccessToken().getTokenValue();
            Assert.fail((String)String.format("Should throw exception with message containing '%s'", "Scopes cannot be null"));
        }
        catch (IllegalStateException expected) {
            Assert.assertTrue((boolean)expected.getMessage().contains("Scopes cannot be null"));
        }
    }

    @Test
    public void refreshAccessToken_success() throws IOException, IllegalStateException {
        this.mockTransportFactory.transport.setTargetPrincipal(IMPERSONATED_CLIENT_EMAIL);
        this.mockTransportFactory.transport.setAccessToken(ACCESS_TOKEN);
        this.mockTransportFactory.transport.setExpireTime(ImpersonatedCredentialsTest.getDefaultExpireTime());
        ImpersonatedCredentials targetCredentials = ImpersonatedCredentials.create((GoogleCredentials)this.sourceCredentials, (String)IMPERSONATED_CLIENT_EMAIL, null, IMMUTABLE_SCOPES_LIST, (int)300, (HttpTransportFactory)this.mockTransportFactory);
        Assert.assertEquals((Object)ACCESS_TOKEN, (Object)targetCredentials.refreshAccessToken().getTokenValue());
        Assert.assertEquals((Object)DEFAULT_IMPERSONATION_URL, (Object)this.mockTransportFactory.transport.getRequest().getUrl());
    }

    @Test
    public void refreshAccessToken_endpointOverride() throws IOException, IllegalStateException {
        this.mockTransportFactory.transport.setTargetPrincipal(IMPERSONATED_CLIENT_EMAIL);
        this.mockTransportFactory.transport.setAccessToken(ACCESS_TOKEN);
        this.mockTransportFactory.transport.setExpireTime(ImpersonatedCredentialsTest.getDefaultExpireTime());
        this.mockTransportFactory.transport.setAccessTokenEndpoint(IMPERSONATION_URL);
        ImpersonatedCredentials targetCredentials = ImpersonatedCredentials.create((GoogleCredentials)this.sourceCredentials, (String)IMPERSONATED_CLIENT_EMAIL, null, IMMUTABLE_SCOPES_LIST, (int)300, (HttpTransportFactory)this.mockTransportFactory, (String)QUOTA_PROJECT_ID, (String)IMPERSONATION_URL);
        Assert.assertEquals((Object)ACCESS_TOKEN, (Object)targetCredentials.refreshAccessToken().getTokenValue());
        Assert.assertEquals((Object)IMPERSONATION_URL, (Object)this.mockTransportFactory.transport.getRequest().getUrl());
    }

    @Test
    public void getRequestMetadata_withQuotaProjectId() throws IOException, IllegalStateException {
        this.mockTransportFactory.transport.setTargetPrincipal(IMPERSONATED_CLIENT_EMAIL);
        this.mockTransportFactory.transport.setAccessToken(ACCESS_TOKEN);
        this.mockTransportFactory.transport.setExpireTime(ImpersonatedCredentialsTest.getDefaultExpireTime());
        ImpersonatedCredentials targetCredentials = ImpersonatedCredentials.create((GoogleCredentials)this.sourceCredentials, (String)IMPERSONATED_CLIENT_EMAIL, null, IMMUTABLE_SCOPES_LIST, (int)300, (HttpTransportFactory)this.mockTransportFactory, (String)QUOTA_PROJECT_ID);
        Map metadata = targetCredentials.getRequestMetadata();
        Assert.assertTrue((boolean)metadata.containsKey("x-goog-user-project"));
        List headerValues = (List)metadata.get("x-goog-user-project");
        Assert.assertEquals((long)1L, (long)headerValues.size());
        Assert.assertEquals((Object)QUOTA_PROJECT_ID, headerValues.get(0));
    }

    @Test
    public void getRequestMetadata_withoutQuotaProjectId() throws IOException, IllegalStateException {
        this.mockTransportFactory.transport.setTargetPrincipal(IMPERSONATED_CLIENT_EMAIL);
        this.mockTransportFactory.transport.setAccessToken(ACCESS_TOKEN);
        this.mockTransportFactory.transport.setExpireTime(ImpersonatedCredentialsTest.getDefaultExpireTime());
        ImpersonatedCredentials targetCredentials = ImpersonatedCredentials.create((GoogleCredentials)this.sourceCredentials, (String)IMPERSONATED_CLIENT_EMAIL, null, IMMUTABLE_SCOPES_LIST, (int)300, (HttpTransportFactory)this.mockTransportFactory);
        Map metadata = targetCredentials.getRequestMetadata();
        Assert.assertFalse((boolean)metadata.containsKey("x-goog-user-project"));
    }

    @Test
    public void refreshAccessToken_delegates_success() throws IOException, IllegalStateException {
        this.mockTransportFactory.transport.setTargetPrincipal(IMPERSONATED_CLIENT_EMAIL);
        this.mockTransportFactory.transport.setAccessToken(ACCESS_TOKEN);
        this.mockTransportFactory.transport.setExpireTime(ImpersonatedCredentialsTest.getDefaultExpireTime());
        List<String> delegates = Arrays.asList("delegate-account@iam.gserviceaccount.com");
        ImpersonatedCredentials targetCredentials = ImpersonatedCredentials.create((GoogleCredentials)this.sourceCredentials, (String)IMPERSONATED_CLIENT_EMAIL, delegates, IMMUTABLE_SCOPES_LIST, (int)300, (HttpTransportFactory)this.mockTransportFactory);
        Assert.assertEquals((Object)ACCESS_TOKEN, (Object)targetCredentials.refreshAccessToken().getTokenValue());
    }

    @Test
    public void refreshAccessToken_GMT_dateParsedCorrectly() throws IOException, IllegalStateException {
        Calendar c = Calendar.getInstance();
        c.add(13, 300);
        this.mockTransportFactory.transport.setTargetPrincipal(IMPERSONATED_CLIENT_EMAIL);
        this.mockTransportFactory.transport.setAccessToken(ACCESS_TOKEN);
        this.mockTransportFactory.transport.setExpireTime(ImpersonatedCredentialsTest.getFormattedTime(c.getTime()));
        ImpersonatedCredentials targetCredentials = ImpersonatedCredentials.create((GoogleCredentials)this.sourceCredentials, (String)IMPERSONATED_CLIENT_EMAIL, null, IMMUTABLE_SCOPES_LIST, (int)300, (HttpTransportFactory)this.mockTransportFactory).createWithCustomCalendar(Calendar.getInstance(TimeZone.getTimeZone("GMT")));
        Assert.assertTrue((c.getTime().toInstant().truncatedTo(ChronoUnit.SECONDS).toEpochMilli() == targetCredentials.refreshAccessToken().getExpirationTimeMillis().longValue() ? 1 : 0) != 0);
    }

    @Test
    public void refreshAccessToken_nonGMT_dateParsedCorrectly() throws IOException, IllegalStateException {
        Calendar c = Calendar.getInstance();
        c.add(13, 300);
        this.mockTransportFactory.transport.setTargetPrincipal(IMPERSONATED_CLIENT_EMAIL);
        this.mockTransportFactory.transport.setAccessToken(ACCESS_TOKEN);
        this.mockTransportFactory.transport.setExpireTime(ImpersonatedCredentialsTest.getFormattedTime(c.getTime()));
        ImpersonatedCredentials targetCredentials = ImpersonatedCredentials.create((GoogleCredentials)this.sourceCredentials, (String)IMPERSONATED_CLIENT_EMAIL, null, IMMUTABLE_SCOPES_LIST, (int)300, (HttpTransportFactory)this.mockTransportFactory).createWithCustomCalendar(Calendar.getInstance(TimeZone.getTimeZone("America/Los_Angeles")));
        Assert.assertTrue((c.getTime().toInstant().truncatedTo(ChronoUnit.SECONDS).toEpochMilli() == targetCredentials.refreshAccessToken().getExpirationTimeMillis().longValue() ? 1 : 0) != 0);
    }

    @Test
    public void refreshAccessToken_invalidDate() throws IllegalStateException {
        String expectedMessage = "Unparseable date";
        this.mockTransportFactory.transport.setTargetPrincipal(IMPERSONATED_CLIENT_EMAIL);
        this.mockTransportFactory.transport.setAccessToken("foo");
        this.mockTransportFactory.transport.setExpireTime("1973-09-29T15:01:23");
        ImpersonatedCredentials targetCredentials = ImpersonatedCredentials.create((GoogleCredentials)this.sourceCredentials, (String)IMPERSONATED_CLIENT_EMAIL, null, IMMUTABLE_SCOPES_LIST, (int)300, (HttpTransportFactory)this.mockTransportFactory);
        try {
            targetCredentials.refreshAccessToken().getTokenValue();
            Assert.fail((String)String.format("Should throw exception with message containing '%s'", expectedMessage));
        }
        catch (IOException expected) {
            Assert.assertTrue((boolean)expected.getMessage().contains(expectedMessage));
        }
    }

    @Test
    public void getAccount_sameAs() {
        this.mockTransportFactory.transport.setTargetPrincipal(IMPERSONATED_CLIENT_EMAIL);
        this.mockTransportFactory.transport.setAccessToken(ACCESS_TOKEN);
        this.mockTransportFactory.transport.setExpireTime(ImpersonatedCredentialsTest.getDefaultExpireTime());
        ImpersonatedCredentials targetCredentials = ImpersonatedCredentials.create((GoogleCredentials)this.sourceCredentials, (String)IMPERSONATED_CLIENT_EMAIL, null, IMMUTABLE_SCOPES_LIST, (int)300, (HttpTransportFactory)this.mockTransportFactory);
        Assert.assertEquals((Object)IMPERSONATED_CLIENT_EMAIL, (Object)targetCredentials.getAccount());
    }

    @Test
    public void sign_sameAs() {
        this.mockTransportFactory.transport.setTargetPrincipal(IMPERSONATED_CLIENT_EMAIL);
        this.mockTransportFactory.transport.setAccessToken(ACCESS_TOKEN);
        this.mockTransportFactory.transport.setExpireTime(ImpersonatedCredentialsTest.getDefaultExpireTime());
        ImpersonatedCredentials targetCredentials = ImpersonatedCredentials.create((GoogleCredentials)this.sourceCredentials, (String)IMPERSONATED_CLIENT_EMAIL, null, IMMUTABLE_SCOPES_LIST, (int)300, (HttpTransportFactory)this.mockTransportFactory);
        byte[] expectedSignature = new byte[]{13, 14, 10, 13};
        this.mockTransportFactory.transport.setTargetPrincipal(IMPERSONATED_CLIENT_EMAIL);
        this.mockTransportFactory.transport.setSignedBlob(expectedSignature);
        Assert.assertArrayEquals((byte[])expectedSignature, (byte[])targetCredentials.sign(expectedSignature));
    }

    @Test
    public void sign_requestIncludesDelegates() throws IOException {
        this.mockTransportFactory.transport.setTargetPrincipal(IMPERSONATED_CLIENT_EMAIL);
        this.mockTransportFactory.transport.setAccessToken(ACCESS_TOKEN);
        this.mockTransportFactory.transport.setExpireTime(ImpersonatedCredentialsTest.getDefaultExpireTime());
        ImpersonatedCredentials targetCredentials = ImpersonatedCredentials.create((GoogleCredentials)this.sourceCredentials, (String)IMPERSONATED_CLIENT_EMAIL, (List)ImmutableList.of((Object)"delegate@example.com"), IMMUTABLE_SCOPES_LIST, (int)300, (HttpTransportFactory)this.mockTransportFactory);
        byte[] expectedSignature = new byte[]{13, 14, 10, 13};
        this.mockTransportFactory.transport.setTargetPrincipal(IMPERSONATED_CLIENT_EMAIL);
        this.mockTransportFactory.transport.setSignedBlob(expectedSignature);
        Assert.assertArrayEquals((byte[])expectedSignature, (byte[])targetCredentials.sign(expectedSignature));
        MockLowLevelHttpRequest request = this.mockTransportFactory.transport.getRequest();
        GenericJson body = (GenericJson)JSON_FACTORY.createJsonParser(request.getContentAsString()).parseAndClose(GenericJson.class);
        ArrayList<String> delegates = new ArrayList<String>();
        delegates.add("delegate@example.com");
        Assert.assertEquals(delegates, (Object)body.get((Object)"delegates"));
    }

    @Test
    public void sign_usesSourceCredentials() {
        Calendar c = Calendar.getInstance();
        c.add(5, 1);
        Date expiry = c.getTime();
        GoogleCredentials sourceCredentials = new GoogleCredentials.Builder().setAccessToken(new AccessToken("source-token", expiry)).build();
        this.mockTransportFactory.transport.setTargetPrincipal(IMPERSONATED_CLIENT_EMAIL);
        this.mockTransportFactory.transport.setAccessToken(ACCESS_TOKEN);
        this.mockTransportFactory.transport.setExpireTime(ImpersonatedCredentialsTest.getDefaultExpireTime());
        ImpersonatedCredentials targetCredentials = ImpersonatedCredentials.create((GoogleCredentials)sourceCredentials, (String)IMPERSONATED_CLIENT_EMAIL, (List)ImmutableList.of((Object)"delegate@example.com"), IMMUTABLE_SCOPES_LIST, (int)300, (HttpTransportFactory)this.mockTransportFactory);
        byte[] expectedSignature = new byte[]{13, 14, 10, 13};
        this.mockTransportFactory.transport.setTargetPrincipal(IMPERSONATED_CLIENT_EMAIL);
        this.mockTransportFactory.transport.setSignedBlob(expectedSignature);
        Assert.assertArrayEquals((byte[])expectedSignature, (byte[])targetCredentials.sign(expectedSignature));
        MockLowLevelHttpRequest request = this.mockTransportFactory.transport.getRequest();
        Assert.assertEquals((Object)"Bearer source-token", (Object)request.getFirstHeaderValue("Authorization"));
    }

    @Test
    public void sign_accessDenied_throws() {
        this.mockTransportFactory.transport.setTargetPrincipal(IMPERSONATED_CLIENT_EMAIL);
        this.mockTransportFactory.transport.setAccessToken(ACCESS_TOKEN);
        this.mockTransportFactory.transport.setExpireTime(ImpersonatedCredentialsTest.getDefaultExpireTime());
        ImpersonatedCredentials targetCredentials = ImpersonatedCredentials.create((GoogleCredentials)this.sourceCredentials, (String)IMPERSONATED_CLIENT_EMAIL, null, IMMUTABLE_SCOPES_LIST, (int)300, (HttpTransportFactory)this.mockTransportFactory);
        byte[] expectedSignature = new byte[]{13, 14, 10, 13};
        this.mockTransportFactory.transport.setTargetPrincipal(IMPERSONATED_CLIENT_EMAIL);
        this.mockTransportFactory.transport.setSignedBlob(expectedSignature);
        this.mockTransportFactory.transport.setErrorResponseCodeAndMessage(403, "Sign Error");
        try {
            byte[] bytes = new byte[]{13, 14, 10, 13};
            targetCredentials.sign(bytes);
            Assert.fail((String)"Signing should have failed");
        }
        catch (ServiceAccountSigner.SigningException e) {
            Assert.assertEquals((Object)"Failed to sign the provided bytes", (Object)e.getMessage());
            Assert.assertNotNull((Object)e.getCause());
            Assert.assertTrue((boolean)e.getCause().getMessage().contains("403"));
        }
    }

    @Test
    public void sign_serverError_throws() {
        this.mockTransportFactory.transport.setTargetPrincipal(IMPERSONATED_CLIENT_EMAIL);
        this.mockTransportFactory.transport.setAccessToken(ACCESS_TOKEN);
        this.mockTransportFactory.transport.setExpireTime(ImpersonatedCredentialsTest.getDefaultExpireTime());
        ImpersonatedCredentials targetCredentials = ImpersonatedCredentials.create((GoogleCredentials)this.sourceCredentials, (String)IMPERSONATED_CLIENT_EMAIL, null, IMMUTABLE_SCOPES_LIST, (int)300, (HttpTransportFactory)this.mockTransportFactory);
        byte[] expectedSignature = new byte[]{13, 14, 10, 13};
        this.mockTransportFactory.transport.setTargetPrincipal(IMPERSONATED_CLIENT_EMAIL);
        this.mockTransportFactory.transport.setSignedBlob(expectedSignature);
        this.mockTransportFactory.transport.setErrorResponseCodeAndMessage(500, "Sign Error");
        try {
            byte[] bytes = new byte[]{13, 14, 10, 13};
            targetCredentials.sign(bytes);
            Assert.fail((String)"Signing should have failed");
        }
        catch (ServiceAccountSigner.SigningException e) {
            Assert.assertEquals((Object)"Failed to sign the provided bytes", (Object)e.getMessage());
            Assert.assertNotNull((Object)e.getCause());
            Assert.assertTrue((boolean)e.getCause().getMessage().contains("500"));
        }
    }

    @Test
    public void idTokenWithAudience_sameAs() throws IOException {
        this.mockTransportFactory.transport.setTargetPrincipal(IMPERSONATED_CLIENT_EMAIL);
        this.mockTransportFactory.transport.setAccessToken(ACCESS_TOKEN);
        this.mockTransportFactory.transport.setExpireTime(ImpersonatedCredentialsTest.getDefaultExpireTime());
        ImpersonatedCredentials targetCredentials = ImpersonatedCredentials.create((GoogleCredentials)this.sourceCredentials, (String)IMPERSONATED_CLIENT_EMAIL, null, IMMUTABLE_SCOPES_LIST, (int)300, (HttpTransportFactory)this.mockTransportFactory);
        this.mockTransportFactory.transport.setIdToken(STANDARD_ID_TOKEN);
        String targetAudience = "https://foo.bar";
        IdTokenCredentials tokenCredential = IdTokenCredentials.newBuilder().setIdTokenProvider((IdTokenProvider)targetCredentials).setTargetAudience(targetAudience).build();
        tokenCredential.refresh();
        Assert.assertEquals((Object)STANDARD_ID_TOKEN, (Object)tokenCredential.getAccessToken().getTokenValue());
        Assert.assertEquals((Object)STANDARD_ID_TOKEN, (Object)tokenCredential.getIdToken().getTokenValue());
        Assert.assertEquals((Object)targetAudience, (Object)((String)tokenCredential.getIdToken().getJsonWebSignature().getPayload().getAudience()));
    }

    @Test
    public void idTokenWithAudience_withEmail() throws IOException {
        this.mockTransportFactory.transport.setTargetPrincipal(IMPERSONATED_CLIENT_EMAIL);
        this.mockTransportFactory.transport.setAccessToken(ACCESS_TOKEN);
        this.mockTransportFactory.transport.setExpireTime(ImpersonatedCredentialsTest.getDefaultExpireTime());
        ImpersonatedCredentials targetCredentials = ImpersonatedCredentials.create((GoogleCredentials)this.sourceCredentials, (String)IMPERSONATED_CLIENT_EMAIL, null, IMMUTABLE_SCOPES_LIST, (int)300, (HttpTransportFactory)this.mockTransportFactory);
        this.mockTransportFactory.transport.setIdToken(TOKEN_WITH_EMAIL);
        String targetAudience = "https://foo.bar";
        IdTokenCredentials tokenCredential = IdTokenCredentials.newBuilder().setIdTokenProvider((IdTokenProvider)targetCredentials).setTargetAudience(targetAudience).setOptions(Arrays.asList(IdTokenProvider.Option.INCLUDE_EMAIL)).build();
        tokenCredential.refresh();
        Assert.assertEquals((Object)TOKEN_WITH_EMAIL, (Object)tokenCredential.getAccessToken().getTokenValue());
        JsonWebToken.Payload p = tokenCredential.getIdToken().getJsonWebSignature().getPayload();
        Assert.assertTrue((boolean)p.containsKey((Object)"email"));
    }

    @Test
    public void idToken_withServerError() {
        this.mockTransportFactory.transport.setTargetPrincipal(IMPERSONATED_CLIENT_EMAIL);
        this.mockTransportFactory.transport.setAccessToken(ACCESS_TOKEN);
        this.mockTransportFactory.transport.setExpireTime(ImpersonatedCredentialsTest.getDefaultExpireTime());
        ImpersonatedCredentials targetCredentials = ImpersonatedCredentials.create((GoogleCredentials)this.sourceCredentials, (String)IMPERSONATED_CLIENT_EMAIL, null, IMMUTABLE_SCOPES_LIST, (int)300, (HttpTransportFactory)this.mockTransportFactory);
        this.mockTransportFactory.transport.setIdToken(STANDARD_ID_TOKEN);
        this.mockTransportFactory.transport.setErrorResponseCodeAndMessage(500, "Internal Server Error");
        String targetAudience = "https://foo.bar";
        IdTokenCredentials tokenCredential = IdTokenCredentials.newBuilder().setIdTokenProvider((IdTokenProvider)targetCredentials).setTargetAudience(targetAudience).build();
        try {
            tokenCredential.refresh();
            Assert.fail((String)"Should not be able to use credential without exception.");
        }
        catch (IOException e) {
            Assert.assertTrue((boolean)e.getMessage().contains("Error code 500 trying to getIDToken"));
        }
    }

    @Test
    public void idToken_withOtherError() {
        this.mockTransportFactory.transport.setTargetPrincipal(IMPERSONATED_CLIENT_EMAIL);
        this.mockTransportFactory.transport.setAccessToken(ACCESS_TOKEN);
        this.mockTransportFactory.transport.setExpireTime(ImpersonatedCredentialsTest.getDefaultExpireTime());
        ImpersonatedCredentials targetCredentials = ImpersonatedCredentials.create((GoogleCredentials)this.sourceCredentials, (String)IMPERSONATED_CLIENT_EMAIL, null, IMMUTABLE_SCOPES_LIST, (int)300, (HttpTransportFactory)this.mockTransportFactory);
        this.mockTransportFactory.transport.setIdToken(STANDARD_ID_TOKEN);
        this.mockTransportFactory.transport.setErrorResponseCodeAndMessage(301, "Redirect");
        String targetAudience = "https://foo.bar";
        IdTokenCredentials tokenCredential = IdTokenCredentials.newBuilder().setIdTokenProvider((IdTokenProvider)targetCredentials).setTargetAudience(targetAudience).build();
        try {
            tokenCredential.refresh();
            Assert.fail((String)"Should not be able to use credential without exception.");
        }
        catch (IOException e) {
            Assert.assertTrue((boolean)e.getMessage().contains("Unexpected Error code 301 trying to getIDToken"));
        }
    }

    @Test
    public void hashCode_equals() throws IOException {
        this.mockTransportFactory.transport.setTargetPrincipal(IMPERSONATED_CLIENT_EMAIL);
        this.mockTransportFactory.transport.setAccessToken(ACCESS_TOKEN);
        this.mockTransportFactory.transport.setExpireTime(ImpersonatedCredentialsTest.getDefaultExpireTime());
        ImpersonatedCredentials credentials = ImpersonatedCredentials.create((GoogleCredentials)this.sourceCredentials, (String)IMPERSONATED_CLIENT_EMAIL, null, IMMUTABLE_SCOPES_LIST, (int)300, (HttpTransportFactory)this.mockTransportFactory);
        ImpersonatedCredentials otherCredentials = ImpersonatedCredentials.create((GoogleCredentials)this.sourceCredentials, (String)IMPERSONATED_CLIENT_EMAIL, null, IMMUTABLE_SCOPES_LIST, (int)300, (HttpTransportFactory)this.mockTransportFactory);
        Assert.assertEquals((long)credentials.hashCode(), (long)otherCredentials.hashCode());
    }

    @Test
    public void serialize() throws IOException, ClassNotFoundException {
        this.mockTransportFactory.transport.setTargetPrincipal(IMPERSONATED_CLIENT_EMAIL);
        this.mockTransportFactory.transport.setAccessToken(ACCESS_TOKEN);
        this.mockTransportFactory.transport.setExpireTime(ImpersonatedCredentialsTest.getDefaultExpireTime());
        ImpersonatedCredentials targetCredentials = ImpersonatedCredentials.create((GoogleCredentials)this.sourceCredentials, (String)IMPERSONATED_CLIENT_EMAIL, null, IMMUTABLE_SCOPES_LIST, (int)300, (HttpTransportFactory)this.mockTransportFactory);
        GoogleCredentials deserializedCredentials = (GoogleCredentials)this.serializeAndDeserialize(targetCredentials);
        Assert.assertEquals((Object)targetCredentials, (Object)deserializedCredentials);
        Assert.assertEquals((long)targetCredentials.hashCode(), (long)deserializedCredentials.hashCode());
        Assert.assertEquals((Object)targetCredentials.toString(), (Object)deserializedCredentials.toString());
        Assert.assertSame((Object)deserializedCredentials.clock, (Object)Clock.SYSTEM);
    }

    public static String getDefaultExpireTime() {
        Calendar c = Calendar.getInstance();
        c.add(13, 300);
        return ImpersonatedCredentialsTest.getFormattedTime(c.getTime());
    }

    private static String getFormattedTime(Date date) {
        SimpleDateFormat formatter = new SimpleDateFormat(RFC3339);
        formatter.setTimeZone(TimeZone.getTimeZone("GMT"));
        return formatter.format(date);
    }

    private String generateErrorJson(int errorCode, String errorMessage, String errorDomain, String errorReason) throws IOException {
        GsonFactory factory = new GsonFactory();
        ByteArrayOutputStream bout = new ByteArrayOutputStream();
        JsonGenerator generator = factory.createJsonGenerator((OutputStream)bout, Charset.defaultCharset());
        generator.enablePrettyPrint();
        generator.writeStartObject();
        generator.writeFieldName("error");
        generator.writeStartObject();
        generator.writeFieldName("code");
        generator.writeNumber(errorCode);
        generator.writeFieldName("message");
        generator.writeString(errorMessage);
        generator.writeFieldName("errors");
        generator.writeStartArray();
        generator.writeStartObject();
        generator.writeFieldName("message");
        generator.writeString(errorMessage);
        generator.writeFieldName("domain");
        generator.writeString(errorDomain);
        generator.writeFieldName("reason");
        generator.writeString(errorReason);
        generator.writeEndObject();
        generator.writeEndArray();
        generator.writeFieldName("status");
        generator.writeString("PERMISSION_DENIED");
        generator.writeEndObject();
        generator.writeEndObject();
        generator.close();
        return bout.toString();
    }

    static GenericJson buildImpersonationCredentialsJson(String impersonationUrl, List<String> delegates, String quotaProjectId, String sourceClientId, String sourceClientSecret, String sourceRefreshToken) {
        GenericJson sourceJson = new GenericJson();
        sourceJson.put("client_id", (Object)sourceClientId);
        sourceJson.put("client_secret", (Object)sourceClientSecret);
        sourceJson.put("refresh_token", (Object)sourceRefreshToken);
        sourceJson.put("type", (Object)"authorized_user");
        GenericJson json = new GenericJson();
        json.put("service_account_impersonation_url", (Object)impersonationUrl);
        json.put("delegates", delegates);
        if (quotaProjectId != null) {
            json.put("quota_project_id", (Object)quotaProjectId);
        }
        json.put("source_credentials", (Object)sourceJson);
        json.put("type", (Object)"impersonated_service_account");
        return json;
    }

    static GenericJson buildImpersonationCredentialsJson(String impersonationUrl, List<String> delegates, String quotaProjectId) {
        GenericJson sourceJson = new GenericJson();
        sourceJson.put("type", (Object)"service_account");
        sourceJson.put("project_id", (Object)PROJECT_ID);
        sourceJson.put("private_key_id", (Object)SA_PRIVATE_KEY_ID);
        sourceJson.put("private_key", (Object)SA_PRIVATE_KEY_PKCS8);
        sourceJson.put("client_email", (Object)SA_CLIENT_EMAIL);
        sourceJson.put("client_id", (Object)"10848832332323213");
        sourceJson.put("auth_uri", (Object)"https://oauth2.googleapis.com/o/oauth2/auth");
        sourceJson.put("token_uri", (Object)"https://oauth2.googleapis.com/token");
        sourceJson.put("auth_provider_x509_cert_url", (Object)"https://www.googleapis.com/oauth2/v1/certs");
        sourceJson.put("client_x509_cert_url", (Object)"https://www.googleapis.com/robot/v1/metadata/x509/chaoren-test-sc%40cloudsdktest.iam.gserviceaccount.com");
        GenericJson json = new GenericJson();
        json.put("source_credentials", (Object)sourceJson);
        json.put("service_account_impersonation_url", (Object)impersonationUrl);
        json.put("delegates", delegates);
        if (quotaProjectId != null) {
            json.put("quota_project_id", (Object)quotaProjectId);
        }
        json.put("type", (Object)"impersonated_service_account");
        return json;
    }

    static GenericJson buildInvalidCredentialsJson() {
        GenericJson json = new GenericJson();
        json.put("service_account_impersonation_url", (Object)"mock_url");
        return json;
    }

    static InputStream writeImpersonationCredentialsStream(String impersonationUrl, List<String> delegates, String quotaProjectId) throws IOException {
        GenericJson json = ImpersonatedCredentialsTest.buildImpersonationCredentialsJson(impersonationUrl, delegates, quotaProjectId);
        return TestUtils.jsonToInputStream(json);
    }

    static class MockIAMCredentialsServiceTransportFactory
    implements HttpTransportFactory {
        MockIAMCredentialsServiceTransport transport = new MockIAMCredentialsServiceTransport();

        MockIAMCredentialsServiceTransportFactory() {
        }

        public HttpTransport create() {
            return this.transport;
        }
    }
}

