/*
 * Decompiled with CFR 0.152.
 */
package com.google.auth.oauth2;

import com.google.api.client.http.HttpResponseException;
import com.google.api.client.http.LowLevelHttpResponse;
import com.google.api.client.json.GenericJson;
import com.google.api.client.json.JsonFactory;
import com.google.api.client.json.gson.GsonFactory;
import com.google.api.client.json.webtoken.JsonWebSignature;
import com.google.api.client.json.webtoken.JsonWebToken;
import com.google.api.client.testing.http.FixedClock;
import com.google.api.client.testing.http.MockLowLevelHttpResponse;
import com.google.api.client.util.Clock;
import com.google.api.client.util.Joiner;
import com.google.auth.RequestMetadataCallback;
import com.google.auth.TestUtils;
import com.google.auth.http.HttpTransportFactory;
import com.google.auth.oauth2.AccessToken;
import com.google.auth.oauth2.BaseSerializationTest;
import com.google.auth.oauth2.GoogleAuthException;
import com.google.auth.oauth2.GoogleCredentials;
import com.google.auth.oauth2.GoogleCredentialsTest;
import com.google.auth.oauth2.IdTokenCredentials;
import com.google.auth.oauth2.IdTokenProvider;
import com.google.auth.oauth2.MockTokenServerTransport;
import com.google.auth.oauth2.OAuth2Utils;
import com.google.auth.oauth2.ServiceAccountCredentials;
import cz.o2.proxima.internal.shaded.com.google.common.collect.ImmutableSet;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.Signature;
import java.security.SignatureException;
import java.time.Duration;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicBoolean;
import org.junit.Assert;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.JUnit4;

@RunWith(value=JUnit4.class)
public class ServiceAccountCredentialsTest
extends BaseSerializationTest {
    private static final String CLIENT_EMAIL = "36680232662-vrd7ji19qe3nelgchd0ah2csanun6bnr@developer.gserviceaccount.com";
    private static final String CLIENT_ID = "36680232662-vrd7ji19qe3nelgchd0ah2csanun6bnr.apps.googleusercontent.com";
    private static final String PRIVATE_KEY_ID = "d84a4fefcf50791d4a90f2d7af17469d6282df9d";
    static final String PRIVATE_KEY_PKCS8 = "-----BEGIN PRIVATE KEY-----\nMIICdgIBADANBgkqhkiG9w0BAQEFAASCAmAwggJcAgEAAoGBALX0PQoe1igW12ikv1bN/r9lN749y2ijmbc/mFHPyS3hNTyOCjDvBbXYbDhQJzWVUikh4mvGBA07qTj79Xc3yBDfKP2IeyYQIFe0t0zkd7R9Zdn98Y2rIQC47aAbDfubtkU1U72t4zL11kHvoa0/RuFZjncvlr42X7be7lYh4p3NAgMBAAECgYASk5wDw4Az2ZkmeuN6Fk/y9H+Lcb2pskJIXjrL533vrDWGOC48LrsThMQPv8cxBky8HFSEklPpkfTF95tpD43iVwJRB/GrCtGTw65IfJ4/tI09h6zGc4yqvIo1cHX/LQ+SxKLGyir/dQM925rGt/VojxY5ryJR7GLbCzxPnJm/oQJBANwOCO6D2hy1LQYJhXh7O+RLtA/tSnT1xyMQsGT+uUCMiKS2bSKx2wxo9k7h3OegNJIu1q6nZ6AbxDK8H3+d0dUCQQDTrPSXagBxzp8PecbaCHjzNRSQE2in81qYnrAFNB4o3DpHyMMY6s5ALLeHKscEWnqP8Ur6X4PvzZecCWU9BKAZAkAutLPknAuxSCsUOvUfS1i87ex77Ot+w6POp34pEX+UWb+u5iFn2cQacDTHLV1LtE80L8jVLSbrbrlH43H0DjU5AkEAgidhycxS86dxpEljnOMCw8CKoUBd5I880IUahEiUltk7OLJYS/Ts1wbn3kPOVX3wyJs8WBDtBkFrDHW2ezth2QJADj3e1YhMVdjJW5jqwlD/VNddGjgzyunmiZg0uOXsHXbytYmsA545S8KRQFaJKFXYYFo2kOjqOiC1T2cAzMDjCQ==\n-----END PRIVATE KEY-----\n";
    private static final String ACCESS_TOKEN = "1/MkSJoj1xsli0AccessToken_NKPY2";
    private static final Collection<String> SCOPES = Collections.singletonList("dummy.scope");
    private static final Collection<String> DEFAULT_SCOPES = Collections.singletonList("dummy.default.scope");
    private static final String USER = "user@example.com";
    private static final String PROJECT_ID = "project-id";
    private static final Collection<String> EMPTY_SCOPES = Collections.emptyList();
    private static final URI CALL_URI = URI.create("http://googleapis.com/testapi/v1/foo");
    private static final String JWT_AUDIENCE = "http://googleapis.com/";
    private static final HttpTransportFactory DUMMY_TRANSPORT_FACTORY = new GoogleCredentialsTest.MockTokenServerTransportFactory();
    public static final String DEFAULT_ID_TOKEN = "eyJhbGciOiJSUzI1NiIsImtpZCI6ImRmMzc1ODkwOGI3OTIyOTNhZDk3N2EwYjk5MWQ5OGE3N2Y0ZWVlY2QiLCJ0eXAiOiJKV1QifQ.eyJhdWQiOiJodHRwczovL2Zvby5iYXIiLCJhenAiOiIxMDIxMDE1NTA4MzQyMDA3MDg1NjgiLCJleHAiOjE1NjQ0NzUwNTEsImlhdCI6MTU2NDQ3MTQ1MSwiaXNzIjoiaHR0cHM6Ly9hY2NvdW50cy5nb29nbGUuY29tIiwic3ViIjoiMTAyMTAxNTUwODM0MjAwNzA4NTY4In0.redacted";
    private static final String QUOTA_PROJECT = "sample-quota-project-id";
    private static final int DEFAULT_LIFETIME_IN_SECONDS = 3600;
    private static final int INVALID_LIFETIME = 43210;
    private static final String JWT_ACCESS_PREFIX = "Bearer ";

    private ServiceAccountCredentials.Builder createDefaultBuilder() throws IOException {
        PrivateKey privateKey = ServiceAccountCredentials.privateKeyFromPkcs8((String)PRIVATE_KEY_PKCS8);
        return ServiceAccountCredentials.newBuilder().setClientId(CLIENT_ID).setClientEmail(CLIENT_EMAIL).setPrivateKey(privateKey).setPrivateKeyId(PRIVATE_KEY_ID).setScopes(SCOPES).setServiceAccountUser(USER).setProjectId(PROJECT_ID);
    }

    @Test
    public void setLifetime() throws IOException {
        ServiceAccountCredentials.Builder builder = this.createDefaultBuilder();
        Assert.assertEquals((long)3600L, (long)builder.getLifetime());
        Assert.assertEquals((long)3600L, (long)builder.build().getLifetime());
        builder.setLifetime(4000);
        Assert.assertEquals((long)4000L, (long)builder.getLifetime());
        Assert.assertEquals((long)4000L, (long)builder.build().getLifetime());
        builder.setLifetime(0);
        Assert.assertEquals((long)3600L, (long)builder.build().getLifetime());
    }

    @Test
    public void setLifetime_invalid_lifetime() throws IOException, IllegalStateException {
        try {
            this.createDefaultBuilder().setLifetime(43210).build();
            Assert.fail((String)String.format("Should throw exception with message containing '%s'", "lifetime must be less than or equal to 43200"));
        }
        catch (IllegalStateException expected) {
            Assert.assertTrue((boolean)expected.getMessage().contains("lifetime must be less than or equal to 43200"));
        }
    }

    @Test
    public void createWithCustomLifetime() throws IOException {
        ServiceAccountCredentials credentials = this.createDefaultBuilder().build();
        credentials = credentials.createWithCustomLifetime(4000);
        Assert.assertEquals((long)4000L, (long)credentials.getLifetime());
    }

    @Test
    public void createdScoped_clones() throws IOException {
        PrivateKey privateKey = ServiceAccountCredentials.privateKeyFromPkcs8((String)PRIVATE_KEY_PKCS8);
        ServiceAccountCredentials credentials = ServiceAccountCredentials.newBuilder().setClientId(CLIENT_ID).setClientEmail(CLIENT_EMAIL).setPrivateKey(privateKey).setPrivateKeyId(PRIVATE_KEY_ID).setScopes(SCOPES).setServiceAccountUser(USER).setProjectId(PROJECT_ID).build();
        List<String> newScopes = Arrays.asList("scope1", "scope2");
        ServiceAccountCredentials newCredentials = (ServiceAccountCredentials)credentials.createScoped(newScopes);
        Assert.assertEquals((Object)CLIENT_ID, (Object)newCredentials.getClientId());
        Assert.assertEquals((Object)CLIENT_EMAIL, (Object)newCredentials.getClientEmail());
        Assert.assertEquals((Object)privateKey, (Object)newCredentials.getPrivateKey());
        Assert.assertEquals((Object)PRIVATE_KEY_ID, (Object)newCredentials.getPrivateKeyId());
        Assert.assertArrayEquals((Object[])newScopes.toArray(), (Object[])newCredentials.getScopes().toArray());
        Assert.assertEquals((Object)USER, (Object)newCredentials.getServiceAccountUser());
        Assert.assertEquals((Object)PROJECT_ID, (Object)newCredentials.getProjectId());
        Assert.assertArrayEquals((Object[])SCOPES.toArray(), (Object[])credentials.getScopes().toArray());
    }

    @Test
    public void createdDelegated_clones() throws IOException {
        PrivateKey privateKey = ServiceAccountCredentials.privateKeyFromPkcs8((String)PRIVATE_KEY_PKCS8);
        ServiceAccountCredentials credentials = ServiceAccountCredentials.newBuilder().setClientId(CLIENT_ID).setClientEmail(CLIENT_EMAIL).setPrivateKey(privateKey).setPrivateKeyId(PRIVATE_KEY_ID).setScopes(SCOPES).setServiceAccountUser(USER).setProjectId(PROJECT_ID).setQuotaProjectId(QUOTA_PROJECT).build();
        String newServiceAccountUser = "stranger@other.org";
        ServiceAccountCredentials newCredentials = (ServiceAccountCredentials)credentials.createDelegated(newServiceAccountUser);
        Assert.assertEquals((Object)CLIENT_ID, (Object)newCredentials.getClientId());
        Assert.assertEquals((Object)CLIENT_EMAIL, (Object)newCredentials.getClientEmail());
        Assert.assertEquals((Object)privateKey, (Object)newCredentials.getPrivateKey());
        Assert.assertEquals((Object)PRIVATE_KEY_ID, (Object)newCredentials.getPrivateKeyId());
        Assert.assertArrayEquals((Object[])SCOPES.toArray(), (Object[])newCredentials.getScopes().toArray());
        Assert.assertEquals((Object)newServiceAccountUser, (Object)newCredentials.getServiceAccountUser());
        Assert.assertEquals((Object)PROJECT_ID, (Object)newCredentials.getProjectId());
        Assert.assertEquals((Object)QUOTA_PROJECT, (Object)newCredentials.getQuotaProjectId());
        Assert.assertEquals((Object)USER, (Object)credentials.getServiceAccountUser());
    }

    @Test
    public void createAssertion_correct() throws IOException {
        PrivateKey privateKey = ServiceAccountCredentials.privateKeyFromPkcs8((String)PRIVATE_KEY_PKCS8);
        List<String> scopes = Arrays.asList("scope1", "scope2");
        ServiceAccountCredentials credentials = ServiceAccountCredentials.newBuilder().setClientId(CLIENT_ID).setClientEmail(CLIENT_EMAIL).setPrivateKey(privateKey).setPrivateKeyId(PRIVATE_KEY_ID).setScopes(scopes).setServiceAccountUser(USER).setProjectId(PROJECT_ID).build();
        JsonFactory jsonFactory = OAuth2Utils.JSON_FACTORY;
        long currentTimeMillis = Clock.SYSTEM.currentTimeMillis();
        String assertion = credentials.createAssertion(jsonFactory, currentTimeMillis);
        JsonWebSignature signature = JsonWebSignature.parse((JsonFactory)jsonFactory, (String)assertion);
        JsonWebToken.Payload payload = signature.getPayload();
        Assert.assertEquals((Object)CLIENT_EMAIL, (Object)payload.getIssuer());
        Assert.assertEquals((Object)OAuth2Utils.TOKEN_SERVER_URI.toString(), (Object)payload.getAudience());
        Assert.assertEquals((long)(currentTimeMillis / 1000L), (long)payload.getIssuedAtTimeSeconds());
        Assert.assertEquals((long)(currentTimeMillis / 1000L + 3600L), (long)payload.getExpirationTimeSeconds());
        Assert.assertEquals((Object)USER, (Object)payload.getSubject());
        Assert.assertEquals((Object)Joiner.on((char)' ').join(scopes), (Object)payload.get((Object)"scope"));
    }

    @Test
    public void createAssertion_defaultScopes_correct() throws IOException {
        PrivateKey privateKey = ServiceAccountCredentials.privateKeyFromPkcs8((String)PRIVATE_KEY_PKCS8);
        List<String> scopes = Arrays.asList("scope1", "scope2");
        ServiceAccountCredentials.Builder builder = ServiceAccountCredentials.newBuilder().setClientId(CLIENT_ID).setClientEmail(CLIENT_EMAIL).setPrivateKey(privateKey).setPrivateKeyId(PRIVATE_KEY_ID).setScopes(null, scopes).setServiceAccountUser(USER).setProjectId(PROJECT_ID);
        Assert.assertEquals((long)2L, (long)builder.getDefaultScopes().size());
        ServiceAccountCredentials credentials = builder.build();
        JsonFactory jsonFactory = OAuth2Utils.JSON_FACTORY;
        long currentTimeMillis = Clock.SYSTEM.currentTimeMillis();
        String assertion = credentials.createAssertion(jsonFactory, currentTimeMillis);
        JsonWebSignature signature = JsonWebSignature.parse((JsonFactory)jsonFactory, (String)assertion);
        JsonWebToken.Payload payload = signature.getPayload();
        Assert.assertEquals((Object)CLIENT_EMAIL, (Object)payload.getIssuer());
        Assert.assertEquals((Object)OAuth2Utils.TOKEN_SERVER_URI.toString(), (Object)payload.getAudience());
        Assert.assertEquals((long)(currentTimeMillis / 1000L), (long)payload.getIssuedAtTimeSeconds());
        Assert.assertEquals((long)(currentTimeMillis / 1000L + 3600L), (long)payload.getExpirationTimeSeconds());
        Assert.assertEquals((Object)USER, (Object)payload.getSubject());
        Assert.assertEquals((Object)Joiner.on((char)' ').join(scopes), (Object)payload.get((Object)"scope"));
    }

    @Test
    public void createAssertion_custom_lifetime() throws IOException {
        ServiceAccountCredentials credentials = this.createDefaultBuilder().setLifetime(4000).build();
        JsonFactory jsonFactory = OAuth2Utils.JSON_FACTORY;
        long currentTimeMillis = Clock.SYSTEM.currentTimeMillis();
        String assertion = credentials.createAssertion(jsonFactory, currentTimeMillis);
        JsonWebSignature signature = JsonWebSignature.parse((JsonFactory)jsonFactory, (String)assertion);
        JsonWebToken.Payload payload = signature.getPayload();
        Assert.assertEquals((long)(currentTimeMillis / 1000L + 4000L), (long)payload.getExpirationTimeSeconds());
    }

    @Test
    public void createAssertionForIdToken_correct() throws IOException {
        PrivateKey privateKey = ServiceAccountCredentials.privateKeyFromPkcs8((String)PRIVATE_KEY_PKCS8);
        ServiceAccountCredentials credentials = ServiceAccountCredentials.newBuilder().setClientId(CLIENT_ID).setClientEmail(CLIENT_EMAIL).setPrivateKey(privateKey).setPrivateKeyId(PRIVATE_KEY_ID).setServiceAccountUser(USER).setProjectId(PROJECT_ID).build();
        JsonFactory jsonFactory = OAuth2Utils.JSON_FACTORY;
        long currentTimeMillis = Clock.SYSTEM.currentTimeMillis();
        String assertion = credentials.createAssertionForIdToken(jsonFactory, currentTimeMillis, null, "https://foo.com/bar");
        JsonWebSignature signature = JsonWebSignature.parse((JsonFactory)jsonFactory, (String)assertion);
        JsonWebToken.Payload payload = signature.getPayload();
        Assert.assertEquals((Object)CLIENT_EMAIL, (Object)payload.getIssuer());
        Assert.assertEquals((Object)"https://foo.com/bar", (Object)((String)payload.getUnknownKeys().get("target_audience")));
        Assert.assertEquals((long)(currentTimeMillis / 1000L), (long)payload.getIssuedAtTimeSeconds());
        Assert.assertEquals((long)(currentTimeMillis / 1000L + 3600L), (long)payload.getExpirationTimeSeconds());
        Assert.assertEquals((Object)USER, (Object)payload.getSubject());
    }

    @Test
    public void createAssertionForIdToken_custom_lifetime() throws IOException {
        ServiceAccountCredentials credentials = this.createDefaultBuilder().setLifetime(4000).build();
        JsonFactory jsonFactory = OAuth2Utils.JSON_FACTORY;
        long currentTimeMillis = Clock.SYSTEM.currentTimeMillis();
        String assertion = credentials.createAssertionForIdToken(jsonFactory, currentTimeMillis, null, "https://foo.com/bar");
        JsonWebSignature signature = JsonWebSignature.parse((JsonFactory)jsonFactory, (String)assertion);
        JsonWebToken.Payload payload = signature.getPayload();
        Assert.assertEquals((long)(currentTimeMillis / 1000L + 4000L), (long)payload.getExpirationTimeSeconds());
    }

    @Test
    public void createAssertionForIdToken_incorrect() throws IOException {
        PrivateKey privateKey = ServiceAccountCredentials.privateKeyFromPkcs8((String)PRIVATE_KEY_PKCS8);
        ServiceAccountCredentials credentials = ServiceAccountCredentials.newBuilder().setClientId(CLIENT_ID).setClientEmail(CLIENT_EMAIL).setPrivateKey(privateKey).setPrivateKeyId(PRIVATE_KEY_ID).setServiceAccountUser(USER).setProjectId(PROJECT_ID).build();
        JsonFactory jsonFactory = OAuth2Utils.JSON_FACTORY;
        long currentTimeMillis = Clock.SYSTEM.currentTimeMillis();
        String assertion = credentials.createAssertionForIdToken(jsonFactory, currentTimeMillis, null, "https://foo.com/bar");
        JsonWebSignature signature = JsonWebSignature.parse((JsonFactory)jsonFactory, (String)assertion);
        JsonWebToken.Payload payload = signature.getPayload();
        Assert.assertEquals((Object)CLIENT_EMAIL, (Object)payload.getIssuer());
        Assert.assertNotEquals((Object)"https://bar.com/foo", (Object)((String)payload.getUnknownKeys().get("target_audience")));
        Assert.assertEquals((long)(currentTimeMillis / 1000L), (long)payload.getIssuedAtTimeSeconds());
        Assert.assertEquals((long)(currentTimeMillis / 1000L + 3600L), (long)payload.getExpirationTimeSeconds());
        Assert.assertEquals((Object)USER, (Object)payload.getSubject());
    }

    @Test
    public void createdScoped_enablesAccessTokens() throws IOException {
        GoogleCredentialsTest.MockTokenServerTransportFactory transportFactory = new GoogleCredentialsTest.MockTokenServerTransportFactory();
        transportFactory.transport.addServiceAccount(CLIENT_EMAIL, ACCESS_TOKEN);
        ServiceAccountCredentials credentials = ServiceAccountCredentials.fromPkcs8((String)CLIENT_ID, (String)CLIENT_EMAIL, (String)PRIVATE_KEY_PKCS8, (String)PRIVATE_KEY_ID, null, (HttpTransportFactory)transportFactory, null);
        try {
            credentials.getRequestMetadata(null);
            Assert.fail((String)"Should not be able to get token without scopes");
        }
        catch (IOException e) {
            Assert.assertTrue((String)"expected to fail with exception", (boolean)e.getMessage().contains("Scopes and uri are not configured for service account"));
        }
        GoogleCredentials scopedCredentials = credentials.createScoped(SCOPES);
        Map metadata = scopedCredentials.getRequestMetadata(CALL_URI);
        TestUtils.assertContainsBearerToken(metadata, ACCESS_TOKEN);
    }

    @Test
    public void createdScoped_defaultScopes() throws IOException {
        URI TOKEN_SERVER = URI.create("https://foo.com/bar");
        GoogleCredentialsTest.MockTokenServerTransportFactory transportFactory = new GoogleCredentialsTest.MockTokenServerTransportFactory();
        transportFactory.transport.addServiceAccount(CLIENT_EMAIL, ACCESS_TOKEN);
        transportFactory.transport.setTokenServerUri(TOKEN_SERVER);
        ServiceAccountCredentials credentials = ServiceAccountCredentials.fromPkcs8((String)CLIENT_ID, (String)CLIENT_EMAIL, (String)PRIVATE_KEY_PKCS8, (String)PRIVATE_KEY_ID, SCOPES, DEFAULT_SCOPES);
        Assert.assertEquals((long)1L, (long)credentials.getDefaultScopes().size());
        Assert.assertEquals((Object)"dummy.default.scope", (Object)credentials.getDefaultScopes().toArray()[0]);
        credentials = ServiceAccountCredentials.fromPkcs8((String)CLIENT_ID, (String)CLIENT_EMAIL, (String)PRIVATE_KEY_PKCS8, (String)PRIVATE_KEY_ID, SCOPES, DEFAULT_SCOPES, (HttpTransportFactory)transportFactory, (URI)TOKEN_SERVER);
        Assert.assertEquals((long)1L, (long)credentials.getDefaultScopes().size());
        Assert.assertEquals((Object)"dummy.default.scope", (Object)credentials.getDefaultScopes().toArray()[0]);
        credentials = ServiceAccountCredentials.fromPkcs8((String)CLIENT_ID, (String)CLIENT_EMAIL, (String)PRIVATE_KEY_PKCS8, (String)PRIVATE_KEY_ID, SCOPES, DEFAULT_SCOPES, (HttpTransportFactory)transportFactory, (URI)TOKEN_SERVER, (String)"service_account_user");
        Assert.assertEquals((long)1L, (long)credentials.getDefaultScopes().size());
        Assert.assertEquals((Object)"dummy.default.scope", (Object)credentials.getDefaultScopes().toArray()[0]);
    }

    @Test
    public void createScopedRequired_emptyScopes() throws IOException {
        ServiceAccountCredentials credentials = ServiceAccountCredentials.fromPkcs8((String)CLIENT_ID, (String)CLIENT_EMAIL, (String)PRIVATE_KEY_PKCS8, (String)PRIVATE_KEY_ID, EMPTY_SCOPES);
        Assert.assertTrue((boolean)credentials.createScopedRequired());
    }

    @Test
    public void createScopedRequired_nonEmptyScopes() throws IOException {
        ServiceAccountCredentials credentials = ServiceAccountCredentials.fromPkcs8((String)CLIENT_ID, (String)CLIENT_EMAIL, (String)PRIVATE_KEY_PKCS8, (String)PRIVATE_KEY_ID, SCOPES);
        Assert.assertFalse((boolean)credentials.createScopedRequired());
    }

    @Test
    public void createScopedRequired_nonEmptyDefaultScopes() throws IOException {
        ServiceAccountCredentials credentials = ServiceAccountCredentials.fromPkcs8((String)CLIENT_ID, (String)CLIENT_EMAIL, (String)PRIVATE_KEY_PKCS8, (String)PRIVATE_KEY_ID, null, SCOPES);
        Assert.assertFalse((boolean)credentials.createScopedRequired());
    }

    @Test
    public void fromJSON_getProjectId() throws IOException {
        GoogleCredentialsTest.MockTokenServerTransportFactory transportFactory = new GoogleCredentialsTest.MockTokenServerTransportFactory();
        transportFactory.transport.addServiceAccount(CLIENT_EMAIL, ACCESS_TOKEN);
        GenericJson json = ServiceAccountCredentialsTest.writeServiceAccountJson(CLIENT_ID, CLIENT_EMAIL, PRIVATE_KEY_PKCS8, PRIVATE_KEY_ID, PROJECT_ID, null);
        ServiceAccountCredentials credentials = ServiceAccountCredentials.fromJson((Map)json, (HttpTransportFactory)transportFactory);
        Assert.assertEquals((Object)PROJECT_ID, (Object)credentials.getProjectId());
    }

    @Test
    public void fromJSON_getProjectIdNull() throws IOException {
        GoogleCredentialsTest.MockTokenServerTransportFactory transportFactory = new GoogleCredentialsTest.MockTokenServerTransportFactory();
        transportFactory.transport.addServiceAccount(CLIENT_EMAIL, ACCESS_TOKEN);
        GenericJson json = ServiceAccountCredentialsTest.writeServiceAccountJson(CLIENT_ID, CLIENT_EMAIL, PRIVATE_KEY_PKCS8, PRIVATE_KEY_ID, null, null);
        ServiceAccountCredentials credentials = ServiceAccountCredentials.fromJson((Map)json, (HttpTransportFactory)transportFactory);
        Assert.assertNull((Object)credentials.getProjectId());
    }

    @Test
    public void fromJSON_hasAccessToken() throws IOException {
        GoogleCredentialsTest.MockTokenServerTransportFactory transportFactory = new GoogleCredentialsTest.MockTokenServerTransportFactory();
        transportFactory.transport.addServiceAccount(CLIENT_EMAIL, ACCESS_TOKEN);
        GenericJson json = ServiceAccountCredentialsTest.writeServiceAccountJson(CLIENT_ID, CLIENT_EMAIL, PRIVATE_KEY_PKCS8, PRIVATE_KEY_ID, PROJECT_ID, null);
        ServiceAccountCredentials credentials = ServiceAccountCredentials.fromJson((Map)json, (HttpTransportFactory)transportFactory);
        credentials = credentials.createScoped(SCOPES);
        Map metadata = credentials.getRequestMetadata(CALL_URI);
        TestUtils.assertContainsBearerToken(metadata, ACCESS_TOKEN);
    }

    @Test
    public void fromJSON_tokenServerUri() throws IOException {
        String tokenServerUri = "https://foo.com/bar";
        GoogleCredentialsTest.MockTokenServerTransportFactory transportFactory = new GoogleCredentialsTest.MockTokenServerTransportFactory();
        transportFactory.transport.addServiceAccount(CLIENT_EMAIL, ACCESS_TOKEN);
        GenericJson json = ServiceAccountCredentialsTest.writeServiceAccountJson(CLIENT_ID, CLIENT_EMAIL, PRIVATE_KEY_PKCS8, PRIVATE_KEY_ID, PROJECT_ID, null);
        json.put("token_uri", (Object)"https://foo.com/bar");
        ServiceAccountCredentials credentials = ServiceAccountCredentials.fromJson((Map)json, (HttpTransportFactory)transportFactory);
        Assert.assertEquals((Object)URI.create("https://foo.com/bar"), (Object)credentials.getTokenServerUri());
    }

    @Test
    public void fromJson_hasQuotaProjectId() throws IOException {
        GoogleCredentialsTest.MockTokenServerTransportFactory transportFactory = new GoogleCredentialsTest.MockTokenServerTransportFactory();
        transportFactory.transport.addServiceAccount(CLIENT_EMAIL, ACCESS_TOKEN);
        GenericJson json = ServiceAccountCredentialsTest.writeServiceAccountJson(CLIENT_ID, CLIENT_EMAIL, PRIVATE_KEY_PKCS8, PRIVATE_KEY_ID, PROJECT_ID, QUOTA_PROJECT);
        ServiceAccountCredentials credentials = ServiceAccountCredentials.fromJson((Map)json, (HttpTransportFactory)transportFactory);
        credentials = credentials.createScoped(SCOPES);
        Map metadata = credentials.getRequestMetadata(CALL_URI);
        Assert.assertTrue((boolean)metadata.containsKey("x-goog-user-project"));
        Assert.assertEquals(metadata.get("x-goog-user-project"), Collections.singletonList(QUOTA_PROJECT));
    }

    @Test
    public void getRequestMetadata_hasAccessToken() throws IOException {
        GoogleCredentialsTest.MockTokenServerTransportFactory transportFactory = new GoogleCredentialsTest.MockTokenServerTransportFactory();
        transportFactory.transport.addServiceAccount(CLIENT_EMAIL, ACCESS_TOKEN);
        ServiceAccountCredentials credentials = ServiceAccountCredentials.fromPkcs8((String)CLIENT_ID, (String)CLIENT_EMAIL, (String)PRIVATE_KEY_PKCS8, (String)PRIVATE_KEY_ID, SCOPES, (HttpTransportFactory)transportFactory, null);
        Map metadata = credentials.getRequestMetadata(CALL_URI);
        TestUtils.assertContainsBearerToken(metadata, ACCESS_TOKEN);
    }

    @Test
    public void getRequestMetadata_customTokenServer_hasAccessToken() throws IOException {
        URI TOKEN_SERVER = URI.create("https://foo.com/bar");
        GoogleCredentialsTest.MockTokenServerTransportFactory transportFactory = new GoogleCredentialsTest.MockTokenServerTransportFactory();
        transportFactory.transport.addServiceAccount(CLIENT_EMAIL, ACCESS_TOKEN);
        transportFactory.transport.setTokenServerUri(TOKEN_SERVER);
        ServiceAccountCredentials credentials = ServiceAccountCredentials.fromPkcs8((String)CLIENT_ID, (String)CLIENT_EMAIL, (String)PRIVATE_KEY_PKCS8, (String)PRIVATE_KEY_ID, SCOPES, (HttpTransportFactory)transportFactory, (URI)TOKEN_SERVER);
        Map metadata = credentials.getRequestMetadata(CALL_URI);
        TestUtils.assertContainsBearerToken(metadata, ACCESS_TOKEN);
    }

    @Test
    public void refreshAccessToken_refreshesToken() throws IOException {
        String accessToken1 = ACCESS_TOKEN;
        String accessToken2 = "2/MkSJoj1xsli0AccessToken_NKPY2";
        GoogleCredentialsTest.MockTokenServerTransportFactory transportFactory = new GoogleCredentialsTest.MockTokenServerTransportFactory();
        MockTokenServerTransport transport = transportFactory.transport;
        ServiceAccountCredentials credentials = ServiceAccountCredentials.fromPkcs8((String)CLIENT_ID, (String)CLIENT_EMAIL, (String)PRIVATE_KEY_PKCS8, (String)PRIVATE_KEY_ID, SCOPES, (HttpTransportFactory)transportFactory, null);
        transport.addServiceAccount(CLIENT_EMAIL, ACCESS_TOKEN);
        TestUtils.assertContainsBearerToken(credentials.getRequestMetadata(CALL_URI), ACCESS_TOKEN);
        transport.addServiceAccount(CLIENT_EMAIL, "2/MkSJoj1xsli0AccessToken_NKPY2");
        credentials.refresh();
        TestUtils.assertContainsBearerToken(credentials.getRequestMetadata(CALL_URI), "2/MkSJoj1xsli0AccessToken_NKPY2");
    }

    @Test
    public void refreshAccessToken_tokenExpiry() throws IOException {
        String tokenString = ACCESS_TOKEN;
        GoogleCredentialsTest.MockTokenServerTransportFactory transportFactory = new GoogleCredentialsTest.MockTokenServerTransportFactory();
        MockTokenServerTransport transport = transportFactory.transport;
        ServiceAccountCredentials credentials = ServiceAccountCredentials.fromPkcs8((String)CLIENT_ID, (String)CLIENT_EMAIL, (String)PRIVATE_KEY_PKCS8, (String)PRIVATE_KEY_ID, SCOPES, (HttpTransportFactory)transportFactory, null);
        credentials.clock = new FixedClock(0L);
        transport.addServiceAccount(CLIENT_EMAIL, ACCESS_TOKEN);
        AccessToken accessToken = credentials.refreshAccessToken();
        Assert.assertEquals((Object)ACCESS_TOKEN, (Object)accessToken.getTokenValue());
        Assert.assertEquals((long)3600000L, (long)accessToken.getExpirationTimeMillis());
        transport.setExpiresInSeconds(3600000);
        accessToken = credentials.refreshAccessToken();
        Assert.assertEquals((Object)ACCESS_TOKEN, (Object)accessToken.getTokenValue());
        Assert.assertEquals((long)3600000000L, (long)accessToken.getExpirationTimeMillis());
    }

    @Test
    public void refreshAccessToken_IOException_Retry() throws IOException {
        String accessToken1 = ACCESS_TOKEN;
        String accessToken2 = "2/MkSJoj1xsli0AccessToken_NKPY2";
        GoogleCredentialsTest.MockTokenServerTransportFactory transportFactory = new GoogleCredentialsTest.MockTokenServerTransportFactory();
        MockTokenServerTransport transport = transportFactory.transport;
        ServiceAccountCredentials credentials = ServiceAccountCredentials.fromPkcs8((String)CLIENT_ID, (String)CLIENT_EMAIL, (String)PRIVATE_KEY_PKCS8, (String)PRIVATE_KEY_ID, SCOPES, (HttpTransportFactory)transportFactory, null);
        transport.addServiceAccount(CLIENT_EMAIL, ACCESS_TOKEN);
        TestUtils.assertContainsBearerToken(credentials.getRequestMetadata(CALL_URI), ACCESS_TOKEN);
        transport.addResponseErrorSequence(new IOException());
        transport.addServiceAccount(CLIENT_EMAIL, "2/MkSJoj1xsli0AccessToken_NKPY2");
        credentials.refresh();
        TestUtils.assertContainsBearerToken(credentials.getRequestMetadata(CALL_URI), "2/MkSJoj1xsli0AccessToken_NKPY2");
    }

    @Test
    public void refreshAccessToken_retriesServerErrors() throws IOException {
        String accessToken1 = ACCESS_TOKEN;
        String accessToken2 = "2/MkSJoj1xsli0AccessToken_NKPY2";
        GoogleCredentialsTest.MockTokenServerTransportFactory transportFactory = new GoogleCredentialsTest.MockTokenServerTransportFactory();
        MockTokenServerTransport transport = transportFactory.transport;
        ServiceAccountCredentials credentials = ServiceAccountCredentials.fromPkcs8((String)CLIENT_ID, (String)CLIENT_EMAIL, (String)PRIVATE_KEY_PKCS8, (String)PRIVATE_KEY_ID, SCOPES, (HttpTransportFactory)transportFactory, null);
        transport.addServiceAccount(CLIENT_EMAIL, ACCESS_TOKEN);
        TestUtils.assertContainsBearerToken(credentials.getRequestMetadata(CALL_URI), ACCESS_TOKEN);
        MockLowLevelHttpResponse response500 = new MockLowLevelHttpResponse().setStatusCode(500);
        MockLowLevelHttpResponse response503 = new MockLowLevelHttpResponse().setStatusCode(503);
        transport.addResponseSequence(new LowLevelHttpResponse[]{response500, response503});
        transport.addServiceAccount(CLIENT_EMAIL, "2/MkSJoj1xsli0AccessToken_NKPY2");
        credentials.refresh();
        TestUtils.assertContainsBearerToken(credentials.getRequestMetadata(CALL_URI), "2/MkSJoj1xsli0AccessToken_NKPY2");
    }

    @Test
    public void refreshAccessToken_retriesTimeoutAndThrottled() throws IOException {
        String accessToken1 = ACCESS_TOKEN;
        String accessToken2 = "2/MkSJoj1xsli0AccessToken_NKPY2";
        GoogleCredentialsTest.MockTokenServerTransportFactory transportFactory = new GoogleCredentialsTest.MockTokenServerTransportFactory();
        MockTokenServerTransport transport = transportFactory.transport;
        ServiceAccountCredentials credentials = ServiceAccountCredentials.fromPkcs8((String)CLIENT_ID, (String)CLIENT_EMAIL, (String)PRIVATE_KEY_PKCS8, (String)PRIVATE_KEY_ID, SCOPES, (HttpTransportFactory)transportFactory, null);
        transport.addServiceAccount(CLIENT_EMAIL, ACCESS_TOKEN);
        TestUtils.assertContainsBearerToken(credentials.getRequestMetadata(CALL_URI), ACCESS_TOKEN);
        MockLowLevelHttpResponse response408 = new MockLowLevelHttpResponse().setStatusCode(408);
        MockLowLevelHttpResponse response429 = new MockLowLevelHttpResponse().setStatusCode(429);
        transport.addResponseSequence(new LowLevelHttpResponse[]{response408, response429});
        transport.addServiceAccount(CLIENT_EMAIL, "2/MkSJoj1xsli0AccessToken_NKPY2");
        credentials.refresh();
        TestUtils.assertContainsBearerToken(credentials.getRequestMetadata(CALL_URI), "2/MkSJoj1xsli0AccessToken_NKPY2");
    }

    @Test
    public void refreshAccessToken_defaultRetriesDisabled() throws IOException {
        String accessToken1 = ACCESS_TOKEN;
        String accessToken2 = "2/MkSJoj1xsli0AccessToken_NKPY2";
        GoogleCredentialsTest.MockTokenServerTransportFactory transportFactory = new GoogleCredentialsTest.MockTokenServerTransportFactory();
        MockTokenServerTransport transport = transportFactory.transport;
        ServiceAccountCredentials credentials = ServiceAccountCredentials.fromPkcs8((String)CLIENT_ID, (String)CLIENT_EMAIL, (String)PRIVATE_KEY_PKCS8, (String)PRIVATE_KEY_ID, SCOPES, (HttpTransportFactory)transportFactory, null).createWithCustomRetryStrategy(false);
        transport.addServiceAccount(CLIENT_EMAIL, ACCESS_TOKEN);
        TestUtils.assertContainsBearerToken(credentials.getRequestMetadata(CALL_URI), ACCESS_TOKEN);
        MockLowLevelHttpResponse response408 = new MockLowLevelHttpResponse().setStatusCode(408);
        MockLowLevelHttpResponse response429 = new MockLowLevelHttpResponse().setStatusCode(429);
        transport.addServiceAccount(CLIENT_EMAIL, "2/MkSJoj1xsli0AccessToken_NKPY2");
        try {
            transport.addResponseSequence(new LowLevelHttpResponse[]{response408, response429});
            credentials.refresh();
            Assert.fail((String)"Should not be able to use credential without exception.");
        }
        catch (GoogleAuthException ex) {
            Assert.assertTrue((boolean)ex.getMessage().contains("Error getting access token for service account: 408"));
            Assert.assertTrue((boolean)ex.isRetryable());
            Assert.assertEquals((long)3L, (long)ex.getRetryCount());
        }
    }

    @Test
    public void refreshAccessToken_maxRetries_maxDelay() throws IOException {
        String accessToken1 = ACCESS_TOKEN;
        GoogleCredentialsTest.MockTokenServerTransportFactory transportFactory = new GoogleCredentialsTest.MockTokenServerTransportFactory();
        MockTokenServerTransport transport = transportFactory.transport;
        ServiceAccountCredentials credentials = ServiceAccountCredentials.fromPkcs8((String)CLIENT_ID, (String)CLIENT_EMAIL, (String)PRIVATE_KEY_PKCS8, (String)PRIVATE_KEY_ID, SCOPES, (HttpTransportFactory)transportFactory, null);
        transport.addServiceAccount(CLIENT_EMAIL, ACCESS_TOKEN);
        TestUtils.assertContainsBearerToken(credentials.getRequestMetadata(CALL_URI), ACCESS_TOKEN);
        MockLowLevelHttpResponse response408 = new MockLowLevelHttpResponse().setStatusCode(408);
        MockLowLevelHttpResponse response429 = new MockLowLevelHttpResponse().setStatusCode(429);
        MockLowLevelHttpResponse response500 = new MockLowLevelHttpResponse().setStatusCode(500);
        MockLowLevelHttpResponse response503 = new MockLowLevelHttpResponse().setStatusCode(503);
        Instant start = Instant.now();
        try {
            transport.addResponseSequence(new LowLevelHttpResponse[]{response408, response429, response500, response503});
            credentials.refresh();
            Assert.fail((String)"Should not be able to use credential without exception.");
        }
        catch (GoogleAuthException ex) {
            Instant finish = Instant.now();
            long timeElapsed = Duration.between(start, finish).toMillis();
            Assert.assertTrue((timeElapsed > 5500L && timeElapsed < 10000L ? 1 : 0) != 0);
            Assert.assertTrue((boolean)ex.getMessage().contains("Error getting access token for service account: 503"));
            Assert.assertTrue((boolean)ex.isRetryable());
            Assert.assertEquals((long)3L, (long)ex.getRetryCount());
            Assert.assertTrue((boolean)(ex.getCause() instanceof HttpResponseException));
        }
    }

    @Test
    public void refreshAccessToken_RequestFailure_retried() throws IOException {
        String accessToken1 = ACCESS_TOKEN;
        GoogleCredentialsTest.MockTokenServerTransportFactory transportFactory = new GoogleCredentialsTest.MockTokenServerTransportFactory();
        MockTokenServerTransport transport = transportFactory.transport;
        ServiceAccountCredentials credentials = ServiceAccountCredentials.fromPkcs8((String)CLIENT_ID, (String)CLIENT_EMAIL, (String)PRIVATE_KEY_PKCS8, (String)PRIVATE_KEY_ID, SCOPES, (HttpTransportFactory)transportFactory, null);
        transport.addServiceAccount(CLIENT_EMAIL, ACCESS_TOKEN);
        TestUtils.assertContainsBearerToken(credentials.getRequestMetadata(CALL_URI), ACCESS_TOKEN);
        IOException error = new IOException("Invalid grant: Account not found");
        MockLowLevelHttpResponse response503 = new MockLowLevelHttpResponse().setStatusCode(503);
        Instant start = Instant.now();
        try {
            transport.addResponseSequence(new LowLevelHttpResponse[]{response503});
            transport.addResponseErrorSequence(error, error, error);
            credentials.refresh();
            Assert.fail((String)"Should not be able to use credential without exception.");
        }
        catch (GoogleAuthException ex) {
            Instant finish = Instant.now();
            long timeElapsed = Duration.between(start, finish).toMillis();
            Assert.assertTrue((timeElapsed > 5500L && timeElapsed < 10000L ? 1 : 0) != 0);
            Assert.assertTrue((boolean)ex.getMessage().contains("Error getting access token for service account: Invalid grant"));
            Assert.assertTrue((boolean)ex.isRetryable());
            Assert.assertEquals((long)3L, (long)ex.getRetryCount());
            Assert.assertTrue((boolean)(ex.getCause() instanceof IOException));
        }
    }

    @Test
    public void refreshAccessToken_4xx_5xx_NonRetryableFails() throws IOException {
        String accessToken1 = ACCESS_TOKEN;
        String accessToken2 = "2/MkSJoj1xsli0AccessToken_NKPY2";
        GoogleCredentialsTest.MockTokenServerTransportFactory transportFactory = new GoogleCredentialsTest.MockTokenServerTransportFactory();
        MockTokenServerTransport transport = transportFactory.transport;
        ServiceAccountCredentials credentials = ServiceAccountCredentials.fromPkcs8((String)CLIENT_ID, (String)CLIENT_EMAIL, (String)PRIVATE_KEY_PKCS8, (String)PRIVATE_KEY_ID, SCOPES, (HttpTransportFactory)transportFactory, null);
        transport.addServiceAccount(CLIENT_EMAIL, ACCESS_TOKEN);
        TestUtils.assertContainsBearerToken(credentials.getRequestMetadata(CALL_URI), ACCESS_TOKEN);
        for (int status = 400; status < 600; ++status) {
            if (OAuth2Utils.TOKEN_ENDPOINT_RETRYABLE_STATUS_CODES.contains(status)) continue;
            MockLowLevelHttpResponse mockResponse = new MockLowLevelHttpResponse().setStatusCode(status);
            try {
                transport.addResponseSequence(new LowLevelHttpResponse[]{mockResponse});
                transport.addServiceAccount(CLIENT_EMAIL, "2/MkSJoj1xsli0AccessToken_NKPY2");
                credentials.refresh();
                Assert.fail((String)"Should not be able to use credential without exception.");
                continue;
            }
            catch (GoogleAuthException ex) {
                Assert.assertFalse((boolean)ex.isRetryable());
                Assert.assertEquals((long)3L, (long)ex.getRetryCount());
            }
        }
    }

    @Test
    public void idTokenWithAudience_correct() throws IOException {
        String accessToken1 = ACCESS_TOKEN;
        GoogleCredentialsTest.MockTokenServerTransportFactory transportFactory = new GoogleCredentialsTest.MockTokenServerTransportFactory();
        MockTokenServerTransport transport = transportFactory.transport;
        ServiceAccountCredentials credentials = ServiceAccountCredentials.fromPkcs8((String)CLIENT_ID, (String)CLIENT_EMAIL, (String)PRIVATE_KEY_PKCS8, (String)PRIVATE_KEY_ID, SCOPES, (HttpTransportFactory)transportFactory, null);
        transport.addServiceAccount(CLIENT_EMAIL, accessToken1);
        TestUtils.assertContainsBearerToken(credentials.getRequestMetadata(CALL_URI), accessToken1);
        String targetAudience = "https://foo.bar";
        IdTokenCredentials tokenCredential = IdTokenCredentials.newBuilder().setIdTokenProvider((IdTokenProvider)credentials).setTargetAudience(targetAudience).build();
        tokenCredential.refresh();
        Assert.assertEquals((Object)DEFAULT_ID_TOKEN, (Object)tokenCredential.getAccessToken().getTokenValue());
        Assert.assertEquals((Object)DEFAULT_ID_TOKEN, (Object)tokenCredential.getIdToken().getTokenValue());
        Assert.assertEquals((Object)targetAudience, (Object)((String)tokenCredential.getIdToken().getJsonWebSignature().getPayload().getAudience()));
    }

    @Test
    public void idTokenWithAudience_incorrect() throws IOException {
        String accessToken1 = ACCESS_TOKEN;
        GoogleCredentialsTest.MockTokenServerTransportFactory transportFactory = new GoogleCredentialsTest.MockTokenServerTransportFactory();
        MockTokenServerTransport transport = transportFactory.transport;
        ServiceAccountCredentials credentials = ServiceAccountCredentials.fromPkcs8((String)CLIENT_ID, (String)CLIENT_EMAIL, (String)PRIVATE_KEY_PKCS8, (String)PRIVATE_KEY_ID, SCOPES, (HttpTransportFactory)transportFactory, null);
        transport.addServiceAccount(CLIENT_EMAIL, accessToken1);
        TestUtils.assertContainsBearerToken(credentials.getRequestMetadata(CALL_URI), accessToken1);
        String targetAudience = "https://bar";
        IdTokenCredentials tokenCredential = IdTokenCredentials.newBuilder().setIdTokenProvider((IdTokenProvider)credentials).setTargetAudience(targetAudience).build();
        tokenCredential.refresh();
        Assert.assertNotEquals((Object)targetAudience, (Object)((String)tokenCredential.getIdToken().getJsonWebSignature().getPayload().getAudience()));
    }

    @Test
    public void getScopes_nullReturnsEmpty() throws IOException {
        ServiceAccountCredentials credentials = ServiceAccountCredentials.fromPkcs8((String)CLIENT_ID, (String)CLIENT_EMAIL, (String)PRIVATE_KEY_PKCS8, (String)PRIVATE_KEY_ID, null);
        Collection scopes = credentials.getScopes();
        Assert.assertNotNull((Object)scopes);
        Assert.assertTrue((boolean)scopes.isEmpty());
    }

    @Test
    public void getAccount_sameAs() throws IOException {
        ServiceAccountCredentials credentials = ServiceAccountCredentials.fromPkcs8((String)CLIENT_ID, (String)CLIENT_EMAIL, (String)PRIVATE_KEY_PKCS8, (String)PRIVATE_KEY_ID, null);
        Assert.assertEquals((Object)CLIENT_EMAIL, (Object)credentials.getAccount());
    }

    @Test
    public void sign_sameAs() throws IOException, NoSuchAlgorithmException, InvalidKeyException, SignatureException {
        byte[] toSign = new byte[]{13, 14, 10, 13};
        ServiceAccountCredentials credentials = ServiceAccountCredentials.fromPkcs8((String)CLIENT_ID, (String)CLIENT_EMAIL, (String)PRIVATE_KEY_PKCS8, (String)PRIVATE_KEY_ID, null);
        byte[] signedBytes = credentials.sign(toSign);
        Signature signature = Signature.getInstance("SHA256withRSA");
        signature.initSign(credentials.getPrivateKey());
        signature.update(toSign);
        Assert.assertArrayEquals((byte[])signature.sign(), (byte[])signedBytes);
    }

    @Test
    public void equals_true() throws IOException {
        URI tokenServer = URI.create("https://foo.com/bar");
        GoogleCredentialsTest.MockTokenServerTransportFactory transportFactory = new GoogleCredentialsTest.MockTokenServerTransportFactory();
        ServiceAccountCredentials credentials = ServiceAccountCredentials.fromPkcs8((String)CLIENT_ID, (String)CLIENT_EMAIL, (String)PRIVATE_KEY_PKCS8, (String)PRIVATE_KEY_ID, SCOPES, (HttpTransportFactory)transportFactory, (URI)tokenServer);
        ServiceAccountCredentials otherCredentials = ServiceAccountCredentials.fromPkcs8((String)CLIENT_ID, (String)CLIENT_EMAIL, (String)PRIVATE_KEY_PKCS8, (String)PRIVATE_KEY_ID, SCOPES, (HttpTransportFactory)transportFactory, (URI)tokenServer);
        Assert.assertTrue((boolean)credentials.equals((Object)otherCredentials));
        Assert.assertTrue((boolean)otherCredentials.equals((Object)credentials));
    }

    @Test
    public void equals_false_clientId() throws IOException {
        URI tokenServer1 = URI.create("https://foo1.com/bar");
        GoogleCredentialsTest.MockTokenServerTransportFactory serverTransportFactory = new GoogleCredentialsTest.MockTokenServerTransportFactory();
        ServiceAccountCredentials credentials = ServiceAccountCredentials.fromPkcs8((String)CLIENT_ID, (String)CLIENT_EMAIL, (String)PRIVATE_KEY_PKCS8, (String)PRIVATE_KEY_ID, SCOPES, (HttpTransportFactory)serverTransportFactory, (URI)tokenServer1);
        ServiceAccountCredentials otherCredentials = ServiceAccountCredentials.fromPkcs8((String)"otherClientId", (String)CLIENT_EMAIL, (String)PRIVATE_KEY_PKCS8, (String)PRIVATE_KEY_ID, SCOPES, (HttpTransportFactory)serverTransportFactory, (URI)tokenServer1);
        Assert.assertFalse((boolean)credentials.equals((Object)otherCredentials));
        Assert.assertFalse((boolean)otherCredentials.equals((Object)credentials));
    }

    @Test
    public void equals_false_email() throws IOException {
        URI tokenServer1 = URI.create("https://foo1.com/bar");
        GoogleCredentialsTest.MockTokenServerTransportFactory serverTransportFactory = new GoogleCredentialsTest.MockTokenServerTransportFactory();
        ServiceAccountCredentials credentials = ServiceAccountCredentials.fromPkcs8((String)CLIENT_ID, (String)CLIENT_EMAIL, (String)PRIVATE_KEY_PKCS8, (String)PRIVATE_KEY_ID, SCOPES, (HttpTransportFactory)serverTransportFactory, (URI)tokenServer1);
        ServiceAccountCredentials otherCredentials = ServiceAccountCredentials.fromPkcs8((String)CLIENT_ID, (String)"otherEmail", (String)PRIVATE_KEY_PKCS8, (String)PRIVATE_KEY_ID, SCOPES, (HttpTransportFactory)serverTransportFactory, (URI)tokenServer1);
        Assert.assertFalse((boolean)credentials.equals((Object)otherCredentials));
        Assert.assertFalse((boolean)otherCredentials.equals((Object)credentials));
    }

    @Test
    public void equals_false_keyId() throws IOException {
        URI tokenServer1 = URI.create("https://foo1.com/bar");
        GoogleCredentialsTest.MockTokenServerTransportFactory serverTransportFactory = new GoogleCredentialsTest.MockTokenServerTransportFactory();
        ServiceAccountCredentials credentials = ServiceAccountCredentials.fromPkcs8((String)CLIENT_ID, (String)CLIENT_EMAIL, (String)PRIVATE_KEY_PKCS8, (String)PRIVATE_KEY_ID, SCOPES, (HttpTransportFactory)serverTransportFactory, (URI)tokenServer1);
        ServiceAccountCredentials otherCredentials = ServiceAccountCredentials.fromPkcs8((String)CLIENT_ID, (String)CLIENT_EMAIL, (String)PRIVATE_KEY_PKCS8, (String)"otherId", SCOPES, (HttpTransportFactory)serverTransportFactory, (URI)tokenServer1);
        Assert.assertFalse((boolean)credentials.equals((Object)otherCredentials));
        Assert.assertFalse((boolean)otherCredentials.equals((Object)credentials));
    }

    @Test
    public void equals_false_scopes() throws IOException {
        URI tokenServer1 = URI.create("https://foo1.com/bar");
        GoogleCredentialsTest.MockTokenServerTransportFactory serverTransportFactory = new GoogleCredentialsTest.MockTokenServerTransportFactory();
        ServiceAccountCredentials credentials = ServiceAccountCredentials.fromPkcs8((String)CLIENT_ID, (String)CLIENT_EMAIL, (String)PRIVATE_KEY_PKCS8, (String)PRIVATE_KEY_ID, SCOPES, (HttpTransportFactory)serverTransportFactory, (URI)tokenServer1);
        ServiceAccountCredentials otherCredentials = ServiceAccountCredentials.fromPkcs8((String)CLIENT_ID, (String)CLIENT_EMAIL, (String)PRIVATE_KEY_PKCS8, (String)PRIVATE_KEY_ID, (Collection)ImmutableSet.of(), (HttpTransportFactory)serverTransportFactory, (URI)tokenServer1);
        Assert.assertFalse((boolean)credentials.equals((Object)otherCredentials));
        Assert.assertFalse((boolean)otherCredentials.equals((Object)credentials));
    }

    @Test
    public void equals_false_transportFactory() throws IOException {
        URI tokenServer1 = URI.create("https://foo1.com/bar");
        GoogleCredentialsTest.MockHttpTransportFactory httpTransportFactory = new GoogleCredentialsTest.MockHttpTransportFactory();
        GoogleCredentialsTest.MockTokenServerTransportFactory serverTransportFactory = new GoogleCredentialsTest.MockTokenServerTransportFactory();
        ServiceAccountCredentials credentials = ServiceAccountCredentials.fromPkcs8((String)CLIENT_ID, (String)CLIENT_EMAIL, (String)PRIVATE_KEY_PKCS8, (String)PRIVATE_KEY_ID, SCOPES, (HttpTransportFactory)serverTransportFactory, (URI)tokenServer1);
        ServiceAccountCredentials otherCredentials = ServiceAccountCredentials.fromPkcs8((String)CLIENT_ID, (String)CLIENT_EMAIL, (String)PRIVATE_KEY_PKCS8, (String)PRIVATE_KEY_ID, SCOPES, (HttpTransportFactory)httpTransportFactory, (URI)tokenServer1);
        Assert.assertFalse((boolean)credentials.equals((Object)otherCredentials));
        Assert.assertFalse((boolean)otherCredentials.equals((Object)credentials));
    }

    @Test
    public void equals_false_tokenServer() throws IOException {
        URI tokenServer1 = URI.create("https://foo1.com/bar");
        URI tokenServer2 = URI.create("https://foo2.com/bar");
        GoogleCredentialsTest.MockTokenServerTransportFactory serverTransportFactory = new GoogleCredentialsTest.MockTokenServerTransportFactory();
        ServiceAccountCredentials credentials = ServiceAccountCredentials.fromPkcs8((String)CLIENT_ID, (String)CLIENT_EMAIL, (String)PRIVATE_KEY_PKCS8, (String)PRIVATE_KEY_ID, SCOPES, (HttpTransportFactory)serverTransportFactory, (URI)tokenServer1);
        ServiceAccountCredentials otherCredentials = ServiceAccountCredentials.fromPkcs8((String)CLIENT_ID, (String)CLIENT_EMAIL, (String)PRIVATE_KEY_PKCS8, (String)PRIVATE_KEY_ID, SCOPES, (HttpTransportFactory)serverTransportFactory, (URI)tokenServer2);
        Assert.assertFalse((boolean)credentials.equals((Object)otherCredentials));
        Assert.assertFalse((boolean)otherCredentials.equals((Object)credentials));
    }

    @Test
    public void toString_containsFields() throws IOException {
        URI tokenServer = URI.create("https://foo.com/bar");
        GoogleCredentialsTest.MockTokenServerTransportFactory transportFactory = new GoogleCredentialsTest.MockTokenServerTransportFactory();
        ServiceAccountCredentials.Builder builder = ServiceAccountCredentials.newBuilder().setClientId(CLIENT_ID).setClientEmail(CLIENT_EMAIL).setPrivateKeyId(PRIVATE_KEY_ID).setScopes(SCOPES, DEFAULT_SCOPES).setHttpTransportFactory((HttpTransportFactory)transportFactory).setTokenServerUri(tokenServer).setServiceAccountUser(USER).setQuotaProjectId(QUOTA_PROJECT);
        ServiceAccountCredentials credentials = ServiceAccountCredentials.fromPkcs8((String)PRIVATE_KEY_PKCS8, (ServiceAccountCredentials.Builder)builder);
        String expectedToString = String.format("ServiceAccountCredentials{clientId=%s, clientEmail=%s, privateKeyId=%s, transportFactoryClassName=%s, tokenServerUri=%s, scopes=%s, defaultScopes=%s, serviceAccountUser=%s, quotaProjectId=%s, lifetime=3600, useJwtAccessWithScope=false, defaultRetriesEnabled=true}", CLIENT_ID, CLIENT_EMAIL, PRIVATE_KEY_ID, GoogleCredentialsTest.MockTokenServerTransportFactory.class.getName(), tokenServer, SCOPES, DEFAULT_SCOPES, USER, QUOTA_PROJECT);
        Assert.assertEquals((Object)expectedToString, (Object)credentials.toString());
    }

    @Test
    public void hashCode_equals() throws IOException {
        URI tokenServer = URI.create("https://foo.com/bar");
        GoogleCredentialsTest.MockTokenServerTransportFactory transportFactory = new GoogleCredentialsTest.MockTokenServerTransportFactory();
        ServiceAccountCredentials credentials = ServiceAccountCredentials.fromPkcs8((String)CLIENT_ID, (String)CLIENT_EMAIL, (String)PRIVATE_KEY_PKCS8, (String)PRIVATE_KEY_ID, SCOPES, (HttpTransportFactory)transportFactory, (URI)tokenServer);
        ServiceAccountCredentials otherCredentials = ServiceAccountCredentials.fromPkcs8((String)CLIENT_ID, (String)CLIENT_EMAIL, (String)PRIVATE_KEY_PKCS8, (String)PRIVATE_KEY_ID, SCOPES, (HttpTransportFactory)transportFactory, (URI)tokenServer);
        Assert.assertEquals((long)credentials.hashCode(), (long)otherCredentials.hashCode());
    }

    @Test
    public void serialize() throws IOException, ClassNotFoundException {
        URI tokenServer = URI.create("https://foo.com/bar");
        GoogleCredentialsTest.MockTokenServerTransportFactory transportFactory = new GoogleCredentialsTest.MockTokenServerTransportFactory();
        ServiceAccountCredentials credentials = ServiceAccountCredentials.fromPkcs8((String)CLIENT_ID, (String)CLIENT_EMAIL, (String)PRIVATE_KEY_PKCS8, (String)PRIVATE_KEY_ID, SCOPES, (HttpTransportFactory)transportFactory, (URI)tokenServer);
        ServiceAccountCredentials deserializedCredentials = this.serializeAndDeserialize(credentials);
        Assert.assertEquals((Object)credentials, (Object)deserializedCredentials);
        Assert.assertEquals((long)credentials.hashCode(), (long)deserializedCredentials.hashCode());
        Assert.assertEquals((Object)credentials.toString(), (Object)deserializedCredentials.toString());
        Assert.assertSame((Object)deserializedCredentials.clock, (Object)Clock.SYSTEM);
        Assert.assertEquals(GoogleCredentialsTest.MockTokenServerTransportFactory.class, deserializedCredentials.toBuilder().getHttpTransportFactory().getClass());
    }

    @Test
    public void fromStream_nullTransport_throws() throws IOException {
        ByteArrayInputStream stream = new ByteArrayInputStream("foo".getBytes());
        try {
            ServiceAccountCredentials.fromStream((InputStream)stream, null);
            Assert.fail((String)"Should throw if HttpTransportFactory is null");
        }
        catch (NullPointerException nullPointerException) {
            // empty catch block
        }
    }

    @Test
    public void fromStream_nullStream_throws() throws IOException {
        GoogleCredentialsTest.MockHttpTransportFactory transportFactory = new GoogleCredentialsTest.MockHttpTransportFactory();
        try {
            ServiceAccountCredentials.fromStream(null, (HttpTransportFactory)transportFactory);
            Assert.fail((String)"Should throw if InputStream is null");
        }
        catch (NullPointerException nullPointerException) {
            // empty catch block
        }
    }

    @Test
    public void fromStream_providesToken() throws IOException {
        GoogleCredentialsTest.MockTokenServerTransportFactory transportFactory = new GoogleCredentialsTest.MockTokenServerTransportFactory();
        transportFactory.transport.addServiceAccount(CLIENT_EMAIL, ACCESS_TOKEN);
        InputStream serviceAccountStream = ServiceAccountCredentialsTest.writeServiceAccountStream(CLIENT_ID, CLIENT_EMAIL, PRIVATE_KEY_PKCS8, PRIVATE_KEY_ID);
        ServiceAccountCredentials credentials = ServiceAccountCredentials.fromStream((InputStream)serviceAccountStream, (HttpTransportFactory)transportFactory);
        Assert.assertNotNull((Object)credentials);
        credentials = credentials.createScoped(SCOPES);
        Map metadata = credentials.getRequestMetadata(CALL_URI);
        TestUtils.assertContainsBearerToken(metadata, ACCESS_TOKEN);
    }

    @Test
    public void fromStream_noClientId_throws() throws IOException {
        InputStream serviceAccountStream = ServiceAccountCredentialsTest.writeServiceAccountStream(null, CLIENT_EMAIL, PRIVATE_KEY_PKCS8, PRIVATE_KEY_ID);
        ServiceAccountCredentialsTest.testFromStreamException(serviceAccountStream, "client_id");
    }

    @Test
    public void fromStream_noClientEmail_throws() throws IOException {
        InputStream serviceAccountStream = ServiceAccountCredentialsTest.writeServiceAccountStream(CLIENT_ID, null, PRIVATE_KEY_PKCS8, PRIVATE_KEY_ID);
        ServiceAccountCredentialsTest.testFromStreamException(serviceAccountStream, "client_email");
    }

    @Test
    public void getIdTokenWithAudience_badEmailError_issClaimTraced() throws IOException {
        GoogleCredentialsTest.MockTokenServerTransportFactory transportFactory = new GoogleCredentialsTest.MockTokenServerTransportFactory();
        MockTokenServerTransport transport = transportFactory.transport;
        transport.setError(new IOException("Invalid grant: Account not found"));
        ServiceAccountCredentials credentials = ServiceAccountCredentials.fromPkcs8((String)CLIENT_ID, (String)CLIENT_EMAIL, (String)PRIVATE_KEY_PKCS8, (String)PRIVATE_KEY_ID, SCOPES, (HttpTransportFactory)transportFactory, null);
        String targetAudience = "https://bar";
        IdTokenCredentials tokenCredential = IdTokenCredentials.newBuilder().setIdTokenProvider((IdTokenProvider)credentials).setTargetAudience(targetAudience).build();
        String expectedErrorMessage = String.format("iss: %s", CLIENT_EMAIL);
        try {
            tokenCredential.refresh();
            Assert.fail((String)"Should not be able to use credential without exception.");
        }
        catch (IOException expected) {
            Assert.assertTrue((boolean)expected.getMessage().contains(expectedErrorMessage));
        }
    }

    @Test
    public void fromStream_noPrivateKey_throws() throws IOException {
        InputStream serviceAccountStream = ServiceAccountCredentialsTest.writeServiceAccountStream(CLIENT_ID, CLIENT_EMAIL, null, PRIVATE_KEY_ID);
        ServiceAccountCredentialsTest.testFromStreamException(serviceAccountStream, "private_key");
    }

    @Test
    public void fromStream_noPrivateKeyId_throws() throws IOException {
        InputStream serviceAccountStream = ServiceAccountCredentialsTest.writeServiceAccountStream(CLIENT_ID, CLIENT_EMAIL, PRIVATE_KEY_PKCS8, null);
        ServiceAccountCredentialsTest.testFromStreamException(serviceAccountStream, "private_key_id");
    }

    @Test
    public void getUriForSelfSignedJWT() {
        Assert.assertNull((Object)ServiceAccountCredentials.getUriForSelfSignedJWT(null));
        URI uri = URI.create("https://compute.googleapis.com/compute/v1/projects/");
        URI expected = URI.create("https://compute.googleapis.com/");
        Assert.assertEquals((Object)expected, (Object)ServiceAccountCredentials.getUriForSelfSignedJWT((URI)uri));
    }

    @Test
    public void getUriForSelfSignedJWT_noHost() {
        URI uri = URI.create("file:foo");
        URI expected = URI.create("file:foo");
        Assert.assertEquals((Object)expected, (Object)ServiceAccountCredentials.getUriForSelfSignedJWT((URI)uri));
    }

    @Test
    public void getUriForSelfSignedJWT_forStaticAudience_returnsURI() {
        URI uri = URI.create("compute.googleapis.com");
        URI expected = URI.create("compute.googleapis.com");
        Assert.assertEquals((Object)expected, (Object)ServiceAccountCredentials.getUriForSelfSignedJWT((URI)uri));
    }

    @Test
    public void getRequestMetadataSetsQuotaProjectId() throws IOException {
        GoogleCredentialsTest.MockTokenServerTransportFactory transportFactory = new GoogleCredentialsTest.MockTokenServerTransportFactory();
        transportFactory.transport.addClient(CLIENT_ID, "unused-client-secret");
        transportFactory.transport.addServiceAccount(CLIENT_EMAIL, ACCESS_TOKEN);
        PrivateKey privateKey = ServiceAccountCredentials.privateKeyFromPkcs8((String)PRIVATE_KEY_PKCS8);
        ServiceAccountCredentials credentials = ServiceAccountCredentials.newBuilder().setClientId(CLIENT_ID).setClientEmail(CLIENT_EMAIL).setPrivateKey(privateKey).setPrivateKeyId(PRIVATE_KEY_ID).setScopes(SCOPES).setServiceAccountUser(USER).setProjectId(PROJECT_ID).setQuotaProjectId("my-quota-project-id").setHttpTransportFactory((HttpTransportFactory)transportFactory).build();
        Map metadata = credentials.getRequestMetadata(CALL_URI);
        Assert.assertTrue((boolean)metadata.containsKey("x-goog-user-project"));
        List headerValues = (List)metadata.get("x-goog-user-project");
        Assert.assertEquals((long)1L, (long)headerValues.size());
        Assert.assertEquals((Object)"my-quota-project-id", headerValues.get(0));
    }

    @Test
    public void getRequestMetadataNoQuotaProjectId() throws IOException {
        GoogleCredentialsTest.MockTokenServerTransportFactory transportFactory = new GoogleCredentialsTest.MockTokenServerTransportFactory();
        transportFactory.transport.addClient(CLIENT_ID, "unused-client-secret");
        transportFactory.transport.addServiceAccount(CLIENT_EMAIL, ACCESS_TOKEN);
        PrivateKey privateKey = ServiceAccountCredentials.privateKeyFromPkcs8((String)PRIVATE_KEY_PKCS8);
        ServiceAccountCredentials credentials = ServiceAccountCredentials.newBuilder().setClientId(CLIENT_ID).setClientEmail(CLIENT_EMAIL).setPrivateKey(privateKey).setPrivateKeyId(PRIVATE_KEY_ID).setScopes(SCOPES).setServiceAccountUser(USER).setProjectId(PROJECT_ID).setHttpTransportFactory((HttpTransportFactory)transportFactory).build();
        Map metadata = credentials.getRequestMetadata(CALL_URI);
        Assert.assertFalse((boolean)metadata.containsKey("x-goog-user-project"));
    }

    @Test
    public void getRequestMetadataWithCallback() throws IOException {
        GoogleCredentialsTest.MockTokenServerTransportFactory transportFactory = new GoogleCredentialsTest.MockTokenServerTransportFactory();
        transportFactory.transport.addClient(CLIENT_ID, "unused-client-secret");
        transportFactory.transport.addServiceAccount(CLIENT_EMAIL, ACCESS_TOKEN);
        PrivateKey privateKey = ServiceAccountCredentials.privateKeyFromPkcs8((String)PRIVATE_KEY_PKCS8);
        ServiceAccountCredentials credentials = ServiceAccountCredentials.newBuilder().setClientId(CLIENT_ID).setClientEmail(CLIENT_EMAIL).setPrivateKey(privateKey).setPrivateKeyId(PRIVATE_KEY_ID).setScopes(SCOPES).setServiceAccountUser(USER).setProjectId(PROJECT_ID).setQuotaProjectId("my-quota-project-id").setHttpTransportFactory((HttpTransportFactory)transportFactory).build();
        final Map plainMetadata = credentials.getRequestMetadata();
        final AtomicBoolean success = new AtomicBoolean(false);
        credentials.getRequestMetadata(null, null, new RequestMetadataCallback(){

            public void onSuccess(Map<String, List<String>> metadata) {
                Assert.assertEquals((Object)plainMetadata, metadata);
                success.set(true);
            }

            public void onFailure(Throwable exception) {
                Assert.fail((String)"Should not throw a failure.");
            }
        });
        Assert.assertTrue((String)"Should have run onSuccess() callback", (boolean)success.get());
    }

    @Test
    public void getRequestMetadata_selfSignedJWT_withScopes() throws IOException {
        PrivateKey privateKey = ServiceAccountCredentials.privateKeyFromPkcs8((String)PRIVATE_KEY_PKCS8);
        ServiceAccountCredentials credentials = ServiceAccountCredentials.newBuilder().setClientId(CLIENT_ID).setClientEmail(CLIENT_EMAIL).setPrivateKey(privateKey).setPrivateKeyId(PRIVATE_KEY_ID).setScopes(SCOPES).setProjectId(PROJECT_ID).setHttpTransportFactory((HttpTransportFactory)new GoogleCredentialsTest.MockTokenServerTransportFactory()).setUseJwtAccessWithScope(true).build();
        Map metadata = credentials.getRequestMetadata(CALL_URI);
        this.verifyJwtAccess(metadata, "dummy.scope");
    }

    @Test
    public void refreshAccessToken_withDomainDelegation_selfSignedJWT_disabled() throws IOException {
        String accessToken1 = ACCESS_TOKEN;
        String accessToken2 = "2/MkSJoj1xsli0AccessToken_NKPY2";
        GoogleCredentialsTest.MockTokenServerTransportFactory transportFactory = new GoogleCredentialsTest.MockTokenServerTransportFactory();
        MockTokenServerTransport transport = transportFactory.transport;
        PrivateKey privateKey = ServiceAccountCredentials.privateKeyFromPkcs8((String)PRIVATE_KEY_PKCS8);
        ServiceAccountCredentials credentials = ServiceAccountCredentials.newBuilder().setClientId(CLIENT_ID).setClientEmail(CLIENT_EMAIL).setPrivateKey(privateKey).setPrivateKeyId(PRIVATE_KEY_ID).setScopes(SCOPES).setServiceAccountUser(USER).setProjectId(PROJECT_ID).setHttpTransportFactory((HttpTransportFactory)transportFactory).setUseJwtAccessWithScope(true).build();
        transport.addServiceAccount(CLIENT_EMAIL, ACCESS_TOKEN);
        Map metadata = credentials.getRequestMetadata(CALL_URI);
        TestUtils.assertContainsBearerToken(metadata, ACCESS_TOKEN);
        try {
            this.verifyJwtAccess(metadata, "dummy.scope");
            Assert.fail((String)"jwt access should fail with ServiceAccountUser");
        }
        catch (Exception exception) {
            // empty catch block
        }
        transport.addServiceAccount(CLIENT_EMAIL, "2/MkSJoj1xsli0AccessToken_NKPY2");
        credentials.refresh();
        TestUtils.assertContainsBearerToken(credentials.getRequestMetadata(CALL_URI), "2/MkSJoj1xsli0AccessToken_NKPY2");
    }

    @Test
    public void getRequestMetadata_selfSignedJWT_withAudience() throws IOException {
        PrivateKey privateKey = ServiceAccountCredentials.privateKeyFromPkcs8((String)PRIVATE_KEY_PKCS8);
        ServiceAccountCredentials credentials = ServiceAccountCredentials.newBuilder().setClientId(CLIENT_ID).setClientEmail(CLIENT_EMAIL).setPrivateKey(privateKey).setPrivateKeyId(PRIVATE_KEY_ID).setProjectId(PROJECT_ID).setHttpTransportFactory((HttpTransportFactory)new GoogleCredentialsTest.MockTokenServerTransportFactory()).build();
        Map metadata = credentials.getRequestMetadata(CALL_URI);
        this.verifyJwtAccess(metadata, null);
    }

    @Test
    public void getRequestMetadata_selfSignedJWT_withDefaultScopes() throws IOException {
        PrivateKey privateKey = ServiceAccountCredentials.privateKeyFromPkcs8((String)PRIVATE_KEY_PKCS8);
        ServiceAccountCredentials credentials = ServiceAccountCredentials.newBuilder().setClientId(CLIENT_ID).setClientEmail(CLIENT_EMAIL).setPrivateKey(privateKey).setPrivateKeyId(PRIVATE_KEY_ID).setScopes(null, SCOPES).setProjectId(PROJECT_ID).setHttpTransportFactory((HttpTransportFactory)new GoogleCredentialsTest.MockTokenServerTransportFactory()).setUseJwtAccessWithScope(true).build();
        Map metadata = credentials.getRequestMetadata(null);
        this.verifyJwtAccess(metadata, "dummy.scope");
    }

    @Test
    public void getRequestMetadataWithCallback_selfSignedJWT() throws IOException {
        PrivateKey privateKey = ServiceAccountCredentials.privateKeyFromPkcs8((String)PRIVATE_KEY_PKCS8);
        ServiceAccountCredentials credentials = ServiceAccountCredentials.newBuilder().setClientId(CLIENT_ID).setClientEmail(CLIENT_EMAIL).setPrivateKey(privateKey).setPrivateKeyId(PRIVATE_KEY_ID).setProjectId(PROJECT_ID).setQuotaProjectId("my-quota-project-id").setHttpTransportFactory((HttpTransportFactory)new GoogleCredentialsTest.MockTokenServerTransportFactory()).setUseJwtAccessWithScope(true).setScopes(SCOPES).build();
        final AtomicBoolean success = new AtomicBoolean(false);
        credentials.getRequestMetadata(CALL_URI, null, new RequestMetadataCallback(){

            public void onSuccess(Map<String, List<String>> metadata) {
                try {
                    ServiceAccountCredentialsTest.this.verifyJwtAccess(metadata, "dummy.scope");
                }
                catch (IOException e) {
                    Assert.fail((String)"Should not throw a failure");
                }
                success.set(true);
            }

            public void onFailure(Throwable exception) {
                Assert.fail((String)"Should not throw a failure.");
            }
        });
        Assert.assertTrue((String)"Should have run onSuccess() callback", (boolean)success.get());
    }

    private void verifyJwtAccess(Map<String, List<String>> metadata, String expectedScopeClaim) throws IOException {
        Assert.assertNotNull(metadata);
        List<String> authorizations = metadata.get("Authorization");
        Assert.assertNotNull((String)"Authorization headers not found", authorizations);
        String assertion = null;
        for (String authorization : authorizations) {
            if (!authorization.startsWith(JWT_ACCESS_PREFIX)) continue;
            Assert.assertNull((String)"Multiple bearer assertions found", assertion);
            assertion = authorization.substring(JWT_ACCESS_PREFIX.length());
        }
        Assert.assertNotNull((String)"Bearer assertion not found", assertion);
        JsonWebSignature signature = JsonWebSignature.parse((JsonFactory)GsonFactory.getDefaultInstance(), assertion);
        Assert.assertEquals((Object)CLIENT_EMAIL, (Object)signature.getPayload().getIssuer());
        Assert.assertEquals((Object)CLIENT_EMAIL, (Object)signature.getPayload().getSubject());
        if (expectedScopeClaim != null) {
            Assert.assertEquals((Object)expectedScopeClaim, (Object)signature.getPayload().get((Object)"scope"));
            Assert.assertFalse((boolean)signature.getPayload().containsKey((Object)"aud"));
        } else {
            Assert.assertEquals((Object)JWT_AUDIENCE, (Object)signature.getPayload().getAudience());
            Assert.assertFalse((boolean)signature.getPayload().containsKey((Object)"scope"));
        }
        Assert.assertEquals((Object)PRIVATE_KEY_ID, (Object)signature.getHeader().getKeyId());
    }

    static GenericJson writeServiceAccountJson(String clientId, String clientEmail, String privateKeyPkcs8, String privateKeyId, String projectId, String quotaProjectId) {
        GenericJson json = new GenericJson();
        if (clientId != null) {
            json.put("client_id", (Object)clientId);
        }
        if (clientEmail != null) {
            json.put("client_email", (Object)clientEmail);
        }
        if (privateKeyPkcs8 != null) {
            json.put("private_key", (Object)privateKeyPkcs8);
        }
        if (privateKeyId != null) {
            json.put("private_key_id", (Object)privateKeyId);
        }
        if (projectId != null) {
            json.put("project_id", (Object)projectId);
        }
        if (quotaProjectId != null) {
            json.put("quota_project_id", (Object)quotaProjectId);
        }
        json.put("type", (Object)"service_account");
        return json;
    }

    static InputStream writeServiceAccountStream(String clientId, String clientEmail, String privateKeyPkcs8, String privateKeyId) throws IOException {
        GenericJson json = ServiceAccountCredentialsTest.writeServiceAccountJson(clientId, clientEmail, privateKeyPkcs8, privateKeyId, null, null);
        return TestUtils.jsonToInputStream(json);
    }

    private static void testFromStreamException(InputStream stream, String expectedMessageContent) {
        try {
            ServiceAccountCredentials.fromStream((InputStream)stream, (HttpTransportFactory)DUMMY_TRANSPORT_FACTORY);
            Assert.fail((String)String.format("Should throw exception with message containing '%s'", expectedMessageContent));
        }
        catch (IOException expected) {
            Assert.assertTrue((boolean)expected.getMessage().contains(expectedMessageContent));
        }
    }
}

