/*
 * Decompiled with CFR 0.152.
 */
package com.google.cloud.hadoop.gcsio.integration;

import com.google.cloud.hadoop.gcsio.GoogleCloudStorage;
import com.google.cloud.hadoop.gcsio.GoogleCloudStorageImpl;
import com.google.cloud.hadoop.gcsio.GoogleCloudStorageItemInfo;
import com.google.cloud.hadoop.gcsio.GoogleCloudStorageOptions;
import com.google.cloud.hadoop.gcsio.GoogleCloudStorageReadOptions;
import com.google.cloud.hadoop.gcsio.StorageResourceId;
import com.google.cloud.hadoop.gcsio.integration.GoogleCloudStorageTestHelper;
import com.google.cloud.hadoop.util.AsyncWriteChannelOptions;
import cz.o2.proxima.internal.shaded.com.google.common.flogger.GoogleLogger;
import cz.o2.proxima.internal.shaded.com.google.common.truth.Truth;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.ClosedChannelException;
import java.nio.channels.SeekableByteChannel;
import java.util.Arrays;
import org.junit.AfterClass;
import org.junit.Assert;
import org.junit.BeforeClass;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

@RunWith(value=Parameterized.class)
public class GoogleCloudStorageGrpcIntegrationTest {
    private static final GoogleLogger logger = GoogleLogger.forEnclosingClass();
    private static final String BUCKET_NAME_PREFIX = "gcs-grpc-team";
    private static final GoogleCloudStorageTestHelper.TestBucketHelper BUCKET_HELPER = new GoogleCloudStorageTestHelper.TestBucketHelper("gcs-grpc-team");
    private static final String BUCKET_NAME = BUCKET_HELPER.getUniqueBucketName("shared");
    private final boolean tdEnabled;

    @Parameterized.Parameters
    public static Iterable<Boolean> tdEnabled() {
        return Arrays.asList(false, true);
    }

    public GoogleCloudStorageGrpcIntegrationTest(boolean tdEnabled) {
        this.tdEnabled = tdEnabled;
    }

    private static GoogleCloudStorageOptions.Builder configureDefaultOptions() {
        GoogleCloudStorageOptions.Builder optionsBuilder = GoogleCloudStorageTestHelper.getStandardOptionBuilder().setGrpcEnabled(true);
        String grpcServerAddress = System.getenv("GCS_TEST_GRPC_SERVER_ADDRESS_OVERRIDE");
        if (grpcServerAddress != null) {
            optionsBuilder.setGrpcServerAddress(grpcServerAddress);
            ((GoogleLogger.Api)logger.atInfo()).log("Overriding gRPC server address to %s", (Object)grpcServerAddress);
        }
        return optionsBuilder;
    }

    private GoogleCloudStorageOptions.Builder configureOptionsWithTD() {
        ((GoogleLogger.Api)logger.atInfo()).log("Creating client with tdEnabled %s", (Object)this.tdEnabled);
        return GoogleCloudStorageGrpcIntegrationTest.configureDefaultOptions().setTrafficDirectorEnabled(this.tdEnabled);
    }

    private GoogleCloudStorage createGoogleCloudStorage() throws IOException {
        return new GoogleCloudStorageImpl(this.configureOptionsWithTD().build(), GoogleCloudStorageTestHelper.getCredential());
    }

    private GoogleCloudStorage createGoogleCloudStorage(AsyncWriteChannelOptions asyncWriteChannelOptions) throws IOException {
        return new GoogleCloudStorageImpl(this.configureOptionsWithTD().setWriteChannelOptions(asyncWriteChannelOptions).build(), GoogleCloudStorageTestHelper.getCredential());
    }

    @BeforeClass
    public static void createBuckets() throws IOException {
        GoogleCloudStorageImpl rawStorage = new GoogleCloudStorageImpl(GoogleCloudStorageGrpcIntegrationTest.configureDefaultOptions().build(), GoogleCloudStorageTestHelper.getCredential());
        rawStorage.createBucket(BUCKET_NAME);
    }

    @AfterClass
    public static void cleanupBuckets() throws IOException {
        GoogleCloudStorageImpl rawStorage = new GoogleCloudStorageImpl(GoogleCloudStorageGrpcIntegrationTest.configureDefaultOptions().build(), GoogleCloudStorageTestHelper.getCredential());
        BUCKET_HELPER.cleanup((GoogleCloudStorage)rawStorage);
    }

    @Test
    public void testCreateObject() throws IOException {
        GoogleCloudStorage rawStorage = this.createGoogleCloudStorage();
        StorageResourceId objectToCreate = new StorageResourceId(BUCKET_NAME, "testCreateObject_Object");
        byte[] objectBytes = GoogleCloudStorageTestHelper.writeObject(rawStorage, objectToCreate, 512);
        GoogleCloudStorageTestHelper.assertObjectContent(rawStorage, objectToCreate, objectBytes);
    }

    @Test
    public void testCreateExistingObject() throws IOException {
        GoogleCloudStorage rawStorage = this.createGoogleCloudStorage();
        StorageResourceId objectToCreate = new StorageResourceId(BUCKET_NAME, "testCreateExistingObject_Object");
        GoogleCloudStorageTestHelper.writeObject(rawStorage, objectToCreate, 128);
        GoogleCloudStorageItemInfo createdItemInfo = rawStorage.getItemInfo(objectToCreate);
        Truth.assertThat((Boolean)createdItemInfo.exists()).isTrue();
        Truth.assertThat((Long)createdItemInfo.getSize()).isEqualTo((Object)128);
        byte[] overwriteBytesToWrite = GoogleCloudStorageTestHelper.writeObject(rawStorage, objectToCreate, 256);
        GoogleCloudStorageItemInfo overwrittenItemInfo = rawStorage.getItemInfo(objectToCreate);
        Truth.assertThat((Boolean)overwrittenItemInfo.exists()).isTrue();
        Truth.assertThat((Long)overwrittenItemInfo.getSize()).isEqualTo((Object)256);
        GoogleCloudStorageTestHelper.assertObjectContent(rawStorage, objectToCreate, overwriteBytesToWrite);
    }

    @Test
    public void testCreateEmptyObject() throws IOException {
        GoogleCloudStorage rawStorage = this.createGoogleCloudStorage();
        StorageResourceId objectToCreate = new StorageResourceId(BUCKET_NAME, "testCreateEmptyObject_Object");
        rawStorage.createEmptyObject(objectToCreate);
        GoogleCloudStorageItemInfo itemInfo = rawStorage.getItemInfo(objectToCreate);
        Truth.assertThat((Boolean)itemInfo.exists()).isTrue();
        Truth.assertThat((Long)itemInfo.getSize()).isEqualTo((Object)0);
    }

    @Test
    public void testCreateInvalidObject() throws IOException {
        GoogleCloudStorage rawStorage = this.createGoogleCloudStorage();
        StorageResourceId objectToCreate = new StorageResourceId(BUCKET_NAME, "testCreateInvalidObject_InvalidObject\n");
        Assert.assertThrows(IOException.class, () -> GoogleCloudStorageTestHelper.writeObject(rawStorage, objectToCreate, 10));
    }

    @Test
    public void testOpen() throws IOException {
        GoogleCloudStorage rawStorage = this.createGoogleCloudStorage();
        StorageResourceId objectToCreate = new StorageResourceId(BUCKET_NAME, "testOpen_Object");
        byte[] objectBytes = GoogleCloudStorageTestHelper.writeObject(rawStorage, objectToCreate, 100);
        GoogleCloudStorageTestHelper.assertObjectContent(rawStorage, objectToCreate, objectBytes);
    }

    @Test
    public void testOpenWithMetricsEnabled() throws IOException {
        GoogleCloudStorageImpl rawStorage = new GoogleCloudStorageImpl(GoogleCloudStorageTestHelper.getStandardOptionBuilder().setMetricsSink(GoogleCloudStorageOptions.MetricsSink.CLOUD_MONITORING).build(), GoogleCloudStorageTestHelper.getCredential());
        StorageResourceId objectToCreate = new StorageResourceId(BUCKET_NAME, "testOpen_Object");
        byte[] objectBytes = GoogleCloudStorageTestHelper.writeObject((GoogleCloudStorage)rawStorage, objectToCreate, 100);
        GoogleCloudStorageTestHelper.assertObjectContent((GoogleCloudStorage)rawStorage, objectToCreate, objectBytes);
    }

    @Test
    public void testOpenNonExistentItem() throws IOException {
        GoogleCloudStorage rawStorage = this.createGoogleCloudStorage();
        Throwable throwable = Assert.assertThrows(IOException.class, () -> rawStorage.open(new StorageResourceId(BUCKET_NAME, "testOpenNonExistentItem_Object")));
        Truth.assertThat((Throwable)throwable).hasMessageThat().contains((CharSequence)"Item not found");
    }

    @Test
    public void testOpenEmptyObject() throws IOException {
        GoogleCloudStorage rawStorage = this.createGoogleCloudStorage();
        StorageResourceId resourceId = new StorageResourceId(BUCKET_NAME, "testOpenEmptyObject_Object");
        rawStorage.createEmptyObject(resourceId);
        GoogleCloudStorageTestHelper.assertObjectContent(rawStorage, resourceId, new byte[0]);
    }

    @Test
    public void testOpenLargeObject() throws IOException {
        GoogleCloudStorage rawStorage = this.createGoogleCloudStorage();
        StorageResourceId resourceId = new StorageResourceId(BUCKET_NAME, "testOpenLargeObject_Object");
        int partitionsCount = 50;
        byte[] partition = GoogleCloudStorageTestHelper.writeObject(rawStorage, resourceId, 0xA00000, partitionsCount);
        GoogleCloudStorageTestHelper.assertObjectContent(rawStorage, resourceId, partition, partitionsCount);
    }

    @Test
    public void testOpenObjectWithChecksum() throws IOException {
        AsyncWriteChannelOptions asyncWriteChannelOptions = AsyncWriteChannelOptions.builder().setGrpcChecksumsEnabled(true).build();
        GoogleCloudStorage rawStorage = this.createGoogleCloudStorage(asyncWriteChannelOptions);
        StorageResourceId objectToCreate = new StorageResourceId(BUCKET_NAME, "testOpenObjectWithChecksum_Object");
        byte[] objectBytes = GoogleCloudStorageTestHelper.writeObject(rawStorage, objectToCreate, 100);
        GoogleCloudStorageReadOptions readOptions = GoogleCloudStorageReadOptions.builder().setGrpcChecksumsEnabled(true).build();
        GoogleCloudStorageTestHelper.assertObjectContent(rawStorage, objectToCreate, readOptions, objectBytes);
    }

    @Test
    public void testOpenObjectWithSeek() throws IOException {
        AsyncWriteChannelOptions asyncWriteChannelOptions = AsyncWriteChannelOptions.builder().setGrpcChecksumsEnabled(true).build();
        GoogleCloudStorage rawStorage = this.createGoogleCloudStorage(asyncWriteChannelOptions);
        StorageResourceId objectToCreate = new StorageResourceId(BUCKET_NAME, "testOpenObjectWithSeek_Object");
        byte[] objectBytes = GoogleCloudStorageTestHelper.writeObject(rawStorage, objectToCreate, 100);
        int offset = 10;
        byte[] trimmedObjectBytes = Arrays.copyOfRange(objectBytes, offset, objectBytes.length);
        GoogleCloudStorageReadOptions readOptions = GoogleCloudStorageReadOptions.builder().setGrpcChecksumsEnabled(true).build();
        GoogleCloudStorageTestHelper.assertObjectContent(rawStorage, objectToCreate, readOptions, trimmedObjectBytes, 1, offset);
    }

    @Test
    public void testOpenObjectWithSeekOverBounds() throws IOException {
        AsyncWriteChannelOptions asyncWriteChannelOptions = AsyncWriteChannelOptions.builder().setGrpcChecksumsEnabled(true).build();
        GoogleCloudStorage rawStorage = this.createGoogleCloudStorage(asyncWriteChannelOptions);
        StorageResourceId objectToCreate = new StorageResourceId(BUCKET_NAME, "testOpenObjectWithSeekOverBounds_Object");
        byte[] objectBytes = GoogleCloudStorageTestHelper.writeObject(rawStorage, objectToCreate, 0x400000);
        int offset = 0x300000;
        byte[] trimmedObjectBytes = Arrays.copyOfRange(objectBytes, offset, objectBytes.length);
        GoogleCloudStorageReadOptions readOptions = GoogleCloudStorageReadOptions.builder().setGrpcChecksumsEnabled(true).build();
        GoogleCloudStorageTestHelper.assertObjectContent(rawStorage, objectToCreate, readOptions, trimmedObjectBytes, 1, offset);
    }

    @Test
    public void testOpenObjectWithSeekLimits() throws IOException {
        AsyncWriteChannelOptions asyncWriteChannelOptions = AsyncWriteChannelOptions.builder().setGrpcChecksumsEnabled(true).build();
        GoogleCloudStorage rawStorage = this.createGoogleCloudStorage(asyncWriteChannelOptions);
        StorageResourceId objectToCreate = new StorageResourceId(BUCKET_NAME, "testOpenObjectWithSeekOverBounds_Object");
        byte[] objectBytes = GoogleCloudStorageTestHelper.writeObject(rawStorage, objectToCreate, 1024);
        int offset = 100;
        byte[] trimmedObjectBytes = Arrays.copyOfRange(objectBytes, offset, objectBytes.length);
        GoogleCloudStorageReadOptions readOptions = GoogleCloudStorageReadOptions.builder().setInplaceSeekLimit(50L).setGrpcChecksumsEnabled(true).build();
        GoogleCloudStorageTestHelper.assertObjectContent(rawStorage, objectToCreate, readOptions, trimmedObjectBytes, 1, offset);
    }

    @Test
    public void testReadFooterDataWithGrpcChecksums() throws IOException {
        AsyncWriteChannelOptions asyncWriteChannelOptions = AsyncWriteChannelOptions.builder().setGrpcChecksumsEnabled(true).build();
        GoogleCloudStorage rawStorage = this.createGoogleCloudStorage(asyncWriteChannelOptions);
        StorageResourceId objectToCreate = new StorageResourceId(BUCKET_NAME, "testOpenObjectWithSeekToFooter_Object");
        int objectSize = 1024;
        byte[] objectBytes = GoogleCloudStorageTestHelper.writeObject(rawStorage, objectToCreate, objectSize);
        int minRangeRequestSize = 200;
        int offset = objectSize - minRangeRequestSize / 2;
        byte[] trimmedObjectBytes = Arrays.copyOfRange(objectBytes, offset, objectBytes.length);
        GoogleCloudStorageReadOptions readOptions = GoogleCloudStorageReadOptions.builder().setInplaceSeekLimit(50L).setMinRangeRequestSize(minRangeRequestSize).setFadvise(GoogleCloudStorageReadOptions.Fadvise.RANDOM).setGrpcChecksumsEnabled(true).build();
        GoogleCloudStorageTestHelper.assertObjectContent(rawStorage, objectToCreate, readOptions, trimmedObjectBytes, 1, offset);
    }

    @Test
    public void testReadCachedFooterData() throws IOException {
        AsyncWriteChannelOptions asyncWriteChannelOptions = AsyncWriteChannelOptions.builder().build();
        GoogleCloudStorage rawStorage = this.createGoogleCloudStorage(asyncWriteChannelOptions);
        StorageResourceId objectToCreate = new StorageResourceId(BUCKET_NAME, "testReadCachedFooterData_Object");
        int objectSize = 0xA00000;
        byte[] objectBytes = GoogleCloudStorageTestHelper.writeObject(rawStorage, objectToCreate, objectSize);
        int minRangeRequestSize = 200;
        int footerOffset = objectSize - minRangeRequestSize / 2;
        byte[] trimmedObjectBytes = Arrays.copyOfRange(objectBytes, footerOffset, objectBytes.length);
        GoogleCloudStorageReadOptions readOptions = GoogleCloudStorageReadOptions.builder().setMinRangeRequestSize(minRangeRequestSize).build();
        GoogleCloudStorageTestHelper.assertObjectContent(rawStorage, objectToCreate, readOptions, trimmedObjectBytes, 1, footerOffset);
    }

    @Test
    public void testReadSeekToFooterData() throws IOException {
        AsyncWriteChannelOptions asyncWriteChannelOptions = AsyncWriteChannelOptions.builder().build();
        GoogleCloudStorage rawStorage = this.createGoogleCloudStorage(asyncWriteChannelOptions);
        StorageResourceId objectToCreate = new StorageResourceId(BUCKET_NAME, "testReadSeekToFooterData_Object");
        int objectSize = 1024;
        byte[] objectBytes = GoogleCloudStorageTestHelper.writeObject(rawStorage, objectToCreate, objectSize);
        int minRangeRequestSize = 200;
        int offset = objectSize - minRangeRequestSize / 4;
        byte[] trimmedObjectBytes = Arrays.copyOfRange(objectBytes, offset, objectBytes.length);
        GoogleCloudStorageReadOptions readOptions = GoogleCloudStorageReadOptions.builder().setMinRangeRequestSize(minRangeRequestSize).build();
        GoogleCloudStorageTestHelper.assertObjectContent(rawStorage, objectToCreate, readOptions, trimmedObjectBytes, 1, offset);
    }

    @Test
    public void testReadObjectCachedAsFooter() throws IOException {
        AsyncWriteChannelOptions asyncWriteChannelOptions = AsyncWriteChannelOptions.builder().build();
        GoogleCloudStorage rawStorage = this.createGoogleCloudStorage(asyncWriteChannelOptions);
        StorageResourceId objectToCreate = new StorageResourceId(BUCKET_NAME, "testReadSeekToFooterData_Object");
        int objectSize = 1024;
        byte[] objectBytes = GoogleCloudStorageTestHelper.writeObject(rawStorage, objectToCreate, objectSize);
        int minRangeRequestSize = 4096;
        int offset = 0;
        byte[] trimmedObjectBytes = Arrays.copyOfRange(objectBytes, offset, objectBytes.length);
        GoogleCloudStorageReadOptions readOptions = GoogleCloudStorageReadOptions.builder().setMinRangeRequestSize(minRangeRequestSize).build();
        GoogleCloudStorageTestHelper.assertObjectContent(rawStorage, objectToCreate, readOptions, trimmedObjectBytes, 1, offset);
    }

    @Test
    public void testPartialReadFooterDataWithSingleChunk() throws IOException {
        AsyncWriteChannelOptions asyncWriteChannelOptions = AsyncWriteChannelOptions.builder().build();
        GoogleCloudStorage rawStorage = this.createGoogleCloudStorage(asyncWriteChannelOptions);
        StorageResourceId objectToCreate = new StorageResourceId(BUCKET_NAME, "testPartialReadFooterDataWithSingleChunk_Object");
        int objectSize = 0x200000;
        byte[] objectBytes = GoogleCloudStorageTestHelper.writeObject(rawStorage, objectToCreate, objectSize);
        int minRangeRequestSize = 1024;
        int readOffset = objectSize / 2;
        byte[] trimmedObjectBytes = Arrays.copyOfRange(objectBytes, readOffset, objectBytes.length);
        GoogleCloudStorageReadOptions readOptions = GoogleCloudStorageReadOptions.builder().setMinRangeRequestSize(minRangeRequestSize).build();
        GoogleCloudStorageTestHelper.assertObjectContent(rawStorage, objectToCreate, readOptions, trimmedObjectBytes, 1, readOffset);
    }

    @Test
    public void testPartialReadFooterDataWithMultipleChunks() throws IOException {
        AsyncWriteChannelOptions asyncWriteChannelOptions = AsyncWriteChannelOptions.builder().build();
        GoogleCloudStorage rawStorage = this.createGoogleCloudStorage(asyncWriteChannelOptions);
        StorageResourceId objectToCreate = new StorageResourceId(BUCKET_NAME, "testPartialReadFooterDataWithMultipleChunks_Object");
        int objectSize = 0xA00000;
        byte[] objectBytes = GoogleCloudStorageTestHelper.writeObject(rawStorage, objectToCreate, objectSize);
        int minRangeRequestSize = 4096;
        int readOffset = objectSize / 2;
        byte[] trimmedObjectBytes = Arrays.copyOfRange(objectBytes, readOffset, objectBytes.length);
        GoogleCloudStorageReadOptions readOptions = GoogleCloudStorageReadOptions.builder().setMinRangeRequestSize(minRangeRequestSize).build();
        GoogleCloudStorageTestHelper.assertObjectContent(rawStorage, objectToCreate, readOptions, trimmedObjectBytes, 1, readOffset);
    }

    @Test
    public void testPartialReadFooterDataWithinSegment() throws IOException {
        AsyncWriteChannelOptions asyncWriteChannelOptions = AsyncWriteChannelOptions.builder().build();
        GoogleCloudStorage rawStorage = this.createGoogleCloudStorage(asyncWriteChannelOptions);
        StorageResourceId objectToCreate = new StorageResourceId(BUCKET_NAME, "testPartialReadFooterDataWithinSegment_Object");
        int objectSize = 10240;
        byte[] objectBytes = GoogleCloudStorageTestHelper.writeObject(rawStorage, objectToCreate, objectSize);
        int minRangeRequestSize = 4096;
        GoogleCloudStorageReadOptions readOptions = GoogleCloudStorageReadOptions.builder().setMinRangeRequestSize(minRangeRequestSize).build();
        int readOffset = 7168;
        try (SeekableByteChannel readChannel = rawStorage.open(objectToCreate, readOptions);){
            byte[] segmentBytes = new byte[100];
            ByteBuffer segmentBuffer = ByteBuffer.wrap(segmentBytes);
            readChannel.position(readOffset);
            readChannel.read(segmentBuffer);
            byte[] expectedSegment = Arrays.copyOfRange(objectBytes, readOffset, readOffset + segmentBytes.length);
            Truth.assertWithMessage((String)"Unexpected segment data read.").that(segmentBytes).isEqualTo((Object)expectedSegment);
        }
    }

    @Test
    public void testPartialRead() throws IOException {
        GoogleCloudStorage rawStorage = this.createGoogleCloudStorage();
        int segmentSize = 553;
        int segmentCount = 5;
        StorageResourceId resourceId = new StorageResourceId(BUCKET_NAME, "testReadPartialObjects_Object");
        byte[] data = GoogleCloudStorageTestHelper.writeObject(rawStorage, resourceId, segmentCount * segmentSize);
        byte[][] readSegments = new byte[segmentCount][segmentSize];
        try (SeekableByteChannel readChannel = rawStorage.open(resourceId);){
            for (int i = 0; i < segmentCount; ++i) {
                ByteBuffer segmentBuffer = ByteBuffer.wrap(readSegments[i]);
                int bytesRead = readChannel.read(segmentBuffer);
                Truth.assertThat((Integer)bytesRead).isEqualTo((Object)segmentSize);
                byte[] expectedSegment = Arrays.copyOfRange(data, i * segmentSize, i * segmentSize + segmentSize);
                Truth.assertWithMessage((String)"Unexpected segment data read.").that(readSegments[i]).isEqualTo((Object)expectedSegment);
            }
        }
    }

    @Test
    public void testReadSeekToOffsetGreaterThanMinRangeRequestSize() throws IOException {
        AsyncWriteChannelOptions asyncWriteChannelOptions = AsyncWriteChannelOptions.builder().build();
        GoogleCloudStorage rawStorage = this.createGoogleCloudStorage(asyncWriteChannelOptions);
        StorageResourceId objectToCreate = new StorageResourceId(BUCKET_NAME, "testReadSeekToOffsetGreaterThanMinRangeRequestSize_Object");
        int objectSize = 20480;
        int inPlaceSeekLimit = 8192;
        int minRangeRequestSize = 4096;
        byte[] objectBytes = GoogleCloudStorageTestHelper.writeObject(rawStorage, objectToCreate, objectSize);
        int totalBytes = 1;
        byte[] readArray = new byte[totalBytes];
        GoogleCloudStorageReadOptions readOptions = GoogleCloudStorageReadOptions.builder().setInplaceSeekLimit((long)inPlaceSeekLimit).setMinRangeRequestSize(minRangeRequestSize).setGrpcChecksumsEnabled(false).setFadvise(GoogleCloudStorageReadOptions.Fadvise.RANDOM).build();
        SeekableByteChannel readableByteChannel = rawStorage.open(objectToCreate, readOptions);
        int newPosition = 7168;
        readableByteChannel.position(newPosition);
        ByteBuffer readBuffer = ByteBuffer.wrap(readArray);
        int bytesRead = readableByteChannel.read(readBuffer);
        byte[] trimmedObjectBytes = Arrays.copyOfRange(objectBytes, newPosition, newPosition + totalBytes);
        byte[] readBufferByteArray = Arrays.copyOf(readBuffer.array(), readBuffer.limit());
        Assert.assertEquals((long)totalBytes, (long)bytesRead);
        GoogleCloudStorageTestHelper.assertByteArrayEquals(trimmedObjectBytes, readBufferByteArray);
    }

    @Test
    public void testReadBeyondRangeWithFadviseRandom() throws IOException {
        AsyncWriteChannelOptions asyncWriteChannelOptions = AsyncWriteChannelOptions.builder().build();
        GoogleCloudStorage rawStorage = this.createGoogleCloudStorage(asyncWriteChannelOptions);
        StorageResourceId objectToCreate = new StorageResourceId(BUCKET_NAME, "testReadBeyondRangeWithFadviseRandom_Object");
        int objectSize = 20480;
        int inPlaceSeekLimit = 8192;
        int minRangeRequestSize = 4096;
        byte[] objectBytes = GoogleCloudStorageTestHelper.writeObject(rawStorage, objectToCreate, objectSize);
        int totalBytes = 2048;
        byte[] readArray = new byte[totalBytes];
        GoogleCloudStorageReadOptions readOptions = GoogleCloudStorageReadOptions.builder().setInplaceSeekLimit((long)inPlaceSeekLimit).setMinRangeRequestSize(minRangeRequestSize).setGrpcChecksumsEnabled(false).setFadvise(GoogleCloudStorageReadOptions.Fadvise.RANDOM).build();
        SeekableByteChannel readableByteChannel = rawStorage.open(objectToCreate, readOptions);
        int newPosition = 7168;
        readableByteChannel.position(newPosition);
        ByteBuffer readBuffer = ByteBuffer.wrap(readArray);
        int bytesRead = readableByteChannel.read(readBuffer);
        byte[] trimmedObjectBytes = Arrays.copyOfRange(objectBytes, newPosition, newPosition + totalBytes);
        byte[] readBufferByteArray = Arrays.copyOf(readBuffer.array(), readBuffer.limit());
        Assert.assertEquals((long)totalBytes, (long)bytesRead);
        GoogleCloudStorageTestHelper.assertByteArrayEquals(trimmedObjectBytes, readBufferByteArray);
    }

    @Test
    public void testReadBeyondRangeWithFadviseAuto() throws IOException {
        AsyncWriteChannelOptions asyncWriteChannelOptions = AsyncWriteChannelOptions.builder().build();
        GoogleCloudStorage rawStorage = this.createGoogleCloudStorage(asyncWriteChannelOptions);
        StorageResourceId objectToCreate = new StorageResourceId(BUCKET_NAME, "testReadBeyondRangeWithFadviseAuto_Object");
        int objectSize = 20480;
        int inPlaceSeekLimit = 8192;
        int minRangeRequestSize = 4096;
        byte[] objectBytes = GoogleCloudStorageTestHelper.writeObject(rawStorage, objectToCreate, objectSize);
        int totalBytes = 2048;
        byte[] readArray = new byte[totalBytes];
        GoogleCloudStorageReadOptions readOptions = GoogleCloudStorageReadOptions.builder().setInplaceSeekLimit((long)inPlaceSeekLimit).setMinRangeRequestSize(minRangeRequestSize).setGrpcChecksumsEnabled(false).setFadvise(GoogleCloudStorageReadOptions.Fadvise.AUTO).build();
        SeekableByteChannel readableByteChannel = rawStorage.open(objectToCreate, readOptions);
        int newPosition = 7168;
        readableByteChannel.position(newPosition);
        ByteBuffer readBuffer = ByteBuffer.wrap(readArray);
        int bytesRead = readableByteChannel.read(readBuffer);
        byte[] trimmedObjectBytes = Arrays.copyOfRange(objectBytes, newPosition, newPosition + totalBytes);
        byte[] readBufferByteArray = Arrays.copyOf(readBuffer.array(), readBuffer.limit());
        Assert.assertEquals((long)totalBytes, (long)bytesRead);
        GoogleCloudStorageTestHelper.assertByteArrayEquals(trimmedObjectBytes, readBufferByteArray);
    }

    @Test
    public void testReadBeyondRangeWithFadviseSequential() throws IOException {
        AsyncWriteChannelOptions asyncWriteChannelOptions = AsyncWriteChannelOptions.builder().build();
        GoogleCloudStorage rawStorage = this.createGoogleCloudStorage(asyncWriteChannelOptions);
        StorageResourceId objectToCreate = new StorageResourceId(BUCKET_NAME, "testReadBeyondRangeWithFadviseSequential_Object");
        int objectSize = 20480;
        int inPlaceSeekLimit = 8192;
        int minRangeRequestSize = 4096;
        byte[] objectBytes = GoogleCloudStorageTestHelper.writeObject(rawStorage, objectToCreate, objectSize);
        int totalBytes = 2048;
        byte[] readArray = new byte[totalBytes];
        GoogleCloudStorageReadOptions readOptions = GoogleCloudStorageReadOptions.builder().setInplaceSeekLimit((long)inPlaceSeekLimit).setMinRangeRequestSize(minRangeRequestSize).setGrpcChecksumsEnabled(false).setFadvise(GoogleCloudStorageReadOptions.Fadvise.SEQUENTIAL).build();
        SeekableByteChannel readableByteChannel = rawStorage.open(objectToCreate, readOptions);
        int newPosition = 7168;
        readableByteChannel.position(newPosition);
        ByteBuffer readBuffer = ByteBuffer.wrap(readArray);
        int bytesRead = readableByteChannel.read(readBuffer);
        byte[] trimmedObjectBytes = Arrays.copyOfRange(objectBytes, newPosition, newPosition + totalBytes);
        byte[] readBufferByteArray = Arrays.copyOf(readBuffer.array(), readBuffer.limit());
        Assert.assertEquals((long)totalBytes, (long)bytesRead);
        GoogleCloudStorageTestHelper.assertByteArrayEquals(trimmedObjectBytes, readBufferByteArray);
    }

    @Test
    public void testChannelClosedException() throws IOException {
        GoogleCloudStorage rawStorage = this.createGoogleCloudStorage();
        int totalBytes = 1200;
        StorageResourceId resourceId = new StorageResourceId(BUCKET_NAME, "testChannelClosedException_Object");
        GoogleCloudStorageTestHelper.writeObject(rawStorage, resourceId, totalBytes);
        byte[] readArray = new byte[totalBytes];
        SeekableByteChannel readableByteChannel = rawStorage.open(resourceId);
        ByteBuffer readBuffer = ByteBuffer.wrap(readArray);
        readBuffer.limit(5);
        readableByteChannel.read(readBuffer);
        Truth.assertThat((Long)readableByteChannel.position()).isEqualTo((Object)readBuffer.position());
        readableByteChannel.close();
        readBuffer.clear();
        Assert.assertThrows(ClosedChannelException.class, () -> readableByteChannel.read(readBuffer));
    }
}

