/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.cluster.node;

import java.io.IOException;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.SortedSet;
import java.util.TreeSet;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.elasticsearch.Version;
import org.elasticsearch.cluster.node.DiscoveryNodeRole;
import org.elasticsearch.common.UUIDs;
import org.elasticsearch.common.io.stream.StreamInput;
import org.elasticsearch.common.io.stream.StreamOutput;
import org.elasticsearch.common.io.stream.Writeable;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.transport.TransportAddress;
import org.elasticsearch.node.Node;
import org.elasticsearch.node.NodeRoleSettings;
import org.elasticsearch.xcontent.ToXContent;
import org.elasticsearch.xcontent.ToXContentFragment;
import org.elasticsearch.xcontent.XContentBuilder;

public class DiscoveryNode
implements Writeable,
ToXContentFragment {
    static final String COORDINATING_ONLY = "coordinating_only";
    private final String nodeName;
    private final String nodeId;
    private final String ephemeralId;
    private final String hostName;
    private final String hostAddress;
    private final TransportAddress address;
    private final Map<String, String> attributes;
    private final Version version;
    private final SortedSet<DiscoveryNodeRole> roles;
    private static Map<String, DiscoveryNodeRole> roleMap = DiscoveryNode.rolesToMap(DiscoveryNodeRole.BUILT_IN_ROLES.stream());

    public static boolean nodeRequiresLocalStorage(Settings settings) {
        boolean localStorageEnable = Node.NODE_LOCAL_STORAGE_SETTING.get(settings);
        if (!localStorageEnable && (DiscoveryNode.canContainData(settings) || DiscoveryNode.isMasterNode(settings))) {
            throw new IllegalArgumentException("storage can not be disabled for master and data nodes");
        }
        return localStorageEnable;
    }

    public static boolean hasRole(Settings settings, DiscoveryNodeRole role) {
        if (settings.hasValue("node.roles")) {
            return settings.getAsList("node.roles").contains(role.roleName());
        }
        if (role.legacySetting() != null && settings.hasValue(role.legacySetting().getKey())) {
            return role.legacySetting().get(settings);
        }
        return role.isEnabledByDefault(settings);
    }

    public static boolean isMasterNode(Settings settings) {
        return DiscoveryNode.hasRole(settings, DiscoveryNodeRole.MASTER_ROLE);
    }

    public static boolean hasDataRole(Settings settings) {
        return DiscoveryNode.hasRole(settings, DiscoveryNodeRole.DATA_ROLE);
    }

    public static boolean canContainData(Settings settings) {
        return DiscoveryNode.getRolesFromSettings(settings).stream().anyMatch(DiscoveryNodeRole::canContainData);
    }

    public static boolean isIngestNode(Settings settings) {
        return DiscoveryNode.hasRole(settings, DiscoveryNodeRole.INGEST_ROLE);
    }

    public static boolean isRemoteClusterClient(Settings settings) {
        return DiscoveryNode.hasRole(settings, DiscoveryNodeRole.REMOTE_CLUSTER_CLIENT_ROLE);
    }

    private static boolean isDedicatedFrozenRoles(Set<DiscoveryNodeRole> roles) {
        return roles.contains(DiscoveryNodeRole.DATA_FROZEN_NODE_ROLE) && !roles.stream().filter(DiscoveryNodeRole::canContainData).anyMatch(r -> r != DiscoveryNodeRole.DATA_FROZEN_NODE_ROLE);
    }

    public static boolean isDedicatedFrozenNode(Settings settings) {
        return DiscoveryNode.isDedicatedFrozenRoles(DiscoveryNode.getRolesFromSettings(settings));
    }

    public DiscoveryNode(String id, TransportAddress address, Version version) {
        this(id, address, Collections.emptyMap(), DiscoveryNodeRole.BUILT_IN_ROLES, version);
    }

    public DiscoveryNode(String id, TransportAddress address, Map<String, String> attributes, Set<DiscoveryNodeRole> roles, Version version) {
        this("", id, address, attributes, roles, version);
    }

    public DiscoveryNode(String nodeName, String nodeId, TransportAddress address, Map<String, String> attributes, Set<DiscoveryNodeRole> roles, Version version) {
        this(nodeName, nodeId, UUIDs.randomBase64UUID(), address.address().getHostString(), address.getAddress(), address, attributes, roles, version);
    }

    public DiscoveryNode(String nodeName, String nodeId, String ephemeralId, String hostName, String hostAddress, TransportAddress address, Map<String, String> attributes, Set<DiscoveryNodeRole> roles, Version version) {
        this.nodeName = nodeName != null ? nodeName.intern() : "";
        this.nodeId = nodeId.intern();
        this.ephemeralId = ephemeralId.intern();
        this.hostName = hostName.intern();
        this.hostAddress = hostAddress.intern();
        this.address = address;
        this.version = version == null ? Version.CURRENT : version;
        this.attributes = Collections.unmodifiableMap(attributes);
        assert (roleMap.values().stream().noneMatch(role -> attributes.containsKey(role.roleName()))) : "Node roles must not be provided as attributes but saw attributes " + attributes;
        this.roles = Collections.unmodifiableSortedSet(new TreeSet<DiscoveryNodeRole>(roles));
    }

    public static DiscoveryNode createLocal(Settings settings, TransportAddress publishAddress, String nodeId) {
        Map<String, String> attributes = Node.NODE_ATTRIBUTES.getAsMap(settings);
        Set<DiscoveryNodeRole> roles = DiscoveryNode.getRolesFromSettings(settings);
        return new DiscoveryNode(Node.NODE_NAME_SETTING.get(settings), nodeId, publishAddress, attributes, roles, Version.CURRENT);
    }

    public static Set<DiscoveryNodeRole> getRolesFromSettings(Settings settings) {
        boolean usesLegacySettings = DiscoveryNode.getPossibleRoles().stream().anyMatch(s2 -> s2.legacySetting() != null && s2.legacySetting().exists(settings));
        if (NodeRoleSettings.NODE_ROLES_SETTING.exists(settings) || !usesLegacySettings) {
            DiscoveryNode.validateLegacySettings(settings, roleMap);
            return Collections.unmodifiableSet(new HashSet(NodeRoleSettings.NODE_ROLES_SETTING.get(settings)));
        }
        return roleMap.values().stream().filter(s2 -> s2.legacySetting() != null && s2.legacySetting().get(settings) != false).collect(Collectors.toSet());
    }

    private static void validateLegacySettings(Settings settings, Map<String, DiscoveryNodeRole> roleMap) {
        for (DiscoveryNodeRole role : roleMap.values()) {
            if (role.legacySetting() == null || !role.legacySetting().exists(settings)) continue;
            String message = String.format(Locale.ROOT, "can not explicitly configure node roles and use legacy role setting [%s]=[%s]", role.legacySetting().getKey(), role.legacySetting().get(settings));
            throw new IllegalArgumentException(message);
        }
    }

    public DiscoveryNode(StreamInput in) throws IOException {
        this.nodeName = in.readString().intern();
        this.nodeId = in.readString().intern();
        this.ephemeralId = in.readString().intern();
        this.hostName = in.readString().intern();
        this.hostAddress = in.readString().intern();
        this.address = new TransportAddress(in);
        int size = in.readVInt();
        this.attributes = new HashMap<String, String>(size);
        for (int i = 0; i < size; ++i) {
            this.attributes.put(in.readString(), in.readString());
        }
        int rolesSize = in.readVInt();
        TreeSet<DiscoveryNodeRole> roles = new TreeSet<DiscoveryNodeRole>();
        if (in.getVersion().onOrAfter(Version.V_7_3_0)) {
            for (int i = 0; i < rolesSize; ++i) {
                String roleName = in.readString();
                String roleNameAbbreviation = in.readString();
                boolean canContainData = in.getVersion().onOrAfter(Version.V_7_10_0) ? in.readBoolean() : roleName.equals(DiscoveryNodeRole.DATA_ROLE.roleName());
                DiscoveryNodeRole role = roleMap.get(roleName);
                if (role == null) {
                    roles.add(new DiscoveryNodeRole.UnknownRole(roleName, roleNameAbbreviation, canContainData));
                    continue;
                }
                assert (roleName.equals(role.roleName())) : "role name [" + roleName + "] does not match role [" + role.roleName() + "]";
                assert (roleNameAbbreviation.equals(role.roleNameAbbreviation())) : "role name abbreviation [" + roleName + "] does not match role [" + role.roleNameAbbreviation() + "]";
                roles.add(role);
            }
        } else {
            block7: for (int i = 0; i < rolesSize; ++i) {
                LegacyRole legacyRole = in.readEnum(LegacyRole.class);
                switch (legacyRole) {
                    case MASTER: {
                        roles.add(DiscoveryNodeRole.MASTER_ROLE);
                        continue block7;
                    }
                    case DATA: {
                        roles.add(DiscoveryNodeRole.DATA_ROLE);
                        continue block7;
                    }
                    case INGEST: {
                        roles.add(DiscoveryNodeRole.INGEST_ROLE);
                        continue block7;
                    }
                    default: {
                        throw new AssertionError((Object)legacyRole.roleName());
                    }
                }
            }
        }
        this.roles = Collections.unmodifiableSortedSet(roles);
        this.version = Version.readVersion(in);
    }

    @Override
    public void writeTo(StreamOutput out) throws IOException {
        out.writeString(this.nodeName);
        out.writeString(this.nodeId);
        out.writeString(this.ephemeralId);
        out.writeString(this.hostName);
        out.writeString(this.hostAddress);
        this.address.writeTo(out);
        out.writeVInt(this.attributes.size());
        for (Map.Entry<String, String> entry : this.attributes.entrySet()) {
            out.writeString(entry.getKey());
            out.writeString(entry.getValue());
        }
        if (out.getVersion().onOrAfter(Version.V_7_3_0)) {
            out.writeVInt(this.roles.size());
            for (DiscoveryNodeRole role2 : this.roles) {
                DiscoveryNodeRole compatibleRole = role2.getCompatibilityRole(out.getVersion());
                out.writeString(compatibleRole.roleName());
                out.writeString(compatibleRole.roleNameAbbreviation());
                if (!out.getVersion().onOrAfter(Version.V_7_10_0)) continue;
                out.writeBoolean(compatibleRole.canContainData());
            }
        } else {
            Set rolesToWrite = this.roles.stream().map(role -> role.getCompatibilityRole(out.getVersion())).filter(DiscoveryNodeRole.LEGACY_ROLES::contains).collect(Collectors.toSet());
            out.writeVInt(rolesToWrite.size());
            for (DiscoveryNodeRole role3 : rolesToWrite) {
                if (role3 == DiscoveryNodeRole.MASTER_ROLE) {
                    out.writeEnum(LegacyRole.MASTER);
                    continue;
                }
                if (role3 == DiscoveryNodeRole.DATA_ROLE) {
                    out.writeEnum(LegacyRole.DATA);
                    continue;
                }
                if (role3 != DiscoveryNodeRole.INGEST_ROLE) continue;
                out.writeEnum(LegacyRole.INGEST);
            }
        }
        Version.writeVersion(this.version, out);
    }

    public TransportAddress getAddress() {
        return this.address;
    }

    public String getId() {
        return this.nodeId;
    }

    public String getEphemeralId() {
        return this.ephemeralId;
    }

    public String getName() {
        return this.nodeName;
    }

    public Map<String, String> getAttributes() {
        return this.attributes;
    }

    public boolean canContainData() {
        return this.roles.stream().anyMatch(DiscoveryNodeRole::canContainData);
    }

    public boolean isMasterNode() {
        return this.roles.contains(DiscoveryNodeRole.MASTER_ROLE);
    }

    public boolean isIngestNode() {
        return this.roles.contains(DiscoveryNodeRole.INGEST_ROLE);
    }

    public boolean isRemoteClusterClient() {
        return this.roles.contains(DiscoveryNodeRole.REMOTE_CLUSTER_CLIENT_ROLE);
    }

    public boolean isDedicatedFrozenNode() {
        return DiscoveryNode.isDedicatedFrozenRoles(this.getRoles());
    }

    public Set<DiscoveryNodeRole> getRoles() {
        return this.roles;
    }

    public Version getVersion() {
        return this.version;
    }

    public String getHostName() {
        return this.hostName;
    }

    public String getHostAddress() {
        return this.hostAddress;
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || this.getClass() != o.getClass()) {
            return false;
        }
        DiscoveryNode that = (DiscoveryNode)o;
        return this.ephemeralId.equals(that.ephemeralId);
    }

    public int hashCode() {
        return this.ephemeralId.hashCode();
    }

    public String toString() {
        StringBuilder sb = new StringBuilder();
        this.appendDescriptionWithoutAttributes(sb);
        if (!this.attributes.isEmpty()) {
            sb.append(this.attributes);
        }
        return sb.toString();
    }

    public void appendDescriptionWithoutAttributes(StringBuilder stringBuilder) {
        if (this.nodeName.length() > 0) {
            stringBuilder.append('{').append(this.nodeName).append('}');
        }
        stringBuilder.append('{').append(this.nodeId).append('}');
        stringBuilder.append('{').append(this.ephemeralId).append('}');
        stringBuilder.append('{').append(this.hostName).append('}');
        stringBuilder.append('{').append(this.address).append('}');
        if (!this.roles.isEmpty()) {
            stringBuilder.append('{');
            this.roles.stream().map(DiscoveryNodeRole::roleNameAbbreviation).sorted().forEach(stringBuilder::append);
            stringBuilder.append('}');
        }
    }

    public String descriptionWithoutAttributes() {
        StringBuilder stringBuilder = new StringBuilder();
        this.appendDescriptionWithoutAttributes(stringBuilder);
        return stringBuilder.toString();
    }

    @Override
    public XContentBuilder toXContent(XContentBuilder builder, ToXContent.Params params) throws IOException {
        builder.startObject(this.getId());
        builder.field("name", this.getName());
        builder.field("ephemeral_id", this.getEphemeralId());
        builder.field("transport_address", this.getAddress().toString());
        builder.startObject("attributes");
        for (Map.Entry<String, String> entry : this.attributes.entrySet()) {
            builder.field(entry.getKey(), entry.getValue());
        }
        builder.endObject();
        builder.startArray("roles");
        for (DiscoveryNodeRole role : this.roles) {
            builder.value(role.roleName());
        }
        builder.endArray();
        builder.endObject();
        return builder;
    }

    private static Map<String, DiscoveryNodeRole> rolesToMap(Stream<DiscoveryNodeRole> roles) {
        return Collections.unmodifiableMap(roles.collect(Collectors.toMap(DiscoveryNodeRole::roleName, Function.identity())));
    }

    public static DiscoveryNodeRole getRoleFromRoleName(String roleName) {
        if (!roleMap.containsKey(roleName)) {
            throw new IllegalArgumentException("unknown role [" + roleName + "]");
        }
        return roleMap.get(roleName);
    }

    public static Collection<DiscoveryNodeRole> getPossibleRoles() {
        return roleMap.values();
    }

    public static void setAdditionalRoles(Set<DiscoveryNodeRole> additionalRoles) {
        assert (additionalRoles.stream().allMatch(r -> r.legacySetting() == null || r.legacySetting().isDeprecated())) : additionalRoles;
        Map<String, DiscoveryNodeRole> roleNameToPossibleRoles = DiscoveryNode.rolesToMap(Stream.concat(DiscoveryNodeRole.BUILT_IN_ROLES.stream(), additionalRoles.stream()));
        Map roleNameAbbreviationToPossibleRoles = Collections.unmodifiableMap(roleNameToPossibleRoles.values().stream().collect(Collectors.toMap(DiscoveryNodeRole::roleNameAbbreviation, Function.identity())));
        assert (roleNameToPossibleRoles.size() == roleNameAbbreviationToPossibleRoles.size()) : "roles by name [" + roleNameToPossibleRoles + "], roles by name abbreviation [" + roleNameAbbreviationToPossibleRoles + "]";
        roleMap = roleNameToPossibleRoles;
    }

    public static Set<String> getPossibleRoleNames() {
        return roleMap.keySet();
    }

    private static enum LegacyRole {
        MASTER("master"),
        DATA("data"),
        INGEST("ingest");

        private final String roleName;

        private LegacyRole(String roleName) {
            this.roleName = roleName;
        }

        public String roleName() {
            return this.roleName;
        }
    }
}

