/*
 * KUtil
 * Copyright (C) 2021-2022 Moritz Zwerger
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this program. If not, see <https://www.gnu.org/licenses/>.
 */

package de.bixilon.kutil.avg

import de.bixilon.kutil.concurrent.lock.simple.SimpleLock
import de.bixilon.kutil.time.TimeUtil

class DoubleAverage(
    override val avgNanos: Long,
    override val default: Double = 0.0,
) : Average<Double> {
    private val lock = SimpleLock()
    private val data: MutableList<AvgEntry<Double>> = mutableListOf()
    private var updated = false
    private var lastAVG = 0.0

    override val avg: Double
        get() {
            lock.lock()
            cleanup()
            if (!updated) {
                lock.unlock()
                return lastAVG
            }
            if (data.size == 0) {
                lock.unlock()
                return this.default
            }

            var total = 0.0
            for (entry in data) {
                total += entry.value
            }

            lastAVG = total / data.size
            updated = false
            lock.unlock()
            return lastAVG
        }

    private fun cleanup(nanos: Long = TimeUtil.nanos()) {
        this.updated = this.cleanup(nanos, this.data) || this.updated
    }


    override fun add(value: Double) {
        val nanos = TimeUtil.nanos()
        lock.lock()
        cleanup(nanos)
        data += AvgEntry(nanos, value)
        updated = true
        lock.unlock()
    }
}
