/*
 * KUtil
 * Copyright (C) 2021-2022 Moritz Zwerger
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this program. If not, see <https://www.gnu.org/licenses/>.
 */

package de.bixilon.kutil.concurrent.schedule

import de.bixilon.kutil.concurrent.lock.simple.SimpleLock
import de.bixilon.kutil.concurrent.pool.DefaultThreadPool
import de.bixilon.kutil.concurrent.pool.runnable.ForcePooledRunnable
import de.bixilon.kutil.reflection.ReflectionUtil.forceInit
import de.bixilon.kutil.shutdown.AbstractShutdownReason
import de.bixilon.kutil.shutdown.ShutdownManager
import de.bixilon.kutil.time.TimeUtil.millis
import java.util.concurrent.TimeUnit

object TaskScheduler {
    private val tasks: MutableSet<SchedulerTask> = mutableSetOf()
    private val lock = SimpleLock()

    init {
        Thread({ Thread.currentThread().setOptions(); startChecking() }, "TaskSchedulerThread").start()
        DefaultThreadPool::class.java.forceInit()
    }

    private fun Thread.setOptions() {
        priority = Thread.MAX_PRIORITY
        uncaughtExceptionHandler = Thread.UncaughtExceptionHandler { _: Thread, error: Throwable ->
            error.printStackTrace()
            ShutdownManager.shutdown("TaskScheduler quit unexpectedly!", AbstractShutdownReason.CRASH)
        }
    }

    private fun startChecking() {
        while (true) {
            lock.lock()
            val time = millis()
            val iterator = tasks.iterator()
            for (task in iterator) {
                if (task.isExecuting) continue
                val due = task.getNextRunDelay(time)
                if (due > 0) {
                    continue
                }

                DefaultThreadPool += ForcePooledRunnable(priority = task.priority) { run(time, task) }
                if (task is QueuedTask) {
                    iterator.remove()
                }
            }
            lock.unlock()
            Thread.sleep(1)
        }
    }

    private fun run(queuedTime: Long, task: SchedulerTask) {
        if (task is RepeatedTask) {
            if (!task.lock.tryLock(1L, TimeUnit.MILLISECONDS)) {
                return
            }
            if (task.isExecuting) {
                task.lock.unlock()
                return
            }
        }
        val time = millis()

        if (time - queuedTime >= task.maxDelay) {
            if (task is RepeatedTask) task.lock.unlock()
            return
        }
        try {
            task.thread = Thread.currentThread()
            task.runnable.run()
        } catch (exception: Exception) {
            exception.printStackTrace()
        }
        task.thread = null

        if (task is RepeatedTask) {
            task.executions++
            task.lastExecution = queuedTime
            task.lock.unlock()
        }
    }

    fun runLater(delay: Int, runnable: Runnable): QueuedTask {
        val task = QueuedTask(
            delay = delay,
            runnable = runnable,
        )
        addTask(task)
        return task
    }

    fun addTask(task: SchedulerTask) {
        lock.lock()
        tasks += task
        lock.unlock()
    }

    operator fun plusAssign(task: SchedulerTask) = addTask(task)

    fun removeTask(task: SchedulerTask) {
        lock.lock()
        tasks -= task
        lock.unlock()
    }

    operator fun minusAssign(task: SchedulerTask) = removeTask(task)
}
