/*
 * KUtil
 * Copyright (C) 2021-2022 Moritz Zwerger
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this program. If not, see <https://www.gnu.org/licenses/>.
 */

package de.bixilon.kutil.string

import java.nio.charset.StandardCharsets

object StringUtil {
    val DEFAULT_CHARSET = StandardCharsets.UTF_8
    private val SNAKE_CASE_REGEX = "(.)(\\p{Upper})".toRegex()

    fun String.toSnakeCase(): String {
        return replace(SNAKE_CASE_REGEX, "$1_$2").lowercase()
    }

    fun String.isLowercase(): Boolean {
        for (code in codePoints()) {
            if (Character.isUpperCase(code)) {
                return false
            }
        }
        return true
    }

    fun String.isUppercase(): Boolean {
        for (code in codePoints()) {
            if (Character.isLowerCase(code)) {
                return false
            }
        }
        return true
    }

    fun String.formatPlaceholder(format: Map<String, Any?>): String {
        var output = this
        for ((key, value) in format) {
            output = output.replace("\${$key}", value.toString())
        }
        return output
    }

    fun String.formatPlaceholder(vararg format: Pair<String, Any?>): String {
        var output = this
        for ((key, value) in format) {
            output = output.replace("\${$key}", value.toString())
        }
        return output
    }


    fun String.getBetween(first: String, second: String): String {
        val result: String = this.substring(this.indexOf(first) + first.length)
        return result.substring(0, result.indexOf(second))
    }


    fun String.codePointAtOrNull(index: Int): Int? {
        if (index < 0 || index >= this.length) {
            return null
        }
        return this.codePointAt(index)
    }

    fun String.truncate(length: Int): String {
        if (length < 0) {
            throw IllegalArgumentException("Length must not be < 0: $length")
        }
        if (this.length <= length) {
            return this
        }
        return this.substring(0, length)
    }


    fun String.fill(char: Char, length: Int): String {
        if (this.length >= length) return this
        val fill = char.toString().repeat(length - this.length)

        return fill + this
    }
}
