/*
 * Copyright (c) 2016 Leibniz Institute of Plant Genetics and Crop Plant Research (IPK), Gatersleben, Germany.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Creative Commons Attribution-NoDerivatives 4.0 International (CC BY-ND 4.0)
 * which accompanies this distribution, and is available at http://creativecommons.org/licenses/by-nd/4.0/
 *
 * Contributors:
 *      Leibniz Institute of Plant Genetics and Crop Plant Research (IPK), Gatersleben, Germany - initial API and implementation
 */
package de.ipk_gatersleben.bit.bi.edal.primary_data.reference;

import java.io.StringWriter;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.Calendar;
import java.util.UUID;

import javax.xml.bind.JAXBException;

import de.ipk_gatersleben.bit.bi.edal.primary_data.DataManager;
import de.ipk_gatersleben.bit.bi.edal.primary_data.EdalConfiguration;
import de.ipk_gatersleben.bit.bi.edal.primary_data.EdalHttpFunctions;
import de.ipk_gatersleben.bit.bi.edal.primary_data.EdalJettyServer;
import de.ipk_gatersleben.bit.bi.edal.primary_data.file.EdalException;
import de.ipk_gatersleben.bit.bi.edal.primary_data.file.PrimaryDataEntityException;
import de.ipk_gatersleben.bit.bi.edal.primary_data.file.PrimaryDataEntityVersion;
import de.ipk_gatersleben.bit.bi.edal.primary_data.file.PublicReference;
import de.ipk_gatersleben.bit.bi.edal.primary_data.file.implementation.PublicReferenceImplementation;
import de.ipk_gatersleben.bit.bi.edal.primary_data.reference.datacite.DataCiteException;
import de.ipk_gatersleben.bit.bi.edal.primary_data.reference.datacite.DataCiteMDSConnector;
import de.ipk_gatersleben.bit.bi.edal.primary_data.reference.datacite.DataCiteSearchConnector;
import de.ipk_gatersleben.bit.bi.edal.primary_data.reference.datacite.DataCiteXmlMapper;
import de.ipk_gatersleben.bit.bi.edal.primary_data.reference.datacite.XmlFunctions;
import de.ipk_gatersleben.bit.bi.edal.primary_data.reference.datacite.xml.XmlIdentifier;
import de.ipk_gatersleben.bit.bi.edal.primary_data.reference.datacite.xml.XmlResource;

/**
 * DataCite implementation of the {@link EdalReferenceable} interface, to
 * connect the system with the DataCite interface.
 * 
 * @author arendd
 */
@SuppressWarnings("unused")
public class DataCiteReference implements EdalReferenceable {

	private static final String TEST_URL = "http://doi.ipk-gatersleben.de/testdata/demo_doi_landingpage/";
	private static final String TEST_DOI = EdalConfiguration.DATACITE_TESTPREFIX + "/EDALTEST/";

	/** {@inheritDoc} */
	@Override
	public String acceptApprovalRequest(PublicReference publicReference) throws EdalApprovalException {

		synchronized (DataCiteReference.class) {

			if (!DataManager.getConfiguration().isInTestMode()) {

				try {
					int year = Calendar.getInstance().get(Calendar.YEAR);

					String doi = "";
					try {
						doi = new DataCiteSearchConnector(DataManager.getConfiguration()).generateNewDOI(year);
					} catch (DataCiteException e) {
						throw new EdalApprovalException("unable to generate new DOI", e);
					}

					DataManager.getImplProv().getLogger().info("Next Free DOI from DataCite: " + doi);

					StringBuffer dataCiteXml = generateDataCiteXML(publicReference, doi);

					DataManager.getImplProv().getLogger().info("Generated DataCite XML : \n" + dataCiteXml);

					String internalURL = createLandingPageURL(publicReference).toString();

					DataManager.getImplProv().getLogger().info("Validating : " + doi + "...");

					this.validateMetaData(publicReference.getVersion());

					DataManager.getImplProv().getLogger().info("Validation successful !");

					DataCiteMDSConnector connector = new DataCiteMDSConnector(DataManager.getConfiguration());
					try {
						DataManager.getImplProv().getLogger().info("Posting MetaData for : " + doi + "...");
						connector.postMetadata(XmlFunctions.parse(dataCiteXml.toString()));
						DataManager.getImplProv().getLogger().info("Posting URL : " + internalURL + "...");
						connector.postDOI(doi, internalURL);
						DataManager.getImplProv().getLogger().info("Post was successful for : " + doi);
					} catch (DataCiteException e) {
						throw new EdalApprovalException("unable to post metadata and DOI to DataCite : ", e);
					}

					try {
						DataManager.getImplProv().getApprovalServiceProvider().newInstance()
								.storeNewDOI(publicReference, doi, year);
					} catch (InstantiationException | IllegalAccessException e) {
						throw new EdalApprovalException("unable to store a new DOI for the PublicReference");
					}

					return doi;

				} catch (EdalException | EdalPublicationMetaDataException e) {
					e.printStackTrace();
					throw new EdalApprovalException("unable to accept approvalRequest", e);
				}
			} else {
				try {

					int year = Calendar.getInstance().get(Calendar.YEAR);

					String testDoi = TEST_DOI + year + "/" + UUID.randomUUID();

					StringBuffer dataCiteXml = generateDataCiteXML(publicReference, testDoi);

					this.validateMetaData(publicReference.getVersion());

					System.out.println(dataCiteXml);
					System.out.println(testDoi);

					// DataCiteMDSConnector connector = new
					// DataCiteMDSConnector(
					// DataManager.getConfiguration());
					//
					// try {
					// connector.postMetadata(XmlFunctions.parse(dataCiteXml
					// .toString()));
					// connector.postDOI(TEST_DOI, TEST_URL);
					//
					// } catch (DataCiteException e) {
					// throw new EdalApprovalException(
					// "unable to post metadata and DOI to DataCite: "
					// + e.getMessage(), e);
					// }

					try {
						DataManager.getImplProv().getApprovalServiceProvider().newInstance()
								.storeNewDOI(publicReference, testDoi, year);
					} catch (InstantiationException | IllegalAccessException e) {
						throw new EdalApprovalException("unable to store a new DOI for the PublicReference");
					}

					DataManager.getImplProv().getLogger().warn(
							"Your PublicReference was not posted to DataCite, because you are running in Test-Mode");

					return testDoi;

				} catch (EdalException | EdalPublicationMetaDataException e) {
					throw new EdalApprovalException("unable to accept approvalRequest:" + e.getMessage(), e.getCause());
				}
			}
		}
	}

	/**
	 * Generate a DataCiteX XML document as {@link String} from a
	 * {@link PublicReference} object.
	 * 
	 * @param publicReference
	 *            the reference to generate a XML document.
	 * @param doi
	 *            the new DOI for this {@link PublicReference}.
	 * @return the XML document as {@link String}.
	 * @throws EdalPublicationMetaDataException
	 *             if unable to marshal the meta data to XML.
	 */
	private StringBuffer generateDataCiteXML(PublicReference publicReference, String doi)
			throws EdalPublicationMetaDataException {

		DataCiteXmlMapper xmlMapper = new DataCiteXmlMapper(publicReference.getVersion());

		XmlResource xmlResource = xmlMapper.createXmlResource();

		xmlResource.setIdentifier(new XmlIdentifier(doi));

		StringWriter strw = new StringWriter();

		try {
			xmlMapper.createXmlMarshaller().marshal(xmlResource, strw);
		} catch (JAXBException e) {
			throw new EdalPublicationMetaDataException("Unable to marshall meta data from PublicReference", e);
		}
		return strw.getBuffer();
	}

	/**
	 * Create the landing page string without the server part (host and port)
	 * for a given PublicReference.
	 * 
	 * @param reference
	 *            the {@link PublicReference} corresponding to this landing
	 *            page.
	 * @return the landing page
	 */
	private String createLandingPageString(PublicReference reference) {

		String landingpage = EdalJettyServer.EDAL_PATH_SEPARATOR + reference.getIdentifierType().toString()
				+ EdalJettyServer.EDAL_PATH_SEPARATOR + reference.getInternalID() + EdalJettyServer.EDAL_PATH_SEPARATOR
				+ reference.getVersion().getEntity().getID() + EdalJettyServer.EDAL_PATH_SEPARATOR
				+ reference.getVersion().getRevision();

		return landingpage;
	}

	/**
	 * Create the complete landing page {@link URL} including server part to
	 * send it in an email to the requested author.
	 * 
	 * @param reference
	 *            the {@link PublicReference} corresponding to this {@link URL}.
	 * @return the complete URL
	 * @throws EdalApprovalException
	 *             if unable to create the {@link URL}.
	 */
	private URL createLandingPageURL(PublicReference reference) throws EdalApprovalException {
		URL url = null;
		try {
			url = EdalJettyServer.getServerURL();
			return new URL(url, createLandingPageString(reference));

		} catch (EdalException | MalformedURLException e) {
			throw new EdalApprovalException("unable to create URL for the landing page : " + e.getMessage(), e);
		}
	}

	/**
	 * {@inheritDoc}
	 * <p>
	 * no implementation for DataCite necessary: impossible to reserve IDs
	 */
	@Override
	public void rejectApprovalRequest(PublicReference publicReference) throws EdalApprovalException {
	}

	/**
	 * {@inheritDoc}
	 * <p>
	 * Check the
	 * {@link de.ipk_gatersleben.bit.bi.edal.primary_data.metadata.MetaData}
	 * schema against the {@link DataCiteXmlMapper} schema.
	 * 
	 * @throws EdalPublicationMetaDataException
	 *             if validation failed.
	 * 
	 */
	@Override
	public void validateMetaData(PrimaryDataEntityVersion entityVersion) throws EdalPublicationMetaDataException {

		DataCiteXmlMapper mapper = new DataCiteXmlMapper(entityVersion);
		XmlResource resource = mapper.createXmlResource();
		mapper.validateSchema(resource);

	}

	@Override
	public void validate(PrimaryDataEntityVersion entityVersion) throws EdalPublicationMetaDataException {
		// TODO Auto-generated method stub

	}
}