/*
 * Copyright (c) 2016 Leibniz Institute of Plant Genetics and Crop Plant Research (IPK), Gatersleben, Germany.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Creative Commons Attribution-NoDerivatives 4.0 International (CC BY-ND 4.0)
 * which accompanies this distribution, and is available at http://creativecommons.org/licenses/by-nd/4.0/
 *
 * Contributors:
 *      Leibniz Institute of Plant Genetics and Crop Plant Research (IPK), Gatersleben, Germany - initial API and implementation
 */
package de.ipk_gatersleben.bit.bi.edal.primary_data.reference.datacite;

import java.io.IOException;

import org.apache.http.HttpHost;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.solr.client.solrj.SolrServerException;
import org.apache.solr.client.solrj.impl.HttpSolrClient;
import org.apache.solr.common.params.CommonParams;
import org.apache.solr.common.params.ModifiableSolrParams;

import de.ipk_gatersleben.bit.bi.edal.primary_data.DataManager;
import de.ipk_gatersleben.bit.bi.edal.primary_data.EdalConfiguration;
import de.ipk_gatersleben.bit.bi.edal.primary_data.EdalConfigurationException;
import de.ipk_gatersleben.bit.bi.edal.primary_data.file.EdalException;

/**
 * Connector to the Solr-API to search for DataCite objects.
 * 
 * @author arendd
 */
public class DataCiteSearchConnector {

	private String prefix = "";

	private HttpSolrClient dataciteSolrServer = null;

	public HttpSolrClient getDataciteSolrServer() {
		return dataciteSolrServer;
	}

	private EdalConfiguration configuration = null;

	/**
	 * Constructor to initialize a SolrServer to DateCite.
	 * 
	 * @param configuration
	 *            the {@link EdalConfiguration} to use
	 * 
	 * @throws DataCiteException
	 *             if unable to create SolrServer.
	 */
	public DataCiteSearchConnector(EdalConfiguration configuration) throws DataCiteException {

		this.configuration = configuration;

		HttpHost httpProxy = null;

		try {
			// HttpHost httpProxy = new
			// HttpHost(configuration.getHttpProxyHost(),
			// configuration.getHttpProxyPort());
			// httpClient.getParams().setParameter(ConnRoutePNames.DEFAULT_PROXY,
			// httpProxy);

			httpProxy = new HttpHost(configuration.getHttpProxyHost(), configuration.getHttpProxyPort());

		} catch (EdalConfigurationException e) {
			DataManager.getImplProv().getLogger().debug("SolrSearchServer use no proxy");
		}

		RequestConfig requestConfig = RequestConfig.custom().setProxy(httpProxy).build();

		CloseableHttpClient httpClient = HttpClientBuilder.create().useSystemProperties()
				.setDefaultRequestConfig(requestConfig).build();

		try {
			this.prefix = configuration.getDataCitePrefix();
		} catch (EdalConfigurationException e) {

			throw new DataCiteException("unable to load DataCite Prefix", e);
		}

		this.dataciteSolrServer = new HttpSolrClient.Builder(EdalConfiguration.DATACITE_SEARCH_URL)
				.withHttpClient(httpClient).build();

	}

	/**
	 * Get the number of currently registered DOIs
	 * 
	 * @param year
	 *            the current year.
	 * @param dataCiteInfix
	 * @return the number of DOIs
	 * @throws SolrServerException
	 *             if unable to query DOIs
	 */
	private int getNumberOfResolvableDOIs(int year, String dataCiteInfix) throws DataCiteException {

		ModifiableSolrParams solrParams = new ModifiableSolrParams();

		// solrParams.set(CommonParams.Q, this.prefix + " and " + year + " and "
		// + dataCiteInfix+ " and publicationYear:"+year);

		solrParams.set(CommonParams.Q,
				this.prefix + " and doi:" + this.prefix + "*" + dataCiteInfix + "*" + year + "*");

		solrParams.set(CommonParams.FL, "doi");

		solrParams.set(CommonParams.ROWS, Integer.MAX_VALUE);

		/*
		 * important parameter to remove the "/select" path at the beginning:
		 * http://search.datacite.org/api?q=10.5447 and IPK and 2012&fl=doi
		 */
		solrParams.set(CommonParams.QT, "/");

		int numberOfResult = 0;
		try {

			if (this.dataciteSolrServer.query(solrParams).getResults().size() == 0) {
				numberOfResult = 0;
			} else {
				numberOfResult = this.dataciteSolrServer.query(solrParams).getResults().size() - 1;
			}
		} catch (SolrServerException | IOException e) {
			throw new DataCiteException("unable to query the number of stored DOIs", e);
		}

		return numberOfResult;
	}

	/**
	 * Get the string for the data center of the given prefix
	 * 
	 * @param the
	 *            prefix for this data center
	 * @return the data center string
	 * @throws SolrServerException
	 *             if unable to query data center
	 */
	private String getDataCentreStringForPrefix(String prefix) throws DataCiteException {

		ModifiableSolrParams solrParams = new ModifiableSolrParams();
		solrParams.set(CommonParams.Q, prefix);
		solrParams.set(CommonParams.FL, "datacentre");
		solrParams.set(CommonParams.ROWS, 1);

		/*
		 * important parameter to remove the "/select" path at the beginning:
		 * http://search.datacite.org/api?q=10.5447&fl=datacentre&rows=1
		 */
		solrParams.set(CommonParams.QT, "/");

		try {

			String str = (String) this.dataciteSolrServer.query(solrParams).getResults().get(0).get("datacentre");

			String datacentre = str.substring(str.indexOf(".") + 1, str.indexOf("-") - 1);

			return datacentre;
		} catch (SolrServerException | IOException e) {
			throw new DataCiteException("unable to query the number of stored DOIs", e);
		}

	}

	/**
	 * Generate a new DOI by counting current DOI
	 * 
	 * @param year
	 *            the current year.
	 * @return the new DOI
	 * @throws DataCiteException
	 *             if unable to generate new DOI.
	 */
	public String generateNewDOI(int year) throws DataCiteException {

		try {

			if (this.configuration.getDoiInfix() != null) {

				DataCiteMDSConnector connector = new DataCiteMDSConnector(this.configuration);

				int nextFreeDoiNumber = connector.getNextFreeDOI(year,
						getNumberOfResolvableDOIs(year, this.configuration.getDoiInfix()),
						this.configuration.getDoiInfix());

				return this.prefix + "/" + this.configuration.getDoiInfix() + "/" + year + "/" + nextFreeDoiNumber;
			} else {

				DataCiteMDSConnector connector = new DataCiteMDSConnector(this.configuration);

				int nextFreeDoiNumber = connector.getNextFreeDOI(year,
						getNumberOfResolvableDOIs(year, getDataCentreStringForPrefix(this.prefix)),
						getDataCentreStringForPrefix(this.prefix));

				return this.prefix + "/" + getDataCentreStringForPrefix(this.prefix) + "/" + year + "/"
						+ nextFreeDoiNumber;
			}

		} catch (EdalException e) {
			throw new DataCiteException("unable to get number of stored DOIs", e);
		}

	}
}
