/*
 * Licensed to the Apache Software Foundation (ASF) under one or more contributor license
 * agreements. See the NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The ASF licenses this file to You under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance with the License. You may obtain a
 * copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under the License
 * is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express
 * or implied. See the License for the specific language governing permissions and limitations under
 * the License.
 */

package de.knightsoftnet.mtwidgets.client.ui.widget;

import de.knightsoftnet.validators.client.decorators.ExtendedValueBoxEditor;

import com.google.gwt.core.shared.GWT;
import com.google.gwt.editor.client.Editor;
import com.google.gwt.editor.client.IsEditor;
import com.google.gwt.editor.ui.client.adapters.ValueBoxEditor;
import com.google.gwt.event.dom.client.ClickEvent;
import com.google.gwt.event.logical.shared.ValueChangeEvent;
import com.google.gwt.event.logical.shared.ValueChangeHandler;
import com.google.gwt.event.shared.HandlerRegistration;
import com.google.gwt.uibinder.client.UiBinder;
import com.google.gwt.uibinder.client.UiField;
import com.google.gwt.uibinder.client.UiHandler;
import com.google.gwt.user.client.ui.Anchor;
import com.google.gwt.user.client.ui.Composite;
import com.google.gwt.user.client.ui.HasValue;
import com.google.gwt.user.client.ui.Widget;

import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;

/**
 * Input widget for page numbers used for pagination.
 *
 * @author Manfred Tremmel
 *
 */
public class PageNumberWithArrowsWidget extends Composite
    implements Editor<Pageable>, IsEditor<ValueBoxEditor<Pageable>>, HasValue<Pageable> {

  interface Binder extends UiBinder<Widget, PageNumberWithArrowsWidget> {
  }

  private static Binder uiBinder = GWT.create(Binder.class);

  @UiField
  Anchor paginationPrev;

  @UiField
  PageNumberListBox pageNumber;

  @UiField
  Anchor paginationNext;

  private int numPages;
  private int pageSize;
  private Pageable value;
  private boolean valueChangeHandlerInitialized;
  private final ValueBoxEditor<Pageable> editor;

  /**
   * constructor.
   */
  public PageNumberWithArrowsWidget() {
    super();
    numPages = -1;
    pageSize = 20;
    editor = new ExtendedValueBoxEditor<>(this, null);
    initWidget(PageNumberWithArrowsWidget.uiBinder.createAndBindUi(this));
    pageNumber.addValueChangeHandler(value -> setValue(PageRequest.of(value.getValue(), pageSize)));
  }

  @Override
  public final void setValue(final Pageable pnewValue) {
    setValue(pnewValue, false);
  }

  @Override
  public final void setValue(final Pageable pvalue, final boolean pfireEvents) {
    value = pvalue;
    int newPage = pvalue == null ? 0 : pvalue.getPageNumber();
    if (newPage < 0) {
      newPage = 0;
    }
    if (newPage >= numPages) {
      newPage = numPages - 1;
    }
    pageNumber.setValue(Integer.valueOf(newPage), pfireEvents);
    paginationPrev.setEnabled(newPage > 0);
    paginationPrev.getElement().getStyle().setOpacity(paginationPrev.isEnabled() ? 1.0 : 0.4);
    paginationNext.setEnabled(newPage < numPages - 1);
    paginationNext.getElement().getStyle().setOpacity(paginationNext.isEnabled() ? 1.0 : 0.4);
  }

  @Override
  public Pageable getValue() {
    return numPages < 0 ? value : PageRequest.of(pageNumber.getValue(), pageSize);
  }

  @Override
  public HandlerRegistration addValueChangeHandler(final ValueChangeHandler<Pageable> phandler) {
    if (!valueChangeHandlerInitialized) {
      ensureDomEventHandlers();
      valueChangeHandlerInitialized = true;
    }
    return this.addHandler(phandler, ValueChangeEvent.getType());
  }

  protected void ensureDomEventHandlers() {
    final ValueChangeHandler<Integer> changeHandler =
        event -> ValueChangeEvent.fire(this, getValue());
    pageNumber.addValueChangeHandler(changeHandler);
  }

  /**
   * set number pages, for each page a entry is created.
   *
   * @param pnumPages number of pages
   */
  public void setNumPages(final int pnumPages) {
    numPages = pnumPages;
    if (numPages < 1) {
      numPages = 1;
    }
    pageNumber.setNumPages(numPages);
    setValue(value, false);
  }

  public int getPageSize() {
    return pageSize;
  }

  public void setPageSize(final int ppageSize) {
    pageSize = ppageSize;
  }

  /**
   * pressed minus button.
   *
   * @param pevent click event
   */
  @UiHandler("paginationPrev")
  public void pressedMinusButton(final ClickEvent pevent) {
    final Integer newValue;
    if (pageNumber.getValue() == null) {
      newValue = Integer.valueOf(0);
    } else {
      newValue = Integer.valueOf(pageNumber.getValue().intValue() - 1);
    }
    setValue(PageRequest.of(newValue, pageSize), true);
  }

  /**
   * pressed plus button.
   *
   * @param pevent click event
   */
  @UiHandler("paginationNext")
  public void pressedPlusButton(final ClickEvent pevent) {
    final Integer newValue;
    if (pageNumber.getValue() == null) {
      newValue = Integer.valueOf(numPages);
    } else {
      newValue = Integer.valueOf(pageNumber.getValue().intValue() + 1);
    }
    setValue(PageRequest.of(newValue, pageSize), true);
  }

  @Override
  public ValueBoxEditor<Pageable> asEditor() {
    return editor;
  }
}
