package de.knightsoftnet.mtwidgets.client.ui.widget;

import de.knightsoftnet.gwtp.spring.shared.search.TableFieldDefinition;
import de.knightsoftnet.mtwidgets.client.ui.widget.features.HandlesSelectedEntry;
import de.knightsoftnet.mtwidgets.client.ui.widget.styling.WidgetResources;

import com.google.gwt.editor.client.EditorDelegate;
import com.google.gwt.editor.client.ValueAwareEditor;
import com.google.gwt.event.dom.client.ClickEvent;
import com.google.gwt.event.logical.shared.HasValueChangeHandlers;
import com.google.gwt.event.logical.shared.ValueChangeEvent;
import com.google.gwt.event.logical.shared.ValueChangeHandler;
import com.google.gwt.event.shared.HandlerRegistration;
import com.google.gwt.uibinder.client.UiBinder;
import com.google.gwt.uibinder.client.UiFactory;
import com.google.gwt.uibinder.client.UiField;
import com.google.gwt.uibinder.client.UiHandler;
import com.google.gwt.user.client.TakesValue;
import com.google.gwt.user.client.ui.Composite;
import com.google.gwt.user.client.ui.HTMLPanel;
import com.google.gwt.user.client.ui.Widget;
import com.google.inject.Provider;

import org.apache.commons.lang3.StringUtils;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Persistable;
import org.springframework.data.domain.Sort;
import org.springframework.data.domain.Sort.Direction;
import org.springframework.data.domain.Sort.Order;

import java.util.Collection;
import java.util.Map;
import java.util.stream.Collectors;

import javax.inject.Inject;

public class PageableList<T extends Persistable<Long>> extends Composite
    implements ValueAwareEditor<Page<T>>, TakesValue<Page<T>>, HasValueChangeHandlers<Pageable>,
    HandlesSelectedEntry<T> {

  interface Binder extends UiBinder<Widget, PageableList<?>> {
  }

  @UiField
  HTMLPanel header;

  @UiField
  PageableListEditor<T> content;

  @UiField
  PageNumberWithArrowsWidget pageable;

  private final Provider<PageableListEditor<T>> searchResultListProvider;

  /**
   * the default resources.
   */
  private final WidgetResources resources;

  private Map<String, HTMLPanel> headerMap;

  private Page<T> value;

  private Sort sort;

  private HandlesSelectedEntry<T> parent;

  /**
   * Constructor for Pageable List.
   */
  @Inject
  public PageableList(final Binder pbinder, final WidgetResources presources,
      final Provider<PageableListEditor<T>> psearchResultListProvider) {
    super();
    resources = presources;
    resources.pageableStyle().ensureInjected();
    searchResultListProvider = psearchResultListProvider;
    initWidget(pbinder.createAndBindUi(this));
    content.setParent(this);
  }

  @Ignore
  @UiFactory
  public PageableListEditor<T> buildSearchResultList() {
    return searchResultListProvider.get();
  }

  @Override
  public void setDelegate(final EditorDelegate<Page<T>> delegate) {
    delegate.subscribe();
  }

  @Override
  public Page<T> getValue() {
    return value;
  }

  public boolean hasEntries() {
    return value != null && !value.isEmpty();
  }

  @Override
  public void setValue(final Page<T> value) {
    this.value = value;
    content.showList(value.getContent());
    pageable.setValue(value.getPageable());
    pageable.setNumPages(value.getTotalPages());
  }

  @Override
  public void flush() {
    // nothing to do
  }

  @Override
  public void onPropertyChange(final String... paths) {
    // nothing to do
  }

  @Override
  public HandlerRegistration addValueChangeHandler(final ValueChangeHandler<Pageable> handler) {
    return addHandler(handler, ValueChangeEvent.getType());
  }

  @Override
  public void handleSelectedEntry(final T entry) {
    parent.handleSelectedEntry(entry);
  }

  /**
   * pageable changed.
   *
   * @param event value change handler
   */
  @UiHandler("pageable")
  public void pageableChanged(final ValueChangeEvent<Pageable> event) {
    final Pageable newPageable =
        PageRequest.of(event.getValue().getPageNumber(), event.getValue().getPageSize(), sort);
    ValueChangeEvent.fire(this, newPageable);
  }

  /**
   * set table field definitions.
   *
   * @param tableFieldDefinitions collection of field definitions
   */
  public void setTableFieldDefinitions(
      final Collection<TableFieldDefinition<T>> tableFieldDefinitions) {
    content.setTableFieldDefinitions(tableFieldDefinitions);
    headerMap = tableFieldDefinitions.stream() //
        .collect(Collectors.toMap(

            definition -> definition.getFieldName(), //
            definition -> {
              final HTMLPanel headerCell = new HTMLPanel(definition.getFieldDisplayName());
              definition.getStyles().forEach(style -> headerCell.addStyleName(style));
              headerCell.addDomHandler(event -> {
                final Direction direction;
                if (StringUtils.contains(headerCell.getStyleName(),
                    resources.pageableStyle().headerSortUp())) {
                  direction = Direction.DESC;
                } else {
                  direction = Direction.ASC;
                }
                fireSortChanged(Sort.by(direction, definition.getFieldName()));
              }, ClickEvent.getType());
              header.add(headerCell);
              return headerCell;
            }));
  }

  /**
   * clear/reset sorting (no header is selectd).
   */
  public void clearSort() {
    headerMap.forEach((key, headerCell) -> {
      headerCell.setStyleName(resources.pageableStyle().headerSortUp(), false);
      headerCell.setStyleName(resources.pageableStyle().headerSortDown(), false);
    });
  }

  private void fireSortChanged(final Sort sort) {
    this.sort = sort;
    final Pageable currentPageable = pageable.getValue();
    final Pageable newPageable =
        PageRequest.of(currentPageable.getPageNumber(), currentPageable.getPageSize(), sort);
    headerMap.forEach((key, headerCell) -> {
      final Order order = sort.getOrderFor(key);
      headerCell.setStyleName(resources.pageableStyle().headerSortUp(),
          order != null && order.isAscending());
      headerCell.setStyleName(resources.pageableStyle().headerSortDown(),
          order != null && order.isDescending());
    });
    ValueChangeEvent.fire(this, newPageable);
  }

  public void setParent(final HandlesSelectedEntry<T> parent) {
    this.parent = parent;
  }
}
