/*
 *  Color.scala
 *  (SoundProcesses)
 *
 *  Copyright (c) 2010-2022 Hanns Holger Rutz. All rights reserved.
 *
 *	This software is published under the GNU Affero General Public License v3+
 *
 *
 *  For further information, please contact Hanns Holger Rutz at
 *  contact@sciss.de
 */

package de.sciss.lucre.expr.graph

import de.sciss.lucre.Adjunct.{FromAny, HasDefault}
import de.sciss.lucre.expr.Context
import de.sciss.lucre.expr.ExElem.{ProductReader, RefMapIn}
import de.sciss.lucre.expr.graph.impl.MappedIExpr
import de.sciss.lucre.expr.impl.BasicExObjBridgeImpl
import de.sciss.lucre.{Adjunct, IExpr, ITargets, Txn}
import de.sciss.numbers.IntFunctions
import de.sciss.proc.Color.Palette
import de.sciss.proc.ExImport.Color
import de.sciss.proc.{Color => _Color}
import de.sciss.serial.DataInput

object Color {
  private lazy val _init: Unit =
    Adjunct.addFactory(TypeImpl)

  def init(): Unit = _init

  def DarkBlue    : Ex[Color]  = Const(Palette( 0))
  def LightBlue   : Ex[Color]  = Const(Palette( 1))
  def Cyan        : Ex[Color]  = Const(Palette( 2))
  def Mint        : Ex[Color]  = Const(Palette( 3))
  def Green       : Ex[Color]  = Const(Palette( 4))
  def Yellow      : Ex[Color]  = Const(Palette( 5))
  def DarkBeige   : Ex[Color]  = Const(Palette( 6))
  def LightBeige  : Ex[Color]  = Const(Palette( 7))
  def Orange      : Ex[Color]  = Const(Palette( 8))
  def Red         : Ex[Color]  = Const(Palette( 9))
  def Maroon      : Ex[Color]  = Const(Palette(10))
  def Fuchsia     : Ex[Color]  = Const(Palette(11))
  def Purple      : Ex[Color]  = Const(Palette(12))
  def Black       : Ex[Color]  = Const(Palette(13))
  def Silver      : Ex[Color]  = Const(Palette(14))
  def White       : Ex[Color]  = Const(Palette(15))

  def Type: Obj.Bridge[Color] with Obj.CanMake[Color] with HasDefault[Color] with FromAny[Color] = TypeImpl

  private object TypeImpl extends BasicExObjBridgeImpl[_Color, _Color.Obj](_Color.Obj)
    with Ex.Value[_Color] with FromAny[_Color]
    with HasDefault[_Color] with Adjunct.Factory {

    override def toString: String = "Color"

    import _Color.{Obj => tpe}

    final val id = 2009

    // WARNING: this must correspond with the serialization of `AbstractExObjBridgeImpl`!
    override def readIdentifiedAdjunct(in: DataInput): Adjunct = {
      val typeId = in.readInt()
      assert (typeId == tpe.typeId)
      this

    }
    override def defaultValue: _Color = _Color.Black

    override def fromAny(in: Any): Option[_Color] = in match {
      case c: _Color  => Some(c)
      case _          => None
    }
  }

  object Predef extends ProductReader[Ex[Color]] {
    /** There are sixteen predefined colors (identifiers 0 to 15). */
    def apply(id: Ex[Int]): Ex[Color] = Impl(id)

    override def read(in: RefMapIn, key: String, arity: Int, adj: Int): Ex[Color] = {
      require (arity == 1 && adj == 0)
      val _id = in.readEx[Int]()
      Predef(_id)
    }

    private final class Expanded[T <: Txn[T]](id: IExpr[T, Int], tx0: T)(implicit targets: ITargets[T])
      extends MappedIExpr[T, Int, Color](id, tx0) {

      protected def mapValue(inValue: Int)(implicit tx: T): Color = {
        val idx = IntFunctions.mod(inValue, Palette.size)
        Palette(idx)
      }
    }

    private final case class Impl(id: Ex[Int]) extends Ex[Color] {
      type Repr[T <: Txn[T]] = IExpr[T, Color]

      override def productPrefix: String = s"Color$$Predef" // serialization

      protected def mkRepr[T <: Txn[T]](implicit ctx: Context[T], tx: T): Repr[T] = {
        import ctx.targets
        new Expanded(id.expand[T], tx)
      }
    }
  }
}
