/*
 *  BufferWrite.scala
 *  (SoundProcesses)
 *
 *  Copyright (c) 2010-2022 Hanns Holger Rutz. All rights reserved.
 *
 *	This software is published under the GNU Affero General Public License v3+
 *
 *
 *  For further information, please contact Hanns Holger Rutz at
 *  contact@sciss.de
 */

package de.sciss.proc.impl

import de.sciss.asyncfile.Ops.URIOps
import de.sciss.audiofile.{AudioFile, AudioFileSpec, AudioFileType}
import de.sciss.lucre.synth.{Buffer, Executor, RT, Resource, Synth}
import de.sciss.lucre.{Artifact, synth}
import de.sciss.numbers.Implicits.intNumberWrapper
import de.sciss.osc
import de.sciss.proc.{AuralContext, AuralNode, SoundProcesses}
import de.sciss.processor.impl.ProcessorBase
import de.sciss.synth.message.BufferInfo
import de.sciss.synth.proc.graph.Action
import de.sciss.synth.proc.graph.impl.SendReplyResponder
import de.sciss.synth.{GE, message, ugen}

import java.util.concurrent.TimeUnit
import scala.concurrent.stm.Ref
import scala.concurrent.stm.TxnExecutor.{defaultAtomic => atomic}
import scala.concurrent.{Future, TimeoutException}
import scala.collection.{IndexedSeq => CVec}
import scala.math.min

// XXX TODO --- DRY with BufferPrepare
// XXX TODO --- investigate why this is so much slower than `BufferPrepare` (chunked b_setn)
// (one possibility to speed up a bit could be to send b_getn right after receiving b_setn,
//  writing to disk in parallel; although it's unclear if that is involved in the bottle neck)
// cf. https://scsynth.org/t/why-would-b-getn-be-much-slower-than-b-setn/
/** Asynchronously transfers a buffer's content to a file chunks by chunk.
  * This works in two steps, as it is triggered from within the node.
  * First, the `Starter` is installed on the aural node. When it detects
  * the trigger, it calls `BufferWrite.apply` (which is similar to `BufferPrepare()`).
  */
object BufferWrite {
  // via SendReply
  def replyName(key: String): String = s"/$$wb_$key"

  // via n_set
  def doneName (key: String): String = s"/$$wb_done_$key"

  def makeUGen(g: Action.WriteBuf): GE = {
    import g._
    import ugen._
    val bufFrames     = BufFrames     .ir(buf)
    val bufChannels   = BufChannels   .ir(buf)
    val bufSampleRate = BufSampleRate .ir(buf)
    val values: GE = Seq(
      buf           ,
      bufFrames     ,
      bufChannels   ,
      bufSampleRate ,
      numFrames     ,
      startFrame    ,
      fileType      ,
      sampleFormat  ,
    )
    SendReply.kr(trig = trig, values = values, msgName = BufferWrite.replyName(key), id = 0)
    ControlProxyFactory.fromString(BufferWrite.doneName(key)).tr
  }

  final class Starter[T <: synth.Txn[T]](f: Artifact.Value, key: String, nr: AuralNode[T])
                                        (implicit context: AuralContext[T])
    extends SendReplyResponder {

    private[this] val Name    = replyName(key)
    private[this] val NodeId  = synth.peer.id

    override protected def synth: Synth = nr.synth

    override protected def added()(implicit tx: RT): Unit = ()

    override protected val body: Body = {
      case osc.Message(Name, NodeId, 0,
          bufIdF          : Float,
          bufFramesF      : Float,
          bufChannelsF    : Float,
          bufSampleRate   : Float,
          numFramesF      : Float,
          startFrameF     : Float,
          fileTypeIdF     : Float,
          sampleFormatIdF : Float,
        ) =>

        val bufId           = bufIdF          .toInt
        val bufFrames       = bufFramesF      .toInt
        val bufChannels     = bufChannelsF    .toInt
        val startFrame      = startFrameF     .toInt
        val numFrames0      = numFramesF      .toInt
        val numFrames       = if (numFrames0 >= 0) numFrames0 else bufFrames - startFrame
        val fileTypeId      = fileTypeIdF     .toInt
        val sampleFormatId  = sampleFormatIdF .toInt
        val fileType        = if (fileTypeId < 0) {
          val e = f.extL
          AudioFileType.writable.find(_.extensions.contains(e)).getOrElse(AudioFileType.AIFF)
        } else {
          Action.WriteBuf.fileType(fileTypeId.clip(0, Action.WriteBuf.maxFileTypeId))
        }
        val sampleFormat    = Action.WriteBuf.sampleFormat(sampleFormatId.clip(0, Action.WriteBuf.maxSampleFormatId))
        val server          = nr.server
        val sPeer           = server.peer
        val bufInfo         = BufferInfo.Data(bufId = bufId, numFrames = bufFrames, numChannels = bufChannels,
          sampleRate = bufSampleRate)
        val bufPeer         = bufInfo.asBuffer(sPeer)
        val spec            = AudioFileSpec(fileType, sampleFormat, numChannels = bufChannels,
          sampleRate = bufSampleRate, numFrames = numFrames)

        import context.universe.cursor
        SoundProcesses.step[T](s"BufferWrite($synth, $key)") { implicit tx: T =>
          val buf     = Buffer.wrap(server, bufPeer)
          val config  = Config(f, spec, offset = startFrame, buf = buf, key = key)
          val bw      = BufferWrite[T](config)
          nr.addResource(bw)
          tx.afterCommit {
            bw.foreach { _ =>
              val server  = nr.server
              val sPeer   = server.peer
              sPeer ! message.NodeSet(synth.peer.id, BufferWrite.doneName(key) -> 1f)
            } (Executor.executionContext)
          }
        }
    }
  }

  /** The configuration of the buffer preparation.
    *
    * @param f          the audio file to write to
    * @param spec       the file's specification (number of channels and frames)
    * @param offset     the offset into the buffer to start from
    * @param buf        the buffer to write from.
    * @param key        the key of the `graph.Buffer` element, used for setting the synth control eventually
    */
  case class Config(f: Artifact.Value, spec: AudioFileSpec, offset: Int, buf: Buffer, key: String) {
    override def productPrefix = "BufferWrite.Config"
    override def toString: String = {
      import spec.{productPrefix => _, _}
      s"$productPrefix($f, numChannels = $numChannels, numFrames = $numFrames, offset = $offset, key = $key)"
    }
  }

  /** Creates and launches the process. */
  def apply[T <: synth.Txn[T]](config: Config)(implicit tx: T): Future[Any] with Resource = {
    import config._
    if (!buf.isOnline) sys.error("Buffer must be allocated")
    val numFrL = spec.numFrames
    if (numFrL > buf.numFrames) sys.error(s"File $f spec ($numFrL frames) is larger than buffer (${buf.numFrames})")
//    if (numFrL > 0x3FFFFFFF) sys.error(s"File $f is too large ($numFrL frames) for an in-memory buffer")
    import spec.numChannels
    val largeBuf  = Executor.isJS || buf.server.config.transport == osc.TCP
    val blockSize = (if (largeBuf) SAMPLES_PER_PACKET_TCP else SAMPLES_PER_PACKET_UDP) / numChannels
    val res = new GetN[T](f = f, numFrames = numFrL.toInt, off0 = offset,
      spec = config.spec.copy(numFrames = 0L), blockSize = blockSize, buf = buf, key = key)
    tx.afterCommit(res.start()(Executor.executionContext))
    res
  }

  private final val SAMPLES_PER_PACKET_TCP = 6540 // ensure < 32K OSC bundle size
  private final val SAMPLES_PER_PACKET_UDP = 1608 // ensure <  8K OSC bundle size

  private final val MAX_CLUMP = 10

  private final class NextData(/*val fileStartFrame: Long,*/ val numFrames: Int, val bufStartFrame: Int,
                               val endProgress: Double)

  // use b_getn commands
  private final class GetN[T <: synth.Txn[T]](f: Artifact.Value, numFrames: Int, off0: Long, spec: AudioFileSpec,
                                              blockSize: Int, buf: Buffer, key: String)
    extends Impl[T, NextData](
      blockSize   = blockSize,
      numFrames   = numFrames,
      buf         = buf,
      key         = key
    ) {

    private val numChannels: Int = spec.numChannels

    private final val DEBUG = false

    protected def runBody(): Future[Prod] = {
      var SPENT_WRITING   = 0L
      var SPENT_RECEIVING = 0L
      AudioFile.openWriteAsync(f, spec).flatMap { af =>
        if (off0 > 0) af.seek(off0)
        val afBuf = af.buffer(blockSize * MAX_CLUMP)

        def loop(): Future[Unit] = {
          val nextOptSq = Seq.fill(MAX_CLUMP)(nextChunk())
          if (nextOptSq.forall(_.isEmpty)) abort()
          checkAborted()

          val nextSq        = nextOptSq.flatten
//          val firstBufStart = nextSq.head.bufStartFrame
          val T_RECEIVE1  = if (DEBUG) System.currentTimeMillis() else 0L
          val futSendSq: Seq[Future[CVec[Float]]] = nextSq.map { next =>
            val smpOff      = next.bufStartFrame * numChannels
            val chunk       = next.numFrames
            val smpChunk    = chunk * numChannels
            val b           = buf.peer
            b.server.!!(b.getnMsg(smpOff until (smpOff + smpChunk))) {
              case message.BufferSetn(b.id, (`smpOff`, xs)) => xs
            }
          }
          val futSend: Future[Seq[CVec[Float]]] = Future.sequence(futSendSq)

          var T_WRITE1 = 0L
          val fut = futSend.flatMap { dataSq =>
            if (DEBUG) {
              val T_RECEIVE2 = System.currentTimeMillis()
              SPENT_RECEIVING += (T_RECEIVE2 - T_RECEIVE1)
            }
            var chunk = 0
            val itSq = dataSq.iterator
            while (itSq.hasNext) {
              val data  = itSq.next()
              val it    = data.iterator
              while (it.hasNext) {
                var ch = 0
                while (ch < numChannels) {
                  val chBuf = afBuf(ch)
                  chBuf(chunk) = it.next() // de-interleave channel data
                  ch += 1
                }
                chunk += 1
              }
            }
            checkAborted()
            if (DEBUG) T_WRITE1 = System.currentTimeMillis()
            af.write(afBuf, off = 0, len = chunk)
          }

          val endProgress = nextSq.last.endProgress
          awaitFut(fut, endProgress) {
            if (DEBUG) {
              val T_WRITE2 = System.currentTimeMillis()
              SPENT_WRITING += (T_WRITE2 - T_WRITE1)
            }
            loop()
          }
        }

        loop().andThen { case x =>
          if (DEBUG) println(s"BUF WRITE - time spent writing $SPENT_WRITING, spent receiving $SPENT_RECEIVING; $x")
          af.close()
        }
      }
    }

    override protected def prepareNext(offset: Int, chunk: Int, endProgress: Double)(implicit tx: RT): NextData =
      new NextData(/*fileStartFrame = offset + off0,*/ numFrames = chunk, bufStartFrame = offset,
        endProgress = endProgress)

    override def toString = s"BufferWrite.GetN($f, $buf)@${hashCode().toHexString}"
  }

  private abstract class Impl[T <: synth.Txn[T], Next](protected val blockSize: Int,
                                                       numFrames: Int, val buf: Buffer, key: String)
    extends ProcessorBase[Unit, Resource] with Buffer.ProxyResource { impl =>

    protected def prepareNext(offset: Int, chunk: Int, endProgress: Double)(implicit tx: RT): Next

    final type Prod = Unit

    private val offsetRef = Ref(0)

    // ---- processor body ----

    protected final def awaitFut(fut: Future[Any], pr: Double)(loop: => Future[Unit]): Future[Unit] =
      if (fut.isCompleted) {
        progress = pr
        if (progress == 1.0) Future.unit else loop

      } else {
        // check once a second if processor was aborted
        val futTimeOut = Executor.timeOut(fut, 1L, TimeUnit.SECONDS).recover {
          case _: TimeoutException => ()
        }
        futTimeOut.flatMap { _ =>
          checkAborted()
          awaitFut(fut, pr)(loop)
        }
      }

    protected def nextChunk(): Option[Next] =
      atomic { implicit tx =>
        val offset  = offsetRef()
        val chunk   = min(numFrames - offset, blockSize)
        val stop    = offset + chunk
        if (chunk > 0) {
          offsetRef() = stop
          implicit val ptx: RT = RT.wrap(tx)
          // buf might be offline if dispose was called
          if (buf.isOnline) {
            val pr  = if (stop == numFrames) 1.0 else min(0.9999, stop.toDouble / numFrames)
            val res = prepareNext(offset = offset, chunk = chunk, endProgress = pr)
            Some(res)
          } else None
        } else None
      }

    // ----

    def dispose()(implicit tx: RT): Unit =
      tx.afterCommit(abort())
  }
}
