/*
 *  ActionResponder.scala
 *  (SoundProcesses)
 *
 *  Copyright (c) 2010-2022 Hanns Holger Rutz. All rights reserved.
 *
 *	This software is published under the GNU Affero General Public License v3+
 *
 *
 *  For further information, please contact Hanns Holger Rutz at
 *  contact@sciss.de
 */

package de.sciss.synth.proc.graph.impl

import de.sciss.lucre.Source
import de.sciss.lucre.edit.UndoManager
import de.sciss.lucre.expr.graph.Const
import de.sciss.lucre.expr.{Context, IAction, IExprAsRunnerMap}
import de.sciss.lucre.synth.{RT, Synth, Txn}
import de.sciss.proc.{Action, AuralContext, Runner, SoundProcesses}
import de.sciss.synth.GE
import de.sciss.{osc, synth}

import scala.collection.immutable.{IndexedSeq => Vec}

object ActionResponder {
  // via SendReply
  private def replyName(key: String): String = s"/$$act_$key"

  def makeUGen(trig: GE, values: Option[GE], key: String): Unit = {
    import synth._
    import ugen._
    // we cannot make values.size zero, because then in the multi-channel expansion,
    // the SendReply collapses :)
    SendReply.kr(trig = trig, values = values.getOrElse(0) /* Vec.empty[GE] */ , msgName = replyName(key), id = 0)
    ()
  }

  var DEBUG = false

  final class WithAction[T <: Txn[T]](actionH: Source[T, Action[T]],
                                        key: String, synth: Synth)
                                       (implicit context: AuralContext[T])
    extends ActionResponder[T](key, synth) {

    override protected def execute(values: Vec[Double])(implicit tx: T): Unit = {
      val action = actionH()
      import context.universe
      val r = Runner(action)
      import context.universe.{cursor, workspace}
      implicit val undo : UndoManager [T] = UndoManager.dummy
      implicit val ctx  : Context     [T] = Context[T](selfH = Some(actionH))
      import ctx.targets
      r.prepare(new IExprAsRunnerMap[T](
        new Const.Expanded[T, (String, Vec[Double])](("value", values)) :: Nil, tx
      ))
      r.run()
      r   .dispose()
      ctx .dispose()
    }
  }

  /** This does not pass on `values` */
  final class WithIAction[T <: Txn[T]](action: IAction[T],
                                        key: String, synth: Synth)
                                       (implicit context: AuralContext[T])
    extends ActionResponder[T](key, synth) {

    override protected def execute(values: Vec[Double])(implicit tx: T): Unit =
      action.executeAction()
  }
}
abstract class ActionResponder[T <: Txn[T]](key: String, protected val synth: Synth)
                                           (implicit context: AuralContext[T])
  extends SendReplyResponder {

  import ActionResponder._

  protected def execute(values: Vec[Double])(implicit tx: T): Unit

  private[this] val Name    = replyName(key)
  private[this] val NodeId  = synth.peer.id

  protected final val body: Body = {
    case osc.Message(Name, NodeId, 0, raw @ _*) =>
      if (DEBUG) println(s"ActionResponder($key, $NodeId) - received trigger")
      // logAural(m.toString)
      val values: Vec[Double] = if (raw.isEmpty) Vector.empty else raw match {
        case rawV: Vec[Any] =>
          rawV.collect {
            case f: Float => f.toDouble
          }

        case _ =>
          raw.iterator.collect {
            case f: Float => f.toDouble
          }.toIndexedSeq
      }
      import context.universe.cursor
      SoundProcesses.step[T](s"ActionResponder($synth, $key)") { implicit tx: T =>
        execute(values)
      }
  }

  protected final def added()(implicit tx: RT): Unit = ()
}