//     Project: sbt-sjs-annots
//      Module: plugin
// Description: Plugin for generating an JS file with contents from Scala annotations

// Copyright (c) 2016. Distributed under the MIT License (see included LICENSE file).
package sjsx.sbtplugin

import org.scalajs.core.tools.io.{RelativeVirtualFile, VirtualScalaJSIRFile}
import org.scalajs.core.tools.linker.{ClearableLinker, ModuleInitializer}
import org.scalajs.core.tools.linker.backend.OutputMode
import org.scalajs.sbtplugin.ScalaJSPlugin
import org.scalajs.sbtplugin.impl.DependencyBuilders
import sbt.Keys._
import sbt._
import sbt.classpath.ClasspathUtilities

object SJSXPlugin extends sbt.AutoPlugin {
  import SJSXPluginInternal._
  import ScalaJSPlugin.AutoImport.{fastOptJS, fullOptJS}

  override def trigger = allRequirements
  override def requires = ScalaJSPlugin

  object autoImport {

    val sjsxPreamble = settingKey[String]("Preamble to be included in the sjsx file")

    val sjsxFile = settingKey[File]("Name of JavaScript file generated by sjsx")

    val sjsxLoader = settingKey[SJSXLoader.Value]("JS module loader to be used(default: SJSXLoader.None)")

    val sjsxSnippets = taskKey[Seq[SJSXSnippet]]("Additional JavaScript snippets written to the sjsx file")

    val sjsxDeps = taskKey[Seq[SJSXDependency]]("Additional module dependencies")

    val sjsxDebug = settingKey[Boolean]("Enable debug statements")

    val sjsxWriteFile = taskKey[Unit]("Writes the sjsx JavaScript file")

    object SJSXLoader extends Enumeration {
      val None = Value
//      val SystemJS = Value
      val CommonJS = Value
    }

    case class SJSXSnippet(prio: Int, arg: String)

    case class SJSXDependency(id: String, global: String)
  }

  import ScalaJSPlugin.AutoImport._
  import autoImport._
  import org.scalajs.sbtplugin.ScalaJSPluginInternal.scalaJSLinker

  lazy val sjsxConfig = taskKey[SJSXConfig]("assembled SJSX configuration")

  case class SJSXConfig(sjsxPreamble: String, sjsxFile: File, sjsxLoader: SJSXLoader.Value, sjsxSnippets: Seq[SJSXSnippet],
                        sjsxDeps: Seq[SJSXDependency], sjsxDebug: Boolean)

  lazy val scalaJSTools = taskKey[ScalaJSTools]("Holds required ScalaJS tool objects")

  case class ScalaJSTools(ir: Seq[VirtualScalaJSIRFile with RelativeVirtualFile],
                          linker: ClearableLinker,
                          outputMode: OutputMode,
                          withSourceMaps: Boolean,
                          classLoader: ClassLoader,
                          moduleKind:  org.scalajs.core.tools.linker.ModuleKind,
                          moduleInitializers: Seq[ModuleInitializer])

  override def projectSettings = Seq(
    // default config
    sjsxDebug := true,
    sjsxSnippets := Seq(),
    sjsxDeps := Seq(),
    sjsxLoader := SJSXLoader.CommonJS,
    sjsxPreamble := "",
    sjsxFile := (crossTarget in compile).value / s"${(moduleName in compile).value}-sjsx.js",
    scalaJSModuleKind := ModuleKind.CommonJSModule,

    sjsxConfig <<= (sjsxPreamble,sjsxFile,sjsxLoader,sjsxSnippets,sjsxDeps,sjsxDebug) map SJSXConfig.apply,
    scalaJSTools <<= (
      (scalaJSIR in Compile),
      (scalaJSLinker in Compile),
      scalaJSOutputMode,
      emitSourceMaps,
      (fullClasspath in Compile),
      scalaInstance,
      scalaJSModuleKind,
      scalaJSModuleInitializers) map (
      (ir,linker,om,withSourceMaps,fullClasspath,instance,moduleKind,moduleInitializers) =>
        ScalaJSTools(ir.data,linker,om,withSourceMaps,
          ClasspathUtilities.makeLoader(fullClasspath.map(_.data),instance),moduleKind,moduleInitializers)),

    sjsxWriteFile <<= (sjsxConfig,scalaJSTools,streams) map writeAnnotations,
    libraryDependencies += DepBuilder.toScalaJSGroupID("de.surfice") %%% "sjsx" % Version.sjsxVersion,
    (fastOptJS in Compile) <<= (fastOptJS in Compile).dependsOn(sjsxWriteFile),
    (fullOptJS in Compile) <<= (fullOptJS in Compile).dependsOn(sjsxWriteFile)
  )

  private object DepBuilder extends DependencyBuilders

}
