package de.terrestris.shoguncore.service;

import de.terrestris.shoguncore.dao.GenericHibernateDao;
import de.terrestris.shoguncore.dao.WpsPluginDao;
import de.terrestris.shoguncore.dao.WpsProcessExecuteDao;
import de.terrestris.shoguncore.model.wps.WpsPlugin;
import de.terrestris.shoguncore.model.wps.WpsProcessExecute;
import org.hibernate.criterion.Restrictions;
import org.hibernate.criterion.SimpleExpression;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.List;

/**
 * Service class for the {@link WpsProcessExecute} model.
 *
 * @author Nils Bühner
 * @see AbstractCrudService
 */
@Service("wpsProcessExecuteService")
public class WpsProcessExecuteService<E extends WpsProcessExecute, D extends WpsProcessExecuteDao<E>> extends
    WpsReferenceService<E, D> {

    /**
     * The WpsPluginService which we e.g. need when WpsProcessExecutes are deleted.
     */
    @Autowired
    @Qualifier("wpsPluginService")
    private WpsPluginService<WpsPlugin, WpsPluginDao<WpsPlugin>> wpsPluginService;

    /**
     * Default constructor, which calls the type-constructor
     */
    @SuppressWarnings("unchecked")
    public WpsProcessExecuteService() {
        this((Class<E>) WpsProcessExecute.class);
    }

    /**
     * Constructor that sets the concrete entity class for the service.
     * Subclasses MUST call this constructor.
     */
    protected WpsProcessExecuteService(Class<E> entityClass) {
        super(entityClass);
    }

    /**
     * Removes the passed WpsProcessExecute from all WpsPlugins and afterwards
     * deletes the WpsProcessExecute itself. This overrides the generic method
     * to delete {@link AbstractCrudService#delete(de.terrestris.shoguncore.model.PersistentObject)}.
     *
     * @param wpsProcessExecute
     */
    @Override
    @PreAuthorize("hasRole(@configHolder.getSuperAdminRoleName()) or hasPermission(#plugin, 'DELETE')")
    public void delete(E wpsProcessExecute) {
        if (wpsPluginService == null) {
            logger.error("WPSProcessExecute cannot be deleted, failed to autowire WpsPluginService");
            return;
        }

        GenericHibernateDao dao = wpsPluginService.getDao();
        WpsPluginDao<WpsPlugin> wpsPluginDao = null;
        if (!(dao instanceof WpsPluginDao)) {
            logger.error("WPSProcessExecute cannot be deleted, failed to get WpsPluginDao");
            return;
        }
        wpsPluginDao = (WpsPluginDao<WpsPlugin>) dao;

        SimpleExpression eqProcess = Restrictions.eq("process", wpsProcessExecute);
        List<WpsPlugin> wpsPlugins = wpsPluginDao.findByCriteria(eqProcess);

        Integer processId = wpsProcessExecute.getId();
        for (WpsPlugin wpsPlugin : wpsPlugins) {
            WpsProcessExecute process = wpsPlugin.getProcess();
            if (process != null) {
                String msg = String.format(
                    "Remove WpsProcessExecute (id=%s) from WpsPlugin (id=%s)",
                    processId, wpsPlugin.getId()
                );
                logger.debug(msg);
                wpsPlugin.setProcess(null);
                wpsPluginService.saveOrUpdate(wpsPlugin);
            }
        }
        logger.debug(String.format("Delete plugin (id=%s)", processId));

        // Call overridden parent to actually delete the entity itself
        super.delete(wpsProcessExecute);
    }

    /**
     * We have to use {@link Qualifier} to define the correct dao here.
     * Otherwise, spring can not decide which dao has to be autowired here
     * as there are multiple candidates.
     */
    @Override
    @Autowired
    @Qualifier("wpsProcessExecuteDao")
    public void setDao(D dao) {
        this.dao = dao;
    }

    /**
     * @param wpsId
     * @return List of {@link WpsPlugin}s that are connected to the given {@link WpsProcessExecute}
     */
    @PreAuthorize("hasRole(@configHolder.getSuperAdminRoleName()) or hasPermission(#wpsId, 'de.terrestris.shoguncore.model.wps.WpsProcessExecute', 'DELETE')")
    @Transactional(readOnly = true)
    public List<String> preCheckDelete(Integer wpsId) {
        List<String> result = new ArrayList<>();

        E wpsProcessExecute = this.dao.findById(wpsId);

        if (wpsProcessExecute != null) {

            List<WpsPlugin> pluginsWithWps = wpsPluginService.findAllWhereFieldEquals("process", wpsProcessExecute);

            for (WpsPlugin plugin : pluginsWithWps) {
                result.add(plugin.getName());
            }

        }

        return result;
    }

    /**
     * @return the wpsPluginService
     */
    public WpsPluginService<WpsPlugin, WpsPluginDao<WpsPlugin>> getWpsPluginService() {
        return wpsPluginService;
    }
}
