/*
 * Decompiled with CFR 0.152.
 */
package dev.jeka.core.api.file;

import dev.jeka.core.api.file.JkPathMatcher;
import dev.jeka.core.api.file.JkPathTreeSet;
import dev.jeka.core.api.utils.JkUtilsAssert;
import dev.jeka.core.api.utils.JkUtilsIO;
import dev.jeka.core.api.utils.JkUtilsPath;
import java.io.Closeable;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.nio.file.CopyOption;
import java.nio.file.FileVisitOption;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.PathMatcher;
import java.nio.file.Paths;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.util.Arrays;
import java.util.LinkedList;
import java.util.List;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import java.util.stream.Stream;

public final class JkPathTree
implements Closeable {
    private final RootHolder rootHolder;
    private final JkPathMatcher matcher;

    public static JkPathTree of(Path rootDir) {
        return JkPathTree.of(rootDir, false);
    }

    public static JkPathTree of(String rootDir) {
        return JkPathTree.of(Paths.get(rootDir, new String[0]), false);
    }

    public static JkPathTree ofZip(Path zipFile) {
        return JkPathTree.of(zipFile.toAbsolutePath(), true);
    }

    private static JkPathTree of(Path rootDir, boolean zip) {
        return JkPathTree.of(rootDir, JkPathMatcher.ACCEPT_ALL, zip);
    }

    private static JkPathTree of(Path rootDirOrArchive, JkPathMatcher matcher, boolean zipFile) {
        RootHolder rootHolder = zipFile ? RootHolder.ofZip(rootDirOrArchive) : RootHolder.ofDir(rootDirOrArchive);
        return new JkPathTree(rootHolder, matcher);
    }

    private JkPathTree(RootHolder rootHolder, JkPathMatcher matcher) {
        this.rootHolder = rootHolder;
        this.matcher = matcher;
    }

    public Path getRoot() {
        return this.rootHolder.get();
    }

    public Path getRootDirOrZipFile() {
        return this.rootHolder.rootFile();
    }

    public JkPathMatcher getMatcher() {
        return this.matcher;
    }

    public boolean hasFilter() {
        return this.matcher != JkPathMatcher.ACCEPT_ALL;
    }

    private Predicate<Path> excludeRootFilter() {
        return path -> !path.equals(this.getRoot());
    }

    private Function<Path, Path> relativePathFunction() {
        return path -> this.getRoot().relativize((Path)path);
    }

    public boolean exists() {
        return this.rootHolder.exists();
    }

    public JkPathTree createIfNotExist() {
        this.rootHolder.createIfNotExist();
        return this;
    }

    public Stream<Path> stream(FileVisitOption ... options) {
        if (!this.exists()) {
            return new LinkedList().stream();
        }
        JkPathMatcher matcher = JkPathMatcher.of(this.matcher);
        return (Stream)JkUtilsPath.walk(this.getRoot(), options).filter(path -> matcher.matches(this.getRoot().relativize((Path)path))).onClose(() -> this.rootHolder.closeIfNeeded());
    }

    public List<Path> getRelativeFiles() {
        try (Stream<Path> stream = this.stream(new FileVisitOption[0]);){
            List<Path> list = stream.filter(JkPathMatcher.ofNoDirectory(new LinkOption[0]).toPredicate()).map(this.relativePathFunction()).collect(Collectors.toList());
            return list;
        }
    }

    public List<Path> getFiles() {
        try (Stream<Path> stream = this.stream(new FileVisitOption[0]);){
            List<Path> list = stream.filter(JkPathMatcher.ofNoDirectory(new LinkOption[0]).toPredicate()).collect(Collectors.toList());
            return list;
        }
    }

    public JkPathTree goTo(String relativePath) {
        Path path = this.getRoot().resolve(relativePath).normalize();
        if (Files.exists(path, new LinkOption[0]) && !Files.isDirectory(path, new LinkOption[0])) {
            throw new IllegalArgumentException(this.getRoot() + "/" + relativePath + " is not a directory");
        }
        RootHolder rootHolder = new RootHolder(this.rootHolder.zipFile, path);
        return new JkPathTree(rootHolder, this.matcher);
    }

    public JkPathTree resolvedTo(Path newRoot) {
        Path path = newRoot.resolve(this.getRoot()).normalize();
        RootHolder rootHolder = new RootHolder(this.rootHolder.zipFile, path);
        return new JkPathTree(rootHolder, this.matcher);
    }

    public Path get(String relativePath) {
        return this.getRoot().resolve(relativePath);
    }

    public JkPathTree importDir(Path dirToCopy, CopyOption ... copyOptions) {
        return this.importTree(JkPathTree.of(dirToCopy), copyOptions);
    }

    public JkPathTree importTree(JkPathTree tree, CopyOption ... copyOptions) {
        this.createIfNotExist();
        if (tree.exists()) {
            tree.stream(new FileVisitOption[0]).filter(this.excludeRootFilter()).forEach(path -> {
                Path target = this.getRoot().resolve(tree.getRoot().relativize((Path)path).toString());
                if (Files.isDirectory(path, new LinkOption[0])) {
                    JkUtilsPath.createDirectories(target, new FileAttribute[0]);
                } else {
                    JkUtilsPath.copy(path, target, copyOptions);
                }
            });
        }
        return this;
    }

    public JkPathTree importFiles(Iterable<Path> files, StandardCopyOption ... copyOptions) {
        this.createIfNotExist();
        List<Path> paths = JkUtilsPath.disambiguate(files);
        for (Path file : paths) {
            JkUtilsPath.copy(file, this.getRoot().resolve(file.getFileName()), copyOptions);
        }
        return this;
    }

    public JkPathTree importFile(Path src, String targetName, StandardCopyOption ... copyOptions) {
        this.createIfNotExist();
        Path parentTarget = this.getRoot().resolve(targetName).getParent();
        if (parentTarget != null && !Files.exists(parentTarget, new LinkOption[0])) {
            JkUtilsPath.createDirectories(parentTarget, new FileAttribute[0]);
        }
        JkUtilsPath.copy(src, this.getRoot().resolve(targetName), copyOptions);
        return this;
    }

    public JkPathTree deleteContent() {
        if (!Files.exists(this.getRoot(), new LinkOption[0])) {
            return this;
        }
        JkUtilsPath.walkFileTree(this.getRoot(), (FileVisitor<Path>)new SimpleFileVisitor<Path>(){

            @Override
            public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) {
                return this.visitFile(file);
            }

            @Override
            public FileVisitResult postVisitDirectory(Path dir, IOException exc) {
                return this.visitDir(dir);
            }

            private FileVisitResult visitFile(Path path) {
                if (JkPathTree.this.matcher.matches(JkPathTree.this.getRoot().relativize(path))) {
                    JkUtilsPath.deleteFile(path);
                }
                return FileVisitResult.CONTINUE;
            }

            private FileVisitResult visitDir(Path path) {
                if (!JkUtilsPath.isSameFile(JkPathTree.this.getRoot(), path) && JkPathTree.this.matcher.matches(JkPathTree.this.getRoot().relativize(path)) && JkUtilsPath.listDirectChildren(path).isEmpty()) {
                    JkUtilsPath.deleteFile(path);
                }
                return FileVisitResult.CONTINUE;
            }
        });
        return this;
    }

    public JkPathTree deleteRoot() {
        JkUtilsPath.walkFileTree(this.getRoot(), (FileVisitor<Path>)new SimpleFileVisitor<Path>(){

            @Override
            public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                return this.visit(file);
            }

            @Override
            public FileVisitResult postVisitDirectory(Path dir, IOException exc) throws IOException {
                return this.visit(dir);
            }

            private FileVisitResult visit(Path path) {
                JkUtilsPath.deleteFile(path);
                return FileVisitResult.CONTINUE;
            }
        });
        return this;
    }

    public JkPathTree zipTo(Path destination) {
        if (destination.getParent() != null) {
            JkUtilsPath.createDirectories(destination.getParent(), new FileAttribute[0]);
        }
        Path zipRootEntry = JkUtilsPath.zipRoot(destination);
        try (Stream<Path> stream = this.stream(new FileVisitOption[0]);){
            stream.filter(this.excludeRootFilter()).forEach(path -> {
                Path zipEntry = zipRootEntry.resolve(this.getRoot().relativize((Path)path).toString());
                if (!Files.exists(zipEntry, new LinkOption[0]) || !Files.isDirectory(zipEntry, new LinkOption[0])) {
                    JkUtilsPath.createDirectories(zipEntry.getParent(), new FileAttribute[0]);
                    JkUtilsPath.copy(path, zipEntry, StandardCopyOption.REPLACE_EXISTING);
                }
            });
            zipRootEntry.getFileSystem().close();
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
        return this;
    }

    public int copyTo(Path destinationDir, CopyOption ... copyOptions) {
        if (!Files.exists(destinationDir, new LinkOption[0])) {
            JkUtilsPath.createDirectories(destinationDir, new FileAttribute[0]);
        }
        return JkUtilsPath.copyDirContent(this.getRoot(), destinationDir, this.matcher, copyOptions);
    }

    public void copyFile(String sourcePath, Path destinationDir, CopyOption ... copyOptions) {
        if (!Files.exists(destinationDir, new LinkOption[0])) {
            JkUtilsPath.createDirectories(destinationDir, new FileAttribute[0]);
        }
        Path source = this.get(sourcePath);
        Path dest = destinationDir.getFileSystem().getPath(destinationDir.toString() + "/" + source.getFileName(), new String[0]);
        JkUtilsPath.copy(source, dest, copyOptions);
    }

    public JkPathTree andMatcher(PathMatcher pathMatcher) {
        return new JkPathTree(this.rootHolder, this.matcher.and(pathMatcher));
    }

    public JkPathTree withMatcher(JkPathMatcher pathMatcher) {
        return new JkPathTree(this.rootHolder, pathMatcher);
    }

    public JkPathTree andMatching(boolean positive, String ... globPatterns) {
        return this.andMatching(positive, Arrays.asList(globPatterns));
    }

    public JkPathTree andMatching(String ... globPatterns) {
        return this.andMatching(true, globPatterns);
    }

    public JkPathTree andMatching(boolean positive, Iterable<String> globPatterns) {
        return this.andMatcher(JkPathMatcher.of(positive, this.getRoot().getFileSystem(), globPatterns));
    }

    public int count(int max, boolean includeDirectories) {
        if (!this.exists()) {
            return 0;
        }
        return JkUtilsPath.childrenCount(this.getRoot(), max, includeDirectories, this.matcher);
    }

    public boolean containFiles() {
        return this.count(1, false) > 0;
    }

    public JkPathTree resolve(Path path) {
        return new JkPathTree(this.rootHolder.resolve(path), this.matcher);
    }

    public JkPathTreeSet toSet() {
        return JkPathTreeSet.of(this);
    }

    public String toString() {
        return this.hasFilter() ? this.rootHolder + ":" + this.matcher : this.rootHolder.toString();
    }

    @Override
    public void close() {
        if (this.rootHolder.isZip()) {
            JkUtilsIO.closeQuietly(this.rootHolder.get().getFileSystem());
        }
    }

    private static class RootHolder {
        final Path zipFile;
        Path dir;

        static RootHolder ofZip(Path zipFile) {
            JkUtilsAssert.argument(zipFile != null, "zip archive file can't be null.", new Object[0]);
            JkUtilsAssert.argument(!Files.exists(zipFile, new LinkOption[0]) || !Files.isDirectory(zipFile, new LinkOption[0]), "Specified zip file " + zipFile + " can't be a directory", new Object[0]);
            return new RootHolder(zipFile, null);
        }

        static RootHolder ofDir(Path dir) {
            JkUtilsAssert.argument(dir != null, "Directory rootHolder tree can't be null.", new Object[0]);
            JkUtilsAssert.argument(!Files.exists(dir, new LinkOption[0]) || Files.isDirectory(dir, new LinkOption[0]), "Specified zip file " + dir + " must be a directory", new Object[0]);
            return new RootHolder(null, dir);
        }

        private RootHolder(Path zipFile, Path root) {
            this.zipFile = zipFile;
            this.dir = root;
        }

        Path get() {
            if (this.isZip() && this.dir == null) {
                this.dir = JkUtilsPath.zipRoot(this.zipFile);
            }
            return this.dir;
        }

        void createIfNotExist() {
            if (this.zipFile == null) {
                if (!Files.exists(this.dir, new LinkOption[0])) {
                    JkUtilsPath.createDirectories(this.dir, new FileAttribute[0]);
                }
            } else if (!Files.exists(this.zipFile, new LinkOption[0])) {
                JkUtilsPath.createDirectories(this.zipFile.getParent(), new FileAttribute[0]);
                this.dir = JkUtilsPath.zipRoot(this.zipFile);
            } else if (this.dir == null) {
                this.dir = JkUtilsPath.zipRoot(this.zipFile);
            } else if (this.dir.getFileSystem().isOpen()) {
                JkUtilsPath.createDirectories(this.dir, new FileAttribute[0]);
            } else {
                Path zipRoot = JkUtilsPath.zipRoot(this.zipFile);
                this.dir = zipRoot.getFileSystem().getPath(this.dir.toString(), new String[0]);
            }
        }

        boolean exists() {
            if (!this.isZip()) {
                return Files.exists(this.dir, new LinkOption[0]);
            }
            if (!Files.exists(this.zipFile, new LinkOption[0])) {
                return false;
            }
            if (this.dir == null) {
                return true;
            }
            if (this.dir.getFileSystem().isOpen()) {
                return Files.exists(this.dir, new LinkOption[0]);
            }
            Path zipRoot = JkUtilsPath.zipRoot(this.zipFile);
            this.dir = zipRoot.getFileSystem().getPath(this.dir.toString(), new String[0]);
            return Files.exists(this.dir, new LinkOption[0]);
        }

        boolean isZip() {
            return this.zipFile != null;
        }

        void closeIfNeeded() {
            if (this.isZip() && this.dir != null) {
                try {
                    this.dir.getFileSystem().close();
                }
                catch (IOException e) {
                    throw new UncheckedIOException(e);
                }
            }
        }

        RootHolder resolve(Path path) {
            if (this.isZip()) {
                return this;
            }
            return new RootHolder(null, path.resolve(this.dir));
        }

        Path rootFile() {
            if (this.isZip()) {
                return this.zipFile;
            }
            return this.dir;
        }

        public String toString() {
            return this.isZip() ? this.zipFile.toString() : this.dir.toString();
        }
    }
}

