/*
 * Copyright 2023 Shreyas Patil
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package dev.shreyaspatil.ai.client.generativeai.type

import kotlinx.serialization.json.JsonObject

/**
 * Interface representing data sent to and received from requests.
 *
 * One of:
 * * [TextPart] representing text or string based data.
 * * [ImagePart] representing image data.
 * * [BlobPart] representing MIME typed binary data.
 * * [FileDataPart] representing MIME typed binary data.
 * * [FunctionCallPart] representing a requested clientside function call by the model
 * * [FunctionResponsePart] representing the result of a clientside function call
 * * [ExecutableCodePart] representing code generated and executed by the model
 * * [CodeExecutionResultPart] representing the result of running code generated by the model.
 */
interface Part

/** Represents text or string based data sent to and received from requests. */
class TextPart(val text: String) : Part

/**
 * Represents image data sent to and received from requests. When this is sent to the server it is
 * converted to jpeg encoding at 80% quality.
 */
class ImagePart(val image: Bitmap) : Part

/** Represents binary data with an associated MIME type sent to and received from requests. */
class BlobPart(val mimeType: String, val blob: ByteArray) : Part

/** Represents an URI-based data with a specified media type. */
class FileDataPart(val uri: String, val mimeType: String) : Part

/** @return The part as a [BlobPart] if it represents a blob, and null otherwise */
fun Part.asFileDataPartOrNull(): FileDataPart? = this as? FileDataPart

/** Represents function call name and params received from requests. */
class FunctionCallPart(val name: String, val args: Map<String, String?>?) : Part

/** Represents function call output to be returned to the model when it requests a function call */
class FunctionResponsePart(val name: String, val response: JsonObject) : Part

/** Represents an internal function call written by the model */
class ExecutableCodePart(val language: String, val code: String) : Part

/** Represents the results of an internal function call written by the model */
class CodeExecutionResultPart(val outcome: ExecutionOutcome, val output: String) : Part

/** @return The part as a [String] if it represents text, and null otherwise */
fun Part.asTextOrNull(): String? = (this as? TextPart)?.text

/** @return The part as a [Bitmap] if it represents an image, and null otherwise */
fun Part.asImageOrNull(): Bitmap? = (this as? ImagePart)?.image

/** @return The part as a [BlobPart] if it represents a blob, and null otherwise */
fun Part.asBlobPartOrNull(): BlobPart? = this as? BlobPart
