package dev.the_fireplace.lib.chat;

import dev.the_fireplace.annotateddi.api.di.Implementation;
import dev.the_fireplace.lib.api.chat.injectables.MessageQueue;
import dev.the_fireplace.lib.api.chat.injectables.MultilineMessageBuffer;
import dev.the_fireplace.lib.entrypoints.FireplaceLib;
import org.apache.commons.lang3.ArrayUtils;

import javax.annotation.concurrent.ThreadSafe;
import javax.inject.Inject;
import javax.inject.Singleton;
import net.minecraft.class_2165;
import net.minecraft.class_2561;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicInteger;

@ThreadSafe
@Implementation
@Singleton
public final class MultilineMessageBufferImpl implements MultilineMessageBuffer {
    private final Map<Integer, Buffer> messageBuffers = new ConcurrentHashMap<>();
    private final AtomicInteger currentBufferId = new AtomicInteger(Integer.MIN_VALUE);
    private final MessageQueue messageQueue;

    @Inject
    public MultilineMessageBufferImpl(MessageQueue messageQueue) {
        this.messageQueue = messageQueue;
    }

    @Override
    public int create(byte expectedMessageCount, class_2165 target) {
        int bufferId = currentBufferId.getAndIncrement();
        messageBuffers.put(bufferId, new Buffer(bufferId, expectedMessageCount, target));
        return bufferId;
    }

    @Override
    public void put(int bufferId, byte position, class_2561 value) {
        Buffer buffer = messageBuffers.get(bufferId);
        if (buffer != null) {
            buffer.put(position, value);
        } else {
            FireplaceLib.getLogger().warn("Tried to add a message to nonexistent buffer " + bufferId + "!", new Exception("Stack trace"));
        }
    }

    private class Buffer {
        private final int bufferId;
        private final class_2561[] messages;
        private final class_2165 target;

        private Buffer(int bufferId, byte expectedMessageCount, class_2165 target) {
            this.bufferId = bufferId;
            this.messages = new class_2561[expectedMessageCount];
            this.target = target;
        }

        private void put(byte position, class_2561 value) {
            messages[position] = value;
            if (isBufferFull()) {
                sendBufferedMessages();
                cleanup();
            }
        }

        private boolean isBufferFull() {
            return !ArrayUtils.contains(messages, null);
        }

        private void sendBufferedMessages() {
            messageQueue.queueMessages(target, messages);
        }

        private void cleanup() {
            messageBuffers.remove(bufferId);
        }
    }
}
