package dev.the_fireplace.lib.command.helpers;

import com.google.common.collect.Lists;
import com.mojang.brigadier.Command;
import com.mojang.brigadier.CommandDispatcher;
import com.mojang.brigadier.arguments.IntegerArgumentType;
import com.mojang.brigadier.builder.LiteralArgumentBuilder;
import com.mojang.brigadier.context.CommandContext;
import com.mojang.brigadier.tree.CommandNode;
import com.mojang.brigadier.tree.LiteralCommandNode;
import dev.the_fireplace.annotateddi.api.DIContainer;
import dev.the_fireplace.lib.api.chat.injectables.TextPaginator;
import dev.the_fireplace.lib.api.chat.injectables.TextStyles;
import dev.the_fireplace.lib.api.chat.interfaces.Translator;
import dev.the_fireplace.lib.api.command.interfaces.HelpCommand;
import dev.the_fireplace.lib.chat.translation.TranslatorManager;
import it.unimi.dsi.fastutil.ints.IntArraySet;
import it.unimi.dsi.fastutil.ints.IntSet;
import java.util.*;
import java.util.function.Function;
import net.minecraft.class_2168;
import net.minecraft.class_2170;
import net.minecraft.class_2561;

public final class HelpCommandImpl implements HelpCommand
{
    private static final Function<String, Collection<String>> NEW_CHILD_NODE_SET = unused -> new HashSet<>(2);
    private final TextStyles textStyles;
    private final TextPaginator textPaginator;
    private final Translator translator;
    private final String modid;
    private final LiteralArgumentBuilder<class_2168> helpCommandBase;
    private final Map<String, Collection<String>> commands = new HashMap<>();
    private final IntSet grandchildNodeHashes = new IntArraySet(3);

    HelpCommandImpl(String modid, LiteralArgumentBuilder<class_2168> helpCommandBase) {
        this.modid = modid;
        this.helpCommandBase = helpCommandBase;
        this.translator = DIContainer.get().getInstance(TranslatorManager.class).getTranslator(modid);
        this.textStyles = DIContainer.get().getInstance(TextStyles.class);
        this.textPaginator = DIContainer.get().getInstance(TextPaginator.class);
    }

    @Override
    public HelpCommand addCommands(CommandNode<?>... commands) {
        String[] commandNames = Arrays.stream(commands).map(CommandNode::getName).toArray(String[]::new);

        return addCommands(commandNames);
    }

    @Override
    public HelpCommand addCommands(String... commands) {
        for (String command : commands) {
            this.commands.putIfAbsent(command, Collections.emptySet());
        }
        this.commands.putIfAbsent(helpCommandBase.getLiteral(), Collections.emptySet());

        return this;
    }

    @Override
    public HelpCommand addSubCommandsFromCommands(CommandNode<?>... commands) {
        for (CommandNode<?> node : commands) {
            for (Iterator<? extends CommandNode<?>> it = node.getChildren().stream().sorted().iterator(); it.hasNext(); ) {
                CommandNode<?> child = it.next();
                int childPathHash = buildChildPathHash(new StringBuilder(), child);
                if (isNewChild(child, childPathHash)) {
                    this.commands.computeIfAbsent(node.getName(), NEW_CHILD_NODE_SET).add(child.getName());
                    this.grandchildNodeHashes.add(childPathHash);
                }
            }
        }
        this.commands.putIfAbsent(helpCommandBase.getLiteral(), Collections.emptySet());

        return this;
    }

    private boolean isNewChild(CommandNode<?> child, int childPathHash) {
        return child instanceof LiteralCommandNode && !grandchildNodeHashes.contains(childPathHash);
    }

    private int buildChildPathHash(StringBuilder stringBuilder, CommandNode<?> node) {
        if (stringBuilder.length() > 0) {
            stringBuilder.append(node.getName()).append("->");
        } else {
            stringBuilder.append("root->");
        }
        Optional<? extends CommandNode<?>> firstGrandchild = node.getChildren().stream().sorted().findFirst();
        //noinspection OptionalIsPresent
        if (!firstGrandchild.isPresent()) {
            return stringBuilder.append(node.getCommand().toString()).toString().hashCode();
        } else {
            return buildChildPathHash(stringBuilder, firstGrandchild.get());
        }
    }

    @Override
    public CommandNode<class_2168> register(CommandDispatcher<class_2168> commandDispatcher) {
        return commandDispatcher.register(helpCommandBase
            .executes((command) -> runHelpCommand(command, 1))
            .then(class_2170.method_9244("page", IntegerArgumentType.integer(1))
                .executes((command) -> runHelpCommand(command, command.getArgument("page", Integer.class)))
            )
        );
    }

    private int runHelpCommand(CommandContext<class_2168> command, int page) {
        textPaginator.sendPaginatedChat(command.getSource(), "/" + helpCommandBase.getLiteral() + " %s", getHelpsList(command), page);
        return Command.SINGLE_SUCCESS;
    }

    private List<? extends class_2561> getHelpsList(CommandContext<class_2168> command) {
        List<class_2561> helps = Lists.newArrayList();
        for (Map.Entry<String, Collection<String>> commandName : commands.entrySet()) {
            if (commandName.getValue().isEmpty()) {
                class_2561 commandHelp = buildCommandDescription(command, commandName.getKey());
                helps.add(commandHelp);
            } else {
                for (String subCommand : commandName.getValue()) {
                    class_2561 commandHelp = buildSubCommandDescription(command, commandName.getKey(), subCommand);
                    helps.add(commandHelp);
                }
            }
        }
        helps.sort(Comparator.comparing(class_2561::getString));

        int i = 0;
        for (class_2561 helpText : helps) {
            helpText.method_10862(i++ % 2 == 0 ? textStyles.white() : textStyles.grey());
        }

        return helps;
    }

    private class_2561 buildCommandDescription(CommandContext<class_2168> command, String commandName) {
        return translator.getTextForTarget(command.getSource(), "commands." + modid + "." + commandName + ".usage")
            .method_10864(": ")
            .method_10852(translator.getTextForTarget(command.getSource(), "commands." + modid + "." + commandName + ".description"));
    }

    private class_2561 buildSubCommandDescription(CommandContext<class_2168> command, String commandName, String subCommand) {
        return translator.getTextForTarget(command.getSource(), "commands." + modid + "." + commandName + "." + subCommand + ".usage")
            .method_10864(": ")
            .method_10852(translator.getTextForTarget(command.getSource(), "commands." + modid + "." + commandName + "." + subCommand + ".description"));
    }
}
