package zio.temporal.protobuf

import java.time.{Instant, LocalDate, LocalDateTime, LocalTime, ZoneId, ZoneOffset}
import java.{util => ju}
import scala.annotation.implicitNotFound
import zio.temporal.protobuf.internal._

/** Typeclass allowing conversion to protocol buffers supported type
  *
  * @tparam A
  *   value type
  */
@implicitNotFound("""Type ${A} cannot be encoded as protocol buffers type.
It should be an existing type or a type generated by scalapb library""")
trait ProtoType[A] {

  /** Protocol buffers representation of this type */
  type Repr

  /** Converts a value of this type to its protocol buffers representation
    * @param value
    *   input value
    * @return
    *   value protocol buffers representation
    */
  def repr(value: A): Repr

  /** Creates a value of this type from its protocol buffers representation
    * @param repr
    *   value protocol buffers representation
    * @return
    *   value of this type
    */
  def fromRepr(repr: Repr): A

  /** Creates a new protocol buffers type based on this representation. via InvariantFunctor.imap
    *
    * @tparam B
    *   new value type
    * @param f
    *   function projecting this type into the given one
    * @param g
    *   reverse projection function
    * @return
    *   new protocol buffers type
    */
  final def convertTo[B](f: A => B)(g: B => A): ProtoType.Of[B, Repr] =
    new ConvertedType[A, Repr, B](this, f, g)
}

object ProtoType extends LowPriorityProtoTypes {
  type Of[A, Repr0] = ProtoType[A] { type Repr = Repr0 }

  implicit val integerType: ProtoType.Of[Int, Int]               = IntegerType
  implicit val longType: ProtoType.Of[Long, Long]                = LongType
  implicit val booleanType: ProtoType.Of[Boolean, Boolean]       = BooleanType
  implicit val stringType: ProtoType.Of[String, String]          = StringType
  implicit val bytesType: ProtoType.Of[Array[Byte], Array[Byte]] = BytesType
}

sealed trait LowPriorityProtoTypes {

  implicit val uuidType: ProtoType.Of[ju.UUID, UUID]                      = UuidType
  implicit val bigIntType: ProtoType.Of[BigInt, BigInteger]               = BigIntegerType
  implicit val bigDecimalType: ProtoType.Of[scala.BigDecimal, BigDecimal] = BigDecimalType
  implicit val unitType: ProtoType.Of[Unit, ZUnit]                        = ZUnitType

  implicit val instantType: ProtoType.Of[Instant, Long] =
    LongType.convertTo(Instant.ofEpochMilli)(_.toEpochMilli)

  implicit val localDateTimeType: ProtoType.Of[LocalDateTime, Long] =
    instantType.convertTo(_.atOffset(ZoneOffset.UTC).toLocalDateTime)(_.atOffset(ZoneOffset.UTC).toInstant)

  implicit val zoneIdProtoType: ProtoType.Of[ZoneId, String] =
    StringType.convertTo(ZoneId.of)(_.getId)

  implicit val localTimeProtoType: ProtoType.Of[LocalTime, Long] =
    LongType.convertTo(LocalTime.ofNanoOfDay)(_.toNanoOfDay)

  implicit val localDateProtoType: ProtoType.Of[LocalDate, Long] =
    localDateTimeType.convertTo(_.toLocalDate)(_.atStartOfDay())

  implicit val zioDurationProtoType: ProtoType.Of[zio.Duration, Long] =
    LongType.convertTo(zio.Duration.fromNanos)(_.toNanos)

  implicit def optionProtoType[A](implicit protoType: ProtoType[A]): ProtoType.Of[Option[A], Option[protoType.Repr]] =
    new OptionType[A, protoType.Repr](protoType)
}
