package zio.aws.firehose.model
import java.time.Instant
import zio.prelude.data.Optional
import zio.aws.core.{AwsError, BuilderHelper}
import zio.ZIO
import zio.aws.firehose.model.primitives.{
  ReadFromTimestamp,
  TopicName,
  MSKClusterARN
}
import scala.jdk.CollectionConverters._
final case class MSKSourceConfiguration(
    mskClusterARN: MSKClusterARN,
    topicName: TopicName,
    authenticationConfiguration: zio.aws.firehose.model.AuthenticationConfiguration,
    readFromTimestamp: Optional[ReadFromTimestamp] = Optional.Absent
) {
  def buildAwsValue()
      : software.amazon.awssdk.services.firehose.model.MSKSourceConfiguration = {
    import MSKSourceConfiguration.zioAwsBuilderHelper.BuilderOps
    software.amazon.awssdk.services.firehose.model.MSKSourceConfiguration
      .builder()
      .mskClusterARN(MSKClusterARN.unwrap(mskClusterARN): java.lang.String)
      .topicName(TopicName.unwrap(topicName): java.lang.String)
      .authenticationConfiguration(authenticationConfiguration.buildAwsValue())
      .optionallyWith(
        readFromTimestamp.map(value => ReadFromTimestamp.unwrap(value): Instant)
      )(_.readFromTimestamp)
      .build()
  }
  def asReadOnly: zio.aws.firehose.model.MSKSourceConfiguration.ReadOnly =
    zio.aws.firehose.model.MSKSourceConfiguration.wrap(buildAwsValue())
}
object MSKSourceConfiguration {
  private lazy val zioAwsBuilderHelper: BuilderHelper[
    software.amazon.awssdk.services.firehose.model.MSKSourceConfiguration
  ] = BuilderHelper.apply
  trait ReadOnly {
    def asEditable: zio.aws.firehose.model.MSKSourceConfiguration =
      zio.aws.firehose.model.MSKSourceConfiguration(
        mskClusterARN,
        topicName,
        authenticationConfiguration.asEditable,
        readFromTimestamp.map(value => value)
      )
    def mskClusterARN: MSKClusterARN
    def topicName: TopicName
    def authenticationConfiguration
        : zio.aws.firehose.model.AuthenticationConfiguration.ReadOnly
    def readFromTimestamp: Optional[ReadFromTimestamp]
    def getMskClusterARN: ZIO[Any, Nothing, MSKClusterARN] =
      ZIO.succeed(mskClusterARN)
    def getTopicName: ZIO[Any, Nothing, TopicName] = ZIO.succeed(topicName)
    def getAuthenticationConfiguration: ZIO[
      Any,
      Nothing,
      zio.aws.firehose.model.AuthenticationConfiguration.ReadOnly
    ] = ZIO.succeed(authenticationConfiguration)
    def getReadFromTimestamp: ZIO[Any, AwsError, ReadFromTimestamp] =
      AwsError.unwrapOptionField("readFromTimestamp", readFromTimestamp)
  }
  private final class Wrapper(
      impl: software.amazon.awssdk.services.firehose.model.MSKSourceConfiguration
  ) extends zio.aws.firehose.model.MSKSourceConfiguration.ReadOnly {
    override val mskClusterARN: MSKClusterARN =
      zio.aws.firehose.model.primitives.MSKClusterARN(impl.mskClusterARN())
    override val topicName: TopicName =
      zio.aws.firehose.model.primitives.TopicName(impl.topicName())
    override val authenticationConfiguration
        : zio.aws.firehose.model.AuthenticationConfiguration.ReadOnly =
      zio.aws.firehose.model.AuthenticationConfiguration
        .wrap(impl.authenticationConfiguration())
    override val readFromTimestamp: Optional[ReadFromTimestamp] =
      zio.aws.core.internal
        .optionalFromNullable(impl.readFromTimestamp())
        .map(value =>
          zio.aws.firehose.model.primitives.ReadFromTimestamp(value)
        )
  }
  def wrap(
      impl: software.amazon.awssdk.services.firehose.model.MSKSourceConfiguration
  ): zio.aws.firehose.model.MSKSourceConfiguration.ReadOnly = new Wrapper(impl)
}
