package zio.aws.iam.model
import java.time.Instant
import zio.prelude.data.Optional
import zio.aws.core.{AwsError, BuilderHelper}
import zio.ZIO
import zio.aws.iam.model.primitives.{
  DateType,
  AccessKeySecretType,
  AccessKeyIdType,
  UserNameType
}
import scala.jdk.CollectionConverters.*
final case class AccessKey(
    userName: UserNameType,
    accessKeyId: AccessKeyIdType,
    status: zio.aws.iam.model.StatusType,
    secretAccessKey: AccessKeySecretType,
    createDate: Optional[DateType] = Optional.Absent
) {
  def buildAwsValue(): software.amazon.awssdk.services.iam.model.AccessKey = {
    import AccessKey.zioAwsBuilderHelper.BuilderOps
    software.amazon.awssdk.services.iam.model.AccessKey
      .builder()
      .userName(UserNameType.unwrap(userName): java.lang.String)
      .accessKeyId(AccessKeyIdType.unwrap(accessKeyId): java.lang.String)
      .status(status.unwrap)
      .secretAccessKey(
        AccessKeySecretType.unwrap(secretAccessKey): java.lang.String
      )
      .optionallyWith(createDate.map(value => DateType.unwrap(value): Instant))(
        _.createDate
      )
      .build()
  }
  def asReadOnly: zio.aws.iam.model.AccessKey.ReadOnly =
    zio.aws.iam.model.AccessKey.wrap(buildAwsValue())
}
object AccessKey {
  private lazy val zioAwsBuilderHelper
      : BuilderHelper[software.amazon.awssdk.services.iam.model.AccessKey] =
    BuilderHelper.apply
  trait ReadOnly {
    def asEditable: zio.aws.iam.model.AccessKey = zio.aws.iam.model.AccessKey(
      userName,
      accessKeyId,
      status,
      secretAccessKey,
      createDate.map(value => value)
    )
    def userName: UserNameType
    def accessKeyId: AccessKeyIdType
    def status: zio.aws.iam.model.StatusType
    def secretAccessKey: AccessKeySecretType
    def createDate: Optional[DateType]
    def getUserName: ZIO[Any, Nothing, UserNameType] = ZIO.succeed(userName)
    def getAccessKeyId: ZIO[Any, Nothing, AccessKeyIdType] =
      ZIO.succeed(accessKeyId)
    def getStatus: ZIO[Any, Nothing, zio.aws.iam.model.StatusType] =
      ZIO.succeed(status)
    def getSecretAccessKey: ZIO[Any, Nothing, AccessKeySecretType] =
      ZIO.succeed(secretAccessKey)
    def getCreateDate: ZIO[Any, AwsError, DateType] =
      AwsError.unwrapOptionField("createDate", createDate)
  }
  private final class Wrapper(
      impl: software.amazon.awssdk.services.iam.model.AccessKey
  ) extends zio.aws.iam.model.AccessKey.ReadOnly {
    override val userName: UserNameType =
      zio.aws.iam.model.primitives.UserNameType(impl.userName())
    override val accessKeyId: AccessKeyIdType =
      zio.aws.iam.model.primitives.AccessKeyIdType(impl.accessKeyId())
    override val status: zio.aws.iam.model.StatusType =
      zio.aws.iam.model.StatusType.wrap(impl.status())
    override val secretAccessKey: AccessKeySecretType =
      zio.aws.iam.model.primitives.AccessKeySecretType(impl.secretAccessKey())
    override val createDate: Optional[DateType] = zio.aws.core.internal
      .optionalFromNullable(impl.createDate())
      .map(value => zio.aws.iam.model.primitives.DateType(value))
  }
  def wrap(
      impl: software.amazon.awssdk.services.iam.model.AccessKey
  ): zio.aws.iam.model.AccessKey.ReadOnly = new Wrapper(impl)
}
