package zio.aws.location.model
import zio.aws.location.model.primitives.{
  PropertyMapKeyString,
  NearestDistance,
  PropertyMapValueString,
  Timestamp,
  Uuid,
  Id
}
import java.time.Instant
import zio.prelude.data.Optional
import zio.aws.core.{AwsError, BuilderHelper}
import zio.ZIO
import scala.jdk.CollectionConverters._
final case class ForecastedEvent(
    eventId: Uuid,
    geofenceId: Id,
    isDeviceInGeofence: Boolean,
    nearestDistance: NearestDistance,
    eventType: zio.aws.location.model.ForecastedGeofenceEventType,
    forecastedBreachTime: Optional[Timestamp] = Optional.Absent,
    geofenceProperties: Optional[
      Map[PropertyMapKeyString, PropertyMapValueString]
    ] = Optional.Absent
) {
  def buildAwsValue()
      : software.amazon.awssdk.services.location.model.ForecastedEvent = {
    import ForecastedEvent.zioAwsBuilderHelper.BuilderOps
    software.amazon.awssdk.services.location.model.ForecastedEvent
      .builder()
      .eventId(Uuid.unwrap(eventId): java.lang.String)
      .geofenceId(Id.unwrap(geofenceId): java.lang.String)
      .isDeviceInGeofence(isDeviceInGeofence: java.lang.Boolean)
      .nearestDistance(
        NearestDistance.unwrap(nearestDistance): java.lang.Double
      )
      .eventType(eventType.unwrap)
      .optionallyWith(
        forecastedBreachTime.map(value => Timestamp.unwrap(value): Instant)
      )(_.forecastedBreachTime)
      .optionallyWith(
        geofenceProperties.map(value =>
          value
            .map({ case (key, value) =>
              (PropertyMapKeyString
                .unwrap(key): java.lang.String) -> (PropertyMapValueString
                .unwrap(value): java.lang.String)
            })
            .asJava
        )
      )(_.geofenceProperties)
      .build()
  }
  def asReadOnly: zio.aws.location.model.ForecastedEvent.ReadOnly =
    zio.aws.location.model.ForecastedEvent.wrap(buildAwsValue())
}
object ForecastedEvent {
  private lazy val zioAwsBuilderHelper: BuilderHelper[
    software.amazon.awssdk.services.location.model.ForecastedEvent
  ] = BuilderHelper.apply
  trait ReadOnly {
    def asEditable: zio.aws.location.model.ForecastedEvent =
      zio.aws.location.model.ForecastedEvent(
        eventId,
        geofenceId,
        isDeviceInGeofence,
        nearestDistance,
        eventType,
        forecastedBreachTime.map(value => value),
        geofenceProperties.map(value => value)
      )
    def eventId: Uuid
    def geofenceId: Id
    def isDeviceInGeofence: Boolean
    def nearestDistance: NearestDistance
    def eventType: zio.aws.location.model.ForecastedGeofenceEventType
    def forecastedBreachTime: Optional[Timestamp]
    def geofenceProperties
        : Optional[Map[PropertyMapKeyString, PropertyMapValueString]]
    def getEventId: ZIO[Any, Nothing, Uuid] = ZIO.succeed(eventId)
    def getGeofenceId: ZIO[Any, Nothing, Id] = ZIO.succeed(geofenceId)
    def getIsDeviceInGeofence: ZIO[Any, Nothing, Boolean] =
      ZIO.succeed(isDeviceInGeofence)
    def getNearestDistance: ZIO[Any, Nothing, NearestDistance] =
      ZIO.succeed(nearestDistance)
    def getEventType: ZIO[
      Any,
      Nothing,
      zio.aws.location.model.ForecastedGeofenceEventType
    ] = ZIO.succeed(eventType)
    def getForecastedBreachTime: ZIO[Any, AwsError, Timestamp] =
      AwsError.unwrapOptionField("forecastedBreachTime", forecastedBreachTime)
    def getGeofenceProperties: ZIO[Any, AwsError, Map[
      PropertyMapKeyString,
      PropertyMapValueString
    ]] = AwsError.unwrapOptionField("geofenceProperties", geofenceProperties)
  }
  private final class Wrapper(
      impl: software.amazon.awssdk.services.location.model.ForecastedEvent
  ) extends zio.aws.location.model.ForecastedEvent.ReadOnly {
    override val eventId: Uuid =
      zio.aws.location.model.primitives.Uuid(impl.eventId())
    override val geofenceId: Id =
      zio.aws.location.model.primitives.Id(impl.geofenceId())
    override val isDeviceInGeofence: Boolean =
      impl.isDeviceInGeofence(): Boolean
    override val nearestDistance: NearestDistance =
      zio.aws.location.model.primitives.NearestDistance(impl.nearestDistance())
    override val eventType: zio.aws.location.model.ForecastedGeofenceEventType =
      zio.aws.location.model.ForecastedGeofenceEventType.wrap(impl.eventType())
    override val forecastedBreachTime: Optional[Timestamp] =
      zio.aws.core.internal
        .optionalFromNullable(impl.forecastedBreachTime())
        .map(value => zio.aws.location.model.primitives.Timestamp(value))
    override val geofenceProperties
        : Optional[Map[PropertyMapKeyString, PropertyMapValueString]] =
      zio.aws.core.internal
        .optionalFromNullable(impl.geofenceProperties())
        .map(value =>
          value.asScala
            .map({ case (key, value) =>
              zio.aws.location.model.primitives
                .PropertyMapKeyString(key) -> zio.aws.location.model.primitives
                .PropertyMapValueString(value)
            })
            .toMap
        )
  }
  def wrap(
      impl: software.amazon.awssdk.services.location.model.ForecastedEvent
  ): zio.aws.location.model.ForecastedEvent.ReadOnly = new Wrapper(impl)
}
