package zio.aws.xray.model
import java.time.Instant
import zio.aws.xray.model.primitives.{NullableInteger, Timestamp}
import zio.aws.core.{AwsError, BuilderHelper}
import zio.ZIO
import java.lang.Integer
import scala.jdk.CollectionConverters._
final case class TelemetryRecord(
    timestamp: Timestamp,
    segmentsReceivedCount: Option[NullableInteger] = None,
    segmentsSentCount: Option[NullableInteger] = None,
    segmentsSpilloverCount: Option[NullableInteger] = None,
    segmentsRejectedCount: Option[NullableInteger] = None,
    backendConnectionErrors: Option[
      zio.aws.xray.model.BackendConnectionErrors
    ] = None
) {
  def buildAwsValue()
      : software.amazon.awssdk.services.xray.model.TelemetryRecord = {
    import TelemetryRecord.zioAwsBuilderHelper.BuilderOps
    software.amazon.awssdk.services.xray.model.TelemetryRecord
      .builder()
      .timestamp(Timestamp.unwrap(timestamp): Instant)
      .optionallyWith(segmentsReceivedCount.map(value => value: Integer))(
        _.segmentsReceivedCount
      )
      .optionallyWith(segmentsSentCount.map(value => value: Integer))(
        _.segmentsSentCount
      )
      .optionallyWith(segmentsSpilloverCount.map(value => value: Integer))(
        _.segmentsSpilloverCount
      )
      .optionallyWith(segmentsRejectedCount.map(value => value: Integer))(
        _.segmentsRejectedCount
      )
      .optionallyWith(
        backendConnectionErrors.map(value => value.buildAwsValue())
      )(_.backendConnectionErrors)
      .build()
  }
  def asReadOnly: zio.aws.xray.model.TelemetryRecord.ReadOnly =
    zio.aws.xray.model.TelemetryRecord.wrap(buildAwsValue())
}
object TelemetryRecord {
  private lazy val zioAwsBuilderHelper: BuilderHelper[
    software.amazon.awssdk.services.xray.model.TelemetryRecord
  ] = BuilderHelper.apply
  trait ReadOnly {
    def asEditable: zio.aws.xray.model.TelemetryRecord =
      zio.aws.xray.model.TelemetryRecord(
        timestamp,
        segmentsReceivedCount.map(value => value),
        segmentsSentCount.map(value => value),
        segmentsSpilloverCount.map(value => value),
        segmentsRejectedCount.map(value => value),
        backendConnectionErrors.map(value => value.asEditable)
      )
    def timestamp: Timestamp
    def segmentsReceivedCount: Option[NullableInteger]
    def segmentsSentCount: Option[NullableInteger]
    def segmentsSpilloverCount: Option[NullableInteger]
    def segmentsRejectedCount: Option[NullableInteger]
    def backendConnectionErrors
        : Option[zio.aws.xray.model.BackendConnectionErrors.ReadOnly]
    def getTimestamp: ZIO[Any, Nothing, Timestamp] = ZIO.succeed(timestamp)
    def getSegmentsReceivedCount: ZIO[Any, AwsError, NullableInteger] =
      AwsError.unwrapOptionField("segmentsReceivedCount", segmentsReceivedCount)
    def getSegmentsSentCount: ZIO[Any, AwsError, NullableInteger] =
      AwsError.unwrapOptionField("segmentsSentCount", segmentsSentCount)
    def getSegmentsSpilloverCount: ZIO[Any, AwsError, NullableInteger] =
      AwsError.unwrapOptionField(
        "segmentsSpilloverCount",
        segmentsSpilloverCount
      )
    def getSegmentsRejectedCount: ZIO[Any, AwsError, NullableInteger] =
      AwsError.unwrapOptionField("segmentsRejectedCount", segmentsRejectedCount)
    def getBackendConnectionErrors: ZIO[
      Any,
      AwsError,
      zio.aws.xray.model.BackendConnectionErrors.ReadOnly
    ] = AwsError.unwrapOptionField(
      "backendConnectionErrors",
      backendConnectionErrors
    )
  }
  private final class Wrapper(
      impl: software.amazon.awssdk.services.xray.model.TelemetryRecord
  ) extends zio.aws.xray.model.TelemetryRecord.ReadOnly {
    override val timestamp: Timestamp =
      zio.aws.xray.model.primitives.Timestamp(impl.timestamp())
    override val segmentsReceivedCount: Option[NullableInteger] = scala
      .Option(impl.segmentsReceivedCount())
      .map(value => value: NullableInteger)
    override val segmentsSentCount: Option[NullableInteger] = scala
      .Option(impl.segmentsSentCount())
      .map(value => value: NullableInteger)
    override val segmentsSpilloverCount: Option[NullableInteger] = scala
      .Option(impl.segmentsSpilloverCount())
      .map(value => value: NullableInteger)
    override val segmentsRejectedCount: Option[NullableInteger] = scala
      .Option(impl.segmentsRejectedCount())
      .map(value => value: NullableInteger)
    override val backendConnectionErrors
        : Option[zio.aws.xray.model.BackendConnectionErrors.ReadOnly] = scala
      .Option(impl.backendConnectionErrors())
      .map(value => zio.aws.xray.model.BackendConnectionErrors.wrap(value))
  }
  def wrap(
      impl: software.amazon.awssdk.services.xray.model.TelemetryRecord
  ): zio.aws.xray.model.TelemetryRecord.ReadOnly = new Wrapper(impl)
}
