// Copyright (c) 2016-2023 Association of Universities for Research in Astronomy, Inc. (AURA)
// For license information see LICENSE or https://opensource.org/licenses/BSD-3-Clause

package lucuma.schemas

import lucuma.core.enums
import lucuma.core.model
import lucuma.core.model._
import lucuma.core.model.sequence._
import lucuma.core.math.dimensional._
import lucuma.core.util._
import lucuma.core.math.BrightnessUnits._

import clue.BigNumberEncoders._
import io.circe.refined._


sealed trait ObservationDB
object ObservationDB {
  object Scalars {
    type AtomId = Atom.Id
    type ExecutionEventId = ExecutionEvent.Id
    type ObservationId = Observation.Id
    type ProgramId = Program.Id
    type StepId = Step.Id
    type TargetId = Target.Id
    type UserId = User.Id
    type VisitId = Visit.Id
    type BigDecimal = scala.BigDecimal
    type Long = scala.Long
    type DatasetFilename = String
    type DmsString = String
    type EpochString = String
    type HmsString = String
    type NonEmptyString = eu.timepit.refined.types.string.NonEmptyString
    type NonNegBigDecimal = eu.timepit.refined.types.numeric.NonNegBigDecimal
    type NonNegInt = eu.timepit.refined.types.numeric.NonNegInt
    type NonNegLong = eu.timepit.refined.types.numeric.NonNegLong
    type PosBigDecimal = eu.timepit.refined.types.numeric.PosBigDecimal
    type PosInt = eu.timepit.refined.types.numeric.PosInt
    type PosLong = eu.timepit.refined.types.numeric.PosLong
    type Timestamp = lucuma.core.util.Timestamp
    def ignoreUnusedImportScalars(): Unit = ()
  }
  object Enums {
    def ignoreUnusedImportEnums(): Unit = ()
    type Band = enums.Band
    type Breakpoint = enums.Breakpoint
    type BrightnessIntegratedUnits = Units Of Brightness[Integrated]
    type BrightnessSurfaceUnits = Units Of Brightness[Surface]
    type CatalogName = enums.CatalogName
    type CloudExtinction = enums.CloudExtinction
    type CoolStarTemperature = enums.CoolStarTemperature
    type DatasetQaState = enums.DatasetQaState
    type DatasetStage = enums.DatasetStage
    type EphemerisKeyType = enums.EphemerisKeyType
    type FluxDensityContinuumIntegratedUnits = Units Of FluxDensityContinuum[Integrated]
    type FluxDensityContinuumSurfaceUnits = Units Of FluxDensityContinuum[Surface]
    type FocalPlane = enums.FocalPlane
    type GalaxySpectrum = enums.GalaxySpectrum
    type GcalArc = enums.GcalArc
    type GcalContinuum = enums.GcalContinuum
    type GcalDiffuser = enums.GcalDiffuser
    type GcalFilter = enums.GcalFilter
    type GcalShutter = enums.GcalShutter
    type GmosAmpCount = enums.GmosAmpCount
    type GmosAmpGain = enums.GmosAmpGain
    type GmosAmpReadMode = enums.GmosAmpReadMode
    type GmosCustomSlitWidth = enums.GmosCustomSlitWidth
    type GmosGratingOrder = enums.GmosGratingOrder
    type GmosDtax = enums.GmosDtax
    type GmosEOffsetting = enums.GmosEOffsetting
    type GmosNorthBuiltinFpu = enums.GmosNorthFpu
    type GmosNorthDetector = enums.GmosNorthDetector
    type GmosNorthGrating = enums.GmosNorthGrating
    type GmosNorthFilter = enums.GmosNorthFilter
    type GmosNorthStageMode = enums.GmosNorthStageMode
    type GmosRoi = enums.GmosRoi
    type GmosSouthBuiltinFpu = enums.GmosSouthFpu
    type GmosSouthDetector = enums.GmosSouthDetector
    type GmosSouthGrating = enums.GmosSouthGrating
    type GmosSouthFilter = enums.GmosSouthFilter
    type GmosSouthStageMode = enums.GmosSouthStageMode
    type GmosXBinning = enums.GmosXBinning
    type GmosYBinning = enums.GmosYBinning
    type HiiRegionSpectrum = enums.HIIRegionSpectrum
    type ImageQuality = enums.ImageQuality
    type InstrumentType = enums.Instrument
    type LineFluxIntegratedUnits = Units Of LineFlux[Integrated]
    type LineFluxSurfaceUnits = Units Of LineFlux[Surface]
    type MosPreImaging = enums.MosPreImaging
    type ObsActiveStatus = enums.ObsActiveStatus
    type ObsStatus = enums.ObsStatus
    type Partner = model.Partner
    type PlanetSpectrum = enums.PlanetSpectrum
    type PlanetaryNebulaSpectrum = enums.PlanetaryNebulaSpectrum
    type QuasarSpectrum = enums.QuasarSpectrum
    type ScienceMode = enums.ScienceMode
    type SequenceCommand = enums.SequenceCommand
    type SkyBackground = enums.SkyBackground
    type SpectroscopyCapabilities = enums.SpectroscopyCapabilities
    type StellarLibrarySpectrum = enums.StellarLibrarySpectrum
    type StepQaState = enums.StepQaState
    type StepStage = enums.StepStage
    type StepType = enums.StepType
    type TacCategory = enums.TacCategory
    type ToOActivation = enums.ToOActivation
    type WaterVapor = enums.WaterVapor
    sealed trait EditType
    object EditType {
      case object Created extends EditType()
      case object Updated extends EditType()
      implicit val eqEditType: cats.Eq[EditType] = cats.Eq.fromUniversalEquals
      implicit val showEditType: cats.Show[EditType] = cats.Show.fromToString
      implicit val jsonEncoderEditType: io.circe.Encoder[EditType] = io.circe.Encoder.encodeString.contramap[EditType]({
        case Created => "CREATED"
        case Updated => "UPDATED"
      })
      implicit val jsonDecoderEditType: io.circe.Decoder[EditType] = io.circe.Decoder.decodeString.emap(_ match {
        case "CREATED" =>
          Right(Created)
        case "UPDATED" =>
          Right(Updated)
        case other =>
          Left(s"Invalid value [$other]")
      })
    }
    sealed trait ProgramUserRole
    object ProgramUserRole {
      case object Coi extends ProgramUserRole()
      case object Observer extends ProgramUserRole()
      case object Support extends ProgramUserRole()
      implicit val eqProgramUserRole: cats.Eq[ProgramUserRole] = cats.Eq.fromUniversalEquals
      implicit val showProgramUserRole: cats.Show[ProgramUserRole] = cats.Show.fromToString
      implicit val jsonEncoderProgramUserRole: io.circe.Encoder[ProgramUserRole] = io.circe.Encoder.encodeString.contramap[ProgramUserRole]({
        case Coi => "COI"
        case Observer => "OBSERVER"
        case Support => "SUPPORT"
      })
      implicit val jsonDecoderProgramUserRole: io.circe.Decoder[ProgramUserRole] = io.circe.Decoder.decodeString.emap(_ match {
        case "COI" =>
          Right(Coi)
        case "OBSERVER" =>
          Right(Observer)
        case "SUPPORT" =>
          Right(Support)
        case other =>
          Left(s"Invalid value [$other]")
      })
    }
    sealed trait ProgramUserSupportRoleType
    object ProgramUserSupportRoleType {
      case object Staff extends ProgramUserSupportRoleType()
      case object Partner extends ProgramUserSupportRoleType()
      implicit val eqProgramUserSupportRoleType: cats.Eq[ProgramUserSupportRoleType] = cats.Eq.fromUniversalEquals
      implicit val showProgramUserSupportRoleType: cats.Show[ProgramUserSupportRoleType] = cats.Show.fromToString
      implicit val jsonEncoderProgramUserSupportRoleType: io.circe.Encoder[ProgramUserSupportRoleType] = io.circe.Encoder.encodeString.contramap[ProgramUserSupportRoleType]({
        case Staff => "STAFF"
        case Partner => "PARTNER"
      })
      implicit val jsonDecoderProgramUserSupportRoleType: io.circe.Decoder[ProgramUserSupportRoleType] = io.circe.Decoder.decodeString.emap(_ match {
        case "STAFF" =>
          Right(Staff)
        case "PARTNER" =>
          Right(Partner)
        case other =>
          Left(s"Invalid value [$other]")
      })
    }
    sealed trait ObservingModeType
    object ObservingModeType {
      case object GmosNorthLongSlit extends ObservingModeType()
      case object GmosSouthLongSlit extends ObservingModeType()
      implicit val eqObservingModeType: cats.Eq[ObservingModeType] = cats.Eq.fromUniversalEquals
      implicit val showObservingModeType: cats.Show[ObservingModeType] = cats.Show.fromToString
      implicit val jsonEncoderObservingModeType: io.circe.Encoder[ObservingModeType] = io.circe.Encoder.encodeString.contramap[ObservingModeType]({
        case GmosNorthLongSlit => "GMOS_NORTH_LONG_SLIT"
        case GmosSouthLongSlit => "GMOS_SOUTH_LONG_SLIT"
      })
      implicit val jsonDecoderObservingModeType: io.circe.Decoder[ObservingModeType] = io.circe.Decoder.decodeString.emap(_ match {
        case "GMOS_NORTH_LONG_SLIT" =>
          Right(GmosNorthLongSlit)
        case "GMOS_SOUTH_LONG_SLIT" =>
          Right(GmosSouthLongSlit)
        case other =>
          Left(s"Invalid value [$other]")
      })
    }
    sealed trait Existence
    object Existence {
      case object Present extends Existence()
      case object Deleted extends Existence()
      implicit val eqExistence: cats.Eq[Existence] = cats.Eq.fromUniversalEquals
      implicit val showExistence: cats.Show[Existence] = cats.Show.fromToString
      implicit val jsonEncoderExistence: io.circe.Encoder[Existence] = io.circe.Encoder.encodeString.contramap[Existence]({
        case Present => "PRESENT"
        case Deleted => "DELETED"
      })
      implicit val jsonDecoderExistence: io.circe.Decoder[Existence] = io.circe.Decoder.decodeString.emap(_ match {
        case "PRESENT" =>
          Right(Present)
        case "DELETED" =>
          Right(Deleted)
        case other =>
          Left(s"Invalid value [$other]")
      })
    }
    sealed trait Instrument
    object Instrument {
      case object AcqCam extends Instrument()
      case object Bhros extends Instrument()
      case object Flamingos2 extends Instrument()
      case object Ghost extends Instrument()
      case object GmosNorth extends Instrument()
      case object GmosSouth extends Instrument()
      case object Gnirs extends Instrument()
      case object Gpi extends Instrument()
      case object Gsaoi extends Instrument()
      case object Michelle extends Instrument()
      case object Nici extends Instrument()
      case object Nifs extends Instrument()
      case object Niri extends Instrument()
      case object Phoenix extends Instrument()
      case object Trecs extends Instrument()
      case object Visitor extends Instrument()
      case object Scorpio extends Instrument()
      case object Alopeke extends Instrument()
      case object Zorro extends Instrument()
      implicit val eqInstrument: cats.Eq[Instrument] = cats.Eq.fromUniversalEquals
      implicit val showInstrument: cats.Show[Instrument] = cats.Show.fromToString
      implicit val jsonEncoderInstrument: io.circe.Encoder[Instrument] = io.circe.Encoder.encodeString.contramap[Instrument]({
        case AcqCam => "ACQ_CAM"
        case Bhros => "BHROS"
        case Flamingos2 => "FLAMINGOS2"
        case Ghost => "GHOST"
        case GmosNorth => "GMOS_NORTH"
        case GmosSouth => "GMOS_SOUTH"
        case Gnirs => "GNIRS"
        case Gpi => "GPI"
        case Gsaoi => "GSAOI"
        case Michelle => "MICHELLE"
        case Nici => "NICI"
        case Nifs => "NIFS"
        case Niri => "NIRI"
        case Phoenix => "PHOENIX"
        case Trecs => "TRECS"
        case Visitor => "VISITOR"
        case Scorpio => "SCORPIO"
        case Alopeke => "ALOPEKE"
        case Zorro => "ZORRO"
      })
      implicit val jsonDecoderInstrument: io.circe.Decoder[Instrument] = io.circe.Decoder.decodeString.emap(_ match {
        case "ACQ_CAM" =>
          Right(AcqCam)
        case "BHROS" =>
          Right(Bhros)
        case "FLAMINGOS2" =>
          Right(Flamingos2)
        case "GHOST" =>
          Right(Ghost)
        case "GMOS_NORTH" =>
          Right(GmosNorth)
        case "GMOS_SOUTH" =>
          Right(GmosSouth)
        case "GNIRS" =>
          Right(Gnirs)
        case "GPI" =>
          Right(Gpi)
        case "GSAOI" =>
          Right(Gsaoi)
        case "MICHELLE" =>
          Right(Michelle)
        case "NICI" =>
          Right(Nici)
        case "NIFS" =>
          Right(Nifs)
        case "NIRI" =>
          Right(Niri)
        case "PHOENIX" =>
          Right(Phoenix)
        case "TRECS" =>
          Right(Trecs)
        case "VISITOR" =>
          Right(Visitor)
        case "SCORPIO" =>
          Right(Scorpio)
        case "ALOPEKE" =>
          Right(Alopeke)
        case "ZORRO" =>
          Right(Zorro)
        case other =>
          Left(s"Invalid value [$other]")
      })
    }
    sealed trait ItcStatus
    object ItcStatus {
      case object Success extends ItcStatus()
      case object MissingParams extends ItcStatus()
      case object ServiceError extends ItcStatus()
      implicit val eqItcStatus: cats.Eq[ItcStatus] = cats.Eq.fromUniversalEquals
      implicit val showItcStatus: cats.Show[ItcStatus] = cats.Show.fromToString
      implicit val jsonEncoderItcStatus: io.circe.Encoder[ItcStatus] = io.circe.Encoder.encodeString.contramap[ItcStatus]({
        case Success => "SUCCESS"
        case MissingParams => "MISSING_PARAMS"
        case ServiceError => "SERVICE_ERROR"
      })
      implicit val jsonDecoderItcStatus: io.circe.Decoder[ItcStatus] = io.circe.Decoder.decodeString.emap(_ match {
        case "SUCCESS" =>
          Right(Success)
        case "MISSING_PARAMS" =>
          Right(MissingParams)
        case "SERVICE_ERROR" =>
          Right(ServiceError)
        case other =>
          Left(s"Invalid value [$other]")
      })
    }
    sealed trait PosAngleConstraintMode
    object PosAngleConstraintMode {
      case object Unbounded extends PosAngleConstraintMode()
      case object Fixed extends PosAngleConstraintMode()
      case object AllowFlip extends PosAngleConstraintMode()
      case object AverageParallactic extends PosAngleConstraintMode()
      case object ParallacticOverride extends PosAngleConstraintMode()
      implicit val eqPosAngleConstraintMode: cats.Eq[PosAngleConstraintMode] = cats.Eq.fromUniversalEquals
      implicit val showPosAngleConstraintMode: cats.Show[PosAngleConstraintMode] = cats.Show.fromToString
      implicit val jsonEncoderPosAngleConstraintMode: io.circe.Encoder[PosAngleConstraintMode] = io.circe.Encoder.encodeString.contramap[PosAngleConstraintMode]({
        case Unbounded => "UNBOUNDED"
        case Fixed => "FIXED"
        case AllowFlip => "ALLOW_FLIP"
        case AverageParallactic => "AVERAGE_PARALLACTIC"
        case ParallacticOverride => "PARALLACTIC_OVERRIDE"
      })
      implicit val jsonDecoderPosAngleConstraintMode: io.circe.Decoder[PosAngleConstraintMode] = io.circe.Decoder.decodeString.emap(_ match {
        case "UNBOUNDED" =>
          Right(Unbounded)
        case "FIXED" =>
          Right(Fixed)
        case "ALLOW_FLIP" =>
          Right(AllowFlip)
        case "AVERAGE_PARALLACTIC" =>
          Right(AverageParallactic)
        case "PARALLACTIC_OVERRIDE" =>
          Right(ParallacticOverride)
        case other =>
          Left(s"Invalid value [$other]")
      })
    }
    sealed trait ProposalClassEnum
    object ProposalClassEnum {
      case object Classical extends ProposalClassEnum()
      case object DemoScience extends ProposalClassEnum()
      case object DirectorsTime extends ProposalClassEnum()
      case object Exchange extends ProposalClassEnum()
      case object FastTurnaround extends ProposalClassEnum()
      case object Intensive extends ProposalClassEnum()
      case object LargeProgram extends ProposalClassEnum()
      case object PoorWeather extends ProposalClassEnum()
      case object Queue extends ProposalClassEnum()
      case object SystemVerification extends ProposalClassEnum()
      implicit val eqProposalClassEnum: cats.Eq[ProposalClassEnum] = cats.Eq.fromUniversalEquals
      implicit val showProposalClassEnum: cats.Show[ProposalClassEnum] = cats.Show.fromToString
      implicit val jsonEncoderProposalClassEnum: io.circe.Encoder[ProposalClassEnum] = io.circe.Encoder.encodeString.contramap[ProposalClassEnum]({
        case Classical => "CLASSICAL"
        case DemoScience => "DEMO_SCIENCE"
        case DirectorsTime => "DIRECTORS_TIME"
        case Exchange => "EXCHANGE"
        case FastTurnaround => "FAST_TURNAROUND"
        case Intensive => "INTENSIVE"
        case LargeProgram => "LARGE_PROGRAM"
        case PoorWeather => "POOR_WEATHER"
        case Queue => "QUEUE"
        case SystemVerification => "SYSTEM_VERIFICATION"
      })
      implicit val jsonDecoderProposalClassEnum: io.circe.Decoder[ProposalClassEnum] = io.circe.Decoder.decodeString.emap(_ match {
        case "CLASSICAL" =>
          Right(Classical)
        case "DEMO_SCIENCE" =>
          Right(DemoScience)
        case "DIRECTORS_TIME" =>
          Right(DirectorsTime)
        case "EXCHANGE" =>
          Right(Exchange)
        case "FAST_TURNAROUND" =>
          Right(FastTurnaround)
        case "INTENSIVE" =>
          Right(Intensive)
        case "LARGE_PROGRAM" =>
          Right(LargeProgram)
        case "POOR_WEATHER" =>
          Right(PoorWeather)
        case "QUEUE" =>
          Right(Queue)
        case "SYSTEM_VERIFICATION" =>
          Right(SystemVerification)
        case other =>
          Left(s"Invalid value [$other]")
      })
    }
    sealed trait SequenceType
    object SequenceType {
      case object Acquisition extends SequenceType()
      case object Science extends SequenceType()
      implicit val eqSequenceType: cats.Eq[SequenceType] = cats.Eq.fromUniversalEquals
      implicit val showSequenceType: cats.Show[SequenceType] = cats.Show.fromToString
      implicit val jsonEncoderSequenceType: io.circe.Encoder[SequenceType] = io.circe.Encoder.encodeString.contramap[SequenceType]({
        case Acquisition => "ACQUISITION"
        case Science => "SCIENCE"
      })
      implicit val jsonDecoderSequenceType: io.circe.Decoder[SequenceType] = io.circe.Decoder.decodeString.emap(_ match {
        case "ACQUISITION" =>
          Right(Acquisition)
        case "SCIENCE" =>
          Right(Science)
        case other =>
          Left(s"Invalid value [$other]")
      })
    }
    sealed trait UserType
    object UserType {
      case object Guest extends UserType()
      case object Standard extends UserType()
      case object Service extends UserType()
      implicit val eqUserType: cats.Eq[UserType] = cats.Eq.fromUniversalEquals
      implicit val showUserType: cats.Show[UserType] = cats.Show.fromToString
      implicit val jsonEncoderUserType: io.circe.Encoder[UserType] = io.circe.Encoder.encodeString.contramap[UserType]({
        case Guest => "GUEST"
        case Standard => "STANDARD"
        case Service => "SERVICE"
      })
      implicit val jsonDecoderUserType: io.circe.Decoder[UserType] = io.circe.Decoder.decodeString.emap(_ match {
        case "GUEST" =>
          Right(Guest)
        case "STANDARD" =>
          Right(Standard)
        case "SERVICE" =>
          Right(Service)
        case other =>
          Left(s"Invalid value [$other]")
      })
    }
    sealed trait FilterType
    object FilterType {
      case object BroadBand extends FilterType()
      case object Combination extends FilterType()
      case object Engineering extends FilterType()
      case object NarrowBand extends FilterType()
      case object Spectroscopic extends FilterType()
      implicit val eqFilterType: cats.Eq[FilterType] = cats.Eq.fromUniversalEquals
      implicit val showFilterType: cats.Show[FilterType] = cats.Show.fromToString
      implicit val jsonEncoderFilterType: io.circe.Encoder[FilterType] = io.circe.Encoder.encodeString.contramap[FilterType]({
        case BroadBand => "BroadBand"
        case Combination => "Combination"
        case Engineering => "Engineering"
        case NarrowBand => "NarrowBand"
        case Spectroscopic => "Spectroscopic"
      })
      implicit val jsonDecoderFilterType: io.circe.Decoder[FilterType] = io.circe.Decoder.decodeString.emap(_ match {
        case "BroadBand" =>
          Right(BroadBand)
        case "Combination" =>
          Right(Combination)
        case "Engineering" =>
          Right(Engineering)
        case "NarrowBand" =>
          Right(NarrowBand)
        case "Spectroscopic" =>
          Right(Spectroscopic)
        case other =>
          Left(s"Invalid value [$other]")
      })
    }
  }
  object Types {
    import Scalars._
    ignoreUnusedImportScalars()
    import Enums._
    ignoreUnusedImportEnums()
    def ignoreUnusedImportTypes(): Unit = ()
    case class AddDatasetEventInput(val visitId: VisitId, val location: DatasetIdInput, val payload: DatasetEventPayloadInput)
    object AddDatasetEventInput {
      val visitId: monocle.Lens[AddDatasetEventInput, VisitId] = monocle.macros.GenLens[AddDatasetEventInput](_.visitId)
      val location: monocle.Lens[AddDatasetEventInput, DatasetIdInput] = monocle.macros.GenLens[AddDatasetEventInput](_.location)
      val payload: monocle.Lens[AddDatasetEventInput, DatasetEventPayloadInput] = monocle.macros.GenLens[AddDatasetEventInput](_.payload)
      implicit val eqAddDatasetEventInput: cats.Eq[AddDatasetEventInput] = cats.Eq.fromUniversalEquals
      implicit val showAddDatasetEventInput: cats.Show[AddDatasetEventInput] = cats.Show.fromToString
      implicit val jsonEncoderAddDatasetEventInput: io.circe.Encoder[AddDatasetEventInput] = io.circe.generic.semiauto.deriveEncoder[AddDatasetEventInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class AddSequenceEventInput(val visitId: VisitId, val location: SequenceEventLocationInput, val payload: SequenceEventPayloadInput)
    object AddSequenceEventInput {
      val visitId: monocle.Lens[AddSequenceEventInput, VisitId] = monocle.macros.GenLens[AddSequenceEventInput](_.visitId)
      val location: monocle.Lens[AddSequenceEventInput, SequenceEventLocationInput] = monocle.macros.GenLens[AddSequenceEventInput](_.location)
      val payload: monocle.Lens[AddSequenceEventInput, SequenceEventPayloadInput] = monocle.macros.GenLens[AddSequenceEventInput](_.payload)
      implicit val eqAddSequenceEventInput: cats.Eq[AddSequenceEventInput] = cats.Eq.fromUniversalEquals
      implicit val showAddSequenceEventInput: cats.Show[AddSequenceEventInput] = cats.Show.fromToString
      implicit val jsonEncoderAddSequenceEventInput: io.circe.Encoder[AddSequenceEventInput] = io.circe.generic.semiauto.deriveEncoder[AddSequenceEventInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class AddStepEventInput(val visitId: VisitId, val location: StepEventLocationInput, val payload: StepEventPayloadInput)
    object AddStepEventInput {
      val visitId: monocle.Lens[AddStepEventInput, VisitId] = monocle.macros.GenLens[AddStepEventInput](_.visitId)
      val location: monocle.Lens[AddStepEventInput, StepEventLocationInput] = monocle.macros.GenLens[AddStepEventInput](_.location)
      val payload: monocle.Lens[AddStepEventInput, StepEventPayloadInput] = monocle.macros.GenLens[AddStepEventInput](_.payload)
      implicit val eqAddStepEventInput: cats.Eq[AddStepEventInput] = cats.Eq.fromUniversalEquals
      implicit val showAddStepEventInput: cats.Show[AddStepEventInput] = cats.Show.fromToString
      implicit val jsonEncoderAddStepEventInput: io.circe.Encoder[AddStepEventInput] = io.circe.generic.semiauto.deriveEncoder[AddStepEventInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class AirMassRangeInput(val min: clue.data.Input[PosBigDecimal] = clue.data.Ignore, val max: clue.data.Input[PosBigDecimal] = clue.data.Ignore)
    object AirMassRangeInput {
      val min: monocle.Lens[AirMassRangeInput, clue.data.Input[PosBigDecimal]] = monocle.macros.GenLens[AirMassRangeInput](_.min)
      val max: monocle.Lens[AirMassRangeInput, clue.data.Input[PosBigDecimal]] = monocle.macros.GenLens[AirMassRangeInput](_.max)
      implicit val eqAirMassRangeInput: cats.Eq[AirMassRangeInput] = cats.Eq.fromUniversalEquals
      implicit val showAirMassRangeInput: cats.Show[AirMassRangeInput] = cats.Show.fromToString
      implicit val jsonEncoderAirMassRangeInput: io.circe.Encoder[AirMassRangeInput] = io.circe.generic.semiauto.deriveEncoder[AirMassRangeInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class AngleInput(val microarcseconds: clue.data.Input[Long] = clue.data.Ignore, val microseconds: clue.data.Input[BigDecimal] = clue.data.Ignore, val milliarcseconds: clue.data.Input[BigDecimal] = clue.data.Ignore, val milliseconds: clue.data.Input[BigDecimal] = clue.data.Ignore, val arcseconds: clue.data.Input[BigDecimal] = clue.data.Ignore, val seconds: clue.data.Input[BigDecimal] = clue.data.Ignore, val arcminutes: clue.data.Input[BigDecimal] = clue.data.Ignore, val minutes: clue.data.Input[BigDecimal] = clue.data.Ignore, val degrees: clue.data.Input[BigDecimal] = clue.data.Ignore, val hours: clue.data.Input[BigDecimal] = clue.data.Ignore, val dms: clue.data.Input[String] = clue.data.Ignore, val hms: clue.data.Input[String] = clue.data.Ignore)
    object AngleInput {
      val microarcseconds: monocle.Lens[AngleInput, clue.data.Input[Long]] = monocle.macros.GenLens[AngleInput](_.microarcseconds)
      val microseconds: monocle.Lens[AngleInput, clue.data.Input[BigDecimal]] = monocle.macros.GenLens[AngleInput](_.microseconds)
      val milliarcseconds: monocle.Lens[AngleInput, clue.data.Input[BigDecimal]] = monocle.macros.GenLens[AngleInput](_.milliarcseconds)
      val milliseconds: monocle.Lens[AngleInput, clue.data.Input[BigDecimal]] = monocle.macros.GenLens[AngleInput](_.milliseconds)
      val arcseconds: monocle.Lens[AngleInput, clue.data.Input[BigDecimal]] = monocle.macros.GenLens[AngleInput](_.arcseconds)
      val seconds: monocle.Lens[AngleInput, clue.data.Input[BigDecimal]] = monocle.macros.GenLens[AngleInput](_.seconds)
      val arcminutes: monocle.Lens[AngleInput, clue.data.Input[BigDecimal]] = monocle.macros.GenLens[AngleInput](_.arcminutes)
      val minutes: monocle.Lens[AngleInput, clue.data.Input[BigDecimal]] = monocle.macros.GenLens[AngleInput](_.minutes)
      val degrees: monocle.Lens[AngleInput, clue.data.Input[BigDecimal]] = monocle.macros.GenLens[AngleInput](_.degrees)
      val hours: monocle.Lens[AngleInput, clue.data.Input[BigDecimal]] = monocle.macros.GenLens[AngleInput](_.hours)
      val dms: monocle.Lens[AngleInput, clue.data.Input[String]] = monocle.macros.GenLens[AngleInput](_.dms)
      val hms: monocle.Lens[AngleInput, clue.data.Input[String]] = monocle.macros.GenLens[AngleInput](_.hms)
      implicit val eqAngleInput: cats.Eq[AngleInput] = cats.Eq.fromUniversalEquals
      implicit val showAngleInput: cats.Show[AngleInput] = cats.Show.fromToString
      implicit val jsonEncoderAngleInput: io.circe.Encoder[AngleInput] = io.circe.generic.semiauto.deriveEncoder[AngleInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class BandBrightnessIntegratedInput(val band: Band, val value: clue.data.Input[BigDecimal] = clue.data.Ignore, val units: clue.data.Input[BrightnessIntegratedUnits] = clue.data.Ignore, val error: clue.data.Input[BigDecimal] = clue.data.Ignore)
    object BandBrightnessIntegratedInput {
      val band: monocle.Lens[BandBrightnessIntegratedInput, Band] = monocle.macros.GenLens[BandBrightnessIntegratedInput](_.band)
      val value: monocle.Lens[BandBrightnessIntegratedInput, clue.data.Input[BigDecimal]] = monocle.macros.GenLens[BandBrightnessIntegratedInput](_.value)
      val units: monocle.Lens[BandBrightnessIntegratedInput, clue.data.Input[BrightnessIntegratedUnits]] = monocle.macros.GenLens[BandBrightnessIntegratedInput](_.units)
      val error: monocle.Lens[BandBrightnessIntegratedInput, clue.data.Input[BigDecimal]] = monocle.macros.GenLens[BandBrightnessIntegratedInput](_.error)
      implicit val eqBandBrightnessIntegratedInput: cats.Eq[BandBrightnessIntegratedInput] = cats.Eq.fromUniversalEquals
      implicit val showBandBrightnessIntegratedInput: cats.Show[BandBrightnessIntegratedInput] = cats.Show.fromToString
      implicit val jsonEncoderBandBrightnessIntegratedInput: io.circe.Encoder[BandBrightnessIntegratedInput] = io.circe.generic.semiauto.deriveEncoder[BandBrightnessIntegratedInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class BandBrightnessSurfaceInput(val band: Band, val value: clue.data.Input[BigDecimal] = clue.data.Ignore, val units: clue.data.Input[BrightnessSurfaceUnits] = clue.data.Ignore, val error: clue.data.Input[BigDecimal] = clue.data.Ignore)
    object BandBrightnessSurfaceInput {
      val band: monocle.Lens[BandBrightnessSurfaceInput, Band] = monocle.macros.GenLens[BandBrightnessSurfaceInput](_.band)
      val value: monocle.Lens[BandBrightnessSurfaceInput, clue.data.Input[BigDecimal]] = monocle.macros.GenLens[BandBrightnessSurfaceInput](_.value)
      val units: monocle.Lens[BandBrightnessSurfaceInput, clue.data.Input[BrightnessSurfaceUnits]] = monocle.macros.GenLens[BandBrightnessSurfaceInput](_.units)
      val error: monocle.Lens[BandBrightnessSurfaceInput, clue.data.Input[BigDecimal]] = monocle.macros.GenLens[BandBrightnessSurfaceInput](_.error)
      implicit val eqBandBrightnessSurfaceInput: cats.Eq[BandBrightnessSurfaceInput] = cats.Eq.fromUniversalEquals
      implicit val showBandBrightnessSurfaceInput: cats.Show[BandBrightnessSurfaceInput] = cats.Show.fromToString
      implicit val jsonEncoderBandBrightnessSurfaceInput: io.circe.Encoder[BandBrightnessSurfaceInput] = io.circe.generic.semiauto.deriveEncoder[BandBrightnessSurfaceInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class BandNormalizedIntegratedInput(val sed: clue.data.Input[UnnormalizedSedInput] = clue.data.Ignore, val brightnesses: clue.data.Input[List[BandBrightnessIntegratedInput]] = clue.data.Ignore)
    object BandNormalizedIntegratedInput {
      val sed: monocle.Lens[BandNormalizedIntegratedInput, clue.data.Input[UnnormalizedSedInput]] = monocle.macros.GenLens[BandNormalizedIntegratedInput](_.sed)
      val brightnesses: monocle.Lens[BandNormalizedIntegratedInput, clue.data.Input[List[BandBrightnessIntegratedInput]]] = monocle.macros.GenLens[BandNormalizedIntegratedInput](_.brightnesses)
      implicit val eqBandNormalizedIntegratedInput: cats.Eq[BandNormalizedIntegratedInput] = cats.Eq.fromUniversalEquals
      implicit val showBandNormalizedIntegratedInput: cats.Show[BandNormalizedIntegratedInput] = cats.Show.fromToString
      implicit val jsonEncoderBandNormalizedIntegratedInput: io.circe.Encoder[BandNormalizedIntegratedInput] = io.circe.generic.semiauto.deriveEncoder[BandNormalizedIntegratedInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class BandNormalizedSurfaceInput(val sed: clue.data.Input[UnnormalizedSedInput] = clue.data.Ignore, val brightnesses: clue.data.Input[List[BandBrightnessSurfaceInput]] = clue.data.Ignore)
    object BandNormalizedSurfaceInput {
      val sed: monocle.Lens[BandNormalizedSurfaceInput, clue.data.Input[UnnormalizedSedInput]] = monocle.macros.GenLens[BandNormalizedSurfaceInput](_.sed)
      val brightnesses: monocle.Lens[BandNormalizedSurfaceInput, clue.data.Input[List[BandBrightnessSurfaceInput]]] = monocle.macros.GenLens[BandNormalizedSurfaceInput](_.brightnesses)
      implicit val eqBandNormalizedSurfaceInput: cats.Eq[BandNormalizedSurfaceInput] = cats.Eq.fromUniversalEquals
      implicit val showBandNormalizedSurfaceInput: cats.Show[BandNormalizedSurfaceInput] = cats.Show.fromToString
      implicit val jsonEncoderBandNormalizedSurfaceInput: io.circe.Encoder[BandNormalizedSurfaceInput] = io.circe.generic.semiauto.deriveEncoder[BandNormalizedSurfaceInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class CatalogInfoInput(val name: clue.data.Input[CatalogName] = clue.data.Ignore, val id: clue.data.Input[NonEmptyString] = clue.data.Ignore, val objectType: clue.data.Input[NonEmptyString] = clue.data.Ignore)
    object CatalogInfoInput {
      val name: monocle.Lens[CatalogInfoInput, clue.data.Input[CatalogName]] = monocle.macros.GenLens[CatalogInfoInput](_.name)
      val id: monocle.Lens[CatalogInfoInput, clue.data.Input[NonEmptyString]] = monocle.macros.GenLens[CatalogInfoInput](_.id)
      val objectType: monocle.Lens[CatalogInfoInput, clue.data.Input[NonEmptyString]] = monocle.macros.GenLens[CatalogInfoInput](_.objectType)
      implicit val eqCatalogInfoInput: cats.Eq[CatalogInfoInput] = cats.Eq.fromUniversalEquals
      implicit val showCatalogInfoInput: cats.Show[CatalogInfoInput] = cats.Show.fromToString
      implicit val jsonEncoderCatalogInfoInput: io.circe.Encoder[CatalogInfoInput] = io.circe.generic.semiauto.deriveEncoder[CatalogInfoInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class ClassicalInput(val minPercentTime: clue.data.Input[IntPercent] = clue.data.Ignore)
    object ClassicalInput {
      val minPercentTime: monocle.Lens[ClassicalInput, clue.data.Input[IntPercent]] = monocle.macros.GenLens[ClassicalInput](_.minPercentTime)
      implicit val eqClassicalInput: cats.Eq[ClassicalInput] = cats.Eq.fromUniversalEquals
      implicit val showClassicalInput: cats.Show[ClassicalInput] = cats.Show.fromToString
      implicit val jsonEncoderClassicalInput: io.circe.Encoder[ClassicalInput] = io.circe.generic.semiauto.deriveEncoder[ClassicalInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class CloneObservationInput(val observationId: ObservationId, val SET: clue.data.Input[ObservationPropertiesInput] = clue.data.Ignore)
    object CloneObservationInput {
      val observationId: monocle.Lens[CloneObservationInput, ObservationId] = monocle.macros.GenLens[CloneObservationInput](_.observationId)
      val SET: monocle.Lens[CloneObservationInput, clue.data.Input[ObservationPropertiesInput]] = monocle.macros.GenLens[CloneObservationInput](_.SET)
      implicit val eqCloneObservationInput: cats.Eq[CloneObservationInput] = cats.Eq.fromUniversalEquals
      implicit val showCloneObservationInput: cats.Show[CloneObservationInput] = cats.Show.fromToString
      implicit val jsonEncoderCloneObservationInput: io.circe.Encoder[CloneObservationInput] = io.circe.generic.semiauto.deriveEncoder[CloneObservationInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class CloneTargetInput(val targetId: TargetId, val SET: clue.data.Input[TargetPropertiesInput] = clue.data.Ignore, val REPLACE_IN: clue.data.Input[List[ObservationId]] = clue.data.Ignore)
    object CloneTargetInput {
      val targetId: monocle.Lens[CloneTargetInput, TargetId] = monocle.macros.GenLens[CloneTargetInput](_.targetId)
      val SET: monocle.Lens[CloneTargetInput, clue.data.Input[TargetPropertiesInput]] = monocle.macros.GenLens[CloneTargetInput](_.SET)
      val REPLACE_IN: monocle.Lens[CloneTargetInput, clue.data.Input[List[ObservationId]]] = monocle.macros.GenLens[CloneTargetInput](_.REPLACE_IN)
      implicit val eqCloneTargetInput: cats.Eq[CloneTargetInput] = cats.Eq.fromUniversalEquals
      implicit val showCloneTargetInput: cats.Show[CloneTargetInput] = cats.Show.fromToString
      implicit val jsonEncoderCloneTargetInput: io.circe.Encoder[CloneTargetInput] = io.circe.generic.semiauto.deriveEncoder[CloneTargetInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class ConstraintSetInput(val imageQuality: clue.data.Input[ImageQuality] = clue.data.Ignore, val cloudExtinction: clue.data.Input[CloudExtinction] = clue.data.Ignore, val skyBackground: clue.data.Input[SkyBackground] = clue.data.Ignore, val waterVapor: clue.data.Input[WaterVapor] = clue.data.Ignore, val elevationRange: clue.data.Input[ElevationRangeInput] = clue.data.Ignore)
    object ConstraintSetInput {
      val imageQuality: monocle.Lens[ConstraintSetInput, clue.data.Input[ImageQuality]] = monocle.macros.GenLens[ConstraintSetInput](_.imageQuality)
      val cloudExtinction: monocle.Lens[ConstraintSetInput, clue.data.Input[CloudExtinction]] = monocle.macros.GenLens[ConstraintSetInput](_.cloudExtinction)
      val skyBackground: monocle.Lens[ConstraintSetInput, clue.data.Input[SkyBackground]] = monocle.macros.GenLens[ConstraintSetInput](_.skyBackground)
      val waterVapor: monocle.Lens[ConstraintSetInput, clue.data.Input[WaterVapor]] = monocle.macros.GenLens[ConstraintSetInput](_.waterVapor)
      val elevationRange: monocle.Lens[ConstraintSetInput, clue.data.Input[ElevationRangeInput]] = monocle.macros.GenLens[ConstraintSetInput](_.elevationRange)
      implicit val eqConstraintSetInput: cats.Eq[ConstraintSetInput] = cats.Eq.fromUniversalEquals
      implicit val showConstraintSetInput: cats.Show[ConstraintSetInput] = cats.Show.fromToString
      implicit val jsonEncoderConstraintSetInput: io.circe.Encoder[ConstraintSetInput] = io.circe.generic.semiauto.deriveEncoder[ConstraintSetInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class CoordinatesInput(val ra: clue.data.Input[RightAscensionInput] = clue.data.Ignore, val dec: clue.data.Input[DeclinationInput] = clue.data.Ignore)
    object CoordinatesInput {
      val ra: monocle.Lens[CoordinatesInput, clue.data.Input[RightAscensionInput]] = monocle.macros.GenLens[CoordinatesInput](_.ra)
      val dec: monocle.Lens[CoordinatesInput, clue.data.Input[DeclinationInput]] = monocle.macros.GenLens[CoordinatesInput](_.dec)
      implicit val eqCoordinatesInput: cats.Eq[CoordinatesInput] = cats.Eq.fromUniversalEquals
      implicit val showCoordinatesInput: cats.Show[CoordinatesInput] = cats.Show.fromToString
      implicit val jsonEncoderCoordinatesInput: io.circe.Encoder[CoordinatesInput] = io.circe.generic.semiauto.deriveEncoder[CoordinatesInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class CreateObservationInput(val programId: ProgramId, val SET: clue.data.Input[ObservationPropertiesInput] = clue.data.Ignore)
    object CreateObservationInput {
      val programId: monocle.Lens[CreateObservationInput, ProgramId] = monocle.macros.GenLens[CreateObservationInput](_.programId)
      val SET: monocle.Lens[CreateObservationInput, clue.data.Input[ObservationPropertiesInput]] = monocle.macros.GenLens[CreateObservationInput](_.SET)
      implicit val eqCreateObservationInput: cats.Eq[CreateObservationInput] = cats.Eq.fromUniversalEquals
      implicit val showCreateObservationInput: cats.Show[CreateObservationInput] = cats.Show.fromToString
      implicit val jsonEncoderCreateObservationInput: io.circe.Encoder[CreateObservationInput] = io.circe.generic.semiauto.deriveEncoder[CreateObservationInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class CreateProgramInput(val SET: clue.data.Input[ProgramPropertiesInput] = clue.data.Ignore)
    object CreateProgramInput {
      val SET: monocle.Lens[CreateProgramInput, clue.data.Input[ProgramPropertiesInput]] = monocle.macros.GenLens[CreateProgramInput](_.SET)
      implicit val eqCreateProgramInput: cats.Eq[CreateProgramInput] = cats.Eq.fromUniversalEquals
      implicit val showCreateProgramInput: cats.Show[CreateProgramInput] = cats.Show.fromToString
      implicit val jsonEncoderCreateProgramInput: io.circe.Encoder[CreateProgramInput] = io.circe.generic.semiauto.deriveEncoder[CreateProgramInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class CreateTargetInput(val programId: ProgramId, val SET: TargetPropertiesInput)
    object CreateTargetInput {
      val programId: monocle.Lens[CreateTargetInput, ProgramId] = monocle.macros.GenLens[CreateTargetInput](_.programId)
      val SET: monocle.Lens[CreateTargetInput, TargetPropertiesInput] = monocle.macros.GenLens[CreateTargetInput](_.SET)
      implicit val eqCreateTargetInput: cats.Eq[CreateTargetInput] = cats.Eq.fromUniversalEquals
      implicit val showCreateTargetInput: cats.Show[CreateTargetInput] = cats.Show.fromToString
      implicit val jsonEncoderCreateTargetInput: io.circe.Encoder[CreateTargetInput] = io.circe.generic.semiauto.deriveEncoder[CreateTargetInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class DatasetEventPayloadInput(val datasetStage: DatasetStage, val filename: clue.data.Input[DatasetFilename] = clue.data.Ignore)
    object DatasetEventPayloadInput {
      val datasetStage: monocle.Lens[DatasetEventPayloadInput, DatasetStage] = monocle.macros.GenLens[DatasetEventPayloadInput](_.datasetStage)
      val filename: monocle.Lens[DatasetEventPayloadInput, clue.data.Input[DatasetFilename]] = monocle.macros.GenLens[DatasetEventPayloadInput](_.filename)
      implicit val eqDatasetEventPayloadInput: cats.Eq[DatasetEventPayloadInput] = cats.Eq.fromUniversalEquals
      implicit val showDatasetEventPayloadInput: cats.Show[DatasetEventPayloadInput] = cats.Show.fromToString
      implicit val jsonEncoderDatasetEventPayloadInput: io.circe.Encoder[DatasetEventPayloadInput] = io.circe.generic.semiauto.deriveEncoder[DatasetEventPayloadInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class DatasetPropertiesInput(val qaState: clue.data.Input[DatasetQaState] = clue.data.Ignore)
    object DatasetPropertiesInput {
      val qaState: monocle.Lens[DatasetPropertiesInput, clue.data.Input[DatasetQaState]] = monocle.macros.GenLens[DatasetPropertiesInput](_.qaState)
      implicit val eqDatasetPropertiesInput: cats.Eq[DatasetPropertiesInput] = cats.Eq.fromUniversalEquals
      implicit val showDatasetPropertiesInput: cats.Show[DatasetPropertiesInput] = cats.Show.fromToString
      implicit val jsonEncoderDatasetPropertiesInput: io.circe.Encoder[DatasetPropertiesInput] = io.circe.generic.semiauto.deriveEncoder[DatasetPropertiesInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class DeclinationInput(val microarcseconds: clue.data.Input[Long] = clue.data.Ignore, val degrees: clue.data.Input[BigDecimal] = clue.data.Ignore, val dms: clue.data.Input[DmsString] = clue.data.Ignore)
    object DeclinationInput {
      val microarcseconds: monocle.Lens[DeclinationInput, clue.data.Input[Long]] = monocle.macros.GenLens[DeclinationInput](_.microarcseconds)
      val degrees: monocle.Lens[DeclinationInput, clue.data.Input[BigDecimal]] = monocle.macros.GenLens[DeclinationInput](_.degrees)
      val dms: monocle.Lens[DeclinationInput, clue.data.Input[DmsString]] = monocle.macros.GenLens[DeclinationInput](_.dms)
      implicit val eqDeclinationInput: cats.Eq[DeclinationInput] = cats.Eq.fromUniversalEquals
      implicit val showDeclinationInput: cats.Show[DeclinationInput] = cats.Show.fromToString
      implicit val jsonEncoderDeclinationInput: io.circe.Encoder[DeclinationInput] = io.circe.generic.semiauto.deriveEncoder[DeclinationInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class DeleteObservationsInput(val WHERE: clue.data.Input[WhereObservation] = clue.data.Ignore, val LIMIT: clue.data.Input[NonNegInt] = clue.data.Ignore)
    object DeleteObservationsInput {
      val WHERE: monocle.Lens[DeleteObservationsInput, clue.data.Input[WhereObservation]] = monocle.macros.GenLens[DeleteObservationsInput](_.WHERE)
      val LIMIT: monocle.Lens[DeleteObservationsInput, clue.data.Input[NonNegInt]] = monocle.macros.GenLens[DeleteObservationsInput](_.LIMIT)
      implicit val eqDeleteObservationsInput: cats.Eq[DeleteObservationsInput] = cats.Eq.fromUniversalEquals
      implicit val showDeleteObservationsInput: cats.Show[DeleteObservationsInput] = cats.Show.fromToString
      implicit val jsonEncoderDeleteObservationsInput: io.circe.Encoder[DeleteObservationsInput] = io.circe.generic.semiauto.deriveEncoder[DeleteObservationsInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class DeleteTargetsInput(val WHERE: clue.data.Input[WhereTarget] = clue.data.Ignore, val LIMIT: clue.data.Input[NonNegInt] = clue.data.Ignore)
    object DeleteTargetsInput {
      val WHERE: monocle.Lens[DeleteTargetsInput, clue.data.Input[WhereTarget]] = monocle.macros.GenLens[DeleteTargetsInput](_.WHERE)
      val LIMIT: monocle.Lens[DeleteTargetsInput, clue.data.Input[NonNegInt]] = monocle.macros.GenLens[DeleteTargetsInput](_.LIMIT)
      implicit val eqDeleteTargetsInput: cats.Eq[DeleteTargetsInput] = cats.Eq.fromUniversalEquals
      implicit val showDeleteTargetsInput: cats.Show[DeleteTargetsInput] = cats.Show.fromToString
      implicit val jsonEncoderDeleteTargetsInput: io.circe.Encoder[DeleteTargetsInput] = io.circe.generic.semiauto.deriveEncoder[DeleteTargetsInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class DemoScienceInput(val minPercentTime: clue.data.Input[IntPercent] = clue.data.Ignore)
    object DemoScienceInput {
      val minPercentTime: monocle.Lens[DemoScienceInput, clue.data.Input[IntPercent]] = monocle.macros.GenLens[DemoScienceInput](_.minPercentTime)
      implicit val eqDemoScienceInput: cats.Eq[DemoScienceInput] = cats.Eq.fromUniversalEquals
      implicit val showDemoScienceInput: cats.Show[DemoScienceInput] = cats.Show.fromToString
      implicit val jsonEncoderDemoScienceInput: io.circe.Encoder[DemoScienceInput] = io.circe.generic.semiauto.deriveEncoder[DemoScienceInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class DirectorsTimeInput(val minPercentTime: clue.data.Input[IntPercent] = clue.data.Ignore)
    object DirectorsTimeInput {
      val minPercentTime: monocle.Lens[DirectorsTimeInput, clue.data.Input[IntPercent]] = monocle.macros.GenLens[DirectorsTimeInput](_.minPercentTime)
      implicit val eqDirectorsTimeInput: cats.Eq[DirectorsTimeInput] = cats.Eq.fromUniversalEquals
      implicit val showDirectorsTimeInput: cats.Show[DirectorsTimeInput] = cats.Show.fromToString
      implicit val jsonEncoderDirectorsTimeInput: io.circe.Encoder[DirectorsTimeInput] = io.circe.generic.semiauto.deriveEncoder[DirectorsTimeInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class EditAsterismsPatchInput(val ADD: clue.data.Input[List[TargetId]] = clue.data.Ignore, val DELETE: clue.data.Input[List[TargetId]] = clue.data.Ignore)
    object EditAsterismsPatchInput {
      val ADD: monocle.Lens[EditAsterismsPatchInput, clue.data.Input[List[TargetId]]] = monocle.macros.GenLens[EditAsterismsPatchInput](_.ADD)
      val DELETE: monocle.Lens[EditAsterismsPatchInput, clue.data.Input[List[TargetId]]] = monocle.macros.GenLens[EditAsterismsPatchInput](_.DELETE)
      implicit val eqEditAsterismsPatchInput: cats.Eq[EditAsterismsPatchInput] = cats.Eq.fromUniversalEquals
      implicit val showEditAsterismsPatchInput: cats.Show[EditAsterismsPatchInput] = cats.Show.fromToString
      implicit val jsonEncoderEditAsterismsPatchInput: io.circe.Encoder[EditAsterismsPatchInput] = io.circe.generic.semiauto.deriveEncoder[EditAsterismsPatchInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class ElevationRangeInput(val airMass: clue.data.Input[AirMassRangeInput] = clue.data.Ignore, val hourAngle: clue.data.Input[HourAngleRangeInput] = clue.data.Ignore)
    object ElevationRangeInput {
      val airMass: monocle.Lens[ElevationRangeInput, clue.data.Input[AirMassRangeInput]] = monocle.macros.GenLens[ElevationRangeInput](_.airMass)
      val hourAngle: monocle.Lens[ElevationRangeInput, clue.data.Input[HourAngleRangeInput]] = monocle.macros.GenLens[ElevationRangeInput](_.hourAngle)
      implicit val eqElevationRangeInput: cats.Eq[ElevationRangeInput] = cats.Eq.fromUniversalEquals
      implicit val showElevationRangeInput: cats.Show[ElevationRangeInput] = cats.Show.fromToString
      implicit val jsonEncoderElevationRangeInput: io.circe.Encoder[ElevationRangeInput] = io.circe.generic.semiauto.deriveEncoder[ElevationRangeInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class EmissionLineIntegratedInput(val wavelength: WavelengthInput, val lineWidth: clue.data.Input[PosBigDecimal] = clue.data.Ignore, val lineFlux: clue.data.Input[LineFluxIntegratedInput] = clue.data.Ignore)
    object EmissionLineIntegratedInput {
      val wavelength: monocle.Lens[EmissionLineIntegratedInput, WavelengthInput] = monocle.macros.GenLens[EmissionLineIntegratedInput](_.wavelength)
      val lineWidth: monocle.Lens[EmissionLineIntegratedInput, clue.data.Input[PosBigDecimal]] = monocle.macros.GenLens[EmissionLineIntegratedInput](_.lineWidth)
      val lineFlux: monocle.Lens[EmissionLineIntegratedInput, clue.data.Input[LineFluxIntegratedInput]] = monocle.macros.GenLens[EmissionLineIntegratedInput](_.lineFlux)
      implicit val eqEmissionLineIntegratedInput: cats.Eq[EmissionLineIntegratedInput] = cats.Eq.fromUniversalEquals
      implicit val showEmissionLineIntegratedInput: cats.Show[EmissionLineIntegratedInput] = cats.Show.fromToString
      implicit val jsonEncoderEmissionLineIntegratedInput: io.circe.Encoder[EmissionLineIntegratedInput] = io.circe.generic.semiauto.deriveEncoder[EmissionLineIntegratedInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class EmissionLineSurfaceInput(val wavelength: WavelengthInput, val lineWidth: clue.data.Input[PosBigDecimal] = clue.data.Ignore, val lineFlux: clue.data.Input[LineFluxSurfaceInput] = clue.data.Ignore)
    object EmissionLineSurfaceInput {
      val wavelength: monocle.Lens[EmissionLineSurfaceInput, WavelengthInput] = monocle.macros.GenLens[EmissionLineSurfaceInput](_.wavelength)
      val lineWidth: monocle.Lens[EmissionLineSurfaceInput, clue.data.Input[PosBigDecimal]] = monocle.macros.GenLens[EmissionLineSurfaceInput](_.lineWidth)
      val lineFlux: monocle.Lens[EmissionLineSurfaceInput, clue.data.Input[LineFluxSurfaceInput]] = monocle.macros.GenLens[EmissionLineSurfaceInput](_.lineFlux)
      implicit val eqEmissionLineSurfaceInput: cats.Eq[EmissionLineSurfaceInput] = cats.Eq.fromUniversalEquals
      implicit val showEmissionLineSurfaceInput: cats.Show[EmissionLineSurfaceInput] = cats.Show.fromToString
      implicit val jsonEncoderEmissionLineSurfaceInput: io.circe.Encoder[EmissionLineSurfaceInput] = io.circe.generic.semiauto.deriveEncoder[EmissionLineSurfaceInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class EmissionLinesIntegratedInput(val lines: clue.data.Input[List[EmissionLineIntegratedInput]] = clue.data.Ignore, val fluxDensityContinuum: clue.data.Input[FluxDensityContinuumIntegratedInput] = clue.data.Ignore)
    object EmissionLinesIntegratedInput {
      val lines: monocle.Lens[EmissionLinesIntegratedInput, clue.data.Input[List[EmissionLineIntegratedInput]]] = monocle.macros.GenLens[EmissionLinesIntegratedInput](_.lines)
      val fluxDensityContinuum: monocle.Lens[EmissionLinesIntegratedInput, clue.data.Input[FluxDensityContinuumIntegratedInput]] = monocle.macros.GenLens[EmissionLinesIntegratedInput](_.fluxDensityContinuum)
      implicit val eqEmissionLinesIntegratedInput: cats.Eq[EmissionLinesIntegratedInput] = cats.Eq.fromUniversalEquals
      implicit val showEmissionLinesIntegratedInput: cats.Show[EmissionLinesIntegratedInput] = cats.Show.fromToString
      implicit val jsonEncoderEmissionLinesIntegratedInput: io.circe.Encoder[EmissionLinesIntegratedInput] = io.circe.generic.semiauto.deriveEncoder[EmissionLinesIntegratedInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class EmissionLinesSurfaceInput(val lines: clue.data.Input[List[EmissionLineSurfaceInput]] = clue.data.Ignore, val fluxDensityContinuum: clue.data.Input[FluxDensityContinuumSurfaceInput] = clue.data.Ignore)
    object EmissionLinesSurfaceInput {
      val lines: monocle.Lens[EmissionLinesSurfaceInput, clue.data.Input[List[EmissionLineSurfaceInput]]] = monocle.macros.GenLens[EmissionLinesSurfaceInput](_.lines)
      val fluxDensityContinuum: monocle.Lens[EmissionLinesSurfaceInput, clue.data.Input[FluxDensityContinuumSurfaceInput]] = monocle.macros.GenLens[EmissionLinesSurfaceInput](_.fluxDensityContinuum)
      implicit val eqEmissionLinesSurfaceInput: cats.Eq[EmissionLinesSurfaceInput] = cats.Eq.fromUniversalEquals
      implicit val showEmissionLinesSurfaceInput: cats.Show[EmissionLinesSurfaceInput] = cats.Show.fromToString
      implicit val jsonEncoderEmissionLinesSurfaceInput: io.circe.Encoder[EmissionLinesSurfaceInput] = io.circe.generic.semiauto.deriveEncoder[EmissionLinesSurfaceInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class ExchangeInput(val minPercentTime: clue.data.Input[IntPercent] = clue.data.Ignore)
    object ExchangeInput {
      val minPercentTime: monocle.Lens[ExchangeInput, clue.data.Input[IntPercent]] = monocle.macros.GenLens[ExchangeInput](_.minPercentTime)
      implicit val eqExchangeInput: cats.Eq[ExchangeInput] = cats.Eq.fromUniversalEquals
      implicit val showExchangeInput: cats.Show[ExchangeInput] = cats.Show.fromToString
      implicit val jsonEncoderExchangeInput: io.circe.Encoder[ExchangeInput] = io.circe.generic.semiauto.deriveEncoder[ExchangeInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class ExposureTimeModeInput(val signalToNoise: clue.data.Input[SignalToNoiseModeInput] = clue.data.Ignore, val fixedExposure: clue.data.Input[FixedExposureModeInput] = clue.data.Ignore)
    object ExposureTimeModeInput {
      val signalToNoise: monocle.Lens[ExposureTimeModeInput, clue.data.Input[SignalToNoiseModeInput]] = monocle.macros.GenLens[ExposureTimeModeInput](_.signalToNoise)
      val fixedExposure: monocle.Lens[ExposureTimeModeInput, clue.data.Input[FixedExposureModeInput]] = monocle.macros.GenLens[ExposureTimeModeInput](_.fixedExposure)
      implicit val eqExposureTimeModeInput: cats.Eq[ExposureTimeModeInput] = cats.Eq.fromUniversalEquals
      implicit val showExposureTimeModeInput: cats.Show[ExposureTimeModeInput] = cats.Show.fromToString
      implicit val jsonEncoderExposureTimeModeInput: io.circe.Encoder[ExposureTimeModeInput] = io.circe.generic.semiauto.deriveEncoder[ExposureTimeModeInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class FastTurnaroundInput(val minPercentTime: clue.data.Input[IntPercent] = clue.data.Ignore)
    object FastTurnaroundInput {
      val minPercentTime: monocle.Lens[FastTurnaroundInput, clue.data.Input[IntPercent]] = monocle.macros.GenLens[FastTurnaroundInput](_.minPercentTime)
      implicit val eqFastTurnaroundInput: cats.Eq[FastTurnaroundInput] = cats.Eq.fromUniversalEquals
      implicit val showFastTurnaroundInput: cats.Show[FastTurnaroundInput] = cats.Show.fromToString
      implicit val jsonEncoderFastTurnaroundInput: io.circe.Encoder[FastTurnaroundInput] = io.circe.generic.semiauto.deriveEncoder[FastTurnaroundInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class FixedExposureModeInput(val count: NonNegInt, val time: TimeSpanInput)
    object FixedExposureModeInput {
      val count: monocle.Lens[FixedExposureModeInput, NonNegInt] = monocle.macros.GenLens[FixedExposureModeInput](_.count)
      val time: monocle.Lens[FixedExposureModeInput, TimeSpanInput] = monocle.macros.GenLens[FixedExposureModeInput](_.time)
      implicit val eqFixedExposureModeInput: cats.Eq[FixedExposureModeInput] = cats.Eq.fromUniversalEquals
      implicit val showFixedExposureModeInput: cats.Show[FixedExposureModeInput] = cats.Show.fromToString
      implicit val jsonEncoderFixedExposureModeInput: io.circe.Encoder[FixedExposureModeInput] = io.circe.generic.semiauto.deriveEncoder[FixedExposureModeInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class FluxDensity(val wavelength: WavelengthInput, val density: PosBigDecimal)
    object FluxDensity {
      val wavelength: monocle.Lens[FluxDensity, WavelengthInput] = monocle.macros.GenLens[FluxDensity](_.wavelength)
      val density: monocle.Lens[FluxDensity, PosBigDecimal] = monocle.macros.GenLens[FluxDensity](_.density)
      implicit val eqFluxDensity: cats.Eq[FluxDensity] = cats.Eq.fromUniversalEquals
      implicit val showFluxDensity: cats.Show[FluxDensity] = cats.Show.fromToString
      implicit val jsonEncoderFluxDensity: io.circe.Encoder[FluxDensity] = io.circe.generic.semiauto.deriveEncoder[FluxDensity].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class FluxDensityContinuumIntegratedInput(val value: PosBigDecimal, val units: FluxDensityContinuumIntegratedUnits, val error: clue.data.Input[PosBigDecimal] = clue.data.Ignore)
    object FluxDensityContinuumIntegratedInput {
      val value: monocle.Lens[FluxDensityContinuumIntegratedInput, PosBigDecimal] = monocle.macros.GenLens[FluxDensityContinuumIntegratedInput](_.value)
      val units: monocle.Lens[FluxDensityContinuumIntegratedInput, FluxDensityContinuumIntegratedUnits] = monocle.macros.GenLens[FluxDensityContinuumIntegratedInput](_.units)
      val error: monocle.Lens[FluxDensityContinuumIntegratedInput, clue.data.Input[PosBigDecimal]] = monocle.macros.GenLens[FluxDensityContinuumIntegratedInput](_.error)
      implicit val eqFluxDensityContinuumIntegratedInput: cats.Eq[FluxDensityContinuumIntegratedInput] = cats.Eq.fromUniversalEquals
      implicit val showFluxDensityContinuumIntegratedInput: cats.Show[FluxDensityContinuumIntegratedInput] = cats.Show.fromToString
      implicit val jsonEncoderFluxDensityContinuumIntegratedInput: io.circe.Encoder[FluxDensityContinuumIntegratedInput] = io.circe.generic.semiauto.deriveEncoder[FluxDensityContinuumIntegratedInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class FluxDensityContinuumSurfaceInput(val value: PosBigDecimal, val units: FluxDensityContinuumSurfaceUnits, val error: clue.data.Input[PosBigDecimal] = clue.data.Ignore)
    object FluxDensityContinuumSurfaceInput {
      val value: monocle.Lens[FluxDensityContinuumSurfaceInput, PosBigDecimal] = monocle.macros.GenLens[FluxDensityContinuumSurfaceInput](_.value)
      val units: monocle.Lens[FluxDensityContinuumSurfaceInput, FluxDensityContinuumSurfaceUnits] = monocle.macros.GenLens[FluxDensityContinuumSurfaceInput](_.units)
      val error: monocle.Lens[FluxDensityContinuumSurfaceInput, clue.data.Input[PosBigDecimal]] = monocle.macros.GenLens[FluxDensityContinuumSurfaceInput](_.error)
      implicit val eqFluxDensityContinuumSurfaceInput: cats.Eq[FluxDensityContinuumSurfaceInput] = cats.Eq.fromUniversalEquals
      implicit val showFluxDensityContinuumSurfaceInput: cats.Show[FluxDensityContinuumSurfaceInput] = cats.Show.fromToString
      implicit val jsonEncoderFluxDensityContinuumSurfaceInput: io.circe.Encoder[FluxDensityContinuumSurfaceInput] = io.circe.generic.semiauto.deriveEncoder[FluxDensityContinuumSurfaceInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class GaussianInput(val fwhm: clue.data.Input[AngleInput] = clue.data.Ignore, val spectralDefinition: clue.data.Input[SpectralDefinitionIntegratedInput] = clue.data.Ignore)
    object GaussianInput {
      val fwhm: monocle.Lens[GaussianInput, clue.data.Input[AngleInput]] = monocle.macros.GenLens[GaussianInput](_.fwhm)
      val spectralDefinition: monocle.Lens[GaussianInput, clue.data.Input[SpectralDefinitionIntegratedInput]] = monocle.macros.GenLens[GaussianInput](_.spectralDefinition)
      implicit val eqGaussianInput: cats.Eq[GaussianInput] = cats.Eq.fromUniversalEquals
      implicit val showGaussianInput: cats.Show[GaussianInput] = cats.Show.fromToString
      implicit val jsonEncoderGaussianInput: io.circe.Encoder[GaussianInput] = io.circe.generic.semiauto.deriveEncoder[GaussianInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class GcalConfigurationInput(val continuum: clue.data.Input[GcalContinuum] = clue.data.Ignore, val arcs: List[GcalArc], val diffuser: GcalDiffuser, val shutter: GcalShutter)
    object GcalConfigurationInput {
      val continuum: monocle.Lens[GcalConfigurationInput, clue.data.Input[GcalContinuum]] = monocle.macros.GenLens[GcalConfigurationInput](_.continuum)
      val arcs: monocle.Lens[GcalConfigurationInput, List[GcalArc]] = monocle.macros.GenLens[GcalConfigurationInput](_.arcs)
      val diffuser: monocle.Lens[GcalConfigurationInput, GcalDiffuser] = monocle.macros.GenLens[GcalConfigurationInput](_.diffuser)
      val shutter: monocle.Lens[GcalConfigurationInput, GcalShutter] = monocle.macros.GenLens[GcalConfigurationInput](_.shutter)
      implicit val eqGcalConfigurationInput: cats.Eq[GcalConfigurationInput] = cats.Eq.fromUniversalEquals
      implicit val showGcalConfigurationInput: cats.Show[GcalConfigurationInput] = cats.Show.fromToString
      implicit val jsonEncoderGcalConfigurationInput: io.circe.Encoder[GcalConfigurationInput] = io.circe.generic.semiauto.deriveEncoder[GcalConfigurationInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class GmosCcdReadoutInput(val xBin: GmosXBinning, val yBin: GmosYBinning, val ampCount: GmosAmpCount, val ampGain: GmosAmpGain, val ampRead: GmosAmpReadMode)
    object GmosCcdReadoutInput {
      val xBin: monocle.Lens[GmosCcdReadoutInput, GmosXBinning] = monocle.macros.GenLens[GmosCcdReadoutInput](_.xBin)
      val yBin: monocle.Lens[GmosCcdReadoutInput, GmosYBinning] = monocle.macros.GenLens[GmosCcdReadoutInput](_.yBin)
      val ampCount: monocle.Lens[GmosCcdReadoutInput, GmosAmpCount] = monocle.macros.GenLens[GmosCcdReadoutInput](_.ampCount)
      val ampGain: monocle.Lens[GmosCcdReadoutInput, GmosAmpGain] = monocle.macros.GenLens[GmosCcdReadoutInput](_.ampGain)
      val ampRead: monocle.Lens[GmosCcdReadoutInput, GmosAmpReadMode] = monocle.macros.GenLens[GmosCcdReadoutInput](_.ampRead)
      implicit val eqGmosCcdReadoutInput: cats.Eq[GmosCcdReadoutInput] = cats.Eq.fromUniversalEquals
      implicit val showGmosCcdReadoutInput: cats.Show[GmosCcdReadoutInput] = cats.Show.fromToString
      implicit val jsonEncoderGmosCcdReadoutInput: io.circe.Encoder[GmosCcdReadoutInput] = io.circe.generic.semiauto.deriveEncoder[GmosCcdReadoutInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class GmosCustomMaskInput(val filename: String, val slitWidth: GmosCustomSlitWidth)
    object GmosCustomMaskInput {
      val filename: monocle.Lens[GmosCustomMaskInput, String] = monocle.macros.GenLens[GmosCustomMaskInput](_.filename)
      val slitWidth: monocle.Lens[GmosCustomMaskInput, GmosCustomSlitWidth] = monocle.macros.GenLens[GmosCustomMaskInput](_.slitWidth)
      implicit val eqGmosCustomMaskInput: cats.Eq[GmosCustomMaskInput] = cats.Eq.fromUniversalEquals
      implicit val showGmosCustomMaskInput: cats.Show[GmosCustomMaskInput] = cats.Show.fromToString
      implicit val jsonEncoderGmosCustomMaskInput: io.circe.Encoder[GmosCustomMaskInput] = io.circe.generic.semiauto.deriveEncoder[GmosCustomMaskInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class GmosNodAndShuffleInput(val posA: OffsetInput, val posB: OffsetInput, val eOffset: GmosEOffsetting, val shuffleOffset: Int, val shuffleCycles: Int)
    object GmosNodAndShuffleInput {
      val posA: monocle.Lens[GmosNodAndShuffleInput, OffsetInput] = monocle.macros.GenLens[GmosNodAndShuffleInput](_.posA)
      val posB: monocle.Lens[GmosNodAndShuffleInput, OffsetInput] = monocle.macros.GenLens[GmosNodAndShuffleInput](_.posB)
      val eOffset: monocle.Lens[GmosNodAndShuffleInput, GmosEOffsetting] = monocle.macros.GenLens[GmosNodAndShuffleInput](_.eOffset)
      val shuffleOffset: monocle.Lens[GmosNodAndShuffleInput, Int] = monocle.macros.GenLens[GmosNodAndShuffleInput](_.shuffleOffset)
      val shuffleCycles: monocle.Lens[GmosNodAndShuffleInput, Int] = monocle.macros.GenLens[GmosNodAndShuffleInput](_.shuffleCycles)
      implicit val eqGmosNodAndShuffleInput: cats.Eq[GmosNodAndShuffleInput] = cats.Eq.fromUniversalEquals
      implicit val showGmosNodAndShuffleInput: cats.Show[GmosNodAndShuffleInput] = cats.Show.fromToString
      implicit val jsonEncoderGmosNodAndShuffleInput: io.circe.Encoder[GmosNodAndShuffleInput] = io.circe.generic.semiauto.deriveEncoder[GmosNodAndShuffleInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class GmosNorthBiasInput(val config: GmosNorthDynamicInput)
    object GmosNorthBiasInput {
      val config: monocle.Lens[GmosNorthBiasInput, GmosNorthDynamicInput] = monocle.macros.GenLens[GmosNorthBiasInput](_.config)
      implicit val eqGmosNorthBiasInput: cats.Eq[GmosNorthBiasInput] = cats.Eq.fromUniversalEquals
      implicit val showGmosNorthBiasInput: cats.Show[GmosNorthBiasInput] = cats.Show.fromToString
      implicit val jsonEncoderGmosNorthBiasInput: io.circe.Encoder[GmosNorthBiasInput] = io.circe.generic.semiauto.deriveEncoder[GmosNorthBiasInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class GmosNorthDarkInput(val config: GmosNorthDynamicInput)
    object GmosNorthDarkInput {
      val config: monocle.Lens[GmosNorthDarkInput, GmosNorthDynamicInput] = monocle.macros.GenLens[GmosNorthDarkInput](_.config)
      implicit val eqGmosNorthDarkInput: cats.Eq[GmosNorthDarkInput] = cats.Eq.fromUniversalEquals
      implicit val showGmosNorthDarkInput: cats.Show[GmosNorthDarkInput] = cats.Show.fromToString
      implicit val jsonEncoderGmosNorthDarkInput: io.circe.Encoder[GmosNorthDarkInput] = io.circe.generic.semiauto.deriveEncoder[GmosNorthDarkInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class GmosNorthDynamicInput(val exposure: TimeSpanInput, val readout: GmosCcdReadoutInput, val dtax: GmosDtax, val roi: GmosRoi, val gratingConfig: clue.data.Input[GmosNorthGratingConfigInput] = clue.data.Ignore, val filter: clue.data.Input[GmosNorthFilter] = clue.data.Ignore, val fpu: clue.data.Input[GmosNorthFpuInput] = clue.data.Ignore)
    object GmosNorthDynamicInput {
      val exposure: monocle.Lens[GmosNorthDynamicInput, TimeSpanInput] = monocle.macros.GenLens[GmosNorthDynamicInput](_.exposure)
      val readout: monocle.Lens[GmosNorthDynamicInput, GmosCcdReadoutInput] = monocle.macros.GenLens[GmosNorthDynamicInput](_.readout)
      val dtax: monocle.Lens[GmosNorthDynamicInput, GmosDtax] = monocle.macros.GenLens[GmosNorthDynamicInput](_.dtax)
      val roi: monocle.Lens[GmosNorthDynamicInput, GmosRoi] = monocle.macros.GenLens[GmosNorthDynamicInput](_.roi)
      val gratingConfig: monocle.Lens[GmosNorthDynamicInput, clue.data.Input[GmosNorthGratingConfigInput]] = monocle.macros.GenLens[GmosNorthDynamicInput](_.gratingConfig)
      val filter: monocle.Lens[GmosNorthDynamicInput, clue.data.Input[GmosNorthFilter]] = monocle.macros.GenLens[GmosNorthDynamicInput](_.filter)
      val fpu: monocle.Lens[GmosNorthDynamicInput, clue.data.Input[GmosNorthFpuInput]] = monocle.macros.GenLens[GmosNorthDynamicInput](_.fpu)
      implicit val eqGmosNorthDynamicInput: cats.Eq[GmosNorthDynamicInput] = cats.Eq.fromUniversalEquals
      implicit val showGmosNorthDynamicInput: cats.Show[GmosNorthDynamicInput] = cats.Show.fromToString
      implicit val jsonEncoderGmosNorthDynamicInput: io.circe.Encoder[GmosNorthDynamicInput] = io.circe.generic.semiauto.deriveEncoder[GmosNorthDynamicInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class GmosNorthFpuInput(val customMask: clue.data.Input[GmosCustomMaskInput] = clue.data.Ignore, val builtin: clue.data.Input[GmosNorthBuiltinFpu] = clue.data.Ignore)
    object GmosNorthFpuInput {
      val customMask: monocle.Lens[GmosNorthFpuInput, clue.data.Input[GmosCustomMaskInput]] = monocle.macros.GenLens[GmosNorthFpuInput](_.customMask)
      val builtin: monocle.Lens[GmosNorthFpuInput, clue.data.Input[GmosNorthBuiltinFpu]] = monocle.macros.GenLens[GmosNorthFpuInput](_.builtin)
      implicit val eqGmosNorthFpuInput: cats.Eq[GmosNorthFpuInput] = cats.Eq.fromUniversalEquals
      implicit val showGmosNorthFpuInput: cats.Show[GmosNorthFpuInput] = cats.Show.fromToString
      implicit val jsonEncoderGmosNorthFpuInput: io.circe.Encoder[GmosNorthFpuInput] = io.circe.generic.semiauto.deriveEncoder[GmosNorthFpuInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class GmosNorthGcalInput(val config: GmosNorthDynamicInput, val gcalConfig: GcalConfigurationInput)
    object GmosNorthGcalInput {
      val config: monocle.Lens[GmosNorthGcalInput, GmosNorthDynamicInput] = monocle.macros.GenLens[GmosNorthGcalInput](_.config)
      val gcalConfig: monocle.Lens[GmosNorthGcalInput, GcalConfigurationInput] = monocle.macros.GenLens[GmosNorthGcalInput](_.gcalConfig)
      implicit val eqGmosNorthGcalInput: cats.Eq[GmosNorthGcalInput] = cats.Eq.fromUniversalEquals
      implicit val showGmosNorthGcalInput: cats.Show[GmosNorthGcalInput] = cats.Show.fromToString
      implicit val jsonEncoderGmosNorthGcalInput: io.circe.Encoder[GmosNorthGcalInput] = io.circe.generic.semiauto.deriveEncoder[GmosNorthGcalInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class GmosNorthGratingConfigInput(val grating: GmosNorthGrating, val order: GmosGratingOrder, val wavelength: WavelengthInput)
    object GmosNorthGratingConfigInput {
      val grating: monocle.Lens[GmosNorthGratingConfigInput, GmosNorthGrating] = monocle.macros.GenLens[GmosNorthGratingConfigInput](_.grating)
      val order: monocle.Lens[GmosNorthGratingConfigInput, GmosGratingOrder] = monocle.macros.GenLens[GmosNorthGratingConfigInput](_.order)
      val wavelength: monocle.Lens[GmosNorthGratingConfigInput, WavelengthInput] = monocle.macros.GenLens[GmosNorthGratingConfigInput](_.wavelength)
      implicit val eqGmosNorthGratingConfigInput: cats.Eq[GmosNorthGratingConfigInput] = cats.Eq.fromUniversalEquals
      implicit val showGmosNorthGratingConfigInput: cats.Show[GmosNorthGratingConfigInput] = cats.Show.fromToString
      implicit val jsonEncoderGmosNorthGratingConfigInput: io.circe.Encoder[GmosNorthGratingConfigInput] = io.circe.generic.semiauto.deriveEncoder[GmosNorthGratingConfigInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class GmosNorthLongSlitInput(val grating: clue.data.Input[GmosNorthGrating] = clue.data.Ignore, val filter: clue.data.Input[GmosNorthFilter] = clue.data.Ignore, val fpu: clue.data.Input[GmosNorthBuiltinFpu] = clue.data.Ignore, val centralWavelength: clue.data.Input[WavelengthInput] = clue.data.Ignore, val explicitXBin: clue.data.Input[GmosXBinning] = clue.data.Ignore, val explicitYBin: clue.data.Input[GmosYBinning] = clue.data.Ignore, val explicitAmpReadMode: clue.data.Input[GmosAmpReadMode] = clue.data.Ignore, val explicitAmpGain: clue.data.Input[GmosAmpGain] = clue.data.Ignore, val explicitRoi: clue.data.Input[GmosRoi] = clue.data.Ignore, val explicitWavelengthDithers: clue.data.Input[List[WavelengthDitherInput]] = clue.data.Ignore, val explicitSpatialOffsets: clue.data.Input[List[OffsetComponentInput]] = clue.data.Ignore)
    object GmosNorthLongSlitInput {
      val grating: monocle.Lens[GmosNorthLongSlitInput, clue.data.Input[GmosNorthGrating]] = monocle.macros.GenLens[GmosNorthLongSlitInput](_.grating)
      val filter: monocle.Lens[GmosNorthLongSlitInput, clue.data.Input[GmosNorthFilter]] = monocle.macros.GenLens[GmosNorthLongSlitInput](_.filter)
      val fpu: monocle.Lens[GmosNorthLongSlitInput, clue.data.Input[GmosNorthBuiltinFpu]] = monocle.macros.GenLens[GmosNorthLongSlitInput](_.fpu)
      val centralWavelength: monocle.Lens[GmosNorthLongSlitInput, clue.data.Input[WavelengthInput]] = monocle.macros.GenLens[GmosNorthLongSlitInput](_.centralWavelength)
      val explicitXBin: monocle.Lens[GmosNorthLongSlitInput, clue.data.Input[GmosXBinning]] = monocle.macros.GenLens[GmosNorthLongSlitInput](_.explicitXBin)
      val explicitYBin: monocle.Lens[GmosNorthLongSlitInput, clue.data.Input[GmosYBinning]] = monocle.macros.GenLens[GmosNorthLongSlitInput](_.explicitYBin)
      val explicitAmpReadMode: monocle.Lens[GmosNorthLongSlitInput, clue.data.Input[GmosAmpReadMode]] = monocle.macros.GenLens[GmosNorthLongSlitInput](_.explicitAmpReadMode)
      val explicitAmpGain: monocle.Lens[GmosNorthLongSlitInput, clue.data.Input[GmosAmpGain]] = monocle.macros.GenLens[GmosNorthLongSlitInput](_.explicitAmpGain)
      val explicitRoi: monocle.Lens[GmosNorthLongSlitInput, clue.data.Input[GmosRoi]] = monocle.macros.GenLens[GmosNorthLongSlitInput](_.explicitRoi)
      val explicitWavelengthDithers: monocle.Lens[GmosNorthLongSlitInput, clue.data.Input[List[WavelengthDitherInput]]] = monocle.macros.GenLens[GmosNorthLongSlitInput](_.explicitWavelengthDithers)
      val explicitSpatialOffsets: monocle.Lens[GmosNorthLongSlitInput, clue.data.Input[List[OffsetComponentInput]]] = monocle.macros.GenLens[GmosNorthLongSlitInput](_.explicitSpatialOffsets)
      implicit val eqGmosNorthLongSlitInput: cats.Eq[GmosNorthLongSlitInput] = cats.Eq.fromUniversalEquals
      implicit val showGmosNorthLongSlitInput: cats.Show[GmosNorthLongSlitInput] = cats.Show.fromToString
      implicit val jsonEncoderGmosNorthLongSlitInput: io.circe.Encoder[GmosNorthLongSlitInput] = io.circe.generic.semiauto.deriveEncoder[GmosNorthLongSlitInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class GmosNorthScienceInput(val config: GmosNorthDynamicInput, val offset: OffsetInput)
    object GmosNorthScienceInput {
      val config: monocle.Lens[GmosNorthScienceInput, GmosNorthDynamicInput] = monocle.macros.GenLens[GmosNorthScienceInput](_.config)
      val offset: monocle.Lens[GmosNorthScienceInput, OffsetInput] = monocle.macros.GenLens[GmosNorthScienceInput](_.offset)
      implicit val eqGmosNorthScienceInput: cats.Eq[GmosNorthScienceInput] = cats.Eq.fromUniversalEquals
      implicit val showGmosNorthScienceInput: cats.Show[GmosNorthScienceInput] = cats.Show.fromToString
      implicit val jsonEncoderGmosNorthScienceInput: io.circe.Encoder[GmosNorthScienceInput] = io.circe.generic.semiauto.deriveEncoder[GmosNorthScienceInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class GmosNorthStaticInput(val detector: GmosNorthDetector, val mosPreImaging: MosPreImaging, val nodAndShuffle: clue.data.Input[GmosNodAndShuffleInput] = clue.data.Ignore, val stageMode: GmosNorthStageMode)
    object GmosNorthStaticInput {
      val detector: monocle.Lens[GmosNorthStaticInput, GmosNorthDetector] = monocle.macros.GenLens[GmosNorthStaticInput](_.detector)
      val mosPreImaging: monocle.Lens[GmosNorthStaticInput, MosPreImaging] = monocle.macros.GenLens[GmosNorthStaticInput](_.mosPreImaging)
      val nodAndShuffle: monocle.Lens[GmosNorthStaticInput, clue.data.Input[GmosNodAndShuffleInput]] = monocle.macros.GenLens[GmosNorthStaticInput](_.nodAndShuffle)
      val stageMode: monocle.Lens[GmosNorthStaticInput, GmosNorthStageMode] = monocle.macros.GenLens[GmosNorthStaticInput](_.stageMode)
      implicit val eqGmosNorthStaticInput: cats.Eq[GmosNorthStaticInput] = cats.Eq.fromUniversalEquals
      implicit val showGmosNorthStaticInput: cats.Show[GmosNorthStaticInput] = cats.Show.fromToString
      implicit val jsonEncoderGmosNorthStaticInput: io.circe.Encoder[GmosNorthStaticInput] = io.circe.generic.semiauto.deriveEncoder[GmosNorthStaticInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class GmosNorthStepInput(val bias: clue.data.Input[GmosNorthBiasInput] = clue.data.Ignore, val dark: clue.data.Input[GmosNorthDarkInput] = clue.data.Ignore, val gcal: clue.data.Input[GmosNorthGcalInput] = clue.data.Ignore, val science: clue.data.Input[GmosNorthScienceInput] = clue.data.Ignore)
    object GmosNorthStepInput {
      val bias: monocle.Lens[GmosNorthStepInput, clue.data.Input[GmosNorthBiasInput]] = monocle.macros.GenLens[GmosNorthStepInput](_.bias)
      val dark: monocle.Lens[GmosNorthStepInput, clue.data.Input[GmosNorthDarkInput]] = monocle.macros.GenLens[GmosNorthStepInput](_.dark)
      val gcal: monocle.Lens[GmosNorthStepInput, clue.data.Input[GmosNorthGcalInput]] = monocle.macros.GenLens[GmosNorthStepInput](_.gcal)
      val science: monocle.Lens[GmosNorthStepInput, clue.data.Input[GmosNorthScienceInput]] = monocle.macros.GenLens[GmosNorthStepInput](_.science)
      implicit val eqGmosNorthStepInput: cats.Eq[GmosNorthStepInput] = cats.Eq.fromUniversalEquals
      implicit val showGmosNorthStepInput: cats.Show[GmosNorthStepInput] = cats.Show.fromToString
      implicit val jsonEncoderGmosNorthStepInput: io.circe.Encoder[GmosNorthStepInput] = io.circe.generic.semiauto.deriveEncoder[GmosNorthStepInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class GmosSouthBiasInput(val config: GmosSouthDynamicInput)
    object GmosSouthBiasInput {
      val config: monocle.Lens[GmosSouthBiasInput, GmosSouthDynamicInput] = monocle.macros.GenLens[GmosSouthBiasInput](_.config)
      implicit val eqGmosSouthBiasInput: cats.Eq[GmosSouthBiasInput] = cats.Eq.fromUniversalEquals
      implicit val showGmosSouthBiasInput: cats.Show[GmosSouthBiasInput] = cats.Show.fromToString
      implicit val jsonEncoderGmosSouthBiasInput: io.circe.Encoder[GmosSouthBiasInput] = io.circe.generic.semiauto.deriveEncoder[GmosSouthBiasInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class GmosSouthDarkInput(val config: GmosSouthDynamicInput)
    object GmosSouthDarkInput {
      val config: monocle.Lens[GmosSouthDarkInput, GmosSouthDynamicInput] = monocle.macros.GenLens[GmosSouthDarkInput](_.config)
      implicit val eqGmosSouthDarkInput: cats.Eq[GmosSouthDarkInput] = cats.Eq.fromUniversalEquals
      implicit val showGmosSouthDarkInput: cats.Show[GmosSouthDarkInput] = cats.Show.fromToString
      implicit val jsonEncoderGmosSouthDarkInput: io.circe.Encoder[GmosSouthDarkInput] = io.circe.generic.semiauto.deriveEncoder[GmosSouthDarkInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class GmosSouthDynamicInput(val exposure: TimeSpanInput, val readout: GmosCcdReadoutInput, val dtax: GmosDtax, val roi: GmosRoi, val gratingConfig: clue.data.Input[GmosSouthGratingConfigInput] = clue.data.Ignore, val filter: clue.data.Input[GmosSouthFilter] = clue.data.Ignore, val fpu: clue.data.Input[GmosSouthFpuInput] = clue.data.Ignore)
    object GmosSouthDynamicInput {
      val exposure: monocle.Lens[GmosSouthDynamicInput, TimeSpanInput] = monocle.macros.GenLens[GmosSouthDynamicInput](_.exposure)
      val readout: monocle.Lens[GmosSouthDynamicInput, GmosCcdReadoutInput] = monocle.macros.GenLens[GmosSouthDynamicInput](_.readout)
      val dtax: monocle.Lens[GmosSouthDynamicInput, GmosDtax] = monocle.macros.GenLens[GmosSouthDynamicInput](_.dtax)
      val roi: monocle.Lens[GmosSouthDynamicInput, GmosRoi] = monocle.macros.GenLens[GmosSouthDynamicInput](_.roi)
      val gratingConfig: monocle.Lens[GmosSouthDynamicInput, clue.data.Input[GmosSouthGratingConfigInput]] = monocle.macros.GenLens[GmosSouthDynamicInput](_.gratingConfig)
      val filter: monocle.Lens[GmosSouthDynamicInput, clue.data.Input[GmosSouthFilter]] = monocle.macros.GenLens[GmosSouthDynamicInput](_.filter)
      val fpu: monocle.Lens[GmosSouthDynamicInput, clue.data.Input[GmosSouthFpuInput]] = monocle.macros.GenLens[GmosSouthDynamicInput](_.fpu)
      implicit val eqGmosSouthDynamicInput: cats.Eq[GmosSouthDynamicInput] = cats.Eq.fromUniversalEquals
      implicit val showGmosSouthDynamicInput: cats.Show[GmosSouthDynamicInput] = cats.Show.fromToString
      implicit val jsonEncoderGmosSouthDynamicInput: io.circe.Encoder[GmosSouthDynamicInput] = io.circe.generic.semiauto.deriveEncoder[GmosSouthDynamicInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class GmosSouthFpuInput(val customMask: clue.data.Input[GmosCustomMaskInput] = clue.data.Ignore, val builtin: clue.data.Input[GmosSouthBuiltinFpu] = clue.data.Ignore)
    object GmosSouthFpuInput {
      val customMask: monocle.Lens[GmosSouthFpuInput, clue.data.Input[GmosCustomMaskInput]] = monocle.macros.GenLens[GmosSouthFpuInput](_.customMask)
      val builtin: monocle.Lens[GmosSouthFpuInput, clue.data.Input[GmosSouthBuiltinFpu]] = monocle.macros.GenLens[GmosSouthFpuInput](_.builtin)
      implicit val eqGmosSouthFpuInput: cats.Eq[GmosSouthFpuInput] = cats.Eq.fromUniversalEquals
      implicit val showGmosSouthFpuInput: cats.Show[GmosSouthFpuInput] = cats.Show.fromToString
      implicit val jsonEncoderGmosSouthFpuInput: io.circe.Encoder[GmosSouthFpuInput] = io.circe.generic.semiauto.deriveEncoder[GmosSouthFpuInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class GmosSouthGcalInput(val config: GmosSouthDynamicInput, val gcalConfig: GcalConfigurationInput)
    object GmosSouthGcalInput {
      val config: monocle.Lens[GmosSouthGcalInput, GmosSouthDynamicInput] = monocle.macros.GenLens[GmosSouthGcalInput](_.config)
      val gcalConfig: monocle.Lens[GmosSouthGcalInput, GcalConfigurationInput] = monocle.macros.GenLens[GmosSouthGcalInput](_.gcalConfig)
      implicit val eqGmosSouthGcalInput: cats.Eq[GmosSouthGcalInput] = cats.Eq.fromUniversalEquals
      implicit val showGmosSouthGcalInput: cats.Show[GmosSouthGcalInput] = cats.Show.fromToString
      implicit val jsonEncoderGmosSouthGcalInput: io.circe.Encoder[GmosSouthGcalInput] = io.circe.generic.semiauto.deriveEncoder[GmosSouthGcalInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class GmosSouthGratingConfigInput(val grating: GmosSouthGrating, val order: GmosGratingOrder, val wavelength: WavelengthInput)
    object GmosSouthGratingConfigInput {
      val grating: monocle.Lens[GmosSouthGratingConfigInput, GmosSouthGrating] = monocle.macros.GenLens[GmosSouthGratingConfigInput](_.grating)
      val order: monocle.Lens[GmosSouthGratingConfigInput, GmosGratingOrder] = monocle.macros.GenLens[GmosSouthGratingConfigInput](_.order)
      val wavelength: monocle.Lens[GmosSouthGratingConfigInput, WavelengthInput] = monocle.macros.GenLens[GmosSouthGratingConfigInput](_.wavelength)
      implicit val eqGmosSouthGratingConfigInput: cats.Eq[GmosSouthGratingConfigInput] = cats.Eq.fromUniversalEquals
      implicit val showGmosSouthGratingConfigInput: cats.Show[GmosSouthGratingConfigInput] = cats.Show.fromToString
      implicit val jsonEncoderGmosSouthGratingConfigInput: io.circe.Encoder[GmosSouthGratingConfigInput] = io.circe.generic.semiauto.deriveEncoder[GmosSouthGratingConfigInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class GmosSouthLongSlitInput(val grating: clue.data.Input[GmosSouthGrating] = clue.data.Ignore, val filter: clue.data.Input[GmosSouthFilter] = clue.data.Ignore, val fpu: clue.data.Input[GmosSouthBuiltinFpu] = clue.data.Ignore, val centralWavelength: clue.data.Input[WavelengthInput] = clue.data.Ignore, val explicitXBin: clue.data.Input[GmosXBinning] = clue.data.Ignore, val explicitYBin: clue.data.Input[GmosYBinning] = clue.data.Ignore, val explicitAmpReadMode: clue.data.Input[GmosAmpReadMode] = clue.data.Ignore, val explicitAmpGain: clue.data.Input[GmosAmpGain] = clue.data.Ignore, val explicitRoi: clue.data.Input[GmosRoi] = clue.data.Ignore, val explicitWavelengthDithers: clue.data.Input[List[WavelengthDitherInput]] = clue.data.Ignore, val explicitSpatialOffsets: clue.data.Input[List[OffsetComponentInput]] = clue.data.Ignore)
    object GmosSouthLongSlitInput {
      val grating: monocle.Lens[GmosSouthLongSlitInput, clue.data.Input[GmosSouthGrating]] = monocle.macros.GenLens[GmosSouthLongSlitInput](_.grating)
      val filter: monocle.Lens[GmosSouthLongSlitInput, clue.data.Input[GmosSouthFilter]] = monocle.macros.GenLens[GmosSouthLongSlitInput](_.filter)
      val fpu: monocle.Lens[GmosSouthLongSlitInput, clue.data.Input[GmosSouthBuiltinFpu]] = monocle.macros.GenLens[GmosSouthLongSlitInput](_.fpu)
      val centralWavelength: monocle.Lens[GmosSouthLongSlitInput, clue.data.Input[WavelengthInput]] = monocle.macros.GenLens[GmosSouthLongSlitInput](_.centralWavelength)
      val explicitXBin: monocle.Lens[GmosSouthLongSlitInput, clue.data.Input[GmosXBinning]] = monocle.macros.GenLens[GmosSouthLongSlitInput](_.explicitXBin)
      val explicitYBin: monocle.Lens[GmosSouthLongSlitInput, clue.data.Input[GmosYBinning]] = monocle.macros.GenLens[GmosSouthLongSlitInput](_.explicitYBin)
      val explicitAmpReadMode: monocle.Lens[GmosSouthLongSlitInput, clue.data.Input[GmosAmpReadMode]] = monocle.macros.GenLens[GmosSouthLongSlitInput](_.explicitAmpReadMode)
      val explicitAmpGain: monocle.Lens[GmosSouthLongSlitInput, clue.data.Input[GmosAmpGain]] = monocle.macros.GenLens[GmosSouthLongSlitInput](_.explicitAmpGain)
      val explicitRoi: monocle.Lens[GmosSouthLongSlitInput, clue.data.Input[GmosRoi]] = monocle.macros.GenLens[GmosSouthLongSlitInput](_.explicitRoi)
      val explicitWavelengthDithers: monocle.Lens[GmosSouthLongSlitInput, clue.data.Input[List[WavelengthDitherInput]]] = monocle.macros.GenLens[GmosSouthLongSlitInput](_.explicitWavelengthDithers)
      val explicitSpatialOffsets: monocle.Lens[GmosSouthLongSlitInput, clue.data.Input[List[OffsetComponentInput]]] = monocle.macros.GenLens[GmosSouthLongSlitInput](_.explicitSpatialOffsets)
      implicit val eqGmosSouthLongSlitInput: cats.Eq[GmosSouthLongSlitInput] = cats.Eq.fromUniversalEquals
      implicit val showGmosSouthLongSlitInput: cats.Show[GmosSouthLongSlitInput] = cats.Show.fromToString
      implicit val jsonEncoderGmosSouthLongSlitInput: io.circe.Encoder[GmosSouthLongSlitInput] = io.circe.generic.semiauto.deriveEncoder[GmosSouthLongSlitInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class GmosSouthScienceInput(val config: GmosSouthDynamicInput, val offset: OffsetInput)
    object GmosSouthScienceInput {
      val config: monocle.Lens[GmosSouthScienceInput, GmosSouthDynamicInput] = monocle.macros.GenLens[GmosSouthScienceInput](_.config)
      val offset: monocle.Lens[GmosSouthScienceInput, OffsetInput] = monocle.macros.GenLens[GmosSouthScienceInput](_.offset)
      implicit val eqGmosSouthScienceInput: cats.Eq[GmosSouthScienceInput] = cats.Eq.fromUniversalEquals
      implicit val showGmosSouthScienceInput: cats.Show[GmosSouthScienceInput] = cats.Show.fromToString
      implicit val jsonEncoderGmosSouthScienceInput: io.circe.Encoder[GmosSouthScienceInput] = io.circe.generic.semiauto.deriveEncoder[GmosSouthScienceInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class GmosSouthStaticInput(val detector: GmosNorthDetector, val mosPreImaging: MosPreImaging, val nodAndShuffle: clue.data.Input[GmosNodAndShuffleInput] = clue.data.Ignore, val stageMode: GmosSouthStageMode)
    object GmosSouthStaticInput {
      val detector: monocle.Lens[GmosSouthStaticInput, GmosNorthDetector] = monocle.macros.GenLens[GmosSouthStaticInput](_.detector)
      val mosPreImaging: monocle.Lens[GmosSouthStaticInput, MosPreImaging] = monocle.macros.GenLens[GmosSouthStaticInput](_.mosPreImaging)
      val nodAndShuffle: monocle.Lens[GmosSouthStaticInput, clue.data.Input[GmosNodAndShuffleInput]] = monocle.macros.GenLens[GmosSouthStaticInput](_.nodAndShuffle)
      val stageMode: monocle.Lens[GmosSouthStaticInput, GmosSouthStageMode] = monocle.macros.GenLens[GmosSouthStaticInput](_.stageMode)
      implicit val eqGmosSouthStaticInput: cats.Eq[GmosSouthStaticInput] = cats.Eq.fromUniversalEquals
      implicit val showGmosSouthStaticInput: cats.Show[GmosSouthStaticInput] = cats.Show.fromToString
      implicit val jsonEncoderGmosSouthStaticInput: io.circe.Encoder[GmosSouthStaticInput] = io.circe.generic.semiauto.deriveEncoder[GmosSouthStaticInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class GmosSouthStepInput(val bias: clue.data.Input[GmosSouthBiasInput] = clue.data.Ignore, val dark: clue.data.Input[GmosSouthDarkInput] = clue.data.Ignore, val gcal: clue.data.Input[GmosSouthGcalInput] = clue.data.Ignore, val science: clue.data.Input[GmosSouthScienceInput] = clue.data.Ignore)
    object GmosSouthStepInput {
      val bias: monocle.Lens[GmosSouthStepInput, clue.data.Input[GmosSouthBiasInput]] = monocle.macros.GenLens[GmosSouthStepInput](_.bias)
      val dark: monocle.Lens[GmosSouthStepInput, clue.data.Input[GmosSouthDarkInput]] = monocle.macros.GenLens[GmosSouthStepInput](_.dark)
      val gcal: monocle.Lens[GmosSouthStepInput, clue.data.Input[GmosSouthGcalInput]] = monocle.macros.GenLens[GmosSouthStepInput](_.gcal)
      val science: monocle.Lens[GmosSouthStepInput, clue.data.Input[GmosSouthScienceInput]] = monocle.macros.GenLens[GmosSouthStepInput](_.science)
      implicit val eqGmosSouthStepInput: cats.Eq[GmosSouthStepInput] = cats.Eq.fromUniversalEquals
      implicit val showGmosSouthStepInput: cats.Show[GmosSouthStepInput] = cats.Show.fromToString
      implicit val jsonEncoderGmosSouthStepInput: io.circe.Encoder[GmosSouthStepInput] = io.circe.generic.semiauto.deriveEncoder[GmosSouthStepInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class HourAngleRangeInput(val minHours: clue.data.Input[BigDecimal] = clue.data.Ignore, val maxHours: clue.data.Input[BigDecimal] = clue.data.Ignore)
    object HourAngleRangeInput {
      val minHours: monocle.Lens[HourAngleRangeInput, clue.data.Input[BigDecimal]] = monocle.macros.GenLens[HourAngleRangeInput](_.minHours)
      val maxHours: monocle.Lens[HourAngleRangeInput, clue.data.Input[BigDecimal]] = monocle.macros.GenLens[HourAngleRangeInput](_.maxHours)
      implicit val eqHourAngleRangeInput: cats.Eq[HourAngleRangeInput] = cats.Eq.fromUniversalEquals
      implicit val showHourAngleRangeInput: cats.Show[HourAngleRangeInput] = cats.Show.fromToString
      implicit val jsonEncoderHourAngleRangeInput: io.circe.Encoder[HourAngleRangeInput] = io.circe.generic.semiauto.deriveEncoder[HourAngleRangeInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class IntensiveInput(val minPercentTime: clue.data.Input[IntPercent] = clue.data.Ignore, val minPercentTotalTime: clue.data.Input[IntPercent] = clue.data.Ignore, val totalTime: clue.data.Input[TimeSpanInput] = clue.data.Ignore)
    object IntensiveInput {
      val minPercentTime: monocle.Lens[IntensiveInput, clue.data.Input[IntPercent]] = monocle.macros.GenLens[IntensiveInput](_.minPercentTime)
      val minPercentTotalTime: monocle.Lens[IntensiveInput, clue.data.Input[IntPercent]] = monocle.macros.GenLens[IntensiveInput](_.minPercentTotalTime)
      val totalTime: monocle.Lens[IntensiveInput, clue.data.Input[TimeSpanInput]] = monocle.macros.GenLens[IntensiveInput](_.totalTime)
      implicit val eqIntensiveInput: cats.Eq[IntensiveInput] = cats.Eq.fromUniversalEquals
      implicit val showIntensiveInput: cats.Show[IntensiveInput] = cats.Show.fromToString
      implicit val jsonEncoderIntensiveInput: io.circe.Encoder[IntensiveInput] = io.circe.generic.semiauto.deriveEncoder[IntensiveInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class LargeProgramInput(val minPercentTime: clue.data.Input[IntPercent] = clue.data.Ignore, val minPercentTotalTime: clue.data.Input[IntPercent] = clue.data.Ignore, val totalTime: clue.data.Input[TimeSpanInput] = clue.data.Ignore)
    object LargeProgramInput {
      val minPercentTime: monocle.Lens[LargeProgramInput, clue.data.Input[IntPercent]] = monocle.macros.GenLens[LargeProgramInput](_.minPercentTime)
      val minPercentTotalTime: monocle.Lens[LargeProgramInput, clue.data.Input[IntPercent]] = monocle.macros.GenLens[LargeProgramInput](_.minPercentTotalTime)
      val totalTime: monocle.Lens[LargeProgramInput, clue.data.Input[TimeSpanInput]] = monocle.macros.GenLens[LargeProgramInput](_.totalTime)
      implicit val eqLargeProgramInput: cats.Eq[LargeProgramInput] = cats.Eq.fromUniversalEquals
      implicit val showLargeProgramInput: cats.Show[LargeProgramInput] = cats.Show.fromToString
      implicit val jsonEncoderLargeProgramInput: io.circe.Encoder[LargeProgramInput] = io.circe.generic.semiauto.deriveEncoder[LargeProgramInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class LineFluxIntegratedInput(val value: PosBigDecimal, val units: LineFluxIntegratedUnits)
    object LineFluxIntegratedInput {
      val value: monocle.Lens[LineFluxIntegratedInput, PosBigDecimal] = monocle.macros.GenLens[LineFluxIntegratedInput](_.value)
      val units: monocle.Lens[LineFluxIntegratedInput, LineFluxIntegratedUnits] = monocle.macros.GenLens[LineFluxIntegratedInput](_.units)
      implicit val eqLineFluxIntegratedInput: cats.Eq[LineFluxIntegratedInput] = cats.Eq.fromUniversalEquals
      implicit val showLineFluxIntegratedInput: cats.Show[LineFluxIntegratedInput] = cats.Show.fromToString
      implicit val jsonEncoderLineFluxIntegratedInput: io.circe.Encoder[LineFluxIntegratedInput] = io.circe.generic.semiauto.deriveEncoder[LineFluxIntegratedInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class LineFluxSurfaceInput(val value: PosBigDecimal, val units: LineFluxSurfaceUnits)
    object LineFluxSurfaceInput {
      val value: monocle.Lens[LineFluxSurfaceInput, PosBigDecimal] = monocle.macros.GenLens[LineFluxSurfaceInput](_.value)
      val units: monocle.Lens[LineFluxSurfaceInput, LineFluxSurfaceUnits] = monocle.macros.GenLens[LineFluxSurfaceInput](_.units)
      implicit val eqLineFluxSurfaceInput: cats.Eq[LineFluxSurfaceInput] = cats.Eq.fromUniversalEquals
      implicit val showLineFluxSurfaceInput: cats.Show[LineFluxSurfaceInput] = cats.Show.fromToString
      implicit val jsonEncoderLineFluxSurfaceInput: io.circe.Encoder[LineFluxSurfaceInput] = io.circe.generic.semiauto.deriveEncoder[LineFluxSurfaceInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class LinkUserInput(val programId: ProgramId, val userId: UserId, val role: ProgramUserRole, val supportType: clue.data.Input[ProgramUserSupportRoleType] = clue.data.Ignore, val supportPartner: clue.data.Input[Partner] = clue.data.Ignore)
    object LinkUserInput {
      val programId: monocle.Lens[LinkUserInput, ProgramId] = monocle.macros.GenLens[LinkUserInput](_.programId)
      val userId: monocle.Lens[LinkUserInput, UserId] = monocle.macros.GenLens[LinkUserInput](_.userId)
      val role: monocle.Lens[LinkUserInput, ProgramUserRole] = monocle.macros.GenLens[LinkUserInput](_.role)
      val supportType: monocle.Lens[LinkUserInput, clue.data.Input[ProgramUserSupportRoleType]] = monocle.macros.GenLens[LinkUserInput](_.supportType)
      val supportPartner: monocle.Lens[LinkUserInput, clue.data.Input[Partner]] = monocle.macros.GenLens[LinkUserInput](_.supportPartner)
      implicit val eqLinkUserInput: cats.Eq[LinkUserInput] = cats.Eq.fromUniversalEquals
      implicit val showLinkUserInput: cats.Show[LinkUserInput] = cats.Show.fromToString
      implicit val jsonEncoderLinkUserInput: io.circe.Encoder[LinkUserInput] = io.circe.generic.semiauto.deriveEncoder[LinkUserInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class NonsiderealInput(val keyType: clue.data.Input[EphemerisKeyType] = clue.data.Ignore, val des: clue.data.Input[NonEmptyString] = clue.data.Ignore, val key: clue.data.Input[NonEmptyString] = clue.data.Ignore)
    object NonsiderealInput {
      val keyType: monocle.Lens[NonsiderealInput, clue.data.Input[EphemerisKeyType]] = monocle.macros.GenLens[NonsiderealInput](_.keyType)
      val des: monocle.Lens[NonsiderealInput, clue.data.Input[NonEmptyString]] = monocle.macros.GenLens[NonsiderealInput](_.des)
      val key: monocle.Lens[NonsiderealInput, clue.data.Input[NonEmptyString]] = monocle.macros.GenLens[NonsiderealInput](_.key)
      implicit val eqNonsiderealInput: cats.Eq[NonsiderealInput] = cats.Eq.fromUniversalEquals
      implicit val showNonsiderealInput: cats.Show[NonsiderealInput] = cats.Show.fromToString
      implicit val jsonEncoderNonsiderealInput: io.circe.Encoder[NonsiderealInput] = io.circe.generic.semiauto.deriveEncoder[NonsiderealInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class ObservationPropertiesInput(val subtitle: clue.data.Input[NonEmptyString] = clue.data.Ignore, val status: clue.data.Input[ObsStatus] = clue.data.Ignore, val activeStatus: clue.data.Input[ObsActiveStatus] = clue.data.Ignore, val visualizationTime: clue.data.Input[Timestamp] = clue.data.Ignore, val posAngleConstraint: clue.data.Input[PosAngleConstraintInput] = clue.data.Ignore, val targetEnvironment: clue.data.Input[TargetEnvironmentInput] = clue.data.Ignore, val constraintSet: clue.data.Input[ConstraintSetInput] = clue.data.Ignore, val scienceRequirements: clue.data.Input[ScienceRequirementsInput] = clue.data.Ignore, val observingMode: clue.data.Input[ObservingModeInput] = clue.data.Ignore, val existence: clue.data.Input[Existence] = clue.data.Ignore)
    object ObservationPropertiesInput {
      val subtitle: monocle.Lens[ObservationPropertiesInput, clue.data.Input[NonEmptyString]] = monocle.macros.GenLens[ObservationPropertiesInput](_.subtitle)
      val status: monocle.Lens[ObservationPropertiesInput, clue.data.Input[ObsStatus]] = monocle.macros.GenLens[ObservationPropertiesInput](_.status)
      val activeStatus: monocle.Lens[ObservationPropertiesInput, clue.data.Input[ObsActiveStatus]] = monocle.macros.GenLens[ObservationPropertiesInput](_.activeStatus)
      val visualizationTime: monocle.Lens[ObservationPropertiesInput, clue.data.Input[Timestamp]] = monocle.macros.GenLens[ObservationPropertiesInput](_.visualizationTime)
      val posAngleConstraint: monocle.Lens[ObservationPropertiesInput, clue.data.Input[PosAngleConstraintInput]] = monocle.macros.GenLens[ObservationPropertiesInput](_.posAngleConstraint)
      val targetEnvironment: monocle.Lens[ObservationPropertiesInput, clue.data.Input[TargetEnvironmentInput]] = monocle.macros.GenLens[ObservationPropertiesInput](_.targetEnvironment)
      val constraintSet: monocle.Lens[ObservationPropertiesInput, clue.data.Input[ConstraintSetInput]] = monocle.macros.GenLens[ObservationPropertiesInput](_.constraintSet)
      val scienceRequirements: monocle.Lens[ObservationPropertiesInput, clue.data.Input[ScienceRequirementsInput]] = monocle.macros.GenLens[ObservationPropertiesInput](_.scienceRequirements)
      val observingMode: monocle.Lens[ObservationPropertiesInput, clue.data.Input[ObservingModeInput]] = monocle.macros.GenLens[ObservationPropertiesInput](_.observingMode)
      val existence: monocle.Lens[ObservationPropertiesInput, clue.data.Input[Existence]] = monocle.macros.GenLens[ObservationPropertiesInput](_.existence)
      implicit val eqObservationPropertiesInput: cats.Eq[ObservationPropertiesInput] = cats.Eq.fromUniversalEquals
      implicit val showObservationPropertiesInput: cats.Show[ObservationPropertiesInput] = cats.Show.fromToString
      implicit val jsonEncoderObservationPropertiesInput: io.circe.Encoder[ObservationPropertiesInput] = io.circe.generic.semiauto.deriveEncoder[ObservationPropertiesInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class OffsetComponentInput(val microarcseconds: clue.data.Input[Long] = clue.data.Ignore, val milliarcseconds: clue.data.Input[BigDecimal] = clue.data.Ignore, val arcseconds: clue.data.Input[BigDecimal] = clue.data.Ignore)
    object OffsetComponentInput {
      val microarcseconds: monocle.Lens[OffsetComponentInput, clue.data.Input[Long]] = monocle.macros.GenLens[OffsetComponentInput](_.microarcseconds)
      val milliarcseconds: monocle.Lens[OffsetComponentInput, clue.data.Input[BigDecimal]] = monocle.macros.GenLens[OffsetComponentInput](_.milliarcseconds)
      val arcseconds: monocle.Lens[OffsetComponentInput, clue.data.Input[BigDecimal]] = monocle.macros.GenLens[OffsetComponentInput](_.arcseconds)
      implicit val eqOffsetComponentInput: cats.Eq[OffsetComponentInput] = cats.Eq.fromUniversalEquals
      implicit val showOffsetComponentInput: cats.Show[OffsetComponentInput] = cats.Show.fromToString
      implicit val jsonEncoderOffsetComponentInput: io.circe.Encoder[OffsetComponentInput] = io.circe.generic.semiauto.deriveEncoder[OffsetComponentInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class OffsetInput(val p: OffsetComponentInput, val q: OffsetComponentInput)
    object OffsetInput {
      val p: monocle.Lens[OffsetInput, OffsetComponentInput] = monocle.macros.GenLens[OffsetInput](_.p)
      val q: monocle.Lens[OffsetInput, OffsetComponentInput] = monocle.macros.GenLens[OffsetInput](_.q)
      implicit val eqOffsetInput: cats.Eq[OffsetInput] = cats.Eq.fromUniversalEquals
      implicit val showOffsetInput: cats.Show[OffsetInput] = cats.Show.fromToString
      implicit val jsonEncoderOffsetInput: io.circe.Encoder[OffsetInput] = io.circe.generic.semiauto.deriveEncoder[OffsetInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class ParallaxInput(val microarcseconds: clue.data.Input[Long] = clue.data.Ignore, val milliarcseconds: clue.data.Input[BigDecimal] = clue.data.Ignore)
    object ParallaxInput {
      val microarcseconds: monocle.Lens[ParallaxInput, clue.data.Input[Long]] = monocle.macros.GenLens[ParallaxInput](_.microarcseconds)
      val milliarcseconds: monocle.Lens[ParallaxInput, clue.data.Input[BigDecimal]] = monocle.macros.GenLens[ParallaxInput](_.milliarcseconds)
      implicit val eqParallaxInput: cats.Eq[ParallaxInput] = cats.Eq.fromUniversalEquals
      implicit val showParallaxInput: cats.Show[ParallaxInput] = cats.Show.fromToString
      implicit val jsonEncoderParallaxInput: io.circe.Encoder[ParallaxInput] = io.circe.generic.semiauto.deriveEncoder[ParallaxInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class PartnerSplitInput(val partner: Partner, val percent: IntPercent)
    object PartnerSplitInput {
      val partner: monocle.Lens[PartnerSplitInput, Partner] = monocle.macros.GenLens[PartnerSplitInput](_.partner)
      val percent: monocle.Lens[PartnerSplitInput, IntPercent] = monocle.macros.GenLens[PartnerSplitInput](_.percent)
      implicit val eqPartnerSplitInput: cats.Eq[PartnerSplitInput] = cats.Eq.fromUniversalEquals
      implicit val showPartnerSplitInput: cats.Show[PartnerSplitInput] = cats.Show.fromToString
      implicit val jsonEncoderPartnerSplitInput: io.circe.Encoder[PartnerSplitInput] = io.circe.generic.semiauto.deriveEncoder[PartnerSplitInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class PoorWeatherInput(val minPercentTime: clue.data.Input[IntPercent] = clue.data.Ignore)
    object PoorWeatherInput {
      val minPercentTime: monocle.Lens[PoorWeatherInput, clue.data.Input[IntPercent]] = monocle.macros.GenLens[PoorWeatherInput](_.minPercentTime)
      implicit val eqPoorWeatherInput: cats.Eq[PoorWeatherInput] = cats.Eq.fromUniversalEquals
      implicit val showPoorWeatherInput: cats.Show[PoorWeatherInput] = cats.Show.fromToString
      implicit val jsonEncoderPoorWeatherInput: io.circe.Encoder[PoorWeatherInput] = io.circe.generic.semiauto.deriveEncoder[PoorWeatherInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class PosAngleConstraintInput(val mode: clue.data.Input[PosAngleConstraintMode] = clue.data.Ignore, val angle: clue.data.Input[AngleInput] = clue.data.Ignore)
    object PosAngleConstraintInput {
      val mode: monocle.Lens[PosAngleConstraintInput, clue.data.Input[PosAngleConstraintMode]] = monocle.macros.GenLens[PosAngleConstraintInput](_.mode)
      val angle: monocle.Lens[PosAngleConstraintInput, clue.data.Input[AngleInput]] = monocle.macros.GenLens[PosAngleConstraintInput](_.angle)
      implicit val eqPosAngleConstraintInput: cats.Eq[PosAngleConstraintInput] = cats.Eq.fromUniversalEquals
      implicit val showPosAngleConstraintInput: cats.Show[PosAngleConstraintInput] = cats.Show.fromToString
      implicit val jsonEncoderPosAngleConstraintInput: io.circe.Encoder[PosAngleConstraintInput] = io.circe.generic.semiauto.deriveEncoder[PosAngleConstraintInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class ProgramPropertiesInput(val name: clue.data.Input[NonEmptyString] = clue.data.Ignore, val proposal: clue.data.Input[ProposalInput] = clue.data.Ignore, val existence: clue.data.Input[Existence] = clue.data.Ignore)
    object ProgramPropertiesInput {
      val name: monocle.Lens[ProgramPropertiesInput, clue.data.Input[NonEmptyString]] = monocle.macros.GenLens[ProgramPropertiesInput](_.name)
      val proposal: monocle.Lens[ProgramPropertiesInput, clue.data.Input[ProposalInput]] = monocle.macros.GenLens[ProgramPropertiesInput](_.proposal)
      val existence: monocle.Lens[ProgramPropertiesInput, clue.data.Input[Existence]] = monocle.macros.GenLens[ProgramPropertiesInput](_.existence)
      implicit val eqProgramPropertiesInput: cats.Eq[ProgramPropertiesInput] = cats.Eq.fromUniversalEquals
      implicit val showProgramPropertiesInput: cats.Show[ProgramPropertiesInput] = cats.Show.fromToString
      implicit val jsonEncoderProgramPropertiesInput: io.circe.Encoder[ProgramPropertiesInput] = io.circe.generic.semiauto.deriveEncoder[ProgramPropertiesInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class ProperMotionComponentInput(val microarcsecondsPerYear: clue.data.Input[Long] = clue.data.Ignore, val milliarcsecondsPerYear: clue.data.Input[BigDecimal] = clue.data.Ignore)
    object ProperMotionComponentInput {
      val microarcsecondsPerYear: monocle.Lens[ProperMotionComponentInput, clue.data.Input[Long]] = monocle.macros.GenLens[ProperMotionComponentInput](_.microarcsecondsPerYear)
      val milliarcsecondsPerYear: monocle.Lens[ProperMotionComponentInput, clue.data.Input[BigDecimal]] = monocle.macros.GenLens[ProperMotionComponentInput](_.milliarcsecondsPerYear)
      implicit val eqProperMotionComponentInput: cats.Eq[ProperMotionComponentInput] = cats.Eq.fromUniversalEquals
      implicit val showProperMotionComponentInput: cats.Show[ProperMotionComponentInput] = cats.Show.fromToString
      implicit val jsonEncoderProperMotionComponentInput: io.circe.Encoder[ProperMotionComponentInput] = io.circe.generic.semiauto.deriveEncoder[ProperMotionComponentInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class ProperMotionInput(val ra: ProperMotionComponentInput, val dec: ProperMotionComponentInput)
    object ProperMotionInput {
      val ra: monocle.Lens[ProperMotionInput, ProperMotionComponentInput] = monocle.macros.GenLens[ProperMotionInput](_.ra)
      val dec: monocle.Lens[ProperMotionInput, ProperMotionComponentInput] = monocle.macros.GenLens[ProperMotionInput](_.dec)
      implicit val eqProperMotionInput: cats.Eq[ProperMotionInput] = cats.Eq.fromUniversalEquals
      implicit val showProperMotionInput: cats.Show[ProperMotionInput] = cats.Show.fromToString
      implicit val jsonEncoderProperMotionInput: io.circe.Encoder[ProperMotionInput] = io.circe.generic.semiauto.deriveEncoder[ProperMotionInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class ProposalClassInput(val classical: clue.data.Input[ClassicalInput] = clue.data.Ignore, val demoScience: clue.data.Input[DemoScienceInput] = clue.data.Ignore, val directorsTime: clue.data.Input[DirectorsTimeInput] = clue.data.Ignore, val exchange: clue.data.Input[ExchangeInput] = clue.data.Ignore, val fastTurnaround: clue.data.Input[FastTurnaroundInput] = clue.data.Ignore, val poorWeather: clue.data.Input[PoorWeatherInput] = clue.data.Ignore, val queue: clue.data.Input[QueueInput] = clue.data.Ignore, val systemVerification: clue.data.Input[SystemVerificationInput] = clue.data.Ignore, val largeProgram: clue.data.Input[LargeProgramInput] = clue.data.Ignore, val intensive: clue.data.Input[IntensiveInput] = clue.data.Ignore)
    object ProposalClassInput {
      val classical: monocle.Lens[ProposalClassInput, clue.data.Input[ClassicalInput]] = monocle.macros.GenLens[ProposalClassInput](_.classical)
      val demoScience: monocle.Lens[ProposalClassInput, clue.data.Input[DemoScienceInput]] = monocle.macros.GenLens[ProposalClassInput](_.demoScience)
      val directorsTime: monocle.Lens[ProposalClassInput, clue.data.Input[DirectorsTimeInput]] = monocle.macros.GenLens[ProposalClassInput](_.directorsTime)
      val exchange: monocle.Lens[ProposalClassInput, clue.data.Input[ExchangeInput]] = monocle.macros.GenLens[ProposalClassInput](_.exchange)
      val fastTurnaround: monocle.Lens[ProposalClassInput, clue.data.Input[FastTurnaroundInput]] = monocle.macros.GenLens[ProposalClassInput](_.fastTurnaround)
      val poorWeather: monocle.Lens[ProposalClassInput, clue.data.Input[PoorWeatherInput]] = monocle.macros.GenLens[ProposalClassInput](_.poorWeather)
      val queue: monocle.Lens[ProposalClassInput, clue.data.Input[QueueInput]] = monocle.macros.GenLens[ProposalClassInput](_.queue)
      val systemVerification: monocle.Lens[ProposalClassInput, clue.data.Input[SystemVerificationInput]] = monocle.macros.GenLens[ProposalClassInput](_.systemVerification)
      val largeProgram: monocle.Lens[ProposalClassInput, clue.data.Input[LargeProgramInput]] = monocle.macros.GenLens[ProposalClassInput](_.largeProgram)
      val intensive: monocle.Lens[ProposalClassInput, clue.data.Input[IntensiveInput]] = monocle.macros.GenLens[ProposalClassInput](_.intensive)
      implicit val eqProposalClassInput: cats.Eq[ProposalClassInput] = cats.Eq.fromUniversalEquals
      implicit val showProposalClassInput: cats.Show[ProposalClassInput] = cats.Show.fromToString
      implicit val jsonEncoderProposalClassInput: io.circe.Encoder[ProposalClassInput] = io.circe.generic.semiauto.deriveEncoder[ProposalClassInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class ProposalInput(val title: clue.data.Input[NonEmptyString] = clue.data.Ignore, val proposalClass: clue.data.Input[ProposalClassInput] = clue.data.Ignore, val category: clue.data.Input[TacCategory] = clue.data.Ignore, val toOActivation: clue.data.Input[ToOActivation] = clue.data.Ignore, val `abstract`: clue.data.Input[NonEmptyString] = clue.data.Ignore, val partnerSplits: clue.data.Input[List[PartnerSplitInput]] = clue.data.Ignore)
    object ProposalInput {
      val title: monocle.Lens[ProposalInput, clue.data.Input[NonEmptyString]] = monocle.macros.GenLens[ProposalInput](_.title)
      val proposalClass: monocle.Lens[ProposalInput, clue.data.Input[ProposalClassInput]] = monocle.macros.GenLens[ProposalInput](_.proposalClass)
      val category: monocle.Lens[ProposalInput, clue.data.Input[TacCategory]] = monocle.macros.GenLens[ProposalInput](_.category)
      val toOActivation: monocle.Lens[ProposalInput, clue.data.Input[ToOActivation]] = monocle.macros.GenLens[ProposalInput](_.toOActivation)
      val `abstract`: monocle.Lens[ProposalInput, clue.data.Input[NonEmptyString]] = monocle.macros.GenLens[ProposalInput](_.`abstract`)
      val partnerSplits: monocle.Lens[ProposalInput, clue.data.Input[List[PartnerSplitInput]]] = monocle.macros.GenLens[ProposalInput](_.partnerSplits)
      implicit val eqProposalInput: cats.Eq[ProposalInput] = cats.Eq.fromUniversalEquals
      implicit val showProposalInput: cats.Show[ProposalInput] = cats.Show.fromToString
      implicit val jsonEncoderProposalInput: io.circe.Encoder[ProposalInput] = io.circe.generic.semiauto.deriveEncoder[ProposalInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class QueueInput(val minPercentTime: clue.data.Input[IntPercent] = clue.data.Ignore)
    object QueueInput {
      val minPercentTime: monocle.Lens[QueueInput, clue.data.Input[IntPercent]] = monocle.macros.GenLens[QueueInput](_.minPercentTime)
      implicit val eqQueueInput: cats.Eq[QueueInput] = cats.Eq.fromUniversalEquals
      implicit val showQueueInput: cats.Show[QueueInput] = cats.Show.fromToString
      implicit val jsonEncoderQueueInput: io.circe.Encoder[QueueInput] = io.circe.generic.semiauto.deriveEncoder[QueueInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class RadialVelocityInput(val centimetersPerSecond: clue.data.Input[Long] = clue.data.Ignore, val metersPerSecond: clue.data.Input[BigDecimal] = clue.data.Ignore, val kilometersPerSecond: clue.data.Input[BigDecimal] = clue.data.Ignore)
    object RadialVelocityInput {
      val centimetersPerSecond: monocle.Lens[RadialVelocityInput, clue.data.Input[Long]] = monocle.macros.GenLens[RadialVelocityInput](_.centimetersPerSecond)
      val metersPerSecond: monocle.Lens[RadialVelocityInput, clue.data.Input[BigDecimal]] = monocle.macros.GenLens[RadialVelocityInput](_.metersPerSecond)
      val kilometersPerSecond: monocle.Lens[RadialVelocityInput, clue.data.Input[BigDecimal]] = monocle.macros.GenLens[RadialVelocityInput](_.kilometersPerSecond)
      implicit val eqRadialVelocityInput: cats.Eq[RadialVelocityInput] = cats.Eq.fromUniversalEquals
      implicit val showRadialVelocityInput: cats.Show[RadialVelocityInput] = cats.Show.fromToString
      implicit val jsonEncoderRadialVelocityInput: io.circe.Encoder[RadialVelocityInput] = io.circe.generic.semiauto.deriveEncoder[RadialVelocityInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class RecordGmosNorthStepInput(val observationId: ObservationId, val visitId: VisitId, val stepConfig: GmosNorthStepInput)
    object RecordGmosNorthStepInput {
      val observationId: monocle.Lens[RecordGmosNorthStepInput, ObservationId] = monocle.macros.GenLens[RecordGmosNorthStepInput](_.observationId)
      val visitId: monocle.Lens[RecordGmosNorthStepInput, VisitId] = monocle.macros.GenLens[RecordGmosNorthStepInput](_.visitId)
      val stepConfig: monocle.Lens[RecordGmosNorthStepInput, GmosNorthStepInput] = monocle.macros.GenLens[RecordGmosNorthStepInput](_.stepConfig)
      implicit val eqRecordGmosNorthStepInput: cats.Eq[RecordGmosNorthStepInput] = cats.Eq.fromUniversalEquals
      implicit val showRecordGmosNorthStepInput: cats.Show[RecordGmosNorthStepInput] = cats.Show.fromToString
      implicit val jsonEncoderRecordGmosNorthStepInput: io.circe.Encoder[RecordGmosNorthStepInput] = io.circe.generic.semiauto.deriveEncoder[RecordGmosNorthStepInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class RecordGmosNorthVisitInput(val observationId: ObservationId, val static: GmosNorthStaticInput)
    object RecordGmosNorthVisitInput {
      val observationId: monocle.Lens[RecordGmosNorthVisitInput, ObservationId] = monocle.macros.GenLens[RecordGmosNorthVisitInput](_.observationId)
      val static: monocle.Lens[RecordGmosNorthVisitInput, GmosNorthStaticInput] = monocle.macros.GenLens[RecordGmosNorthVisitInput](_.static)
      implicit val eqRecordGmosNorthVisitInput: cats.Eq[RecordGmosNorthVisitInput] = cats.Eq.fromUniversalEquals
      implicit val showRecordGmosNorthVisitInput: cats.Show[RecordGmosNorthVisitInput] = cats.Show.fromToString
      implicit val jsonEncoderRecordGmosNorthVisitInput: io.circe.Encoder[RecordGmosNorthVisitInput] = io.circe.generic.semiauto.deriveEncoder[RecordGmosNorthVisitInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class RecordGmosSouthStepInput(val observationId: ObservationId, val visitId: VisitId, val stepConfig: GmosSouthStepInput)
    object RecordGmosSouthStepInput {
      val observationId: monocle.Lens[RecordGmosSouthStepInput, ObservationId] = monocle.macros.GenLens[RecordGmosSouthStepInput](_.observationId)
      val visitId: monocle.Lens[RecordGmosSouthStepInput, VisitId] = monocle.macros.GenLens[RecordGmosSouthStepInput](_.visitId)
      val stepConfig: monocle.Lens[RecordGmosSouthStepInput, GmosSouthStepInput] = monocle.macros.GenLens[RecordGmosSouthStepInput](_.stepConfig)
      implicit val eqRecordGmosSouthStepInput: cats.Eq[RecordGmosSouthStepInput] = cats.Eq.fromUniversalEquals
      implicit val showRecordGmosSouthStepInput: cats.Show[RecordGmosSouthStepInput] = cats.Show.fromToString
      implicit val jsonEncoderRecordGmosSouthStepInput: io.circe.Encoder[RecordGmosSouthStepInput] = io.circe.generic.semiauto.deriveEncoder[RecordGmosSouthStepInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class RecordGmosSouthVisitInput(val observationId: ObservationId, val static: GmosSouthStaticInput)
    object RecordGmosSouthVisitInput {
      val observationId: monocle.Lens[RecordGmosSouthVisitInput, ObservationId] = monocle.macros.GenLens[RecordGmosSouthVisitInput](_.observationId)
      val static: monocle.Lens[RecordGmosSouthVisitInput, GmosSouthStaticInput] = monocle.macros.GenLens[RecordGmosSouthVisitInput](_.static)
      implicit val eqRecordGmosSouthVisitInput: cats.Eq[RecordGmosSouthVisitInput] = cats.Eq.fromUniversalEquals
      implicit val showRecordGmosSouthVisitInput: cats.Show[RecordGmosSouthVisitInput] = cats.Show.fromToString
      implicit val jsonEncoderRecordGmosSouthVisitInput: io.circe.Encoder[RecordGmosSouthVisitInput] = io.circe.generic.semiauto.deriveEncoder[RecordGmosSouthVisitInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class RightAscensionInput(val microarcseconds: clue.data.Input[Long] = clue.data.Ignore, val degrees: clue.data.Input[BigDecimal] = clue.data.Ignore, val hours: clue.data.Input[BigDecimal] = clue.data.Ignore, val hms: clue.data.Input[HmsString] = clue.data.Ignore)
    object RightAscensionInput {
      val microarcseconds: monocle.Lens[RightAscensionInput, clue.data.Input[Long]] = monocle.macros.GenLens[RightAscensionInput](_.microarcseconds)
      val degrees: monocle.Lens[RightAscensionInput, clue.data.Input[BigDecimal]] = monocle.macros.GenLens[RightAscensionInput](_.degrees)
      val hours: monocle.Lens[RightAscensionInput, clue.data.Input[BigDecimal]] = monocle.macros.GenLens[RightAscensionInput](_.hours)
      val hms: monocle.Lens[RightAscensionInput, clue.data.Input[HmsString]] = monocle.macros.GenLens[RightAscensionInput](_.hms)
      implicit val eqRightAscensionInput: cats.Eq[RightAscensionInput] = cats.Eq.fromUniversalEquals
      implicit val showRightAscensionInput: cats.Show[RightAscensionInput] = cats.Show.fromToString
      implicit val jsonEncoderRightAscensionInput: io.circe.Encoder[RightAscensionInput] = io.circe.generic.semiauto.deriveEncoder[RightAscensionInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class ObservingModeInput(val gmosNorthLongSlit: clue.data.Input[GmosNorthLongSlitInput] = clue.data.Ignore, val gmosSouthLongSlit: clue.data.Input[GmosSouthLongSlitInput] = clue.data.Ignore)
    object ObservingModeInput {
      val gmosNorthLongSlit: monocle.Lens[ObservingModeInput, clue.data.Input[GmosNorthLongSlitInput]] = monocle.macros.GenLens[ObservingModeInput](_.gmosNorthLongSlit)
      val gmosSouthLongSlit: monocle.Lens[ObservingModeInput, clue.data.Input[GmosSouthLongSlitInput]] = monocle.macros.GenLens[ObservingModeInput](_.gmosSouthLongSlit)
      implicit val eqObservingModeInput: cats.Eq[ObservingModeInput] = cats.Eq.fromUniversalEquals
      implicit val showObservingModeInput: cats.Show[ObservingModeInput] = cats.Show.fromToString
      implicit val jsonEncoderObservingModeInput: io.circe.Encoder[ObservingModeInput] = io.circe.generic.semiauto.deriveEncoder[ObservingModeInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class ScienceRequirementsInput(val mode: clue.data.Input[ScienceMode] = clue.data.Ignore, val spectroscopy: clue.data.Input[SpectroscopyScienceRequirementsInput] = clue.data.Ignore)
    object ScienceRequirementsInput {
      val mode: monocle.Lens[ScienceRequirementsInput, clue.data.Input[ScienceMode]] = monocle.macros.GenLens[ScienceRequirementsInput](_.mode)
      val spectroscopy: monocle.Lens[ScienceRequirementsInput, clue.data.Input[SpectroscopyScienceRequirementsInput]] = monocle.macros.GenLens[ScienceRequirementsInput](_.spectroscopy)
      implicit val eqScienceRequirementsInput: cats.Eq[ScienceRequirementsInput] = cats.Eq.fromUniversalEquals
      implicit val showScienceRequirementsInput: cats.Show[ScienceRequirementsInput] = cats.Show.fromToString
      implicit val jsonEncoderScienceRequirementsInput: io.circe.Encoder[ScienceRequirementsInput] = io.circe.generic.semiauto.deriveEncoder[ScienceRequirementsInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class SequenceEventLocationInput(val observationId: ObservationId)
    object SequenceEventLocationInput {
      val observationId: monocle.Lens[SequenceEventLocationInput, ObservationId] = monocle.macros.GenLens[SequenceEventLocationInput](_.observationId)
      implicit val eqSequenceEventLocationInput: cats.Eq[SequenceEventLocationInput] = cats.Eq.fromUniversalEquals
      implicit val showSequenceEventLocationInput: cats.Show[SequenceEventLocationInput] = cats.Show.fromToString
      implicit val jsonEncoderSequenceEventLocationInput: io.circe.Encoder[SequenceEventLocationInput] = io.circe.generic.semiauto.deriveEncoder[SequenceEventLocationInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class SequenceEventPayloadInput(val command: SequenceCommand)
    object SequenceEventPayloadInput {
      val command: monocle.Lens[SequenceEventPayloadInput, SequenceCommand] = monocle.macros.GenLens[SequenceEventPayloadInput](_.command)
      implicit val eqSequenceEventPayloadInput: cats.Eq[SequenceEventPayloadInput] = cats.Eq.fromUniversalEquals
      implicit val showSequenceEventPayloadInput: cats.Show[SequenceEventPayloadInput] = cats.Show.fromToString
      implicit val jsonEncoderSequenceEventPayloadInput: io.circe.Encoder[SequenceEventPayloadInput] = io.circe.generic.semiauto.deriveEncoder[SequenceEventPayloadInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class SetAllocationInput(val programId: ProgramId, val partner: Partner, val duration: TimeSpanInput)
    object SetAllocationInput {
      val programId: monocle.Lens[SetAllocationInput, ProgramId] = monocle.macros.GenLens[SetAllocationInput](_.programId)
      val partner: monocle.Lens[SetAllocationInput, Partner] = monocle.macros.GenLens[SetAllocationInput](_.partner)
      val duration: monocle.Lens[SetAllocationInput, TimeSpanInput] = monocle.macros.GenLens[SetAllocationInput](_.duration)
      implicit val eqSetAllocationInput: cats.Eq[SetAllocationInput] = cats.Eq.fromUniversalEquals
      implicit val showSetAllocationInput: cats.Show[SetAllocationInput] = cats.Show.fromToString
      implicit val jsonEncoderSetAllocationInput: io.circe.Encoder[SetAllocationInput] = io.circe.generic.semiauto.deriveEncoder[SetAllocationInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class SiderealInput(val ra: clue.data.Input[RightAscensionInput] = clue.data.Ignore, val dec: clue.data.Input[DeclinationInput] = clue.data.Ignore, val epoch: clue.data.Input[EpochString] = clue.data.Ignore, val properMotion: clue.data.Input[ProperMotionInput] = clue.data.Ignore, val radialVelocity: clue.data.Input[RadialVelocityInput] = clue.data.Ignore, val parallax: clue.data.Input[ParallaxInput] = clue.data.Ignore, val catalogInfo: clue.data.Input[CatalogInfoInput] = clue.data.Ignore)
    object SiderealInput {
      val ra: monocle.Lens[SiderealInput, clue.data.Input[RightAscensionInput]] = monocle.macros.GenLens[SiderealInput](_.ra)
      val dec: monocle.Lens[SiderealInput, clue.data.Input[DeclinationInput]] = monocle.macros.GenLens[SiderealInput](_.dec)
      val epoch: monocle.Lens[SiderealInput, clue.data.Input[EpochString]] = monocle.macros.GenLens[SiderealInput](_.epoch)
      val properMotion: monocle.Lens[SiderealInput, clue.data.Input[ProperMotionInput]] = monocle.macros.GenLens[SiderealInput](_.properMotion)
      val radialVelocity: monocle.Lens[SiderealInput, clue.data.Input[RadialVelocityInput]] = monocle.macros.GenLens[SiderealInput](_.radialVelocity)
      val parallax: monocle.Lens[SiderealInput, clue.data.Input[ParallaxInput]] = monocle.macros.GenLens[SiderealInput](_.parallax)
      val catalogInfo: monocle.Lens[SiderealInput, clue.data.Input[CatalogInfoInput]] = monocle.macros.GenLens[SiderealInput](_.catalogInfo)
      implicit val eqSiderealInput: cats.Eq[SiderealInput] = cats.Eq.fromUniversalEquals
      implicit val showSiderealInput: cats.Show[SiderealInput] = cats.Show.fromToString
      implicit val jsonEncoderSiderealInput: io.circe.Encoder[SiderealInput] = io.circe.generic.semiauto.deriveEncoder[SiderealInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class SignalToNoiseModeInput(val value: PosBigDecimal)
    object SignalToNoiseModeInput {
      val value: monocle.Lens[SignalToNoiseModeInput, PosBigDecimal] = monocle.macros.GenLens[SignalToNoiseModeInput](_.value)
      implicit val eqSignalToNoiseModeInput: cats.Eq[SignalToNoiseModeInput] = cats.Eq.fromUniversalEquals
      implicit val showSignalToNoiseModeInput: cats.Show[SignalToNoiseModeInput] = cats.Show.fromToString
      implicit val jsonEncoderSignalToNoiseModeInput: io.circe.Encoder[SignalToNoiseModeInput] = io.circe.generic.semiauto.deriveEncoder[SignalToNoiseModeInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class SourceProfileInput(val point: clue.data.Input[SpectralDefinitionIntegratedInput] = clue.data.Ignore, val uniform: clue.data.Input[SpectralDefinitionSurfaceInput] = clue.data.Ignore, val gaussian: clue.data.Input[GaussianInput] = clue.data.Ignore)
    object SourceProfileInput {
      val point: monocle.Lens[SourceProfileInput, clue.data.Input[SpectralDefinitionIntegratedInput]] = monocle.macros.GenLens[SourceProfileInput](_.point)
      val uniform: monocle.Lens[SourceProfileInput, clue.data.Input[SpectralDefinitionSurfaceInput]] = monocle.macros.GenLens[SourceProfileInput](_.uniform)
      val gaussian: monocle.Lens[SourceProfileInput, clue.data.Input[GaussianInput]] = monocle.macros.GenLens[SourceProfileInput](_.gaussian)
      implicit val eqSourceProfileInput: cats.Eq[SourceProfileInput] = cats.Eq.fromUniversalEquals
      implicit val showSourceProfileInput: cats.Show[SourceProfileInput] = cats.Show.fromToString
      implicit val jsonEncoderSourceProfileInput: io.circe.Encoder[SourceProfileInput] = io.circe.generic.semiauto.deriveEncoder[SourceProfileInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class SpectralDefinitionIntegratedInput(val bandNormalized: clue.data.Input[BandNormalizedIntegratedInput] = clue.data.Ignore, val emissionLines: clue.data.Input[EmissionLinesIntegratedInput] = clue.data.Ignore)
    object SpectralDefinitionIntegratedInput {
      val bandNormalized: monocle.Lens[SpectralDefinitionIntegratedInput, clue.data.Input[BandNormalizedIntegratedInput]] = monocle.macros.GenLens[SpectralDefinitionIntegratedInput](_.bandNormalized)
      val emissionLines: monocle.Lens[SpectralDefinitionIntegratedInput, clue.data.Input[EmissionLinesIntegratedInput]] = monocle.macros.GenLens[SpectralDefinitionIntegratedInput](_.emissionLines)
      implicit val eqSpectralDefinitionIntegratedInput: cats.Eq[SpectralDefinitionIntegratedInput] = cats.Eq.fromUniversalEquals
      implicit val showSpectralDefinitionIntegratedInput: cats.Show[SpectralDefinitionIntegratedInput] = cats.Show.fromToString
      implicit val jsonEncoderSpectralDefinitionIntegratedInput: io.circe.Encoder[SpectralDefinitionIntegratedInput] = io.circe.generic.semiauto.deriveEncoder[SpectralDefinitionIntegratedInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class SpectralDefinitionSurfaceInput(val bandNormalized: clue.data.Input[BandNormalizedSurfaceInput] = clue.data.Ignore, val emissionLines: clue.data.Input[EmissionLinesSurfaceInput] = clue.data.Ignore)
    object SpectralDefinitionSurfaceInput {
      val bandNormalized: monocle.Lens[SpectralDefinitionSurfaceInput, clue.data.Input[BandNormalizedSurfaceInput]] = monocle.macros.GenLens[SpectralDefinitionSurfaceInput](_.bandNormalized)
      val emissionLines: monocle.Lens[SpectralDefinitionSurfaceInput, clue.data.Input[EmissionLinesSurfaceInput]] = monocle.macros.GenLens[SpectralDefinitionSurfaceInput](_.emissionLines)
      implicit val eqSpectralDefinitionSurfaceInput: cats.Eq[SpectralDefinitionSurfaceInput] = cats.Eq.fromUniversalEquals
      implicit val showSpectralDefinitionSurfaceInput: cats.Show[SpectralDefinitionSurfaceInput] = cats.Show.fromToString
      implicit val jsonEncoderSpectralDefinitionSurfaceInput: io.circe.Encoder[SpectralDefinitionSurfaceInput] = io.circe.generic.semiauto.deriveEncoder[SpectralDefinitionSurfaceInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class SpectroscopyScienceRequirementsInput(val wavelength: clue.data.Input[WavelengthInput] = clue.data.Ignore, val resolution: clue.data.Input[PosInt] = clue.data.Ignore, val signalToNoise: clue.data.Input[PosBigDecimal] = clue.data.Ignore, val signalToNoiseAt: clue.data.Input[WavelengthInput] = clue.data.Ignore, val wavelengthCoverage: clue.data.Input[WavelengthInput] = clue.data.Ignore, val focalPlane: clue.data.Input[FocalPlane] = clue.data.Ignore, val focalPlaneAngle: clue.data.Input[AngleInput] = clue.data.Ignore, val capability: clue.data.Input[SpectroscopyCapabilities] = clue.data.Ignore)
    object SpectroscopyScienceRequirementsInput {
      val wavelength: monocle.Lens[SpectroscopyScienceRequirementsInput, clue.data.Input[WavelengthInput]] = monocle.macros.GenLens[SpectroscopyScienceRequirementsInput](_.wavelength)
      val resolution: monocle.Lens[SpectroscopyScienceRequirementsInput, clue.data.Input[PosInt]] = monocle.macros.GenLens[SpectroscopyScienceRequirementsInput](_.resolution)
      val signalToNoise: monocle.Lens[SpectroscopyScienceRequirementsInput, clue.data.Input[PosBigDecimal]] = monocle.macros.GenLens[SpectroscopyScienceRequirementsInput](_.signalToNoise)
      val signalToNoiseAt: monocle.Lens[SpectroscopyScienceRequirementsInput, clue.data.Input[WavelengthInput]] = monocle.macros.GenLens[SpectroscopyScienceRequirementsInput](_.signalToNoiseAt)
      val wavelengthCoverage: monocle.Lens[SpectroscopyScienceRequirementsInput, clue.data.Input[WavelengthInput]] = monocle.macros.GenLens[SpectroscopyScienceRequirementsInput](_.wavelengthCoverage)
      val focalPlane: monocle.Lens[SpectroscopyScienceRequirementsInput, clue.data.Input[FocalPlane]] = monocle.macros.GenLens[SpectroscopyScienceRequirementsInput](_.focalPlane)
      val focalPlaneAngle: monocle.Lens[SpectroscopyScienceRequirementsInput, clue.data.Input[AngleInput]] = monocle.macros.GenLens[SpectroscopyScienceRequirementsInput](_.focalPlaneAngle)
      val capability: monocle.Lens[SpectroscopyScienceRequirementsInput, clue.data.Input[SpectroscopyCapabilities]] = monocle.macros.GenLens[SpectroscopyScienceRequirementsInput](_.capability)
      implicit val eqSpectroscopyScienceRequirementsInput: cats.Eq[SpectroscopyScienceRequirementsInput] = cats.Eq.fromUniversalEquals
      implicit val showSpectroscopyScienceRequirementsInput: cats.Show[SpectroscopyScienceRequirementsInput] = cats.Show.fromToString
      implicit val jsonEncoderSpectroscopyScienceRequirementsInput: io.circe.Encoder[SpectroscopyScienceRequirementsInput] = io.circe.generic.semiauto.deriveEncoder[SpectroscopyScienceRequirementsInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class StepEventLocationInput(val observationId: ObservationId, val stepId: StepId)
    object StepEventLocationInput {
      val observationId: monocle.Lens[StepEventLocationInput, ObservationId] = monocle.macros.GenLens[StepEventLocationInput](_.observationId)
      val stepId: monocle.Lens[StepEventLocationInput, StepId] = monocle.macros.GenLens[StepEventLocationInput](_.stepId)
      implicit val eqStepEventLocationInput: cats.Eq[StepEventLocationInput] = cats.Eq.fromUniversalEquals
      implicit val showStepEventLocationInput: cats.Show[StepEventLocationInput] = cats.Show.fromToString
      implicit val jsonEncoderStepEventLocationInput: io.circe.Encoder[StepEventLocationInput] = io.circe.generic.semiauto.deriveEncoder[StepEventLocationInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class StepEventPayloadInput(val sequenceType: SequenceType, val stepStage: StepStage)
    object StepEventPayloadInput {
      val sequenceType: monocle.Lens[StepEventPayloadInput, SequenceType] = monocle.macros.GenLens[StepEventPayloadInput](_.sequenceType)
      val stepStage: monocle.Lens[StepEventPayloadInput, StepStage] = monocle.macros.GenLens[StepEventPayloadInput](_.stepStage)
      implicit val eqStepEventPayloadInput: cats.Eq[StepEventPayloadInput] = cats.Eq.fromUniversalEquals
      implicit val showStepEventPayloadInput: cats.Show[StepEventPayloadInput] = cats.Show.fromToString
      implicit val jsonEncoderStepEventPayloadInput: io.circe.Encoder[StepEventPayloadInput] = io.circe.generic.semiauto.deriveEncoder[StepEventPayloadInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class ObservationEditInput(val observationId: clue.data.Input[ObservationId] = clue.data.Ignore, val programId: clue.data.Input[ProgramId] = clue.data.Ignore)
    object ObservationEditInput {
      val observationId: monocle.Lens[ObservationEditInput, clue.data.Input[ObservationId]] = monocle.macros.GenLens[ObservationEditInput](_.observationId)
      val programId: monocle.Lens[ObservationEditInput, clue.data.Input[ProgramId]] = monocle.macros.GenLens[ObservationEditInput](_.programId)
      implicit val eqObservationEditInput: cats.Eq[ObservationEditInput] = cats.Eq.fromUniversalEquals
      implicit val showObservationEditInput: cats.Show[ObservationEditInput] = cats.Show.fromToString
      implicit val jsonEncoderObservationEditInput: io.circe.Encoder[ObservationEditInput] = io.circe.generic.semiauto.deriveEncoder[ObservationEditInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class ProgramEditInput(val programId: clue.data.Input[ProgramId] = clue.data.Ignore)
    object ProgramEditInput {
      val programId: monocle.Lens[ProgramEditInput, clue.data.Input[ProgramId]] = monocle.macros.GenLens[ProgramEditInput](_.programId)
      implicit val eqProgramEditInput: cats.Eq[ProgramEditInput] = cats.Eq.fromUniversalEquals
      implicit val showProgramEditInput: cats.Show[ProgramEditInput] = cats.Show.fromToString
      implicit val jsonEncoderProgramEditInput: io.circe.Encoder[ProgramEditInput] = io.circe.generic.semiauto.deriveEncoder[ProgramEditInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class SystemVerificationInput(val minPercentTime: clue.data.Input[IntPercent] = clue.data.Ignore)
    object SystemVerificationInput {
      val minPercentTime: monocle.Lens[SystemVerificationInput, clue.data.Input[IntPercent]] = monocle.macros.GenLens[SystemVerificationInput](_.minPercentTime)
      implicit val eqSystemVerificationInput: cats.Eq[SystemVerificationInput] = cats.Eq.fromUniversalEquals
      implicit val showSystemVerificationInput: cats.Show[SystemVerificationInput] = cats.Show.fromToString
      implicit val jsonEncoderSystemVerificationInput: io.circe.Encoder[SystemVerificationInput] = io.circe.generic.semiauto.deriveEncoder[SystemVerificationInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class TargetEnvironmentInput(val explicitBase: clue.data.Input[CoordinatesInput] = clue.data.Ignore, val asterism: clue.data.Input[List[TargetId]] = clue.data.Ignore)
    object TargetEnvironmentInput {
      val explicitBase: monocle.Lens[TargetEnvironmentInput, clue.data.Input[CoordinatesInput]] = monocle.macros.GenLens[TargetEnvironmentInput](_.explicitBase)
      val asterism: monocle.Lens[TargetEnvironmentInput, clue.data.Input[List[TargetId]]] = monocle.macros.GenLens[TargetEnvironmentInput](_.asterism)
      implicit val eqTargetEnvironmentInput: cats.Eq[TargetEnvironmentInput] = cats.Eq.fromUniversalEquals
      implicit val showTargetEnvironmentInput: cats.Show[TargetEnvironmentInput] = cats.Show.fromToString
      implicit val jsonEncoderTargetEnvironmentInput: io.circe.Encoder[TargetEnvironmentInput] = io.circe.generic.semiauto.deriveEncoder[TargetEnvironmentInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class TargetPropertiesInput(val name: clue.data.Input[NonEmptyString] = clue.data.Ignore, val sidereal: clue.data.Input[SiderealInput] = clue.data.Ignore, val nonsidereal: clue.data.Input[NonsiderealInput] = clue.data.Ignore, val sourceProfile: clue.data.Input[SourceProfileInput] = clue.data.Ignore, val existence: clue.data.Input[Existence] = clue.data.Ignore)
    object TargetPropertiesInput {
      val name: monocle.Lens[TargetPropertiesInput, clue.data.Input[NonEmptyString]] = monocle.macros.GenLens[TargetPropertiesInput](_.name)
      val sidereal: monocle.Lens[TargetPropertiesInput, clue.data.Input[SiderealInput]] = monocle.macros.GenLens[TargetPropertiesInput](_.sidereal)
      val nonsidereal: monocle.Lens[TargetPropertiesInput, clue.data.Input[NonsiderealInput]] = monocle.macros.GenLens[TargetPropertiesInput](_.nonsidereal)
      val sourceProfile: monocle.Lens[TargetPropertiesInput, clue.data.Input[SourceProfileInput]] = monocle.macros.GenLens[TargetPropertiesInput](_.sourceProfile)
      val existence: monocle.Lens[TargetPropertiesInput, clue.data.Input[Existence]] = monocle.macros.GenLens[TargetPropertiesInput](_.existence)
      implicit val eqTargetPropertiesInput: cats.Eq[TargetPropertiesInput] = cats.Eq.fromUniversalEquals
      implicit val showTargetPropertiesInput: cats.Show[TargetPropertiesInput] = cats.Show.fromToString
      implicit val jsonEncoderTargetPropertiesInput: io.circe.Encoder[TargetPropertiesInput] = io.circe.generic.semiauto.deriveEncoder[TargetPropertiesInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class UndeleteObservationsInput(val WHERE: clue.data.Input[WhereObservation] = clue.data.Ignore, val LIMIT: clue.data.Input[NonNegInt] = clue.data.Ignore)
    object UndeleteObservationsInput {
      val WHERE: monocle.Lens[UndeleteObservationsInput, clue.data.Input[WhereObservation]] = monocle.macros.GenLens[UndeleteObservationsInput](_.WHERE)
      val LIMIT: monocle.Lens[UndeleteObservationsInput, clue.data.Input[NonNegInt]] = monocle.macros.GenLens[UndeleteObservationsInput](_.LIMIT)
      implicit val eqUndeleteObservationsInput: cats.Eq[UndeleteObservationsInput] = cats.Eq.fromUniversalEquals
      implicit val showUndeleteObservationsInput: cats.Show[UndeleteObservationsInput] = cats.Show.fromToString
      implicit val jsonEncoderUndeleteObservationsInput: io.circe.Encoder[UndeleteObservationsInput] = io.circe.generic.semiauto.deriveEncoder[UndeleteObservationsInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class UndeleteTargetsInput(val WHERE: clue.data.Input[WhereTarget] = clue.data.Ignore, val LIMIT: clue.data.Input[NonNegInt] = clue.data.Ignore)
    object UndeleteTargetsInput {
      val WHERE: monocle.Lens[UndeleteTargetsInput, clue.data.Input[WhereTarget]] = monocle.macros.GenLens[UndeleteTargetsInput](_.WHERE)
      val LIMIT: monocle.Lens[UndeleteTargetsInput, clue.data.Input[NonNegInt]] = monocle.macros.GenLens[UndeleteTargetsInput](_.LIMIT)
      implicit val eqUndeleteTargetsInput: cats.Eq[UndeleteTargetsInput] = cats.Eq.fromUniversalEquals
      implicit val showUndeleteTargetsInput: cats.Show[UndeleteTargetsInput] = cats.Show.fromToString
      implicit val jsonEncoderUndeleteTargetsInput: io.circe.Encoder[UndeleteTargetsInput] = io.circe.generic.semiauto.deriveEncoder[UndeleteTargetsInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class UnnormalizedSedInput(val stellarLibrary: clue.data.Input[StellarLibrarySpectrum] = clue.data.Ignore, val coolStar: clue.data.Input[CoolStarTemperature] = clue.data.Ignore, val galaxy: clue.data.Input[GalaxySpectrum] = clue.data.Ignore, val planet: clue.data.Input[PlanetSpectrum] = clue.data.Ignore, val quasar: clue.data.Input[QuasarSpectrum] = clue.data.Ignore, val hiiRegion: clue.data.Input[HiiRegionSpectrum] = clue.data.Ignore, val planetaryNebula: clue.data.Input[PlanetaryNebulaSpectrum] = clue.data.Ignore, val powerLaw: clue.data.Input[BigDecimal] = clue.data.Ignore, val blackBodyTempK: clue.data.Input[PosInt] = clue.data.Ignore, val fluxDensities: clue.data.Input[List[FluxDensity]] = clue.data.Ignore)
    object UnnormalizedSedInput {
      val stellarLibrary: monocle.Lens[UnnormalizedSedInput, clue.data.Input[StellarLibrarySpectrum]] = monocle.macros.GenLens[UnnormalizedSedInput](_.stellarLibrary)
      val coolStar: monocle.Lens[UnnormalizedSedInput, clue.data.Input[CoolStarTemperature]] = monocle.macros.GenLens[UnnormalizedSedInput](_.coolStar)
      val galaxy: monocle.Lens[UnnormalizedSedInput, clue.data.Input[GalaxySpectrum]] = monocle.macros.GenLens[UnnormalizedSedInput](_.galaxy)
      val planet: monocle.Lens[UnnormalizedSedInput, clue.data.Input[PlanetSpectrum]] = monocle.macros.GenLens[UnnormalizedSedInput](_.planet)
      val quasar: monocle.Lens[UnnormalizedSedInput, clue.data.Input[QuasarSpectrum]] = monocle.macros.GenLens[UnnormalizedSedInput](_.quasar)
      val hiiRegion: monocle.Lens[UnnormalizedSedInput, clue.data.Input[HiiRegionSpectrum]] = monocle.macros.GenLens[UnnormalizedSedInput](_.hiiRegion)
      val planetaryNebula: monocle.Lens[UnnormalizedSedInput, clue.data.Input[PlanetaryNebulaSpectrum]] = monocle.macros.GenLens[UnnormalizedSedInput](_.planetaryNebula)
      val powerLaw: monocle.Lens[UnnormalizedSedInput, clue.data.Input[BigDecimal]] = monocle.macros.GenLens[UnnormalizedSedInput](_.powerLaw)
      val blackBodyTempK: monocle.Lens[UnnormalizedSedInput, clue.data.Input[PosInt]] = monocle.macros.GenLens[UnnormalizedSedInput](_.blackBodyTempK)
      val fluxDensities: monocle.Lens[UnnormalizedSedInput, clue.data.Input[List[FluxDensity]]] = monocle.macros.GenLens[UnnormalizedSedInput](_.fluxDensities)
      implicit val eqUnnormalizedSedInput: cats.Eq[UnnormalizedSedInput] = cats.Eq.fromUniversalEquals
      implicit val showUnnormalizedSedInput: cats.Show[UnnormalizedSedInput] = cats.Show.fromToString
      implicit val jsonEncoderUnnormalizedSedInput: io.circe.Encoder[UnnormalizedSedInput] = io.circe.generic.semiauto.deriveEncoder[UnnormalizedSedInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class UpdateAsterismsInput(val programId: ProgramId, val SET: EditAsterismsPatchInput, val WHERE: clue.data.Input[WhereObservation] = clue.data.Ignore, val LIMIT: clue.data.Input[NonNegInt] = clue.data.Ignore, val includeDeleted: clue.data.Input[Boolean] = clue.data.Ignore)
    object UpdateAsterismsInput {
      val programId: monocle.Lens[UpdateAsterismsInput, ProgramId] = monocle.macros.GenLens[UpdateAsterismsInput](_.programId)
      val SET: monocle.Lens[UpdateAsterismsInput, EditAsterismsPatchInput] = monocle.macros.GenLens[UpdateAsterismsInput](_.SET)
      val WHERE: monocle.Lens[UpdateAsterismsInput, clue.data.Input[WhereObservation]] = monocle.macros.GenLens[UpdateAsterismsInput](_.WHERE)
      val LIMIT: monocle.Lens[UpdateAsterismsInput, clue.data.Input[NonNegInt]] = monocle.macros.GenLens[UpdateAsterismsInput](_.LIMIT)
      val includeDeleted: monocle.Lens[UpdateAsterismsInput, clue.data.Input[Boolean]] = monocle.macros.GenLens[UpdateAsterismsInput](_.includeDeleted)
      implicit val eqUpdateAsterismsInput: cats.Eq[UpdateAsterismsInput] = cats.Eq.fromUniversalEquals
      implicit val showUpdateAsterismsInput: cats.Show[UpdateAsterismsInput] = cats.Show.fromToString
      implicit val jsonEncoderUpdateAsterismsInput: io.circe.Encoder[UpdateAsterismsInput] = io.circe.generic.semiauto.deriveEncoder[UpdateAsterismsInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class UpdateDatasetsInput(val SET: DatasetPropertiesInput, val WHERE: clue.data.Input[WhereDataset] = clue.data.Ignore, val LIMIT: clue.data.Input[NonNegInt] = clue.data.Ignore)
    object UpdateDatasetsInput {
      val SET: monocle.Lens[UpdateDatasetsInput, DatasetPropertiesInput] = monocle.macros.GenLens[UpdateDatasetsInput](_.SET)
      val WHERE: monocle.Lens[UpdateDatasetsInput, clue.data.Input[WhereDataset]] = monocle.macros.GenLens[UpdateDatasetsInput](_.WHERE)
      val LIMIT: monocle.Lens[UpdateDatasetsInput, clue.data.Input[NonNegInt]] = monocle.macros.GenLens[UpdateDatasetsInput](_.LIMIT)
      implicit val eqUpdateDatasetsInput: cats.Eq[UpdateDatasetsInput] = cats.Eq.fromUniversalEquals
      implicit val showUpdateDatasetsInput: cats.Show[UpdateDatasetsInput] = cats.Show.fromToString
      implicit val jsonEncoderUpdateDatasetsInput: io.circe.Encoder[UpdateDatasetsInput] = io.circe.generic.semiauto.deriveEncoder[UpdateDatasetsInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class UpdateObservationsInput(val programId: ProgramId, val SET: ObservationPropertiesInput, val WHERE: clue.data.Input[WhereObservation] = clue.data.Ignore, val LIMIT: clue.data.Input[NonNegInt] = clue.data.Ignore, val includeDeleted: clue.data.Input[Boolean] = clue.data.Ignore)
    object UpdateObservationsInput {
      val programId: monocle.Lens[UpdateObservationsInput, ProgramId] = monocle.macros.GenLens[UpdateObservationsInput](_.programId)
      val SET: monocle.Lens[UpdateObservationsInput, ObservationPropertiesInput] = monocle.macros.GenLens[UpdateObservationsInput](_.SET)
      val WHERE: monocle.Lens[UpdateObservationsInput, clue.data.Input[WhereObservation]] = monocle.macros.GenLens[UpdateObservationsInput](_.WHERE)
      val LIMIT: monocle.Lens[UpdateObservationsInput, clue.data.Input[NonNegInt]] = monocle.macros.GenLens[UpdateObservationsInput](_.LIMIT)
      val includeDeleted: monocle.Lens[UpdateObservationsInput, clue.data.Input[Boolean]] = monocle.macros.GenLens[UpdateObservationsInput](_.includeDeleted)
      implicit val eqUpdateObservationsInput: cats.Eq[UpdateObservationsInput] = cats.Eq.fromUniversalEquals
      implicit val showUpdateObservationsInput: cats.Show[UpdateObservationsInput] = cats.Show.fromToString
      implicit val jsonEncoderUpdateObservationsInput: io.circe.Encoder[UpdateObservationsInput] = io.circe.generic.semiauto.deriveEncoder[UpdateObservationsInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class UpdateProgramsInput(val SET: ProgramPropertiesInput, val WHERE: clue.data.Input[WhereProgram] = clue.data.Ignore, val LIMIT: clue.data.Input[NonNegInt] = clue.data.Ignore, val includeDeleted: clue.data.Input[Boolean] = clue.data.Ignore)
    object UpdateProgramsInput {
      val SET: monocle.Lens[UpdateProgramsInput, ProgramPropertiesInput] = monocle.macros.GenLens[UpdateProgramsInput](_.SET)
      val WHERE: monocle.Lens[UpdateProgramsInput, clue.data.Input[WhereProgram]] = monocle.macros.GenLens[UpdateProgramsInput](_.WHERE)
      val LIMIT: monocle.Lens[UpdateProgramsInput, clue.data.Input[NonNegInt]] = monocle.macros.GenLens[UpdateProgramsInput](_.LIMIT)
      val includeDeleted: monocle.Lens[UpdateProgramsInput, clue.data.Input[Boolean]] = monocle.macros.GenLens[UpdateProgramsInput](_.includeDeleted)
      implicit val eqUpdateProgramsInput: cats.Eq[UpdateProgramsInput] = cats.Eq.fromUniversalEquals
      implicit val showUpdateProgramsInput: cats.Show[UpdateProgramsInput] = cats.Show.fromToString
      implicit val jsonEncoderUpdateProgramsInput: io.circe.Encoder[UpdateProgramsInput] = io.circe.generic.semiauto.deriveEncoder[UpdateProgramsInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class UpdateTargetsInput(val SET: TargetPropertiesInput, val WHERE: clue.data.Input[WhereTarget] = clue.data.Ignore, val LIMIT: clue.data.Input[NonNegInt] = clue.data.Ignore, val includeDeleted: clue.data.Input[Boolean] = clue.data.Ignore)
    object UpdateTargetsInput {
      val SET: monocle.Lens[UpdateTargetsInput, TargetPropertiesInput] = monocle.macros.GenLens[UpdateTargetsInput](_.SET)
      val WHERE: monocle.Lens[UpdateTargetsInput, clue.data.Input[WhereTarget]] = monocle.macros.GenLens[UpdateTargetsInput](_.WHERE)
      val LIMIT: monocle.Lens[UpdateTargetsInput, clue.data.Input[NonNegInt]] = monocle.macros.GenLens[UpdateTargetsInput](_.LIMIT)
      val includeDeleted: monocle.Lens[UpdateTargetsInput, clue.data.Input[Boolean]] = monocle.macros.GenLens[UpdateTargetsInput](_.includeDeleted)
      implicit val eqUpdateTargetsInput: cats.Eq[UpdateTargetsInput] = cats.Eq.fromUniversalEquals
      implicit val showUpdateTargetsInput: cats.Show[UpdateTargetsInput] = cats.Show.fromToString
      implicit val jsonEncoderUpdateTargetsInput: io.circe.Encoder[UpdateTargetsInput] = io.circe.generic.semiauto.deriveEncoder[UpdateTargetsInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class WavelengthInput(val picometers: clue.data.Input[PosInt] = clue.data.Ignore, val angstroms: clue.data.Input[PosBigDecimal] = clue.data.Ignore, val nanometers: clue.data.Input[PosBigDecimal] = clue.data.Ignore, val micrometers: clue.data.Input[PosBigDecimal] = clue.data.Ignore)
    object WavelengthInput {
      val picometers: monocle.Lens[WavelengthInput, clue.data.Input[PosInt]] = monocle.macros.GenLens[WavelengthInput](_.picometers)
      val angstroms: monocle.Lens[WavelengthInput, clue.data.Input[PosBigDecimal]] = monocle.macros.GenLens[WavelengthInput](_.angstroms)
      val nanometers: monocle.Lens[WavelengthInput, clue.data.Input[PosBigDecimal]] = monocle.macros.GenLens[WavelengthInput](_.nanometers)
      val micrometers: monocle.Lens[WavelengthInput, clue.data.Input[PosBigDecimal]] = monocle.macros.GenLens[WavelengthInput](_.micrometers)
      implicit val eqWavelengthInput: cats.Eq[WavelengthInput] = cats.Eq.fromUniversalEquals
      implicit val showWavelengthInput: cats.Show[WavelengthInput] = cats.Show.fromToString
      implicit val jsonEncoderWavelengthInput: io.circe.Encoder[WavelengthInput] = io.circe.generic.semiauto.deriveEncoder[WavelengthInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class WavelengthDitherInput(val picometers: clue.data.Input[Int] = clue.data.Ignore, val angstroms: clue.data.Input[BigDecimal] = clue.data.Ignore, val nanometers: clue.data.Input[BigDecimal] = clue.data.Ignore, val micrometers: clue.data.Input[BigDecimal] = clue.data.Ignore)
    object WavelengthDitherInput {
      val picometers: monocle.Lens[WavelengthDitherInput, clue.data.Input[Int]] = monocle.macros.GenLens[WavelengthDitherInput](_.picometers)
      val angstroms: monocle.Lens[WavelengthDitherInput, clue.data.Input[BigDecimal]] = monocle.macros.GenLens[WavelengthDitherInput](_.angstroms)
      val nanometers: monocle.Lens[WavelengthDitherInput, clue.data.Input[BigDecimal]] = monocle.macros.GenLens[WavelengthDitherInput](_.nanometers)
      val micrometers: monocle.Lens[WavelengthDitherInput, clue.data.Input[BigDecimal]] = monocle.macros.GenLens[WavelengthDitherInput](_.micrometers)
      implicit val eqWavelengthDitherInput: cats.Eq[WavelengthDitherInput] = cats.Eq.fromUniversalEquals
      implicit val showWavelengthDitherInput: cats.Show[WavelengthDitherInput] = cats.Show.fromToString
      implicit val jsonEncoderWavelengthDitherInput: io.circe.Encoder[WavelengthDitherInput] = io.circe.generic.semiauto.deriveEncoder[WavelengthDitherInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class DatasetIdInput(val observationId: ObservationId, val stepId: StepId, val index: PosInt)
    object DatasetIdInput {
      val observationId: monocle.Lens[DatasetIdInput, ObservationId] = monocle.macros.GenLens[DatasetIdInput](_.observationId)
      val stepId: monocle.Lens[DatasetIdInput, StepId] = monocle.macros.GenLens[DatasetIdInput](_.stepId)
      val index: monocle.Lens[DatasetIdInput, PosInt] = monocle.macros.GenLens[DatasetIdInput](_.index)
      implicit val eqDatasetIdInput: cats.Eq[DatasetIdInput] = cats.Eq.fromUniversalEquals
      implicit val showDatasetIdInput: cats.Show[DatasetIdInput] = cats.Show.fromToString
      implicit val jsonEncoderDatasetIdInput: io.circe.Encoder[DatasetIdInput] = io.circe.generic.semiauto.deriveEncoder[DatasetIdInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class TimeSpanInput(val microseconds: clue.data.Input[Long] = clue.data.Ignore, val milliseconds: clue.data.Input[BigDecimal] = clue.data.Ignore, val seconds: clue.data.Input[BigDecimal] = clue.data.Ignore, val minutes: clue.data.Input[BigDecimal] = clue.data.Ignore, val hours: clue.data.Input[BigDecimal] = clue.data.Ignore, val iso: clue.data.Input[String] = clue.data.Ignore)
    object TimeSpanInput {
      val microseconds: monocle.Lens[TimeSpanInput, clue.data.Input[Long]] = monocle.macros.GenLens[TimeSpanInput](_.microseconds)
      val milliseconds: monocle.Lens[TimeSpanInput, clue.data.Input[BigDecimal]] = monocle.macros.GenLens[TimeSpanInput](_.milliseconds)
      val seconds: monocle.Lens[TimeSpanInput, clue.data.Input[BigDecimal]] = monocle.macros.GenLens[TimeSpanInput](_.seconds)
      val minutes: monocle.Lens[TimeSpanInput, clue.data.Input[BigDecimal]] = monocle.macros.GenLens[TimeSpanInput](_.minutes)
      val hours: monocle.Lens[TimeSpanInput, clue.data.Input[BigDecimal]] = monocle.macros.GenLens[TimeSpanInput](_.hours)
      val iso: monocle.Lens[TimeSpanInput, clue.data.Input[String]] = monocle.macros.GenLens[TimeSpanInput](_.iso)
      implicit val eqTimeSpanInput: cats.Eq[TimeSpanInput] = cats.Eq.fromUniversalEquals
      implicit val showTimeSpanInput: cats.Show[TimeSpanInput] = cats.Show.fromToString
      implicit val jsonEncoderTimeSpanInput: io.circe.Encoder[TimeSpanInput] = io.circe.generic.semiauto.deriveEncoder[TimeSpanInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class UnlinkUserInput(val programId: ProgramId, val userId: UserId)
    object UnlinkUserInput {
      val programId: monocle.Lens[UnlinkUserInput, ProgramId] = monocle.macros.GenLens[UnlinkUserInput](_.programId)
      val userId: monocle.Lens[UnlinkUserInput, UserId] = monocle.macros.GenLens[UnlinkUserInput](_.userId)
      implicit val eqUnlinkUserInput: cats.Eq[UnlinkUserInput] = cats.Eq.fromUniversalEquals
      implicit val showUnlinkUserInput: cats.Show[UnlinkUserInput] = cats.Show.fromToString
      implicit val jsonEncoderUnlinkUserInput: io.circe.Encoder[UnlinkUserInput] = io.circe.generic.semiauto.deriveEncoder[UnlinkUserInput].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class WhereDataset(val observationId: clue.data.Input[WhereOrderObservationId] = clue.data.Ignore, val stepId: clue.data.Input[WhereEqStepId] = clue.data.Ignore, val index: clue.data.Input[WhereOrderDatasetIndex] = clue.data.Ignore, val filename: clue.data.Input[WhereString] = clue.data.Ignore, val qaState: clue.data.Input[WhereOptionEqQaState] = clue.data.Ignore)
    object WhereDataset {
      val observationId: monocle.Lens[WhereDataset, clue.data.Input[WhereOrderObservationId]] = monocle.macros.GenLens[WhereDataset](_.observationId)
      val stepId: monocle.Lens[WhereDataset, clue.data.Input[WhereEqStepId]] = monocle.macros.GenLens[WhereDataset](_.stepId)
      val index: monocle.Lens[WhereDataset, clue.data.Input[WhereOrderDatasetIndex]] = monocle.macros.GenLens[WhereDataset](_.index)
      val filename: monocle.Lens[WhereDataset, clue.data.Input[WhereString]] = monocle.macros.GenLens[WhereDataset](_.filename)
      val qaState: monocle.Lens[WhereDataset, clue.data.Input[WhereOptionEqQaState]] = monocle.macros.GenLens[WhereDataset](_.qaState)
      implicit val eqWhereDataset: cats.Eq[WhereDataset] = cats.Eq.fromUniversalEquals
      implicit val showWhereDataset: cats.Show[WhereDataset] = cats.Show.fromToString
      implicit val jsonEncoderWhereDataset: io.circe.Encoder[WhereDataset] = io.circe.generic.semiauto.deriveEncoder[WhereDataset].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class WhereDatasetEvent(val stepId: clue.data.Input[WhereEqStepId] = clue.data.Ignore, val index: clue.data.Input[WhereOrderDatasetIndex] = clue.data.Ignore, val stage: clue.data.Input[WhereOrderDatasetStage] = clue.data.Ignore, val filename: clue.data.Input[WhereOptionString] = clue.data.Ignore)
    object WhereDatasetEvent {
      val stepId: monocle.Lens[WhereDatasetEvent, clue.data.Input[WhereEqStepId]] = monocle.macros.GenLens[WhereDatasetEvent](_.stepId)
      val index: monocle.Lens[WhereDatasetEvent, clue.data.Input[WhereOrderDatasetIndex]] = monocle.macros.GenLens[WhereDatasetEvent](_.index)
      val stage: monocle.Lens[WhereDatasetEvent, clue.data.Input[WhereOrderDatasetStage]] = monocle.macros.GenLens[WhereDatasetEvent](_.stage)
      val filename: monocle.Lens[WhereDatasetEvent, clue.data.Input[WhereOptionString]] = monocle.macros.GenLens[WhereDatasetEvent](_.filename)
      implicit val eqWhereDatasetEvent: cats.Eq[WhereDatasetEvent] = cats.Eq.fromUniversalEquals
      implicit val showWhereDatasetEvent: cats.Show[WhereDatasetEvent] = cats.Show.fromToString
      implicit val jsonEncoderWhereDatasetEvent: io.circe.Encoder[WhereDatasetEvent] = io.circe.generic.semiauto.deriveEncoder[WhereDatasetEvent].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class WhereEqPartner(val EQ: clue.data.Input[Partner] = clue.data.Ignore, val NEQ: clue.data.Input[Partner] = clue.data.Ignore, val IN: clue.data.Input[List[Partner]] = clue.data.Ignore, val NIN: clue.data.Input[List[Partner]] = clue.data.Ignore)
    object WhereEqPartner {
      val EQ: monocle.Lens[WhereEqPartner, clue.data.Input[Partner]] = monocle.macros.GenLens[WhereEqPartner](_.EQ)
      val NEQ: monocle.Lens[WhereEqPartner, clue.data.Input[Partner]] = monocle.macros.GenLens[WhereEqPartner](_.NEQ)
      val IN: monocle.Lens[WhereEqPartner, clue.data.Input[List[Partner]]] = monocle.macros.GenLens[WhereEqPartner](_.IN)
      val NIN: monocle.Lens[WhereEqPartner, clue.data.Input[List[Partner]]] = monocle.macros.GenLens[WhereEqPartner](_.NIN)
      implicit val eqWhereEqPartner: cats.Eq[WhereEqPartner] = cats.Eq.fromUniversalEquals
      implicit val showWhereEqPartner: cats.Show[WhereEqPartner] = cats.Show.fromToString
      implicit val jsonEncoderWhereEqPartner: io.circe.Encoder[WhereEqPartner] = io.circe.generic.semiauto.deriveEncoder[WhereEqPartner].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class WhereEqProposalClassType(val EQ: clue.data.Input[ProposalClassEnum] = clue.data.Ignore, val NEQ: clue.data.Input[ProposalClassEnum] = clue.data.Ignore, val IN: clue.data.Input[List[ProposalClassEnum]] = clue.data.Ignore, val NIN: clue.data.Input[List[ProposalClassEnum]] = clue.data.Ignore)
    object WhereEqProposalClassType {
      val EQ: monocle.Lens[WhereEqProposalClassType, clue.data.Input[ProposalClassEnum]] = monocle.macros.GenLens[WhereEqProposalClassType](_.EQ)
      val NEQ: monocle.Lens[WhereEqProposalClassType, clue.data.Input[ProposalClassEnum]] = monocle.macros.GenLens[WhereEqProposalClassType](_.NEQ)
      val IN: monocle.Lens[WhereEqProposalClassType, clue.data.Input[List[ProposalClassEnum]]] = monocle.macros.GenLens[WhereEqProposalClassType](_.IN)
      val NIN: monocle.Lens[WhereEqProposalClassType, clue.data.Input[List[ProposalClassEnum]]] = monocle.macros.GenLens[WhereEqProposalClassType](_.NIN)
      implicit val eqWhereEqProposalClassType: cats.Eq[WhereEqProposalClassType] = cats.Eq.fromUniversalEquals
      implicit val showWhereEqProposalClassType: cats.Show[WhereEqProposalClassType] = cats.Show.fromToString
      implicit val jsonEncoderWhereEqProposalClassType: io.circe.Encoder[WhereEqProposalClassType] = io.circe.generic.semiauto.deriveEncoder[WhereEqProposalClassType].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class WhereEqStepId(val EQ: clue.data.Input[StepId] = clue.data.Ignore, val NEQ: clue.data.Input[StepId] = clue.data.Ignore, val IN: clue.data.Input[List[StepId]] = clue.data.Ignore, val NIN: clue.data.Input[List[StepId]] = clue.data.Ignore)
    object WhereEqStepId {
      val EQ: monocle.Lens[WhereEqStepId, clue.data.Input[StepId]] = monocle.macros.GenLens[WhereEqStepId](_.EQ)
      val NEQ: monocle.Lens[WhereEqStepId, clue.data.Input[StepId]] = monocle.macros.GenLens[WhereEqStepId](_.NEQ)
      val IN: monocle.Lens[WhereEqStepId, clue.data.Input[List[StepId]]] = monocle.macros.GenLens[WhereEqStepId](_.IN)
      val NIN: monocle.Lens[WhereEqStepId, clue.data.Input[List[StepId]]] = monocle.macros.GenLens[WhereEqStepId](_.NIN)
      implicit val eqWhereEqStepId: cats.Eq[WhereEqStepId] = cats.Eq.fromUniversalEquals
      implicit val showWhereEqStepId: cats.Show[WhereEqStepId] = cats.Show.fromToString
      implicit val jsonEncoderWhereEqStepId: io.circe.Encoder[WhereEqStepId] = io.circe.generic.semiauto.deriveEncoder[WhereEqStepId].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class WhereEqToOActivation(val EQ: clue.data.Input[ToOActivation] = clue.data.Ignore, val NEQ: clue.data.Input[ToOActivation] = clue.data.Ignore, val IN: clue.data.Input[List[ToOActivation]] = clue.data.Ignore, val NIN: clue.data.Input[List[ToOActivation]] = clue.data.Ignore)
    object WhereEqToOActivation {
      val EQ: monocle.Lens[WhereEqToOActivation, clue.data.Input[ToOActivation]] = monocle.macros.GenLens[WhereEqToOActivation](_.EQ)
      val NEQ: monocle.Lens[WhereEqToOActivation, clue.data.Input[ToOActivation]] = monocle.macros.GenLens[WhereEqToOActivation](_.NEQ)
      val IN: monocle.Lens[WhereEqToOActivation, clue.data.Input[List[ToOActivation]]] = monocle.macros.GenLens[WhereEqToOActivation](_.IN)
      val NIN: monocle.Lens[WhereEqToOActivation, clue.data.Input[List[ToOActivation]]] = monocle.macros.GenLens[WhereEqToOActivation](_.NIN)
      implicit val eqWhereEqToOActivation: cats.Eq[WhereEqToOActivation] = cats.Eq.fromUniversalEquals
      implicit val showWhereEqToOActivation: cats.Show[WhereEqToOActivation] = cats.Show.fromToString
      implicit val jsonEncoderWhereEqToOActivation: io.circe.Encoder[WhereEqToOActivation] = io.circe.generic.semiauto.deriveEncoder[WhereEqToOActivation].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class WhereEqVisitId(val EQ: clue.data.Input[VisitId] = clue.data.Ignore, val NEQ: clue.data.Input[VisitId] = clue.data.Ignore, val IN: clue.data.Input[List[VisitId]] = clue.data.Ignore, val NIN: clue.data.Input[List[VisitId]] = clue.data.Ignore)
    object WhereEqVisitId {
      val EQ: monocle.Lens[WhereEqVisitId, clue.data.Input[VisitId]] = monocle.macros.GenLens[WhereEqVisitId](_.EQ)
      val NEQ: monocle.Lens[WhereEqVisitId, clue.data.Input[VisitId]] = monocle.macros.GenLens[WhereEqVisitId](_.NEQ)
      val IN: monocle.Lens[WhereEqVisitId, clue.data.Input[List[VisitId]]] = monocle.macros.GenLens[WhereEqVisitId](_.IN)
      val NIN: monocle.Lens[WhereEqVisitId, clue.data.Input[List[VisitId]]] = monocle.macros.GenLens[WhereEqVisitId](_.NIN)
      implicit val eqWhereEqVisitId: cats.Eq[WhereEqVisitId] = cats.Eq.fromUniversalEquals
      implicit val showWhereEqVisitId: cats.Show[WhereEqVisitId] = cats.Show.fromToString
      implicit val jsonEncoderWhereEqVisitId: io.circe.Encoder[WhereEqVisitId] = io.circe.generic.semiauto.deriveEncoder[WhereEqVisitId].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class WhereExecutionEvent(val AND: clue.data.Input[List[WhereExecutionEvent]] = clue.data.Ignore, val OR: clue.data.Input[List[WhereExecutionEvent]] = clue.data.Ignore, val NOT: clue.data.Input[WhereExecutionEvent] = clue.data.Ignore, val id: clue.data.Input[WhereOrderExecutionEventId] = clue.data.Ignore, val visitId: clue.data.Input[WhereEqVisitId] = clue.data.Ignore, val observationId: clue.data.Input[WhereOrderObservationId] = clue.data.Ignore, val received: clue.data.Input[WhereOrderTimestamp] = clue.data.Ignore, val sequenceEvent: clue.data.Input[WhereSequenceEvent] = clue.data.Ignore, val stepEvent: clue.data.Input[WhereStepEvent] = clue.data.Ignore, val datasetEvent: clue.data.Input[WhereDatasetEvent] = clue.data.Ignore)
    object WhereExecutionEvent {
      val AND: monocle.Lens[WhereExecutionEvent, clue.data.Input[List[WhereExecutionEvent]]] = monocle.macros.GenLens[WhereExecutionEvent](_.AND)
      val OR: monocle.Lens[WhereExecutionEvent, clue.data.Input[List[WhereExecutionEvent]]] = monocle.macros.GenLens[WhereExecutionEvent](_.OR)
      val NOT: monocle.Lens[WhereExecutionEvent, clue.data.Input[WhereExecutionEvent]] = monocle.macros.GenLens[WhereExecutionEvent](_.NOT)
      val id: monocle.Lens[WhereExecutionEvent, clue.data.Input[WhereOrderExecutionEventId]] = monocle.macros.GenLens[WhereExecutionEvent](_.id)
      val visitId: monocle.Lens[WhereExecutionEvent, clue.data.Input[WhereEqVisitId]] = monocle.macros.GenLens[WhereExecutionEvent](_.visitId)
      val observationId: monocle.Lens[WhereExecutionEvent, clue.data.Input[WhereOrderObservationId]] = monocle.macros.GenLens[WhereExecutionEvent](_.observationId)
      val received: monocle.Lens[WhereExecutionEvent, clue.data.Input[WhereOrderTimestamp]] = monocle.macros.GenLens[WhereExecutionEvent](_.received)
      val sequenceEvent: monocle.Lens[WhereExecutionEvent, clue.data.Input[WhereSequenceEvent]] = monocle.macros.GenLens[WhereExecutionEvent](_.sequenceEvent)
      val stepEvent: monocle.Lens[WhereExecutionEvent, clue.data.Input[WhereStepEvent]] = monocle.macros.GenLens[WhereExecutionEvent](_.stepEvent)
      val datasetEvent: monocle.Lens[WhereExecutionEvent, clue.data.Input[WhereDatasetEvent]] = monocle.macros.GenLens[WhereExecutionEvent](_.datasetEvent)
      implicit val eqWhereExecutionEvent: cats.Eq[WhereExecutionEvent] = cats.Eq.fromUniversalEquals
      implicit val showWhereExecutionEvent: cats.Show[WhereExecutionEvent] = cats.Show.fromToString
      implicit val jsonEncoderWhereExecutionEvent: io.circe.Encoder[WhereExecutionEvent] = io.circe.generic.semiauto.deriveEncoder[WhereExecutionEvent].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class WhereObservation(val AND: clue.data.Input[List[WhereObservation]] = clue.data.Ignore, val OR: clue.data.Input[List[WhereObservation]] = clue.data.Ignore, val NOT: clue.data.Input[WhereObservation] = clue.data.Ignore, val id: clue.data.Input[WhereOrderObservationId] = clue.data.Ignore, val subtitle: clue.data.Input[WhereOptionString] = clue.data.Ignore, val status: clue.data.Input[WhereOrderObsStatus] = clue.data.Ignore, val activeStatus: clue.data.Input[WhereOrderObsActiveStatus] = clue.data.Ignore)
    object WhereObservation {
      val AND: monocle.Lens[WhereObservation, clue.data.Input[List[WhereObservation]]] = monocle.macros.GenLens[WhereObservation](_.AND)
      val OR: monocle.Lens[WhereObservation, clue.data.Input[List[WhereObservation]]] = monocle.macros.GenLens[WhereObservation](_.OR)
      val NOT: monocle.Lens[WhereObservation, clue.data.Input[WhereObservation]] = monocle.macros.GenLens[WhereObservation](_.NOT)
      val id: monocle.Lens[WhereObservation, clue.data.Input[WhereOrderObservationId]] = monocle.macros.GenLens[WhereObservation](_.id)
      val subtitle: monocle.Lens[WhereObservation, clue.data.Input[WhereOptionString]] = monocle.macros.GenLens[WhereObservation](_.subtitle)
      val status: monocle.Lens[WhereObservation, clue.data.Input[WhereOrderObsStatus]] = monocle.macros.GenLens[WhereObservation](_.status)
      val activeStatus: monocle.Lens[WhereObservation, clue.data.Input[WhereOrderObsActiveStatus]] = monocle.macros.GenLens[WhereObservation](_.activeStatus)
      implicit val eqWhereObservation: cats.Eq[WhereObservation] = cats.Eq.fromUniversalEquals
      implicit val showWhereObservation: cats.Show[WhereObservation] = cats.Show.fromToString
      implicit val jsonEncoderWhereObservation: io.circe.Encoder[WhereObservation] = io.circe.generic.semiauto.deriveEncoder[WhereObservation].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class WhereOptionEqQaState(val IS_NULL: clue.data.Input[Boolean] = clue.data.Ignore, val EQ: clue.data.Input[DatasetQaState] = clue.data.Ignore, val NEQ: clue.data.Input[DatasetQaState] = clue.data.Ignore, val IN: clue.data.Input[List[DatasetQaState]] = clue.data.Ignore, val NIN: clue.data.Input[List[DatasetQaState]] = clue.data.Ignore)
    object WhereOptionEqQaState {
      val IS_NULL: monocle.Lens[WhereOptionEqQaState, clue.data.Input[Boolean]] = monocle.macros.GenLens[WhereOptionEqQaState](_.IS_NULL)
      val EQ: monocle.Lens[WhereOptionEqQaState, clue.data.Input[DatasetQaState]] = monocle.macros.GenLens[WhereOptionEqQaState](_.EQ)
      val NEQ: monocle.Lens[WhereOptionEqQaState, clue.data.Input[DatasetQaState]] = monocle.macros.GenLens[WhereOptionEqQaState](_.NEQ)
      val IN: monocle.Lens[WhereOptionEqQaState, clue.data.Input[List[DatasetQaState]]] = monocle.macros.GenLens[WhereOptionEqQaState](_.IN)
      val NIN: monocle.Lens[WhereOptionEqQaState, clue.data.Input[List[DatasetQaState]]] = monocle.macros.GenLens[WhereOptionEqQaState](_.NIN)
      implicit val eqWhereOptionEqQaState: cats.Eq[WhereOptionEqQaState] = cats.Eq.fromUniversalEquals
      implicit val showWhereOptionEqQaState: cats.Show[WhereOptionEqQaState] = cats.Show.fromToString
      implicit val jsonEncoderWhereOptionEqQaState: io.circe.Encoder[WhereOptionEqQaState] = io.circe.generic.semiauto.deriveEncoder[WhereOptionEqQaState].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class WhereOptionEqTacCategory(val IS_NULL: clue.data.Input[Boolean] = clue.data.Ignore, val EQ: clue.data.Input[TacCategory] = clue.data.Ignore, val NEQ: clue.data.Input[TacCategory] = clue.data.Ignore, val IN: clue.data.Input[List[TacCategory]] = clue.data.Ignore, val NIN: clue.data.Input[List[TacCategory]] = clue.data.Ignore)
    object WhereOptionEqTacCategory {
      val IS_NULL: monocle.Lens[WhereOptionEqTacCategory, clue.data.Input[Boolean]] = monocle.macros.GenLens[WhereOptionEqTacCategory](_.IS_NULL)
      val EQ: monocle.Lens[WhereOptionEqTacCategory, clue.data.Input[TacCategory]] = monocle.macros.GenLens[WhereOptionEqTacCategory](_.EQ)
      val NEQ: monocle.Lens[WhereOptionEqTacCategory, clue.data.Input[TacCategory]] = monocle.macros.GenLens[WhereOptionEqTacCategory](_.NEQ)
      val IN: monocle.Lens[WhereOptionEqTacCategory, clue.data.Input[List[TacCategory]]] = monocle.macros.GenLens[WhereOptionEqTacCategory](_.IN)
      val NIN: monocle.Lens[WhereOptionEqTacCategory, clue.data.Input[List[TacCategory]]] = monocle.macros.GenLens[WhereOptionEqTacCategory](_.NIN)
      implicit val eqWhereOptionEqTacCategory: cats.Eq[WhereOptionEqTacCategory] = cats.Eq.fromUniversalEquals
      implicit val showWhereOptionEqTacCategory: cats.Show[WhereOptionEqTacCategory] = cats.Show.fromToString
      implicit val jsonEncoderWhereOptionEqTacCategory: io.circe.Encoder[WhereOptionEqTacCategory] = io.circe.generic.semiauto.deriveEncoder[WhereOptionEqTacCategory].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class WhereOptionString(val IS_NULL: clue.data.Input[Boolean] = clue.data.Ignore, val EQ: clue.data.Input[NonEmptyString] = clue.data.Ignore, val NEQ: clue.data.Input[NonEmptyString] = clue.data.Ignore, val IN: clue.data.Input[List[NonEmptyString]] = clue.data.Ignore, val NIN: clue.data.Input[List[NonEmptyString]] = clue.data.Ignore, val LIKE: clue.data.Input[NonEmptyString] = clue.data.Ignore, val NLIKE: clue.data.Input[NonEmptyString] = clue.data.Ignore, val MATCH_CASE: clue.data.Input[Boolean] = clue.data.Ignore)
    object WhereOptionString {
      val IS_NULL: monocle.Lens[WhereOptionString, clue.data.Input[Boolean]] = monocle.macros.GenLens[WhereOptionString](_.IS_NULL)
      val EQ: monocle.Lens[WhereOptionString, clue.data.Input[NonEmptyString]] = monocle.macros.GenLens[WhereOptionString](_.EQ)
      val NEQ: monocle.Lens[WhereOptionString, clue.data.Input[NonEmptyString]] = monocle.macros.GenLens[WhereOptionString](_.NEQ)
      val IN: monocle.Lens[WhereOptionString, clue.data.Input[List[NonEmptyString]]] = monocle.macros.GenLens[WhereOptionString](_.IN)
      val NIN: monocle.Lens[WhereOptionString, clue.data.Input[List[NonEmptyString]]] = monocle.macros.GenLens[WhereOptionString](_.NIN)
      val LIKE: monocle.Lens[WhereOptionString, clue.data.Input[NonEmptyString]] = monocle.macros.GenLens[WhereOptionString](_.LIKE)
      val NLIKE: monocle.Lens[WhereOptionString, clue.data.Input[NonEmptyString]] = monocle.macros.GenLens[WhereOptionString](_.NLIKE)
      val MATCH_CASE: monocle.Lens[WhereOptionString, clue.data.Input[Boolean]] = monocle.macros.GenLens[WhereOptionString](_.MATCH_CASE)
      implicit val eqWhereOptionString: cats.Eq[WhereOptionString] = cats.Eq.fromUniversalEquals
      implicit val showWhereOptionString: cats.Show[WhereOptionString] = cats.Show.fromToString
      implicit val jsonEncoderWhereOptionString: io.circe.Encoder[WhereOptionString] = io.circe.generic.semiauto.deriveEncoder[WhereOptionString].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class WhereOrderDatasetIndex(val EQ: clue.data.Input[PosInt] = clue.data.Ignore, val NEQ: clue.data.Input[PosInt] = clue.data.Ignore, val IN: clue.data.Input[List[PosInt]] = clue.data.Ignore, val NIN: clue.data.Input[List[PosInt]] = clue.data.Ignore, val GT: clue.data.Input[PosInt] = clue.data.Ignore, val LT: clue.data.Input[PosInt] = clue.data.Ignore, val GTE: clue.data.Input[PosInt] = clue.data.Ignore, val LTE: clue.data.Input[PosInt] = clue.data.Ignore)
    object WhereOrderDatasetIndex {
      val EQ: monocle.Lens[WhereOrderDatasetIndex, clue.data.Input[PosInt]] = monocle.macros.GenLens[WhereOrderDatasetIndex](_.EQ)
      val NEQ: monocle.Lens[WhereOrderDatasetIndex, clue.data.Input[PosInt]] = monocle.macros.GenLens[WhereOrderDatasetIndex](_.NEQ)
      val IN: monocle.Lens[WhereOrderDatasetIndex, clue.data.Input[List[PosInt]]] = monocle.macros.GenLens[WhereOrderDatasetIndex](_.IN)
      val NIN: monocle.Lens[WhereOrderDatasetIndex, clue.data.Input[List[PosInt]]] = monocle.macros.GenLens[WhereOrderDatasetIndex](_.NIN)
      val GT: monocle.Lens[WhereOrderDatasetIndex, clue.data.Input[PosInt]] = monocle.macros.GenLens[WhereOrderDatasetIndex](_.GT)
      val LT: monocle.Lens[WhereOrderDatasetIndex, clue.data.Input[PosInt]] = monocle.macros.GenLens[WhereOrderDatasetIndex](_.LT)
      val GTE: monocle.Lens[WhereOrderDatasetIndex, clue.data.Input[PosInt]] = monocle.macros.GenLens[WhereOrderDatasetIndex](_.GTE)
      val LTE: monocle.Lens[WhereOrderDatasetIndex, clue.data.Input[PosInt]] = monocle.macros.GenLens[WhereOrderDatasetIndex](_.LTE)
      implicit val eqWhereOrderDatasetIndex: cats.Eq[WhereOrderDatasetIndex] = cats.Eq.fromUniversalEquals
      implicit val showWhereOrderDatasetIndex: cats.Show[WhereOrderDatasetIndex] = cats.Show.fromToString
      implicit val jsonEncoderWhereOrderDatasetIndex: io.circe.Encoder[WhereOrderDatasetIndex] = io.circe.generic.semiauto.deriveEncoder[WhereOrderDatasetIndex].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class WhereOrderDatasetStage(val EQ: clue.data.Input[DatasetStage] = clue.data.Ignore, val NEQ: clue.data.Input[DatasetStage] = clue.data.Ignore, val IN: clue.data.Input[List[DatasetStage]] = clue.data.Ignore, val NIN: clue.data.Input[List[DatasetStage]] = clue.data.Ignore, val GT: clue.data.Input[DatasetStage] = clue.data.Ignore, val LT: clue.data.Input[DatasetStage] = clue.data.Ignore, val GTE: clue.data.Input[DatasetStage] = clue.data.Ignore, val LTE: clue.data.Input[DatasetStage] = clue.data.Ignore)
    object WhereOrderDatasetStage {
      val EQ: monocle.Lens[WhereOrderDatasetStage, clue.data.Input[DatasetStage]] = monocle.macros.GenLens[WhereOrderDatasetStage](_.EQ)
      val NEQ: monocle.Lens[WhereOrderDatasetStage, clue.data.Input[DatasetStage]] = monocle.macros.GenLens[WhereOrderDatasetStage](_.NEQ)
      val IN: monocle.Lens[WhereOrderDatasetStage, clue.data.Input[List[DatasetStage]]] = monocle.macros.GenLens[WhereOrderDatasetStage](_.IN)
      val NIN: monocle.Lens[WhereOrderDatasetStage, clue.data.Input[List[DatasetStage]]] = monocle.macros.GenLens[WhereOrderDatasetStage](_.NIN)
      val GT: monocle.Lens[WhereOrderDatasetStage, clue.data.Input[DatasetStage]] = monocle.macros.GenLens[WhereOrderDatasetStage](_.GT)
      val LT: monocle.Lens[WhereOrderDatasetStage, clue.data.Input[DatasetStage]] = monocle.macros.GenLens[WhereOrderDatasetStage](_.LT)
      val GTE: monocle.Lens[WhereOrderDatasetStage, clue.data.Input[DatasetStage]] = monocle.macros.GenLens[WhereOrderDatasetStage](_.GTE)
      val LTE: monocle.Lens[WhereOrderDatasetStage, clue.data.Input[DatasetStage]] = monocle.macros.GenLens[WhereOrderDatasetStage](_.LTE)
      implicit val eqWhereOrderDatasetStage: cats.Eq[WhereOrderDatasetStage] = cats.Eq.fromUniversalEquals
      implicit val showWhereOrderDatasetStage: cats.Show[WhereOrderDatasetStage] = cats.Show.fromToString
      implicit val jsonEncoderWhereOrderDatasetStage: io.circe.Encoder[WhereOrderDatasetStage] = io.circe.generic.semiauto.deriveEncoder[WhereOrderDatasetStage].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class WhereOrderExecutionEventId(val EQ: clue.data.Input[ExecutionEventId] = clue.data.Ignore, val NEQ: clue.data.Input[ExecutionEventId] = clue.data.Ignore, val IN: clue.data.Input[List[ExecutionEventId]] = clue.data.Ignore, val NIN: clue.data.Input[List[ExecutionEventId]] = clue.data.Ignore, val GT: clue.data.Input[ExecutionEventId] = clue.data.Ignore, val LT: clue.data.Input[ExecutionEventId] = clue.data.Ignore, val GTE: clue.data.Input[ExecutionEventId] = clue.data.Ignore, val LTE: clue.data.Input[ExecutionEventId] = clue.data.Ignore)
    object WhereOrderExecutionEventId {
      val EQ: monocle.Lens[WhereOrderExecutionEventId, clue.data.Input[ExecutionEventId]] = monocle.macros.GenLens[WhereOrderExecutionEventId](_.EQ)
      val NEQ: monocle.Lens[WhereOrderExecutionEventId, clue.data.Input[ExecutionEventId]] = monocle.macros.GenLens[WhereOrderExecutionEventId](_.NEQ)
      val IN: monocle.Lens[WhereOrderExecutionEventId, clue.data.Input[List[ExecutionEventId]]] = monocle.macros.GenLens[WhereOrderExecutionEventId](_.IN)
      val NIN: monocle.Lens[WhereOrderExecutionEventId, clue.data.Input[List[ExecutionEventId]]] = monocle.macros.GenLens[WhereOrderExecutionEventId](_.NIN)
      val GT: monocle.Lens[WhereOrderExecutionEventId, clue.data.Input[ExecutionEventId]] = monocle.macros.GenLens[WhereOrderExecutionEventId](_.GT)
      val LT: monocle.Lens[WhereOrderExecutionEventId, clue.data.Input[ExecutionEventId]] = monocle.macros.GenLens[WhereOrderExecutionEventId](_.LT)
      val GTE: monocle.Lens[WhereOrderExecutionEventId, clue.data.Input[ExecutionEventId]] = monocle.macros.GenLens[WhereOrderExecutionEventId](_.GTE)
      val LTE: monocle.Lens[WhereOrderExecutionEventId, clue.data.Input[ExecutionEventId]] = monocle.macros.GenLens[WhereOrderExecutionEventId](_.LTE)
      implicit val eqWhereOrderExecutionEventId: cats.Eq[WhereOrderExecutionEventId] = cats.Eq.fromUniversalEquals
      implicit val showWhereOrderExecutionEventId: cats.Show[WhereOrderExecutionEventId] = cats.Show.fromToString
      implicit val jsonEncoderWhereOrderExecutionEventId: io.circe.Encoder[WhereOrderExecutionEventId] = io.circe.generic.semiauto.deriveEncoder[WhereOrderExecutionEventId].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class WhereOrderTimestamp(val EQ: clue.data.Input[Timestamp] = clue.data.Ignore, val NEQ: clue.data.Input[Timestamp] = clue.data.Ignore, val IN: clue.data.Input[List[Timestamp]] = clue.data.Ignore, val NIN: clue.data.Input[List[Timestamp]] = clue.data.Ignore, val GT: clue.data.Input[Timestamp] = clue.data.Ignore, val LT: clue.data.Input[Timestamp] = clue.data.Ignore, val GTE: clue.data.Input[Timestamp] = clue.data.Ignore, val LTE: clue.data.Input[Timestamp] = clue.data.Ignore)
    object WhereOrderTimestamp {
      val EQ: monocle.Lens[WhereOrderTimestamp, clue.data.Input[Timestamp]] = monocle.macros.GenLens[WhereOrderTimestamp](_.EQ)
      val NEQ: monocle.Lens[WhereOrderTimestamp, clue.data.Input[Timestamp]] = monocle.macros.GenLens[WhereOrderTimestamp](_.NEQ)
      val IN: monocle.Lens[WhereOrderTimestamp, clue.data.Input[List[Timestamp]]] = monocle.macros.GenLens[WhereOrderTimestamp](_.IN)
      val NIN: monocle.Lens[WhereOrderTimestamp, clue.data.Input[List[Timestamp]]] = monocle.macros.GenLens[WhereOrderTimestamp](_.NIN)
      val GT: monocle.Lens[WhereOrderTimestamp, clue.data.Input[Timestamp]] = monocle.macros.GenLens[WhereOrderTimestamp](_.GT)
      val LT: monocle.Lens[WhereOrderTimestamp, clue.data.Input[Timestamp]] = monocle.macros.GenLens[WhereOrderTimestamp](_.LT)
      val GTE: monocle.Lens[WhereOrderTimestamp, clue.data.Input[Timestamp]] = monocle.macros.GenLens[WhereOrderTimestamp](_.GTE)
      val LTE: monocle.Lens[WhereOrderTimestamp, clue.data.Input[Timestamp]] = monocle.macros.GenLens[WhereOrderTimestamp](_.LTE)
      implicit val eqWhereOrderTimestamp: cats.Eq[WhereOrderTimestamp] = cats.Eq.fromUniversalEquals
      implicit val showWhereOrderTimestamp: cats.Show[WhereOrderTimestamp] = cats.Show.fromToString
      implicit val jsonEncoderWhereOrderTimestamp: io.circe.Encoder[WhereOrderTimestamp] = io.circe.generic.semiauto.deriveEncoder[WhereOrderTimestamp].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class WhereOrderInt(val EQ: clue.data.Input[Int] = clue.data.Ignore, val NEQ: clue.data.Input[Int] = clue.data.Ignore, val IN: clue.data.Input[List[Int]] = clue.data.Ignore, val NIN: clue.data.Input[List[Int]] = clue.data.Ignore, val GT: clue.data.Input[Int] = clue.data.Ignore, val LT: clue.data.Input[Int] = clue.data.Ignore, val GTE: clue.data.Input[Int] = clue.data.Ignore, val LTE: clue.data.Input[Int] = clue.data.Ignore)
    object WhereOrderInt {
      val EQ: monocle.Lens[WhereOrderInt, clue.data.Input[Int]] = monocle.macros.GenLens[WhereOrderInt](_.EQ)
      val NEQ: monocle.Lens[WhereOrderInt, clue.data.Input[Int]] = monocle.macros.GenLens[WhereOrderInt](_.NEQ)
      val IN: monocle.Lens[WhereOrderInt, clue.data.Input[List[Int]]] = monocle.macros.GenLens[WhereOrderInt](_.IN)
      val NIN: monocle.Lens[WhereOrderInt, clue.data.Input[List[Int]]] = monocle.macros.GenLens[WhereOrderInt](_.NIN)
      val GT: monocle.Lens[WhereOrderInt, clue.data.Input[Int]] = monocle.macros.GenLens[WhereOrderInt](_.GT)
      val LT: monocle.Lens[WhereOrderInt, clue.data.Input[Int]] = monocle.macros.GenLens[WhereOrderInt](_.LT)
      val GTE: monocle.Lens[WhereOrderInt, clue.data.Input[Int]] = monocle.macros.GenLens[WhereOrderInt](_.GTE)
      val LTE: monocle.Lens[WhereOrderInt, clue.data.Input[Int]] = monocle.macros.GenLens[WhereOrderInt](_.LTE)
      implicit val eqWhereOrderInt: cats.Eq[WhereOrderInt] = cats.Eq.fromUniversalEquals
      implicit val showWhereOrderInt: cats.Show[WhereOrderInt] = cats.Show.fromToString
      implicit val jsonEncoderWhereOrderInt: io.circe.Encoder[WhereOrderInt] = io.circe.generic.semiauto.deriveEncoder[WhereOrderInt].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class WhereOrderObsActiveStatus(val EQ: clue.data.Input[ObsActiveStatus] = clue.data.Ignore, val NEQ: clue.data.Input[ObsActiveStatus] = clue.data.Ignore, val IN: clue.data.Input[List[ObsActiveStatus]] = clue.data.Ignore, val NIN: clue.data.Input[List[ObsActiveStatus]] = clue.data.Ignore, val GT: clue.data.Input[ObsActiveStatus] = clue.data.Ignore, val LT: clue.data.Input[ObsActiveStatus] = clue.data.Ignore, val GTE: clue.data.Input[ObsActiveStatus] = clue.data.Ignore, val LTE: clue.data.Input[ObsActiveStatus] = clue.data.Ignore)
    object WhereOrderObsActiveStatus {
      val EQ: monocle.Lens[WhereOrderObsActiveStatus, clue.data.Input[ObsActiveStatus]] = monocle.macros.GenLens[WhereOrderObsActiveStatus](_.EQ)
      val NEQ: monocle.Lens[WhereOrderObsActiveStatus, clue.data.Input[ObsActiveStatus]] = monocle.macros.GenLens[WhereOrderObsActiveStatus](_.NEQ)
      val IN: monocle.Lens[WhereOrderObsActiveStatus, clue.data.Input[List[ObsActiveStatus]]] = monocle.macros.GenLens[WhereOrderObsActiveStatus](_.IN)
      val NIN: monocle.Lens[WhereOrderObsActiveStatus, clue.data.Input[List[ObsActiveStatus]]] = monocle.macros.GenLens[WhereOrderObsActiveStatus](_.NIN)
      val GT: monocle.Lens[WhereOrderObsActiveStatus, clue.data.Input[ObsActiveStatus]] = monocle.macros.GenLens[WhereOrderObsActiveStatus](_.GT)
      val LT: monocle.Lens[WhereOrderObsActiveStatus, clue.data.Input[ObsActiveStatus]] = monocle.macros.GenLens[WhereOrderObsActiveStatus](_.LT)
      val GTE: monocle.Lens[WhereOrderObsActiveStatus, clue.data.Input[ObsActiveStatus]] = monocle.macros.GenLens[WhereOrderObsActiveStatus](_.GTE)
      val LTE: monocle.Lens[WhereOrderObsActiveStatus, clue.data.Input[ObsActiveStatus]] = monocle.macros.GenLens[WhereOrderObsActiveStatus](_.LTE)
      implicit val eqWhereOrderObsActiveStatus: cats.Eq[WhereOrderObsActiveStatus] = cats.Eq.fromUniversalEquals
      implicit val showWhereOrderObsActiveStatus: cats.Show[WhereOrderObsActiveStatus] = cats.Show.fromToString
      implicit val jsonEncoderWhereOrderObsActiveStatus: io.circe.Encoder[WhereOrderObsActiveStatus] = io.circe.generic.semiauto.deriveEncoder[WhereOrderObsActiveStatus].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class WhereOrderObsStatus(val EQ: clue.data.Input[ObsStatus] = clue.data.Ignore, val NEQ: clue.data.Input[ObsStatus] = clue.data.Ignore, val IN: clue.data.Input[List[ObsStatus]] = clue.data.Ignore, val NIN: clue.data.Input[List[ObsStatus]] = clue.data.Ignore, val GT: clue.data.Input[ObsStatus] = clue.data.Ignore, val LT: clue.data.Input[ObsStatus] = clue.data.Ignore, val GTE: clue.data.Input[ObsStatus] = clue.data.Ignore, val LTE: clue.data.Input[ObsStatus] = clue.data.Ignore)
    object WhereOrderObsStatus {
      val EQ: monocle.Lens[WhereOrderObsStatus, clue.data.Input[ObsStatus]] = monocle.macros.GenLens[WhereOrderObsStatus](_.EQ)
      val NEQ: monocle.Lens[WhereOrderObsStatus, clue.data.Input[ObsStatus]] = monocle.macros.GenLens[WhereOrderObsStatus](_.NEQ)
      val IN: monocle.Lens[WhereOrderObsStatus, clue.data.Input[List[ObsStatus]]] = monocle.macros.GenLens[WhereOrderObsStatus](_.IN)
      val NIN: monocle.Lens[WhereOrderObsStatus, clue.data.Input[List[ObsStatus]]] = monocle.macros.GenLens[WhereOrderObsStatus](_.NIN)
      val GT: monocle.Lens[WhereOrderObsStatus, clue.data.Input[ObsStatus]] = monocle.macros.GenLens[WhereOrderObsStatus](_.GT)
      val LT: monocle.Lens[WhereOrderObsStatus, clue.data.Input[ObsStatus]] = monocle.macros.GenLens[WhereOrderObsStatus](_.LT)
      val GTE: monocle.Lens[WhereOrderObsStatus, clue.data.Input[ObsStatus]] = monocle.macros.GenLens[WhereOrderObsStatus](_.GTE)
      val LTE: monocle.Lens[WhereOrderObsStatus, clue.data.Input[ObsStatus]] = monocle.macros.GenLens[WhereOrderObsStatus](_.LTE)
      implicit val eqWhereOrderObsStatus: cats.Eq[WhereOrderObsStatus] = cats.Eq.fromUniversalEquals
      implicit val showWhereOrderObsStatus: cats.Show[WhereOrderObsStatus] = cats.Show.fromToString
      implicit val jsonEncoderWhereOrderObsStatus: io.circe.Encoder[WhereOrderObsStatus] = io.circe.generic.semiauto.deriveEncoder[WhereOrderObsStatus].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class WhereOrderObservationId(val EQ: clue.data.Input[ObservationId] = clue.data.Ignore, val NEQ: clue.data.Input[ObservationId] = clue.data.Ignore, val IN: clue.data.Input[List[ObservationId]] = clue.data.Ignore, val NIN: clue.data.Input[List[ObservationId]] = clue.data.Ignore, val GT: clue.data.Input[ObservationId] = clue.data.Ignore, val LT: clue.data.Input[ObservationId] = clue.data.Ignore, val GTE: clue.data.Input[ObservationId] = clue.data.Ignore, val LTE: clue.data.Input[ObservationId] = clue.data.Ignore)
    object WhereOrderObservationId {
      val EQ: monocle.Lens[WhereOrderObservationId, clue.data.Input[ObservationId]] = monocle.macros.GenLens[WhereOrderObservationId](_.EQ)
      val NEQ: monocle.Lens[WhereOrderObservationId, clue.data.Input[ObservationId]] = monocle.macros.GenLens[WhereOrderObservationId](_.NEQ)
      val IN: monocle.Lens[WhereOrderObservationId, clue.data.Input[List[ObservationId]]] = monocle.macros.GenLens[WhereOrderObservationId](_.IN)
      val NIN: monocle.Lens[WhereOrderObservationId, clue.data.Input[List[ObservationId]]] = monocle.macros.GenLens[WhereOrderObservationId](_.NIN)
      val GT: monocle.Lens[WhereOrderObservationId, clue.data.Input[ObservationId]] = monocle.macros.GenLens[WhereOrderObservationId](_.GT)
      val LT: monocle.Lens[WhereOrderObservationId, clue.data.Input[ObservationId]] = monocle.macros.GenLens[WhereOrderObservationId](_.LT)
      val GTE: monocle.Lens[WhereOrderObservationId, clue.data.Input[ObservationId]] = monocle.macros.GenLens[WhereOrderObservationId](_.GTE)
      val LTE: monocle.Lens[WhereOrderObservationId, clue.data.Input[ObservationId]] = monocle.macros.GenLens[WhereOrderObservationId](_.LTE)
      implicit val eqWhereOrderObservationId: cats.Eq[WhereOrderObservationId] = cats.Eq.fromUniversalEquals
      implicit val showWhereOrderObservationId: cats.Show[WhereOrderObservationId] = cats.Show.fromToString
      implicit val jsonEncoderWhereOrderObservationId: io.circe.Encoder[WhereOrderObservationId] = io.circe.generic.semiauto.deriveEncoder[WhereOrderObservationId].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class WhereOrderProgramId(val EQ: clue.data.Input[ProgramId] = clue.data.Ignore, val NEQ: clue.data.Input[ProgramId] = clue.data.Ignore, val IN: clue.data.Input[List[ProgramId]] = clue.data.Ignore, val NIN: clue.data.Input[List[ProgramId]] = clue.data.Ignore, val GT: clue.data.Input[ProgramId] = clue.data.Ignore, val LT: clue.data.Input[ProgramId] = clue.data.Ignore, val GTE: clue.data.Input[ProgramId] = clue.data.Ignore, val LTE: clue.data.Input[ProgramId] = clue.data.Ignore)
    object WhereOrderProgramId {
      val EQ: monocle.Lens[WhereOrderProgramId, clue.data.Input[ProgramId]] = monocle.macros.GenLens[WhereOrderProgramId](_.EQ)
      val NEQ: monocle.Lens[WhereOrderProgramId, clue.data.Input[ProgramId]] = monocle.macros.GenLens[WhereOrderProgramId](_.NEQ)
      val IN: monocle.Lens[WhereOrderProgramId, clue.data.Input[List[ProgramId]]] = monocle.macros.GenLens[WhereOrderProgramId](_.IN)
      val NIN: monocle.Lens[WhereOrderProgramId, clue.data.Input[List[ProgramId]]] = monocle.macros.GenLens[WhereOrderProgramId](_.NIN)
      val GT: monocle.Lens[WhereOrderProgramId, clue.data.Input[ProgramId]] = monocle.macros.GenLens[WhereOrderProgramId](_.GT)
      val LT: monocle.Lens[WhereOrderProgramId, clue.data.Input[ProgramId]] = monocle.macros.GenLens[WhereOrderProgramId](_.LT)
      val GTE: monocle.Lens[WhereOrderProgramId, clue.data.Input[ProgramId]] = monocle.macros.GenLens[WhereOrderProgramId](_.GTE)
      val LTE: monocle.Lens[WhereOrderProgramId, clue.data.Input[ProgramId]] = monocle.macros.GenLens[WhereOrderProgramId](_.LTE)
      implicit val eqWhereOrderProgramId: cats.Eq[WhereOrderProgramId] = cats.Eq.fromUniversalEquals
      implicit val showWhereOrderProgramId: cats.Show[WhereOrderProgramId] = cats.Show.fromToString
      implicit val jsonEncoderWhereOrderProgramId: io.circe.Encoder[WhereOrderProgramId] = io.circe.generic.semiauto.deriveEncoder[WhereOrderProgramId].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class WhereOrderSequenceCommand(val EQ: clue.data.Input[SequenceCommand] = clue.data.Ignore, val NEQ: clue.data.Input[SequenceCommand] = clue.data.Ignore, val IN: clue.data.Input[List[SequenceCommand]] = clue.data.Ignore, val NIN: clue.data.Input[List[SequenceCommand]] = clue.data.Ignore, val GT: clue.data.Input[SequenceCommand] = clue.data.Ignore, val LT: clue.data.Input[SequenceCommand] = clue.data.Ignore, val GTE: clue.data.Input[SequenceCommand] = clue.data.Ignore, val LTE: clue.data.Input[SequenceCommand] = clue.data.Ignore)
    object WhereOrderSequenceCommand {
      val EQ: monocle.Lens[WhereOrderSequenceCommand, clue.data.Input[SequenceCommand]] = monocle.macros.GenLens[WhereOrderSequenceCommand](_.EQ)
      val NEQ: monocle.Lens[WhereOrderSequenceCommand, clue.data.Input[SequenceCommand]] = monocle.macros.GenLens[WhereOrderSequenceCommand](_.NEQ)
      val IN: monocle.Lens[WhereOrderSequenceCommand, clue.data.Input[List[SequenceCommand]]] = monocle.macros.GenLens[WhereOrderSequenceCommand](_.IN)
      val NIN: monocle.Lens[WhereOrderSequenceCommand, clue.data.Input[List[SequenceCommand]]] = monocle.macros.GenLens[WhereOrderSequenceCommand](_.NIN)
      val GT: monocle.Lens[WhereOrderSequenceCommand, clue.data.Input[SequenceCommand]] = monocle.macros.GenLens[WhereOrderSequenceCommand](_.GT)
      val LT: monocle.Lens[WhereOrderSequenceCommand, clue.data.Input[SequenceCommand]] = monocle.macros.GenLens[WhereOrderSequenceCommand](_.LT)
      val GTE: monocle.Lens[WhereOrderSequenceCommand, clue.data.Input[SequenceCommand]] = monocle.macros.GenLens[WhereOrderSequenceCommand](_.GTE)
      val LTE: monocle.Lens[WhereOrderSequenceCommand, clue.data.Input[SequenceCommand]] = monocle.macros.GenLens[WhereOrderSequenceCommand](_.LTE)
      implicit val eqWhereOrderSequenceCommand: cats.Eq[WhereOrderSequenceCommand] = cats.Eq.fromUniversalEquals
      implicit val showWhereOrderSequenceCommand: cats.Show[WhereOrderSequenceCommand] = cats.Show.fromToString
      implicit val jsonEncoderWhereOrderSequenceCommand: io.circe.Encoder[WhereOrderSequenceCommand] = io.circe.generic.semiauto.deriveEncoder[WhereOrderSequenceCommand].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class WhereOrderSequenceType(val EQ: clue.data.Input[SequenceType] = clue.data.Ignore, val NEQ: clue.data.Input[SequenceType] = clue.data.Ignore, val IN: clue.data.Input[List[SequenceType]] = clue.data.Ignore, val NIN: clue.data.Input[List[SequenceType]] = clue.data.Ignore, val GT: clue.data.Input[SequenceType] = clue.data.Ignore, val LT: clue.data.Input[SequenceType] = clue.data.Ignore, val GTE: clue.data.Input[SequenceType] = clue.data.Ignore, val LTE: clue.data.Input[SequenceType] = clue.data.Ignore)
    object WhereOrderSequenceType {
      val EQ: monocle.Lens[WhereOrderSequenceType, clue.data.Input[SequenceType]] = monocle.macros.GenLens[WhereOrderSequenceType](_.EQ)
      val NEQ: monocle.Lens[WhereOrderSequenceType, clue.data.Input[SequenceType]] = monocle.macros.GenLens[WhereOrderSequenceType](_.NEQ)
      val IN: monocle.Lens[WhereOrderSequenceType, clue.data.Input[List[SequenceType]]] = monocle.macros.GenLens[WhereOrderSequenceType](_.IN)
      val NIN: monocle.Lens[WhereOrderSequenceType, clue.data.Input[List[SequenceType]]] = monocle.macros.GenLens[WhereOrderSequenceType](_.NIN)
      val GT: monocle.Lens[WhereOrderSequenceType, clue.data.Input[SequenceType]] = monocle.macros.GenLens[WhereOrderSequenceType](_.GT)
      val LT: monocle.Lens[WhereOrderSequenceType, clue.data.Input[SequenceType]] = monocle.macros.GenLens[WhereOrderSequenceType](_.LT)
      val GTE: monocle.Lens[WhereOrderSequenceType, clue.data.Input[SequenceType]] = monocle.macros.GenLens[WhereOrderSequenceType](_.GTE)
      val LTE: monocle.Lens[WhereOrderSequenceType, clue.data.Input[SequenceType]] = monocle.macros.GenLens[WhereOrderSequenceType](_.LTE)
      implicit val eqWhereOrderSequenceType: cats.Eq[WhereOrderSequenceType] = cats.Eq.fromUniversalEquals
      implicit val showWhereOrderSequenceType: cats.Show[WhereOrderSequenceType] = cats.Show.fromToString
      implicit val jsonEncoderWhereOrderSequenceType: io.circe.Encoder[WhereOrderSequenceType] = io.circe.generic.semiauto.deriveEncoder[WhereOrderSequenceType].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class WhereOrderStepStage(val EQ: clue.data.Input[StepStage] = clue.data.Ignore, val NEQ: clue.data.Input[StepStage] = clue.data.Ignore, val IN: clue.data.Input[List[StepStage]] = clue.data.Ignore, val NIN: clue.data.Input[List[StepStage]] = clue.data.Ignore, val GT: clue.data.Input[StepStage] = clue.data.Ignore, val LT: clue.data.Input[StepStage] = clue.data.Ignore, val GTE: clue.data.Input[StepStage] = clue.data.Ignore, val LTE: clue.data.Input[StepStage] = clue.data.Ignore)
    object WhereOrderStepStage {
      val EQ: monocle.Lens[WhereOrderStepStage, clue.data.Input[StepStage]] = monocle.macros.GenLens[WhereOrderStepStage](_.EQ)
      val NEQ: monocle.Lens[WhereOrderStepStage, clue.data.Input[StepStage]] = monocle.macros.GenLens[WhereOrderStepStage](_.NEQ)
      val IN: monocle.Lens[WhereOrderStepStage, clue.data.Input[List[StepStage]]] = monocle.macros.GenLens[WhereOrderStepStage](_.IN)
      val NIN: monocle.Lens[WhereOrderStepStage, clue.data.Input[List[StepStage]]] = monocle.macros.GenLens[WhereOrderStepStage](_.NIN)
      val GT: monocle.Lens[WhereOrderStepStage, clue.data.Input[StepStage]] = monocle.macros.GenLens[WhereOrderStepStage](_.GT)
      val LT: monocle.Lens[WhereOrderStepStage, clue.data.Input[StepStage]] = monocle.macros.GenLens[WhereOrderStepStage](_.LT)
      val GTE: monocle.Lens[WhereOrderStepStage, clue.data.Input[StepStage]] = monocle.macros.GenLens[WhereOrderStepStage](_.GTE)
      val LTE: monocle.Lens[WhereOrderStepStage, clue.data.Input[StepStage]] = monocle.macros.GenLens[WhereOrderStepStage](_.LTE)
      implicit val eqWhereOrderStepStage: cats.Eq[WhereOrderStepStage] = cats.Eq.fromUniversalEquals
      implicit val showWhereOrderStepStage: cats.Show[WhereOrderStepStage] = cats.Show.fromToString
      implicit val jsonEncoderWhereOrderStepStage: io.circe.Encoder[WhereOrderStepStage] = io.circe.generic.semiauto.deriveEncoder[WhereOrderStepStage].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class WhereOrderTargetId(val EQ: clue.data.Input[TargetId] = clue.data.Ignore, val NEQ: clue.data.Input[TargetId] = clue.data.Ignore, val IN: clue.data.Input[List[TargetId]] = clue.data.Ignore, val NIN: clue.data.Input[List[TargetId]] = clue.data.Ignore, val GT: clue.data.Input[TargetId] = clue.data.Ignore, val LT: clue.data.Input[TargetId] = clue.data.Ignore, val GTE: clue.data.Input[TargetId] = clue.data.Ignore, val LTE: clue.data.Input[TargetId] = clue.data.Ignore)
    object WhereOrderTargetId {
      val EQ: monocle.Lens[WhereOrderTargetId, clue.data.Input[TargetId]] = monocle.macros.GenLens[WhereOrderTargetId](_.EQ)
      val NEQ: monocle.Lens[WhereOrderTargetId, clue.data.Input[TargetId]] = monocle.macros.GenLens[WhereOrderTargetId](_.NEQ)
      val IN: monocle.Lens[WhereOrderTargetId, clue.data.Input[List[TargetId]]] = monocle.macros.GenLens[WhereOrderTargetId](_.IN)
      val NIN: monocle.Lens[WhereOrderTargetId, clue.data.Input[List[TargetId]]] = monocle.macros.GenLens[WhereOrderTargetId](_.NIN)
      val GT: monocle.Lens[WhereOrderTargetId, clue.data.Input[TargetId]] = monocle.macros.GenLens[WhereOrderTargetId](_.GT)
      val LT: monocle.Lens[WhereOrderTargetId, clue.data.Input[TargetId]] = monocle.macros.GenLens[WhereOrderTargetId](_.LT)
      val GTE: monocle.Lens[WhereOrderTargetId, clue.data.Input[TargetId]] = monocle.macros.GenLens[WhereOrderTargetId](_.GTE)
      val LTE: monocle.Lens[WhereOrderTargetId, clue.data.Input[TargetId]] = monocle.macros.GenLens[WhereOrderTargetId](_.LTE)
      implicit val eqWhereOrderTargetId: cats.Eq[WhereOrderTargetId] = cats.Eq.fromUniversalEquals
      implicit val showWhereOrderTargetId: cats.Show[WhereOrderTargetId] = cats.Show.fromToString
      implicit val jsonEncoderWhereOrderTargetId: io.circe.Encoder[WhereOrderTargetId] = io.circe.generic.semiauto.deriveEncoder[WhereOrderTargetId].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class WhereProgram(val AND: clue.data.Input[List[WhereProgram]] = clue.data.Ignore, val OR: clue.data.Input[List[WhereProgram]] = clue.data.Ignore, val NOT: clue.data.Input[WhereProgram] = clue.data.Ignore, val id: clue.data.Input[WhereOrderProgramId] = clue.data.Ignore, val name: clue.data.Input[WhereOptionString] = clue.data.Ignore, val proposal: clue.data.Input[WhereProposal] = clue.data.Ignore)
    object WhereProgram {
      val AND: monocle.Lens[WhereProgram, clue.data.Input[List[WhereProgram]]] = monocle.macros.GenLens[WhereProgram](_.AND)
      val OR: monocle.Lens[WhereProgram, clue.data.Input[List[WhereProgram]]] = monocle.macros.GenLens[WhereProgram](_.OR)
      val NOT: monocle.Lens[WhereProgram, clue.data.Input[WhereProgram]] = monocle.macros.GenLens[WhereProgram](_.NOT)
      val id: monocle.Lens[WhereProgram, clue.data.Input[WhereOrderProgramId]] = monocle.macros.GenLens[WhereProgram](_.id)
      val name: monocle.Lens[WhereProgram, clue.data.Input[WhereOptionString]] = monocle.macros.GenLens[WhereProgram](_.name)
      val proposal: monocle.Lens[WhereProgram, clue.data.Input[WhereProposal]] = monocle.macros.GenLens[WhereProgram](_.proposal)
      implicit val eqWhereProgram: cats.Eq[WhereProgram] = cats.Eq.fromUniversalEquals
      implicit val showWhereProgram: cats.Show[WhereProgram] = cats.Show.fromToString
      implicit val jsonEncoderWhereProgram: io.circe.Encoder[WhereProgram] = io.circe.generic.semiauto.deriveEncoder[WhereProgram].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class WhereProposal(val IS_NULL: clue.data.Input[Boolean] = clue.data.Ignore, val AND: clue.data.Input[List[WhereProposal]] = clue.data.Ignore, val OR: clue.data.Input[List[WhereProposal]] = clue.data.Ignore, val NOT: clue.data.Input[WhereProposal] = clue.data.Ignore, val title: clue.data.Input[WhereOptionString] = clue.data.Ignore, val `class`: clue.data.Input[WhereProposalClass] = clue.data.Ignore, val category: clue.data.Input[WhereOptionEqTacCategory] = clue.data.Ignore, val toOActivation: clue.data.Input[WhereEqToOActivation] = clue.data.Ignore, val `abstract`: clue.data.Input[WhereOptionString] = clue.data.Ignore, val partners: clue.data.Input[WhereProposalPartners] = clue.data.Ignore)
    object WhereProposal {
      val IS_NULL: monocle.Lens[WhereProposal, clue.data.Input[Boolean]] = monocle.macros.GenLens[WhereProposal](_.IS_NULL)
      val AND: monocle.Lens[WhereProposal, clue.data.Input[List[WhereProposal]]] = monocle.macros.GenLens[WhereProposal](_.AND)
      val OR: monocle.Lens[WhereProposal, clue.data.Input[List[WhereProposal]]] = monocle.macros.GenLens[WhereProposal](_.OR)
      val NOT: monocle.Lens[WhereProposal, clue.data.Input[WhereProposal]] = monocle.macros.GenLens[WhereProposal](_.NOT)
      val title: monocle.Lens[WhereProposal, clue.data.Input[WhereOptionString]] = monocle.macros.GenLens[WhereProposal](_.title)
      val `class`: monocle.Lens[WhereProposal, clue.data.Input[WhereProposalClass]] = monocle.macros.GenLens[WhereProposal](_.`class`)
      val category: monocle.Lens[WhereProposal, clue.data.Input[WhereOptionEqTacCategory]] = monocle.macros.GenLens[WhereProposal](_.category)
      val toOActivation: monocle.Lens[WhereProposal, clue.data.Input[WhereEqToOActivation]] = monocle.macros.GenLens[WhereProposal](_.toOActivation)
      val `abstract`: monocle.Lens[WhereProposal, clue.data.Input[WhereOptionString]] = monocle.macros.GenLens[WhereProposal](_.`abstract`)
      val partners: monocle.Lens[WhereProposal, clue.data.Input[WhereProposalPartners]] = monocle.macros.GenLens[WhereProposal](_.partners)
      implicit val eqWhereProposal: cats.Eq[WhereProposal] = cats.Eq.fromUniversalEquals
      implicit val showWhereProposal: cats.Show[WhereProposal] = cats.Show.fromToString
      implicit val jsonEncoderWhereProposal: io.circe.Encoder[WhereProposal] = io.circe.generic.semiauto.deriveEncoder[WhereProposal].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class WhereProposalClass(val `type`: clue.data.Input[WhereEqProposalClassType] = clue.data.Ignore, val minPercent: clue.data.Input[WhereOrderInt] = clue.data.Ignore)
    object WhereProposalClass {
      val `type`: monocle.Lens[WhereProposalClass, clue.data.Input[WhereEqProposalClassType]] = monocle.macros.GenLens[WhereProposalClass](_.`type`)
      val minPercent: monocle.Lens[WhereProposalClass, clue.data.Input[WhereOrderInt]] = monocle.macros.GenLens[WhereProposalClass](_.minPercent)
      implicit val eqWhereProposalClass: cats.Eq[WhereProposalClass] = cats.Eq.fromUniversalEquals
      implicit val showWhereProposalClass: cats.Show[WhereProposalClass] = cats.Show.fromToString
      implicit val jsonEncoderWhereProposalClass: io.circe.Encoder[WhereProposalClass] = io.circe.generic.semiauto.deriveEncoder[WhereProposalClass].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class WhereProposalPartnerEntry(val AND: clue.data.Input[List[WhereProposalPartnerEntry]] = clue.data.Ignore, val OR: clue.data.Input[List[WhereProposalPartnerEntry]] = clue.data.Ignore, val NOT: clue.data.Input[WhereProposalPartnerEntry] = clue.data.Ignore, val partner: clue.data.Input[WhereEqPartner] = clue.data.Ignore, val percent: clue.data.Input[WhereOrderInt] = clue.data.Ignore)
    object WhereProposalPartnerEntry {
      val AND: monocle.Lens[WhereProposalPartnerEntry, clue.data.Input[List[WhereProposalPartnerEntry]]] = monocle.macros.GenLens[WhereProposalPartnerEntry](_.AND)
      val OR: monocle.Lens[WhereProposalPartnerEntry, clue.data.Input[List[WhereProposalPartnerEntry]]] = monocle.macros.GenLens[WhereProposalPartnerEntry](_.OR)
      val NOT: monocle.Lens[WhereProposalPartnerEntry, clue.data.Input[WhereProposalPartnerEntry]] = monocle.macros.GenLens[WhereProposalPartnerEntry](_.NOT)
      val partner: monocle.Lens[WhereProposalPartnerEntry, clue.data.Input[WhereEqPartner]] = monocle.macros.GenLens[WhereProposalPartnerEntry](_.partner)
      val percent: monocle.Lens[WhereProposalPartnerEntry, clue.data.Input[WhereOrderInt]] = monocle.macros.GenLens[WhereProposalPartnerEntry](_.percent)
      implicit val eqWhereProposalPartnerEntry: cats.Eq[WhereProposalPartnerEntry] = cats.Eq.fromUniversalEquals
      implicit val showWhereProposalPartnerEntry: cats.Show[WhereProposalPartnerEntry] = cats.Show.fromToString
      implicit val jsonEncoderWhereProposalPartnerEntry: io.circe.Encoder[WhereProposalPartnerEntry] = io.circe.generic.semiauto.deriveEncoder[WhereProposalPartnerEntry].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class WhereProposalPartners(val MATCH: clue.data.Input[WhereProposalPartnerEntry] = clue.data.Ignore, val EQ: clue.data.Input[List[Partner]] = clue.data.Ignore, val isJoint: clue.data.Input[Boolean] = clue.data.Ignore)
    object WhereProposalPartners {
      val MATCH: monocle.Lens[WhereProposalPartners, clue.data.Input[WhereProposalPartnerEntry]] = monocle.macros.GenLens[WhereProposalPartners](_.MATCH)
      val EQ: monocle.Lens[WhereProposalPartners, clue.data.Input[List[Partner]]] = monocle.macros.GenLens[WhereProposalPartners](_.EQ)
      val isJoint: monocle.Lens[WhereProposalPartners, clue.data.Input[Boolean]] = monocle.macros.GenLens[WhereProposalPartners](_.isJoint)
      implicit val eqWhereProposalPartners: cats.Eq[WhereProposalPartners] = cats.Eq.fromUniversalEquals
      implicit val showWhereProposalPartners: cats.Show[WhereProposalPartners] = cats.Show.fromToString
      implicit val jsonEncoderWhereProposalPartners: io.circe.Encoder[WhereProposalPartners] = io.circe.generic.semiauto.deriveEncoder[WhereProposalPartners].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class WhereSequenceEvent(val command: clue.data.Input[WhereOrderSequenceCommand] = clue.data.Ignore)
    object WhereSequenceEvent {
      val command: monocle.Lens[WhereSequenceEvent, clue.data.Input[WhereOrderSequenceCommand]] = monocle.macros.GenLens[WhereSequenceEvent](_.command)
      implicit val eqWhereSequenceEvent: cats.Eq[WhereSequenceEvent] = cats.Eq.fromUniversalEquals
      implicit val showWhereSequenceEvent: cats.Show[WhereSequenceEvent] = cats.Show.fromToString
      implicit val jsonEncoderWhereSequenceEvent: io.circe.Encoder[WhereSequenceEvent] = io.circe.generic.semiauto.deriveEncoder[WhereSequenceEvent].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class WhereStepEvent(val stepId: clue.data.Input[WhereEqStepId] = clue.data.Ignore, val sequenceType: clue.data.Input[WhereOrderSequenceType] = clue.data.Ignore, val stage: clue.data.Input[WhereOrderStepStage] = clue.data.Ignore)
    object WhereStepEvent {
      val stepId: monocle.Lens[WhereStepEvent, clue.data.Input[WhereEqStepId]] = monocle.macros.GenLens[WhereStepEvent](_.stepId)
      val sequenceType: monocle.Lens[WhereStepEvent, clue.data.Input[WhereOrderSequenceType]] = monocle.macros.GenLens[WhereStepEvent](_.sequenceType)
      val stage: monocle.Lens[WhereStepEvent, clue.data.Input[WhereOrderStepStage]] = monocle.macros.GenLens[WhereStepEvent](_.stage)
      implicit val eqWhereStepEvent: cats.Eq[WhereStepEvent] = cats.Eq.fromUniversalEquals
      implicit val showWhereStepEvent: cats.Show[WhereStepEvent] = cats.Show.fromToString
      implicit val jsonEncoderWhereStepEvent: io.circe.Encoder[WhereStepEvent] = io.circe.generic.semiauto.deriveEncoder[WhereStepEvent].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class WhereString(val EQ: clue.data.Input[NonEmptyString] = clue.data.Ignore, val NEQ: clue.data.Input[NonEmptyString] = clue.data.Ignore, val IN: clue.data.Input[List[NonEmptyString]] = clue.data.Ignore, val NIN: clue.data.Input[List[NonEmptyString]] = clue.data.Ignore, val LIKE: clue.data.Input[NonEmptyString] = clue.data.Ignore, val NLIKE: clue.data.Input[NonEmptyString] = clue.data.Ignore, val MATCH_CASE: clue.data.Input[Boolean] = clue.data.Ignore)
    object WhereString {
      val EQ: monocle.Lens[WhereString, clue.data.Input[NonEmptyString]] = monocle.macros.GenLens[WhereString](_.EQ)
      val NEQ: monocle.Lens[WhereString, clue.data.Input[NonEmptyString]] = monocle.macros.GenLens[WhereString](_.NEQ)
      val IN: monocle.Lens[WhereString, clue.data.Input[List[NonEmptyString]]] = monocle.macros.GenLens[WhereString](_.IN)
      val NIN: monocle.Lens[WhereString, clue.data.Input[List[NonEmptyString]]] = monocle.macros.GenLens[WhereString](_.NIN)
      val LIKE: monocle.Lens[WhereString, clue.data.Input[NonEmptyString]] = monocle.macros.GenLens[WhereString](_.LIKE)
      val NLIKE: monocle.Lens[WhereString, clue.data.Input[NonEmptyString]] = monocle.macros.GenLens[WhereString](_.NLIKE)
      val MATCH_CASE: monocle.Lens[WhereString, clue.data.Input[Boolean]] = monocle.macros.GenLens[WhereString](_.MATCH_CASE)
      implicit val eqWhereString: cats.Eq[WhereString] = cats.Eq.fromUniversalEquals
      implicit val showWhereString: cats.Show[WhereString] = cats.Show.fromToString
      implicit val jsonEncoderWhereString: io.circe.Encoder[WhereString] = io.circe.generic.semiauto.deriveEncoder[WhereString].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
    case class WhereTarget(val AND: clue.data.Input[List[WhereTarget]] = clue.data.Ignore, val OR: clue.data.Input[List[WhereTarget]] = clue.data.Ignore, val NOT: clue.data.Input[WhereTarget] = clue.data.Ignore, val id: clue.data.Input[WhereOrderTargetId] = clue.data.Ignore, val programId: clue.data.Input[WhereOrderProgramId] = clue.data.Ignore, val name: clue.data.Input[WhereString] = clue.data.Ignore)
    object WhereTarget {
      val AND: monocle.Lens[WhereTarget, clue.data.Input[List[WhereTarget]]] = monocle.macros.GenLens[WhereTarget](_.AND)
      val OR: monocle.Lens[WhereTarget, clue.data.Input[List[WhereTarget]]] = monocle.macros.GenLens[WhereTarget](_.OR)
      val NOT: monocle.Lens[WhereTarget, clue.data.Input[WhereTarget]] = monocle.macros.GenLens[WhereTarget](_.NOT)
      val id: monocle.Lens[WhereTarget, clue.data.Input[WhereOrderTargetId]] = monocle.macros.GenLens[WhereTarget](_.id)
      val programId: monocle.Lens[WhereTarget, clue.data.Input[WhereOrderProgramId]] = monocle.macros.GenLens[WhereTarget](_.programId)
      val name: monocle.Lens[WhereTarget, clue.data.Input[WhereString]] = monocle.macros.GenLens[WhereTarget](_.name)
      implicit val eqWhereTarget: cats.Eq[WhereTarget] = cats.Eq.fromUniversalEquals
      implicit val showWhereTarget: cats.Show[WhereTarget] = cats.Show.fromToString
      implicit val jsonEncoderWhereTarget: io.circe.Encoder[WhereTarget] = io.circe.generic.semiauto.deriveEncoder[WhereTarget].mapJson(_.foldWith(clue.data.Input.dropIgnoreFolder))
    }
  }
}
