// Copyright (c) 2016-2023 Association of Universities for Research in Astronomy, Inc. (AURA)
// For license information see LICENSE or https://opensource.org/licenses/BSD-3-Clause

package lucuma.schemas

import lucuma.core.enums
import lucuma.core.math.SignalToNoise
import lucuma.core.math.dimensional.*
import lucuma.core.model
import lucuma.core.model.*
import lucuma.core.model.sequence.*
import lucuma.core.util.*
import lucuma.core.math.BrightnessUnits.*

import clue.BigNumberEncoders._
import io.circe.refined._
import lucuma.schemas.decoders.given


sealed trait ObservationDB
object ObservationDB {
  object Scalars {
    type AtomId = Atom.Id
    type DatasetId = Dataset.Id
    type ExecutionEventId = ExecutionEvent.Id
    type GroupId = Group.Id
    type ObsAttachmentId = ObsAttachment.Id
    type ObservationId = Observation.Id
    type ProgramId = Program.Id
    type StepId = Step.Id
    type TargetId = Target.Id
    type UserId = User.Id
    type VisitId = Visit.Id
    type BigDecimal = scala.BigDecimal
    type Long = scala.Long
    type DmsString = String
    type EpochString = String
    type HmsString = String
    type UserInvitationId = String
    type UserInvitationKey = String
    type Extinction = NonNegBigDecimal
    type NonEmptyString = eu.timepit.refined.types.string.NonEmptyString
    type NonNegBigDecimal = eu.timepit.refined.types.numeric.NonNegBigDecimal
    type NonNegInt = eu.timepit.refined.types.numeric.NonNegInt
    type NonNegLong = eu.timepit.refined.types.numeric.NonNegLong
    type NonNegShort = eu.timepit.refined.types.numeric.NonNegShort
    type PosBigDecimal = eu.timepit.refined.types.numeric.PosBigDecimal
    type PosInt = eu.timepit.refined.types.numeric.PosInt
    type PosLong = eu.timepit.refined.types.numeric.PosLong
    type PosShort = eu.timepit.refined.types.numeric.PosShort
    type DatasetFilename = lucuma.core.model.sequence.Dataset.Filename
    type ProposalReferenceLabel = lucuma.core.model.ProposalReference
    type ProgramReferenceLabel = lucuma.core.model.ProgramReference
    type Semester = lucuma.core.model.Semester
    type SignalToNoise = lucuma.core.math.SignalToNoise
    type Timestamp = lucuma.core.util.Timestamp
    type ObsAttachmentTypeMeta = lucuma.schemas.enums.ObsAttachmentType
    type ProposalAttachmentTypeMeta = lucuma.schemas.enums.ProposalAttachmentType
    type ProposalStatusMeta = lucuma.schemas.enums.ProposalStatus
    def ignoreUnusedImportScalars(): Unit = ()
  }
  object Enums {
    def ignoreUnusedImportEnums(): Unit = ()
    type Band = enums.Band
    type Breakpoint = enums.Breakpoint
    type BrightnessIntegratedUnits = Units Of Brightness[Integrated]
    type BrightnessSurfaceUnits = Units Of Brightness[Surface]
    type CatalogName = enums.CatalogName
    type ChargeClass = enums.ChargeClass
    type CloudExtinction = enums.CloudExtinction
    type CoolStarTemperature = enums.CoolStarTemperature
    type DatasetQaState = enums.DatasetQaState
    type DatasetStage = enums.DatasetStage
    type EphemerisKeyType = enums.EphemerisKeyType
    type FilterType = enums.FilterType
    type FluxDensityContinuumIntegratedUnits = Units Of FluxDensityContinuum[Integrated]
    type FluxDensityContinuumSurfaceUnits = Units Of FluxDensityContinuum[Surface]
    type FocalPlane = enums.FocalPlane
    type GalaxySpectrum = enums.GalaxySpectrum
    type GcalArc = enums.GcalArc
    type GcalContinuum = enums.GcalContinuum
    type GcalDiffuser = enums.GcalDiffuser
    type GcalFilter = enums.GcalFilter
    type GcalShutter = enums.GcalShutter
    type GmosAmpCount = enums.GmosAmpCount
    type GmosAmpGain = enums.GmosAmpGain
    type GmosAmpReadMode = enums.GmosAmpReadMode
    type GmosCustomSlitWidth = enums.GmosCustomSlitWidth
    type GmosGratingOrder = enums.GmosGratingOrder
    type GmosDtax = enums.GmosDtax
    type GmosEOffsetting = enums.GmosEOffsetting
    type GmosNorthBuiltinFpu = enums.GmosNorthFpu
    type GmosNorthDetector = enums.GmosNorthDetector
    type GmosNorthGrating = enums.GmosNorthGrating
    type GmosNorthFilter = enums.GmosNorthFilter
    type GmosNorthStageMode = enums.GmosNorthStageMode
    type GmosRoi = enums.GmosRoi
    type GmosSouthBuiltinFpu = enums.GmosSouthFpu
    type GmosSouthDetector = enums.GmosSouthDetector
    type GmosSouthGrating = enums.GmosSouthGrating
    type GmosSouthFilter = enums.GmosSouthFilter
    type GmosSouthStageMode = enums.GmosSouthStageMode
    type GmosXBinning = enums.GmosXBinning
    type GmosYBinning = enums.GmosYBinning
    type GuideState = enums.GuideState
    type HiiRegionSpectrum = enums.HIIRegionSpectrum
    type ImageQuality = enums.ImageQuality
    type Instrument = enums.Instrument
    type LineFluxIntegratedUnits = Units Of LineFlux[Integrated]
    type LineFluxSurfaceUnits = Units Of LineFlux[Surface]
    type MosPreImaging = enums.MosPreImaging
    type ObsActiveStatus = enums.ObsActiveStatus
    type ObsAttachmentType = lucuma.schemas.enums.ObsAttachmentType
    type ObsStatus = enums.ObsStatus
    type ObserveClass = enums.ObserveClass
    type Partner = model.Partner
    type PlanetSpectrum = enums.PlanetSpectrum
    type PlanetaryNebulaSpectrum = enums.PlanetaryNebulaSpectrum
    type ProposalAttachmentType = lucuma.schemas.enums.ProposalAttachmentType
    type ProposalStatus = lucuma.schemas.enums.ProposalStatus
    type QuasarSpectrum = enums.QuasarSpectrum
    type ScienceMode = enums.ScienceMode
    type SequenceCommand = enums.SequenceCommand
    type SequenceType = enums.SequenceType
    type SkyBackground = enums.SkyBackground
    type SpectroscopyCapabilities = enums.SpectroscopyCapabilities
    type SmartGcalType = enums.SmartGcalType
    type StellarLibrarySpectrum = enums.StellarLibrarySpectrum
    type StepQaState = enums.StepQaState
    type StepStage = enums.StepStage
    type StepType = enums.StepType
    type TacCategory = enums.TacCategory
    type TimingWindowInclusion = enums.TimingWindowInclusion
    type ToOActivation = enums.ToOActivation
    type WaterVapor = enums.WaterVapor
    sealed trait ConditionsMeasurementSource
    object ConditionsMeasurementSource {
      case object Observer extends ConditionsMeasurementSource()
      implicit val eqConditionsMeasurementSource: cats.Eq[ConditionsMeasurementSource] = cats.Eq.fromUniversalEquals
      implicit val showConditionsMeasurementSource: cats.Show[ConditionsMeasurementSource] = cats.Show.fromToString
      implicit val jsonEncoderConditionsMeasurementSource: io.circe.Encoder[ConditionsMeasurementSource] = io.circe.Encoder.encodeString.contramap[ConditionsMeasurementSource]({
        case Observer => "OBSERVER"
      })
      implicit val jsonDecoderConditionsMeasurementSource: io.circe.Decoder[ConditionsMeasurementSource] = io.circe.Decoder.decodeString.emap(_ match {
        case "OBSERVER" =>
          Right(Observer)
        case other =>
          Left(s"Invalid value [$other]")
      })
    }
    sealed trait SeeingTrend
    object SeeingTrend {
      case object GettingBetter extends SeeingTrend()
      case object GettingWorse extends SeeingTrend()
      case object StayingTheSame extends SeeingTrend()
      case object Variable extends SeeingTrend()
      implicit val eqSeeingTrend: cats.Eq[SeeingTrend] = cats.Eq.fromUniversalEquals
      implicit val showSeeingTrend: cats.Show[SeeingTrend] = cats.Show.fromToString
      implicit val jsonEncoderSeeingTrend: io.circe.Encoder[SeeingTrend] = io.circe.Encoder.encodeString.contramap[SeeingTrend]({
        case GettingBetter => "GETTING_BETTER"
        case GettingWorse => "GETTING_WORSE"
        case StayingTheSame => "STAYING_THE_SAME"
        case Variable => "VARIABLE"
      })
      implicit val jsonDecoderSeeingTrend: io.circe.Decoder[SeeingTrend] = io.circe.Decoder.decodeString.emap(_ match {
        case "GETTING_BETTER" =>
          Right(GettingBetter)
        case "GETTING_WORSE" =>
          Right(GettingWorse)
        case "STAYING_THE_SAME" =>
          Right(StayingTheSame)
        case "VARIABLE" =>
          Right(Variable)
        case other =>
          Left(s"Invalid value [$other]")
      })
    }
    sealed trait ConditionsExpectationType
    object ConditionsExpectationType {
      case object ClearSkies extends ConditionsExpectationType()
      case object Fog extends ConditionsExpectationType()
      case object ThickClouds extends ConditionsExpectationType()
      case object ThinClouds extends ConditionsExpectationType()
      implicit val eqConditionsExpectationType: cats.Eq[ConditionsExpectationType] = cats.Eq.fromUniversalEquals
      implicit val showConditionsExpectationType: cats.Show[ConditionsExpectationType] = cats.Show.fromToString
      implicit val jsonEncoderConditionsExpectationType: io.circe.Encoder[ConditionsExpectationType] = io.circe.Encoder.encodeString.contramap[ConditionsExpectationType]({
        case ClearSkies => "CLEAR_SKIES"
        case Fog => "FOG"
        case ThickClouds => "THICK_CLOUDS"
        case ThinClouds => "THIN_CLOUDS"
      })
      implicit val jsonDecoderConditionsExpectationType: io.circe.Decoder[ConditionsExpectationType] = io.circe.Decoder.decodeString.emap(_ match {
        case "CLEAR_SKIES" =>
          Right(ClearSkies)
        case "FOG" =>
          Right(Fog)
        case "THICK_CLOUDS" =>
          Right(ThickClouds)
        case "THIN_CLOUDS" =>
          Right(ThinClouds)
        case other =>
          Left(s"Invalid value [$other]")
      })
    }
    sealed trait EditType
    object EditType {
      case object Created extends EditType()
      case object Updated extends EditType()
      implicit val eqEditType: cats.Eq[EditType] = cats.Eq.fromUniversalEquals
      implicit val showEditType: cats.Show[EditType] = cats.Show.fromToString
      implicit val jsonEncoderEditType: io.circe.Encoder[EditType] = io.circe.Encoder.encodeString.contramap[EditType]({
        case Created => "CREATED"
        case Updated => "UPDATED"
      })
      implicit val jsonDecoderEditType: io.circe.Decoder[EditType] = io.circe.Decoder.decodeString.emap(_ match {
        case "CREATED" =>
          Right(Created)
        case "UPDATED" =>
          Right(Updated)
        case other =>
          Left(s"Invalid value [$other]")
      })
    }
    sealed trait ExecutionEventType
    object ExecutionEventType {
      case object Sequence extends ExecutionEventType()
      case object Step extends ExecutionEventType()
      case object Dataset extends ExecutionEventType()
      implicit val eqExecutionEventType: cats.Eq[ExecutionEventType] = cats.Eq.fromUniversalEquals
      implicit val showExecutionEventType: cats.Show[ExecutionEventType] = cats.Show.fromToString
      implicit val jsonEncoderExecutionEventType: io.circe.Encoder[ExecutionEventType] = io.circe.Encoder.encodeString.contramap[ExecutionEventType]({
        case Sequence => "SEQUENCE"
        case Step => "STEP"
        case Dataset => "DATASET"
      })
      implicit val jsonDecoderExecutionEventType: io.circe.Decoder[ExecutionEventType] = io.circe.Decoder.decodeString.emap(_ match {
        case "SEQUENCE" =>
          Right(Sequence)
        case "STEP" =>
          Right(Step)
        case "DATASET" =>
          Right(Dataset)
        case other =>
          Left(s"Invalid value [$other]")
      })
    }
    sealed trait UserInvitationStatus
    object UserInvitationStatus {
      case object Pending extends UserInvitationStatus()
      case object Redeemed extends UserInvitationStatus()
      case object Declined extends UserInvitationStatus()
      case object Revoked extends UserInvitationStatus()
      implicit val eqUserInvitationStatus: cats.Eq[UserInvitationStatus] = cats.Eq.fromUniversalEquals
      implicit val showUserInvitationStatus: cats.Show[UserInvitationStatus] = cats.Show.fromToString
      implicit val jsonEncoderUserInvitationStatus: io.circe.Encoder[UserInvitationStatus] = io.circe.Encoder.encodeString.contramap[UserInvitationStatus]({
        case Pending => "PENDING"
        case Redeemed => "REDEEMED"
        case Declined => "DECLINED"
        case Revoked => "REVOKED"
      })
      implicit val jsonDecoderUserInvitationStatus: io.circe.Decoder[UserInvitationStatus] = io.circe.Decoder.decodeString.emap(_ match {
        case "PENDING" =>
          Right(Pending)
        case "REDEEMED" =>
          Right(Redeemed)
        case "DECLINED" =>
          Right(Declined)
        case "REVOKED" =>
          Right(Revoked)
        case other =>
          Left(s"Invalid value [$other]")
      })
    }
    sealed trait ProgramUserRole
    object ProgramUserRole {
      case object Coi extends ProgramUserRole()
      case object Observer extends ProgramUserRole()
      case object Support extends ProgramUserRole()
      implicit val eqProgramUserRole: cats.Eq[ProgramUserRole] = cats.Eq.fromUniversalEquals
      implicit val showProgramUserRole: cats.Show[ProgramUserRole] = cats.Show.fromToString
      implicit val jsonEncoderProgramUserRole: io.circe.Encoder[ProgramUserRole] = io.circe.Encoder.encodeString.contramap[ProgramUserRole]({
        case Coi => "COI"
        case Observer => "OBSERVER"
        case Support => "SUPPORT"
      })
      implicit val jsonDecoderProgramUserRole: io.circe.Decoder[ProgramUserRole] = io.circe.Decoder.decodeString.emap(_ match {
        case "COI" =>
          Right(Coi)
        case "OBSERVER" =>
          Right(Observer)
        case "SUPPORT" =>
          Right(Support)
        case other =>
          Left(s"Invalid value [$other]")
      })
    }
    sealed trait ProgramUserSupportRoleType
    object ProgramUserSupportRoleType {
      case object Staff extends ProgramUserSupportRoleType()
      case object Partner extends ProgramUserSupportRoleType()
      implicit val eqProgramUserSupportRoleType: cats.Eq[ProgramUserSupportRoleType] = cats.Eq.fromUniversalEquals
      implicit val showProgramUserSupportRoleType: cats.Show[ProgramUserSupportRoleType] = cats.Show.fromToString
      implicit val jsonEncoderProgramUserSupportRoleType: io.circe.Encoder[ProgramUserSupportRoleType] = io.circe.Encoder.encodeString.contramap[ProgramUserSupportRoleType]({
        case Staff => "STAFF"
        case Partner => "PARTNER"
      })
      implicit val jsonDecoderProgramUserSupportRoleType: io.circe.Decoder[ProgramUserSupportRoleType] = io.circe.Decoder.decodeString.emap(_ match {
        case "STAFF" =>
          Right(Staff)
        case "PARTNER" =>
          Right(Partner)
        case other =>
          Left(s"Invalid value [$other]")
      })
    }
    sealed trait ObservingModeType
    object ObservingModeType {
      case object GmosNorthLongSlit extends ObservingModeType()
      case object GmosSouthLongSlit extends ObservingModeType()
      implicit val eqObservingModeType: cats.Eq[ObservingModeType] = cats.Eq.fromUniversalEquals
      implicit val showObservingModeType: cats.Show[ObservingModeType] = cats.Show.fromToString
      implicit val jsonEncoderObservingModeType: io.circe.Encoder[ObservingModeType] = io.circe.Encoder.encodeString.contramap[ObservingModeType]({
        case GmosNorthLongSlit => "GMOS_NORTH_LONG_SLIT"
        case GmosSouthLongSlit => "GMOS_SOUTH_LONG_SLIT"
      })
      implicit val jsonDecoderObservingModeType: io.circe.Decoder[ObservingModeType] = io.circe.Decoder.decodeString.emap(_ match {
        case "GMOS_NORTH_LONG_SLIT" =>
          Right(GmosNorthLongSlit)
        case "GMOS_SOUTH_LONG_SLIT" =>
          Right(GmosSouthLongSlit)
        case other =>
          Left(s"Invalid value [$other]")
      })
    }
    sealed trait Existence
    object Existence {
      case object Present extends Existence()
      case object Deleted extends Existence()
      implicit val eqExistence: cats.Eq[Existence] = cats.Eq.fromUniversalEquals
      implicit val showExistence: cats.Show[Existence] = cats.Show.fromToString
      implicit val jsonEncoderExistence: io.circe.Encoder[Existence] = io.circe.Encoder.encodeString.contramap[Existence]({
        case Present => "PRESENT"
        case Deleted => "DELETED"
      })
      implicit val jsonDecoderExistence: io.circe.Decoder[Existence] = io.circe.Decoder.decodeString.emap(_ match {
        case "PRESENT" =>
          Right(Present)
        case "DELETED" =>
          Right(Deleted)
        case other =>
          Left(s"Invalid value [$other]")
      })
    }
    sealed trait GuideProbe
    object GuideProbe {
      case object Pwfs1 extends GuideProbe()
      case object Pwfs2 extends GuideProbe()
      case object GmosOiwfs extends GuideProbe()
      implicit val eqGuideProbe: cats.Eq[GuideProbe] = cats.Eq.fromUniversalEquals
      implicit val showGuideProbe: cats.Show[GuideProbe] = cats.Show.fromToString
      implicit val jsonEncoderGuideProbe: io.circe.Encoder[GuideProbe] = io.circe.Encoder.encodeString.contramap[GuideProbe]({
        case Pwfs1 => "PWFS_1"
        case Pwfs2 => "PWFS_2"
        case GmosOiwfs => "GMOS_OIWFS"
      })
      implicit val jsonDecoderGuideProbe: io.circe.Decoder[GuideProbe] = io.circe.Decoder.decodeString.emap(_ match {
        case "PWFS_1" =>
          Right(Pwfs1)
        case "PWFS_2" =>
          Right(Pwfs2)
        case "GMOS_OIWFS" =>
          Right(GmosOiwfs)
        case other =>
          Left(s"Invalid value [$other]")
      })
    }
    sealed trait PosAngleConstraintMode
    object PosAngleConstraintMode {
      case object Unbounded extends PosAngleConstraintMode()
      case object Fixed extends PosAngleConstraintMode()
      case object AllowFlip extends PosAngleConstraintMode()
      case object AverageParallactic extends PosAngleConstraintMode()
      case object ParallacticOverride extends PosAngleConstraintMode()
      implicit val eqPosAngleConstraintMode: cats.Eq[PosAngleConstraintMode] = cats.Eq.fromUniversalEquals
      implicit val showPosAngleConstraintMode: cats.Show[PosAngleConstraintMode] = cats.Show.fromToString
      implicit val jsonEncoderPosAngleConstraintMode: io.circe.Encoder[PosAngleConstraintMode] = io.circe.Encoder.encodeString.contramap[PosAngleConstraintMode]({
        case Unbounded => "UNBOUNDED"
        case Fixed => "FIXED"
        case AllowFlip => "ALLOW_FLIP"
        case AverageParallactic => "AVERAGE_PARALLACTIC"
        case ParallacticOverride => "PARALLACTIC_OVERRIDE"
      })
      implicit val jsonDecoderPosAngleConstraintMode: io.circe.Decoder[PosAngleConstraintMode] = io.circe.Decoder.decodeString.emap(_ match {
        case "UNBOUNDED" =>
          Right(Unbounded)
        case "FIXED" =>
          Right(Fixed)
        case "ALLOW_FLIP" =>
          Right(AllowFlip)
        case "AVERAGE_PARALLACTIC" =>
          Right(AverageParallactic)
        case "PARALLACTIC_OVERRIDE" =>
          Right(ParallacticOverride)
        case other =>
          Left(s"Invalid value [$other]")
      })
    }
    sealed trait ProgramType
    object ProgramType {
      case object Calibration extends ProgramType()
      case object Engineering extends ProgramType()
      case object Example extends ProgramType()
      case object Library extends ProgramType()
      case object Science extends ProgramType()
      implicit val eqProgramType: cats.Eq[ProgramType] = cats.Eq.fromUniversalEquals
      implicit val showProgramType: cats.Show[ProgramType] = cats.Show.fromToString
      implicit val jsonEncoderProgramType: io.circe.Encoder[ProgramType] = io.circe.Encoder.encodeString.contramap[ProgramType]({
        case Calibration => "CALIBRATION"
        case Engineering => "ENGINEERING"
        case Example => "EXAMPLE"
        case Library => "LIBRARY"
        case Science => "SCIENCE"
      })
      implicit val jsonDecoderProgramType: io.circe.Decoder[ProgramType] = io.circe.Decoder.decodeString.emap(_ match {
        case "CALIBRATION" =>
          Right(Calibration)
        case "ENGINEERING" =>
          Right(Engineering)
        case "EXAMPLE" =>
          Right(Example)
        case "LIBRARY" =>
          Right(Library)
        case "SCIENCE" =>
          Right(Science)
        case other =>
          Left(s"Invalid value [$other]")
      })
    }
    sealed trait ProposalClassEnum
    object ProposalClassEnum {
      case object Classical extends ProposalClassEnum()
      case object DemoScience extends ProposalClassEnum()
      case object DirectorsTime extends ProposalClassEnum()
      case object Exchange extends ProposalClassEnum()
      case object FastTurnaround extends ProposalClassEnum()
      case object Intensive extends ProposalClassEnum()
      case object LargeProgram extends ProposalClassEnum()
      case object PoorWeather extends ProposalClassEnum()
      case object Queue extends ProposalClassEnum()
      case object SystemVerification extends ProposalClassEnum()
      implicit val eqProposalClassEnum: cats.Eq[ProposalClassEnum] = cats.Eq.fromUniversalEquals
      implicit val showProposalClassEnum: cats.Show[ProposalClassEnum] = cats.Show.fromToString
      implicit val jsonEncoderProposalClassEnum: io.circe.Encoder[ProposalClassEnum] = io.circe.Encoder.encodeString.contramap[ProposalClassEnum]({
        case Classical => "CLASSICAL"
        case DemoScience => "DEMO_SCIENCE"
        case DirectorsTime => "DIRECTORS_TIME"
        case Exchange => "EXCHANGE"
        case FastTurnaround => "FAST_TURNAROUND"
        case Intensive => "INTENSIVE"
        case LargeProgram => "LARGE_PROGRAM"
        case PoorWeather => "POOR_WEATHER"
        case Queue => "QUEUE"
        case SystemVerification => "SYSTEM_VERIFICATION"
      })
      implicit val jsonDecoderProposalClassEnum: io.circe.Decoder[ProposalClassEnum] = io.circe.Decoder.decodeString.emap(_ match {
        case "CLASSICAL" =>
          Right(Classical)
        case "DEMO_SCIENCE" =>
          Right(DemoScience)
        case "DIRECTORS_TIME" =>
          Right(DirectorsTime)
        case "EXCHANGE" =>
          Right(Exchange)
        case "FAST_TURNAROUND" =>
          Right(FastTurnaround)
        case "INTENSIVE" =>
          Right(Intensive)
        case "LARGE_PROGRAM" =>
          Right(LargeProgram)
        case "POOR_WEATHER" =>
          Right(PoorWeather)
        case "QUEUE" =>
          Right(Queue)
        case "SYSTEM_VERIFICATION" =>
          Right(SystemVerification)
        case other =>
          Left(s"Invalid value [$other]")
      })
    }
    sealed trait ScienceSubtype
    object ScienceSubtype {
      case object Classical extends ScienceSubtype()
      case object DirectorsTime extends ScienceSubtype()
      case object FastTurnaround extends ScienceSubtype()
      case object LargeProgram extends ScienceSubtype()
      case object Queue extends ScienceSubtype()
      case object DemoScience extends ScienceSubtype()
      case object SystemVerification extends ScienceSubtype()
      implicit val eqScienceSubtype: cats.Eq[ScienceSubtype] = cats.Eq.fromUniversalEquals
      implicit val showScienceSubtype: cats.Show[ScienceSubtype] = cats.Show.fromToString
      implicit val jsonEncoderScienceSubtype: io.circe.Encoder[ScienceSubtype] = io.circe.Encoder.encodeString.contramap[ScienceSubtype]({
        case Classical => "CLASSICAL"
        case DirectorsTime => "DIRECTORS_TIME"
        case FastTurnaround => "FAST_TURNAROUND"
        case LargeProgram => "LARGE_PROGRAM"
        case Queue => "QUEUE"
        case DemoScience => "DEMO_SCIENCE"
        case SystemVerification => "SYSTEM_VERIFICATION"
      })
      implicit val jsonDecoderScienceSubtype: io.circe.Decoder[ScienceSubtype] = io.circe.Decoder.decodeString.emap(_ match {
        case "CLASSICAL" =>
          Right(Classical)
        case "DIRECTORS_TIME" =>
          Right(DirectorsTime)
        case "FAST_TURNAROUND" =>
          Right(FastTurnaround)
        case "LARGE_PROGRAM" =>
          Right(LargeProgram)
        case "QUEUE" =>
          Right(Queue)
        case "DEMO_SCIENCE" =>
          Right(DemoScience)
        case "SYSTEM_VERIFICATION" =>
          Right(SystemVerification)
        case other =>
          Left(s"Invalid value [$other]")
      })
    }
    sealed trait Site
    object Site {
      case object Gn extends Site()
      case object Gs extends Site()
      implicit val eqSite: cats.Eq[Site] = cats.Eq.fromUniversalEquals
      implicit val showSite: cats.Show[Site] = cats.Show.fromToString
      implicit val jsonEncoderSite: io.circe.Encoder[Site] = io.circe.Encoder.encodeString.contramap[Site]({
        case Gn => "GN"
        case Gs => "GS"
      })
      implicit val jsonDecoderSite: io.circe.Decoder[Site] = io.circe.Decoder.decodeString.emap(_ match {
        case "GN" =>
          Right(Gn)
        case "GS" =>
          Right(Gs)
        case other =>
          Left(s"Invalid value [$other]")
      })
    }
    sealed trait TimeChargeCorrectionOp
    object TimeChargeCorrectionOp {
      case object Add extends TimeChargeCorrectionOp()
      case object Subtract extends TimeChargeCorrectionOp()
      implicit val eqTimeChargeCorrectionOp: cats.Eq[TimeChargeCorrectionOp] = cats.Eq.fromUniversalEquals
      implicit val showTimeChargeCorrectionOp: cats.Show[TimeChargeCorrectionOp] = cats.Show.fromToString
      implicit val jsonEncoderTimeChargeCorrectionOp: io.circe.Encoder[TimeChargeCorrectionOp] = io.circe.Encoder.encodeString.contramap[TimeChargeCorrectionOp]({
        case Add => "ADD"
        case Subtract => "SUBTRACT"
      })
      implicit val jsonDecoderTimeChargeCorrectionOp: io.circe.Decoder[TimeChargeCorrectionOp] = io.circe.Decoder.decodeString.emap(_ match {
        case "ADD" =>
          Right(Add)
        case "SUBTRACT" =>
          Right(Subtract)
        case other =>
          Left(s"Invalid value [$other]")
      })
    }
    sealed trait UserType
    object UserType {
      case object Guest extends UserType()
      case object Standard extends UserType()
      case object Service extends UserType()
      implicit val eqUserType: cats.Eq[UserType] = cats.Eq.fromUniversalEquals
      implicit val showUserType: cats.Show[UserType] = cats.Show.fromToString
      implicit val jsonEncoderUserType: io.circe.Encoder[UserType] = io.circe.Encoder.encodeString.contramap[UserType]({
        case Guest => "GUEST"
        case Standard => "STANDARD"
        case Service => "SERVICE"
      })
      implicit val jsonDecoderUserType: io.circe.Decoder[UserType] = io.circe.Decoder.decodeString.emap(_ match {
        case "GUEST" =>
          Right(Guest)
        case "STANDARD" =>
          Right(Standard)
        case "SERVICE" =>
          Right(Service)
        case other =>
          Left(s"Invalid value [$other]")
      })
    }
  }
  object Types {
    import Scalars._
    ignoreUnusedImportScalars()
    import Enums._
    ignoreUnusedImportEnums()
    def ignoreUnusedImportTypes(): Unit = ()
    case class AddDatasetEventInput(val datasetId: DatasetId, val datasetStage: DatasetStage)
    object AddDatasetEventInput {
      val datasetId: monocle.Lens[AddDatasetEventInput, DatasetId] = monocle.macros.GenLens[AddDatasetEventInput](_.datasetId)
      val datasetStage: monocle.Lens[AddDatasetEventInput, DatasetStage] = monocle.macros.GenLens[AddDatasetEventInput](_.datasetStage)
      implicit val eqAddDatasetEventInput: cats.Eq[AddDatasetEventInput] = cats.Eq.fromUniversalEquals
      implicit val showAddDatasetEventInput: cats.Show[AddDatasetEventInput] = cats.Show.fromToString
      implicit val jsonEncoderAddDatasetEventInput: io.circe.Encoder.AsObject[AddDatasetEventInput] = io.circe.generic.semiauto.deriveEncoder[AddDatasetEventInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class AddSequenceEventInput(val visitId: VisitId, val command: SequenceCommand)
    object AddSequenceEventInput {
      val visitId: monocle.Lens[AddSequenceEventInput, VisitId] = monocle.macros.GenLens[AddSequenceEventInput](_.visitId)
      val command: monocle.Lens[AddSequenceEventInput, SequenceCommand] = monocle.macros.GenLens[AddSequenceEventInput](_.command)
      implicit val eqAddSequenceEventInput: cats.Eq[AddSequenceEventInput] = cats.Eq.fromUniversalEquals
      implicit val showAddSequenceEventInput: cats.Show[AddSequenceEventInput] = cats.Show.fromToString
      implicit val jsonEncoderAddSequenceEventInput: io.circe.Encoder.AsObject[AddSequenceEventInput] = io.circe.generic.semiauto.deriveEncoder[AddSequenceEventInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class AddStepEventInput(val stepId: StepId, val stepStage: StepStage)
    object AddStepEventInput {
      val stepId: monocle.Lens[AddStepEventInput, StepId] = monocle.macros.GenLens[AddStepEventInput](_.stepId)
      val stepStage: monocle.Lens[AddStepEventInput, StepStage] = monocle.macros.GenLens[AddStepEventInput](_.stepStage)
      implicit val eqAddStepEventInput: cats.Eq[AddStepEventInput] = cats.Eq.fromUniversalEquals
      implicit val showAddStepEventInput: cats.Show[AddStepEventInput] = cats.Show.fromToString
      implicit val jsonEncoderAddStepEventInput: io.circe.Encoder.AsObject[AddStepEventInput] = io.circe.generic.semiauto.deriveEncoder[AddStepEventInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class AddTimeChargeCorrectionInput(val visitId: VisitId, val correction: TimeChargeCorrectionInput)
    object AddTimeChargeCorrectionInput {
      val visitId: monocle.Lens[AddTimeChargeCorrectionInput, VisitId] = monocle.macros.GenLens[AddTimeChargeCorrectionInput](_.visitId)
      val correction: monocle.Lens[AddTimeChargeCorrectionInput, TimeChargeCorrectionInput] = monocle.macros.GenLens[AddTimeChargeCorrectionInput](_.correction)
      implicit val eqAddTimeChargeCorrectionInput: cats.Eq[AddTimeChargeCorrectionInput] = cats.Eq.fromUniversalEquals
      implicit val showAddTimeChargeCorrectionInput: cats.Show[AddTimeChargeCorrectionInput] = cats.Show.fromToString
      implicit val jsonEncoderAddTimeChargeCorrectionInput: io.circe.Encoder.AsObject[AddTimeChargeCorrectionInput] = io.circe.generic.semiauto.deriveEncoder[AddTimeChargeCorrectionInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class AirMassRangeInput(val min: clue.data.Input[PosBigDecimal] = clue.data.Ignore, val max: clue.data.Input[PosBigDecimal] = clue.data.Ignore)
    object AirMassRangeInput {
      val min: monocle.Lens[AirMassRangeInput, clue.data.Input[PosBigDecimal]] = monocle.macros.GenLens[AirMassRangeInput](_.min)
      val max: monocle.Lens[AirMassRangeInput, clue.data.Input[PosBigDecimal]] = monocle.macros.GenLens[AirMassRangeInput](_.max)
      implicit val eqAirMassRangeInput: cats.Eq[AirMassRangeInput] = cats.Eq.fromUniversalEquals
      implicit val showAirMassRangeInput: cats.Show[AirMassRangeInput] = cats.Show.fromToString
      implicit val jsonEncoderAirMassRangeInput: io.circe.Encoder.AsObject[AirMassRangeInput] = io.circe.generic.semiauto.deriveEncoder[AirMassRangeInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class AngleInput(val microarcseconds: clue.data.Input[Long] = clue.data.Ignore, val microseconds: clue.data.Input[BigDecimal] = clue.data.Ignore, val milliarcseconds: clue.data.Input[BigDecimal] = clue.data.Ignore, val milliseconds: clue.data.Input[BigDecimal] = clue.data.Ignore, val arcseconds: clue.data.Input[BigDecimal] = clue.data.Ignore, val seconds: clue.data.Input[BigDecimal] = clue.data.Ignore, val arcminutes: clue.data.Input[BigDecimal] = clue.data.Ignore, val minutes: clue.data.Input[BigDecimal] = clue.data.Ignore, val degrees: clue.data.Input[BigDecimal] = clue.data.Ignore, val hours: clue.data.Input[BigDecimal] = clue.data.Ignore, val dms: clue.data.Input[String] = clue.data.Ignore, val hms: clue.data.Input[String] = clue.data.Ignore)
    object AngleInput {
      val microarcseconds: monocle.Lens[AngleInput, clue.data.Input[Long]] = monocle.macros.GenLens[AngleInput](_.microarcseconds)
      val microseconds: monocle.Lens[AngleInput, clue.data.Input[BigDecimal]] = monocle.macros.GenLens[AngleInput](_.microseconds)
      val milliarcseconds: monocle.Lens[AngleInput, clue.data.Input[BigDecimal]] = monocle.macros.GenLens[AngleInput](_.milliarcseconds)
      val milliseconds: monocle.Lens[AngleInput, clue.data.Input[BigDecimal]] = monocle.macros.GenLens[AngleInput](_.milliseconds)
      val arcseconds: monocle.Lens[AngleInput, clue.data.Input[BigDecimal]] = monocle.macros.GenLens[AngleInput](_.arcseconds)
      val seconds: monocle.Lens[AngleInput, clue.data.Input[BigDecimal]] = monocle.macros.GenLens[AngleInput](_.seconds)
      val arcminutes: monocle.Lens[AngleInput, clue.data.Input[BigDecimal]] = monocle.macros.GenLens[AngleInput](_.arcminutes)
      val minutes: monocle.Lens[AngleInput, clue.data.Input[BigDecimal]] = monocle.macros.GenLens[AngleInput](_.minutes)
      val degrees: monocle.Lens[AngleInput, clue.data.Input[BigDecimal]] = monocle.macros.GenLens[AngleInput](_.degrees)
      val hours: monocle.Lens[AngleInput, clue.data.Input[BigDecimal]] = monocle.macros.GenLens[AngleInput](_.hours)
      val dms: monocle.Lens[AngleInput, clue.data.Input[String]] = monocle.macros.GenLens[AngleInput](_.dms)
      val hms: monocle.Lens[AngleInput, clue.data.Input[String]] = monocle.macros.GenLens[AngleInput](_.hms)
      implicit val eqAngleInput: cats.Eq[AngleInput] = cats.Eq.fromUniversalEquals
      implicit val showAngleInput: cats.Show[AngleInput] = cats.Show.fromToString
      implicit val jsonEncoderAngleInput: io.circe.Encoder.AsObject[AngleInput] = io.circe.generic.semiauto.deriveEncoder[AngleInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class BandBrightnessIntegratedInput(val band: Band, val value: clue.data.Input[BigDecimal] = clue.data.Ignore, val units: clue.data.Input[BrightnessIntegratedUnits] = clue.data.Ignore, val error: clue.data.Input[BigDecimal] = clue.data.Ignore)
    object BandBrightnessIntegratedInput {
      val band: monocle.Lens[BandBrightnessIntegratedInput, Band] = monocle.macros.GenLens[BandBrightnessIntegratedInput](_.band)
      val value: monocle.Lens[BandBrightnessIntegratedInput, clue.data.Input[BigDecimal]] = monocle.macros.GenLens[BandBrightnessIntegratedInput](_.value)
      val units: monocle.Lens[BandBrightnessIntegratedInput, clue.data.Input[BrightnessIntegratedUnits]] = monocle.macros.GenLens[BandBrightnessIntegratedInput](_.units)
      val error: monocle.Lens[BandBrightnessIntegratedInput, clue.data.Input[BigDecimal]] = monocle.macros.GenLens[BandBrightnessIntegratedInput](_.error)
      implicit val eqBandBrightnessIntegratedInput: cats.Eq[BandBrightnessIntegratedInput] = cats.Eq.fromUniversalEquals
      implicit val showBandBrightnessIntegratedInput: cats.Show[BandBrightnessIntegratedInput] = cats.Show.fromToString
      implicit val jsonEncoderBandBrightnessIntegratedInput: io.circe.Encoder.AsObject[BandBrightnessIntegratedInput] = io.circe.generic.semiauto.deriveEncoder[BandBrightnessIntegratedInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class BandBrightnessSurfaceInput(val band: Band, val value: clue.data.Input[BigDecimal] = clue.data.Ignore, val units: clue.data.Input[BrightnessSurfaceUnits] = clue.data.Ignore, val error: clue.data.Input[BigDecimal] = clue.data.Ignore)
    object BandBrightnessSurfaceInput {
      val band: monocle.Lens[BandBrightnessSurfaceInput, Band] = monocle.macros.GenLens[BandBrightnessSurfaceInput](_.band)
      val value: monocle.Lens[BandBrightnessSurfaceInput, clue.data.Input[BigDecimal]] = monocle.macros.GenLens[BandBrightnessSurfaceInput](_.value)
      val units: monocle.Lens[BandBrightnessSurfaceInput, clue.data.Input[BrightnessSurfaceUnits]] = monocle.macros.GenLens[BandBrightnessSurfaceInput](_.units)
      val error: monocle.Lens[BandBrightnessSurfaceInput, clue.data.Input[BigDecimal]] = monocle.macros.GenLens[BandBrightnessSurfaceInput](_.error)
      implicit val eqBandBrightnessSurfaceInput: cats.Eq[BandBrightnessSurfaceInput] = cats.Eq.fromUniversalEquals
      implicit val showBandBrightnessSurfaceInput: cats.Show[BandBrightnessSurfaceInput] = cats.Show.fromToString
      implicit val jsonEncoderBandBrightnessSurfaceInput: io.circe.Encoder.AsObject[BandBrightnessSurfaceInput] = io.circe.generic.semiauto.deriveEncoder[BandBrightnessSurfaceInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class BandNormalizedIntegratedInput(val sed: clue.data.Input[UnnormalizedSedInput] = clue.data.Ignore, val brightnesses: clue.data.Input[List[BandBrightnessIntegratedInput]] = clue.data.Ignore)
    object BandNormalizedIntegratedInput {
      val sed: monocle.Lens[BandNormalizedIntegratedInput, clue.data.Input[UnnormalizedSedInput]] = monocle.macros.GenLens[BandNormalizedIntegratedInput](_.sed)
      val brightnesses: monocle.Lens[BandNormalizedIntegratedInput, clue.data.Input[List[BandBrightnessIntegratedInput]]] = monocle.macros.GenLens[BandNormalizedIntegratedInput](_.brightnesses)
      implicit val eqBandNormalizedIntegratedInput: cats.Eq[BandNormalizedIntegratedInput] = cats.Eq.fromUniversalEquals
      implicit val showBandNormalizedIntegratedInput: cats.Show[BandNormalizedIntegratedInput] = cats.Show.fromToString
      implicit val jsonEncoderBandNormalizedIntegratedInput: io.circe.Encoder.AsObject[BandNormalizedIntegratedInput] = io.circe.generic.semiauto.deriveEncoder[BandNormalizedIntegratedInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class BandNormalizedSurfaceInput(val sed: clue.data.Input[UnnormalizedSedInput] = clue.data.Ignore, val brightnesses: clue.data.Input[List[BandBrightnessSurfaceInput]] = clue.data.Ignore)
    object BandNormalizedSurfaceInput {
      val sed: monocle.Lens[BandNormalizedSurfaceInput, clue.data.Input[UnnormalizedSedInput]] = monocle.macros.GenLens[BandNormalizedSurfaceInput](_.sed)
      val brightnesses: monocle.Lens[BandNormalizedSurfaceInput, clue.data.Input[List[BandBrightnessSurfaceInput]]] = monocle.macros.GenLens[BandNormalizedSurfaceInput](_.brightnesses)
      implicit val eqBandNormalizedSurfaceInput: cats.Eq[BandNormalizedSurfaceInput] = cats.Eq.fromUniversalEquals
      implicit val showBandNormalizedSurfaceInput: cats.Show[BandNormalizedSurfaceInput] = cats.Show.fromToString
      implicit val jsonEncoderBandNormalizedSurfaceInput: io.circe.Encoder.AsObject[BandNormalizedSurfaceInput] = io.circe.generic.semiauto.deriveEncoder[BandNormalizedSurfaceInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class CatalogInfoInput(val name: clue.data.Input[CatalogName] = clue.data.Ignore, val id: clue.data.Input[NonEmptyString] = clue.data.Ignore, val objectType: clue.data.Input[NonEmptyString] = clue.data.Ignore)
    object CatalogInfoInput {
      val name: monocle.Lens[CatalogInfoInput, clue.data.Input[CatalogName]] = monocle.macros.GenLens[CatalogInfoInput](_.name)
      val id: monocle.Lens[CatalogInfoInput, clue.data.Input[NonEmptyString]] = monocle.macros.GenLens[CatalogInfoInput](_.id)
      val objectType: monocle.Lens[CatalogInfoInput, clue.data.Input[NonEmptyString]] = monocle.macros.GenLens[CatalogInfoInput](_.objectType)
      implicit val eqCatalogInfoInput: cats.Eq[CatalogInfoInput] = cats.Eq.fromUniversalEquals
      implicit val showCatalogInfoInput: cats.Show[CatalogInfoInput] = cats.Show.fromToString
      implicit val jsonEncoderCatalogInfoInput: io.circe.Encoder.AsObject[CatalogInfoInput] = io.circe.generic.semiauto.deriveEncoder[CatalogInfoInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class ClassicalInput(val minPercentTime: clue.data.Input[IntPercent] = clue.data.Ignore)
    object ClassicalInput {
      val minPercentTime: monocle.Lens[ClassicalInput, clue.data.Input[IntPercent]] = monocle.macros.GenLens[ClassicalInput](_.minPercentTime)
      implicit val eqClassicalInput: cats.Eq[ClassicalInput] = cats.Eq.fromUniversalEquals
      implicit val showClassicalInput: cats.Show[ClassicalInput] = cats.Show.fromToString
      implicit val jsonEncoderClassicalInput: io.circe.Encoder.AsObject[ClassicalInput] = io.circe.generic.semiauto.deriveEncoder[ClassicalInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class CloneObservationInput(val observationId: ObservationId, val SET: clue.data.Input[ObservationPropertiesInput] = clue.data.Ignore)
    object CloneObservationInput {
      val observationId: monocle.Lens[CloneObservationInput, ObservationId] = monocle.macros.GenLens[CloneObservationInput](_.observationId)
      val SET: monocle.Lens[CloneObservationInput, clue.data.Input[ObservationPropertiesInput]] = monocle.macros.GenLens[CloneObservationInput](_.SET)
      implicit val eqCloneObservationInput: cats.Eq[CloneObservationInput] = cats.Eq.fromUniversalEquals
      implicit val showCloneObservationInput: cats.Show[CloneObservationInput] = cats.Show.fromToString
      implicit val jsonEncoderCloneObservationInput: io.circe.Encoder.AsObject[CloneObservationInput] = io.circe.generic.semiauto.deriveEncoder[CloneObservationInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class CloneTargetInput(val targetId: TargetId, val SET: clue.data.Input[TargetPropertiesInput] = clue.data.Ignore, val REPLACE_IN: clue.data.Input[List[ObservationId]] = clue.data.Ignore)
    object CloneTargetInput {
      val targetId: monocle.Lens[CloneTargetInput, TargetId] = monocle.macros.GenLens[CloneTargetInput](_.targetId)
      val SET: monocle.Lens[CloneTargetInput, clue.data.Input[TargetPropertiesInput]] = monocle.macros.GenLens[CloneTargetInput](_.SET)
      val REPLACE_IN: monocle.Lens[CloneTargetInput, clue.data.Input[List[ObservationId]]] = monocle.macros.GenLens[CloneTargetInput](_.REPLACE_IN)
      implicit val eqCloneTargetInput: cats.Eq[CloneTargetInput] = cats.Eq.fromUniversalEquals
      implicit val showCloneTargetInput: cats.Show[CloneTargetInput] = cats.Show.fromToString
      implicit val jsonEncoderCloneTargetInput: io.circe.Encoder.AsObject[CloneTargetInput] = io.circe.generic.semiauto.deriveEncoder[CloneTargetInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class ConstraintSetInput(val imageQuality: clue.data.Input[ImageQuality] = clue.data.Ignore, val cloudExtinction: clue.data.Input[CloudExtinction] = clue.data.Ignore, val skyBackground: clue.data.Input[SkyBackground] = clue.data.Ignore, val waterVapor: clue.data.Input[WaterVapor] = clue.data.Ignore, val elevationRange: clue.data.Input[ElevationRangeInput] = clue.data.Ignore)
    object ConstraintSetInput {
      val imageQuality: monocle.Lens[ConstraintSetInput, clue.data.Input[ImageQuality]] = monocle.macros.GenLens[ConstraintSetInput](_.imageQuality)
      val cloudExtinction: monocle.Lens[ConstraintSetInput, clue.data.Input[CloudExtinction]] = monocle.macros.GenLens[ConstraintSetInput](_.cloudExtinction)
      val skyBackground: monocle.Lens[ConstraintSetInput, clue.data.Input[SkyBackground]] = monocle.macros.GenLens[ConstraintSetInput](_.skyBackground)
      val waterVapor: monocle.Lens[ConstraintSetInput, clue.data.Input[WaterVapor]] = monocle.macros.GenLens[ConstraintSetInput](_.waterVapor)
      val elevationRange: monocle.Lens[ConstraintSetInput, clue.data.Input[ElevationRangeInput]] = monocle.macros.GenLens[ConstraintSetInput](_.elevationRange)
      implicit val eqConstraintSetInput: cats.Eq[ConstraintSetInput] = cats.Eq.fromUniversalEquals
      implicit val showConstraintSetInput: cats.Show[ConstraintSetInput] = cats.Show.fromToString
      implicit val jsonEncoderConstraintSetInput: io.circe.Encoder.AsObject[ConstraintSetInput] = io.circe.generic.semiauto.deriveEncoder[ConstraintSetInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class ConditionsEntryInput(val measurement: clue.data.Input[ConditionsMeasurementInput] = clue.data.Ignore, val intuition: clue.data.Input[ConditionsIntuitionInput] = clue.data.Ignore)
    object ConditionsEntryInput {
      val measurement: monocle.Lens[ConditionsEntryInput, clue.data.Input[ConditionsMeasurementInput]] = monocle.macros.GenLens[ConditionsEntryInput](_.measurement)
      val intuition: monocle.Lens[ConditionsEntryInput, clue.data.Input[ConditionsIntuitionInput]] = monocle.macros.GenLens[ConditionsEntryInput](_.intuition)
      implicit val eqConditionsEntryInput: cats.Eq[ConditionsEntryInput] = cats.Eq.fromUniversalEquals
      implicit val showConditionsEntryInput: cats.Show[ConditionsEntryInput] = cats.Show.fromToString
      implicit val jsonEncoderConditionsEntryInput: io.circe.Encoder.AsObject[ConditionsEntryInput] = io.circe.generic.semiauto.deriveEncoder[ConditionsEntryInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class ConditionsMeasurementInput(val source: ConditionsMeasurementSource, val seeing: clue.data.Input[AngleInput] = clue.data.Ignore, val extinction: clue.data.Input[Extinction] = clue.data.Ignore, val wavelength: clue.data.Input[WavelengthInput] = clue.data.Ignore, val azimuth: clue.data.Input[AngleInput] = clue.data.Ignore, val elevation: clue.data.Input[AngleInput] = clue.data.Ignore)
    object ConditionsMeasurementInput {
      val source: monocle.Lens[ConditionsMeasurementInput, ConditionsMeasurementSource] = monocle.macros.GenLens[ConditionsMeasurementInput](_.source)
      val seeing: monocle.Lens[ConditionsMeasurementInput, clue.data.Input[AngleInput]] = monocle.macros.GenLens[ConditionsMeasurementInput](_.seeing)
      val extinction: monocle.Lens[ConditionsMeasurementInput, clue.data.Input[Extinction]] = monocle.macros.GenLens[ConditionsMeasurementInput](_.extinction)
      val wavelength: monocle.Lens[ConditionsMeasurementInput, clue.data.Input[WavelengthInput]] = monocle.macros.GenLens[ConditionsMeasurementInput](_.wavelength)
      val azimuth: monocle.Lens[ConditionsMeasurementInput, clue.data.Input[AngleInput]] = monocle.macros.GenLens[ConditionsMeasurementInput](_.azimuth)
      val elevation: monocle.Lens[ConditionsMeasurementInput, clue.data.Input[AngleInput]] = monocle.macros.GenLens[ConditionsMeasurementInput](_.elevation)
      implicit val eqConditionsMeasurementInput: cats.Eq[ConditionsMeasurementInput] = cats.Eq.fromUniversalEquals
      implicit val showConditionsMeasurementInput: cats.Show[ConditionsMeasurementInput] = cats.Show.fromToString
      implicit val jsonEncoderConditionsMeasurementInput: io.circe.Encoder.AsObject[ConditionsMeasurementInput] = io.circe.generic.semiauto.deriveEncoder[ConditionsMeasurementInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class ConditionsIntuitionInput(val expectation: clue.data.Input[ConditionsExpectationInput] = clue.data.Ignore, val seeingTrend: clue.data.Input[SeeingTrend] = clue.data.Ignore)
    object ConditionsIntuitionInput {
      val expectation: monocle.Lens[ConditionsIntuitionInput, clue.data.Input[ConditionsExpectationInput]] = monocle.macros.GenLens[ConditionsIntuitionInput](_.expectation)
      val seeingTrend: monocle.Lens[ConditionsIntuitionInput, clue.data.Input[SeeingTrend]] = monocle.macros.GenLens[ConditionsIntuitionInput](_.seeingTrend)
      implicit val eqConditionsIntuitionInput: cats.Eq[ConditionsIntuitionInput] = cats.Eq.fromUniversalEquals
      implicit val showConditionsIntuitionInput: cats.Show[ConditionsIntuitionInput] = cats.Show.fromToString
      implicit val jsonEncoderConditionsIntuitionInput: io.circe.Encoder.AsObject[ConditionsIntuitionInput] = io.circe.generic.semiauto.deriveEncoder[ConditionsIntuitionInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class ConditionsExpectationInput(val `type`: ConditionsExpectationType, val timeframe: TimeSpanInput)
    object ConditionsExpectationInput {
      val `type`: monocle.Lens[ConditionsExpectationInput, ConditionsExpectationType] = monocle.macros.GenLens[ConditionsExpectationInput](_.`type`)
      val timeframe: monocle.Lens[ConditionsExpectationInput, TimeSpanInput] = monocle.macros.GenLens[ConditionsExpectationInput](_.timeframe)
      implicit val eqConditionsExpectationInput: cats.Eq[ConditionsExpectationInput] = cats.Eq.fromUniversalEquals
      implicit val showConditionsExpectationInput: cats.Show[ConditionsExpectationInput] = cats.Show.fromToString
      implicit val jsonEncoderConditionsExpectationInput: io.circe.Encoder.AsObject[ConditionsExpectationInput] = io.circe.generic.semiauto.deriveEncoder[ConditionsExpectationInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class CoordinatesInput(val ra: clue.data.Input[RightAscensionInput] = clue.data.Ignore, val dec: clue.data.Input[DeclinationInput] = clue.data.Ignore)
    object CoordinatesInput {
      val ra: monocle.Lens[CoordinatesInput, clue.data.Input[RightAscensionInput]] = monocle.macros.GenLens[CoordinatesInput](_.ra)
      val dec: monocle.Lens[CoordinatesInput, clue.data.Input[DeclinationInput]] = monocle.macros.GenLens[CoordinatesInput](_.dec)
      implicit val eqCoordinatesInput: cats.Eq[CoordinatesInput] = cats.Eq.fromUniversalEquals
      implicit val showCoordinatesInput: cats.Show[CoordinatesInput] = cats.Show.fromToString
      implicit val jsonEncoderCoordinatesInput: io.circe.Encoder.AsObject[CoordinatesInput] = io.circe.generic.semiauto.deriveEncoder[CoordinatesInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class CreateObservationInput(val programId: clue.data.Input[ProgramId] = clue.data.Ignore, val proposalReference: clue.data.Input[ProposalReferenceLabel] = clue.data.Ignore, val programReference: clue.data.Input[ProgramReferenceLabel] = clue.data.Ignore, val SET: clue.data.Input[ObservationPropertiesInput] = clue.data.Ignore)
    object CreateObservationInput {
      val programId: monocle.Lens[CreateObservationInput, clue.data.Input[ProgramId]] = monocle.macros.GenLens[CreateObservationInput](_.programId)
      val proposalReference: monocle.Lens[CreateObservationInput, clue.data.Input[ProposalReferenceLabel]] = monocle.macros.GenLens[CreateObservationInput](_.proposalReference)
      val programReference: monocle.Lens[CreateObservationInput, clue.data.Input[ProgramReferenceLabel]] = monocle.macros.GenLens[CreateObservationInput](_.programReference)
      val SET: monocle.Lens[CreateObservationInput, clue.data.Input[ObservationPropertiesInput]] = monocle.macros.GenLens[CreateObservationInput](_.SET)
      implicit val eqCreateObservationInput: cats.Eq[CreateObservationInput] = cats.Eq.fromUniversalEquals
      implicit val showCreateObservationInput: cats.Show[CreateObservationInput] = cats.Show.fromToString
      implicit val jsonEncoderCreateObservationInput: io.circe.Encoder.AsObject[CreateObservationInput] = io.circe.generic.semiauto.deriveEncoder[CreateObservationInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class CreateProgramInput(val SET: clue.data.Input[ProgramPropertiesInput] = clue.data.Ignore)
    object CreateProgramInput {
      val SET: monocle.Lens[CreateProgramInput, clue.data.Input[ProgramPropertiesInput]] = monocle.macros.GenLens[CreateProgramInput](_.SET)
      implicit val eqCreateProgramInput: cats.Eq[CreateProgramInput] = cats.Eq.fromUniversalEquals
      implicit val showCreateProgramInput: cats.Show[CreateProgramInput] = cats.Show.fromToString
      implicit val jsonEncoderCreateProgramInput: io.circe.Encoder.AsObject[CreateProgramInput] = io.circe.generic.semiauto.deriveEncoder[CreateProgramInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class CreateTargetInput(val programId: clue.data.Input[ProgramId] = clue.data.Ignore, val proposalReference: clue.data.Input[ProposalReferenceLabel] = clue.data.Ignore, val programReference: clue.data.Input[ProgramReferenceLabel] = clue.data.Ignore, val SET: TargetPropertiesInput)
    object CreateTargetInput {
      val programId: monocle.Lens[CreateTargetInput, clue.data.Input[ProgramId]] = monocle.macros.GenLens[CreateTargetInput](_.programId)
      val proposalReference: monocle.Lens[CreateTargetInput, clue.data.Input[ProposalReferenceLabel]] = monocle.macros.GenLens[CreateTargetInput](_.proposalReference)
      val programReference: monocle.Lens[CreateTargetInput, clue.data.Input[ProgramReferenceLabel]] = monocle.macros.GenLens[CreateTargetInput](_.programReference)
      val SET: monocle.Lens[CreateTargetInput, TargetPropertiesInput] = monocle.macros.GenLens[CreateTargetInput](_.SET)
      implicit val eqCreateTargetInput: cats.Eq[CreateTargetInput] = cats.Eq.fromUniversalEquals
      implicit val showCreateTargetInput: cats.Show[CreateTargetInput] = cats.Show.fromToString
      implicit val jsonEncoderCreateTargetInput: io.circe.Encoder.AsObject[CreateTargetInput] = io.circe.generic.semiauto.deriveEncoder[CreateTargetInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class DatasetPropertiesInput(val qaState: clue.data.Input[DatasetQaState] = clue.data.Ignore)
    object DatasetPropertiesInput {
      val qaState: monocle.Lens[DatasetPropertiesInput, clue.data.Input[DatasetQaState]] = monocle.macros.GenLens[DatasetPropertiesInput](_.qaState)
      implicit val eqDatasetPropertiesInput: cats.Eq[DatasetPropertiesInput] = cats.Eq.fromUniversalEquals
      implicit val showDatasetPropertiesInput: cats.Show[DatasetPropertiesInput] = cats.Show.fromToString
      implicit val jsonEncoderDatasetPropertiesInput: io.circe.Encoder.AsObject[DatasetPropertiesInput] = io.circe.generic.semiauto.deriveEncoder[DatasetPropertiesInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class DeclinationInput(val microarcseconds: clue.data.Input[Long] = clue.data.Ignore, val degrees: clue.data.Input[BigDecimal] = clue.data.Ignore, val dms: clue.data.Input[DmsString] = clue.data.Ignore)
    object DeclinationInput {
      val microarcseconds: monocle.Lens[DeclinationInput, clue.data.Input[Long]] = monocle.macros.GenLens[DeclinationInput](_.microarcseconds)
      val degrees: monocle.Lens[DeclinationInput, clue.data.Input[BigDecimal]] = monocle.macros.GenLens[DeclinationInput](_.degrees)
      val dms: monocle.Lens[DeclinationInput, clue.data.Input[DmsString]] = monocle.macros.GenLens[DeclinationInput](_.dms)
      implicit val eqDeclinationInput: cats.Eq[DeclinationInput] = cats.Eq.fromUniversalEquals
      implicit val showDeclinationInput: cats.Show[DeclinationInput] = cats.Show.fromToString
      implicit val jsonEncoderDeclinationInput: io.circe.Encoder.AsObject[DeclinationInput] = io.circe.generic.semiauto.deriveEncoder[DeclinationInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class DemoScienceInput(val minPercentTime: clue.data.Input[IntPercent] = clue.data.Ignore)
    object DemoScienceInput {
      val minPercentTime: monocle.Lens[DemoScienceInput, clue.data.Input[IntPercent]] = monocle.macros.GenLens[DemoScienceInput](_.minPercentTime)
      implicit val eqDemoScienceInput: cats.Eq[DemoScienceInput] = cats.Eq.fromUniversalEquals
      implicit val showDemoScienceInput: cats.Show[DemoScienceInput] = cats.Show.fromToString
      implicit val jsonEncoderDemoScienceInput: io.circe.Encoder.AsObject[DemoScienceInput] = io.circe.generic.semiauto.deriveEncoder[DemoScienceInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class DirectorsTimeInput(val minPercentTime: clue.data.Input[IntPercent] = clue.data.Ignore)
    object DirectorsTimeInput {
      val minPercentTime: monocle.Lens[DirectorsTimeInput, clue.data.Input[IntPercent]] = monocle.macros.GenLens[DirectorsTimeInput](_.minPercentTime)
      implicit val eqDirectorsTimeInput: cats.Eq[DirectorsTimeInput] = cats.Eq.fromUniversalEquals
      implicit val showDirectorsTimeInput: cats.Show[DirectorsTimeInput] = cats.Show.fromToString
      implicit val jsonEncoderDirectorsTimeInput: io.circe.Encoder.AsObject[DirectorsTimeInput] = io.circe.generic.semiauto.deriveEncoder[DirectorsTimeInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class EditAsterismsPatchInput(val ADD: clue.data.Input[List[TargetId]] = clue.data.Ignore, val DELETE: clue.data.Input[List[TargetId]] = clue.data.Ignore)
    object EditAsterismsPatchInput {
      val ADD: monocle.Lens[EditAsterismsPatchInput, clue.data.Input[List[TargetId]]] = monocle.macros.GenLens[EditAsterismsPatchInput](_.ADD)
      val DELETE: monocle.Lens[EditAsterismsPatchInput, clue.data.Input[List[TargetId]]] = monocle.macros.GenLens[EditAsterismsPatchInput](_.DELETE)
      implicit val eqEditAsterismsPatchInput: cats.Eq[EditAsterismsPatchInput] = cats.Eq.fromUniversalEquals
      implicit val showEditAsterismsPatchInput: cats.Show[EditAsterismsPatchInput] = cats.Show.fromToString
      implicit val jsonEncoderEditAsterismsPatchInput: io.circe.Encoder.AsObject[EditAsterismsPatchInput] = io.circe.generic.semiauto.deriveEncoder[EditAsterismsPatchInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class ElevationRangeInput(val airMass: clue.data.Input[AirMassRangeInput] = clue.data.Ignore, val hourAngle: clue.data.Input[HourAngleRangeInput] = clue.data.Ignore)
    object ElevationRangeInput {
      val airMass: monocle.Lens[ElevationRangeInput, clue.data.Input[AirMassRangeInput]] = monocle.macros.GenLens[ElevationRangeInput](_.airMass)
      val hourAngle: monocle.Lens[ElevationRangeInput, clue.data.Input[HourAngleRangeInput]] = monocle.macros.GenLens[ElevationRangeInput](_.hourAngle)
      implicit val eqElevationRangeInput: cats.Eq[ElevationRangeInput] = cats.Eq.fromUniversalEquals
      implicit val showElevationRangeInput: cats.Show[ElevationRangeInput] = cats.Show.fromToString
      implicit val jsonEncoderElevationRangeInput: io.circe.Encoder.AsObject[ElevationRangeInput] = io.circe.generic.semiauto.deriveEncoder[ElevationRangeInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class EmissionLineIntegratedInput(val wavelength: WavelengthInput, val lineWidth: clue.data.Input[PosBigDecimal] = clue.data.Ignore, val lineFlux: clue.data.Input[LineFluxIntegratedInput] = clue.data.Ignore)
    object EmissionLineIntegratedInput {
      val wavelength: monocle.Lens[EmissionLineIntegratedInput, WavelengthInput] = monocle.macros.GenLens[EmissionLineIntegratedInput](_.wavelength)
      val lineWidth: monocle.Lens[EmissionLineIntegratedInput, clue.data.Input[PosBigDecimal]] = monocle.macros.GenLens[EmissionLineIntegratedInput](_.lineWidth)
      val lineFlux: monocle.Lens[EmissionLineIntegratedInput, clue.data.Input[LineFluxIntegratedInput]] = monocle.macros.GenLens[EmissionLineIntegratedInput](_.lineFlux)
      implicit val eqEmissionLineIntegratedInput: cats.Eq[EmissionLineIntegratedInput] = cats.Eq.fromUniversalEquals
      implicit val showEmissionLineIntegratedInput: cats.Show[EmissionLineIntegratedInput] = cats.Show.fromToString
      implicit val jsonEncoderEmissionLineIntegratedInput: io.circe.Encoder.AsObject[EmissionLineIntegratedInput] = io.circe.generic.semiauto.deriveEncoder[EmissionLineIntegratedInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class EmissionLineSurfaceInput(val wavelength: WavelengthInput, val lineWidth: clue.data.Input[PosBigDecimal] = clue.data.Ignore, val lineFlux: clue.data.Input[LineFluxSurfaceInput] = clue.data.Ignore)
    object EmissionLineSurfaceInput {
      val wavelength: monocle.Lens[EmissionLineSurfaceInput, WavelengthInput] = monocle.macros.GenLens[EmissionLineSurfaceInput](_.wavelength)
      val lineWidth: monocle.Lens[EmissionLineSurfaceInput, clue.data.Input[PosBigDecimal]] = monocle.macros.GenLens[EmissionLineSurfaceInput](_.lineWidth)
      val lineFlux: monocle.Lens[EmissionLineSurfaceInput, clue.data.Input[LineFluxSurfaceInput]] = monocle.macros.GenLens[EmissionLineSurfaceInput](_.lineFlux)
      implicit val eqEmissionLineSurfaceInput: cats.Eq[EmissionLineSurfaceInput] = cats.Eq.fromUniversalEquals
      implicit val showEmissionLineSurfaceInput: cats.Show[EmissionLineSurfaceInput] = cats.Show.fromToString
      implicit val jsonEncoderEmissionLineSurfaceInput: io.circe.Encoder.AsObject[EmissionLineSurfaceInput] = io.circe.generic.semiauto.deriveEncoder[EmissionLineSurfaceInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class EmissionLinesIntegratedInput(val lines: clue.data.Input[List[EmissionLineIntegratedInput]] = clue.data.Ignore, val fluxDensityContinuum: clue.data.Input[FluxDensityContinuumIntegratedInput] = clue.data.Ignore)
    object EmissionLinesIntegratedInput {
      val lines: monocle.Lens[EmissionLinesIntegratedInput, clue.data.Input[List[EmissionLineIntegratedInput]]] = monocle.macros.GenLens[EmissionLinesIntegratedInput](_.lines)
      val fluxDensityContinuum: monocle.Lens[EmissionLinesIntegratedInput, clue.data.Input[FluxDensityContinuumIntegratedInput]] = monocle.macros.GenLens[EmissionLinesIntegratedInput](_.fluxDensityContinuum)
      implicit val eqEmissionLinesIntegratedInput: cats.Eq[EmissionLinesIntegratedInput] = cats.Eq.fromUniversalEquals
      implicit val showEmissionLinesIntegratedInput: cats.Show[EmissionLinesIntegratedInput] = cats.Show.fromToString
      implicit val jsonEncoderEmissionLinesIntegratedInput: io.circe.Encoder.AsObject[EmissionLinesIntegratedInput] = io.circe.generic.semiauto.deriveEncoder[EmissionLinesIntegratedInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class EmissionLinesSurfaceInput(val lines: clue.data.Input[List[EmissionLineSurfaceInput]] = clue.data.Ignore, val fluxDensityContinuum: clue.data.Input[FluxDensityContinuumSurfaceInput] = clue.data.Ignore)
    object EmissionLinesSurfaceInput {
      val lines: monocle.Lens[EmissionLinesSurfaceInput, clue.data.Input[List[EmissionLineSurfaceInput]]] = monocle.macros.GenLens[EmissionLinesSurfaceInput](_.lines)
      val fluxDensityContinuum: monocle.Lens[EmissionLinesSurfaceInput, clue.data.Input[FluxDensityContinuumSurfaceInput]] = monocle.macros.GenLens[EmissionLinesSurfaceInput](_.fluxDensityContinuum)
      implicit val eqEmissionLinesSurfaceInput: cats.Eq[EmissionLinesSurfaceInput] = cats.Eq.fromUniversalEquals
      implicit val showEmissionLinesSurfaceInput: cats.Show[EmissionLinesSurfaceInput] = cats.Show.fromToString
      implicit val jsonEncoderEmissionLinesSurfaceInput: io.circe.Encoder.AsObject[EmissionLinesSurfaceInput] = io.circe.generic.semiauto.deriveEncoder[EmissionLinesSurfaceInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class ExchangeInput(val minPercentTime: clue.data.Input[IntPercent] = clue.data.Ignore)
    object ExchangeInput {
      val minPercentTime: monocle.Lens[ExchangeInput, clue.data.Input[IntPercent]] = monocle.macros.GenLens[ExchangeInput](_.minPercentTime)
      implicit val eqExchangeInput: cats.Eq[ExchangeInput] = cats.Eq.fromUniversalEquals
      implicit val showExchangeInput: cats.Show[ExchangeInput] = cats.Show.fromToString
      implicit val jsonEncoderExchangeInput: io.circe.Encoder.AsObject[ExchangeInput] = io.circe.generic.semiauto.deriveEncoder[ExchangeInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class ExposureTimeModeInput(val signalToNoise: clue.data.Input[SignalToNoiseModeInput] = clue.data.Ignore, val fixedExposure: clue.data.Input[FixedExposureModeInput] = clue.data.Ignore)
    object ExposureTimeModeInput {
      val signalToNoise: monocle.Lens[ExposureTimeModeInput, clue.data.Input[SignalToNoiseModeInput]] = monocle.macros.GenLens[ExposureTimeModeInput](_.signalToNoise)
      val fixedExposure: monocle.Lens[ExposureTimeModeInput, clue.data.Input[FixedExposureModeInput]] = monocle.macros.GenLens[ExposureTimeModeInput](_.fixedExposure)
      implicit val eqExposureTimeModeInput: cats.Eq[ExposureTimeModeInput] = cats.Eq.fromUniversalEquals
      implicit val showExposureTimeModeInput: cats.Show[ExposureTimeModeInput] = cats.Show.fromToString
      implicit val jsonEncoderExposureTimeModeInput: io.circe.Encoder.AsObject[ExposureTimeModeInput] = io.circe.generic.semiauto.deriveEncoder[ExposureTimeModeInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class FastTurnaroundInput(val minPercentTime: clue.data.Input[IntPercent] = clue.data.Ignore)
    object FastTurnaroundInput {
      val minPercentTime: monocle.Lens[FastTurnaroundInput, clue.data.Input[IntPercent]] = monocle.macros.GenLens[FastTurnaroundInput](_.minPercentTime)
      implicit val eqFastTurnaroundInput: cats.Eq[FastTurnaroundInput] = cats.Eq.fromUniversalEquals
      implicit val showFastTurnaroundInput: cats.Show[FastTurnaroundInput] = cats.Show.fromToString
      implicit val jsonEncoderFastTurnaroundInput: io.circe.Encoder.AsObject[FastTurnaroundInput] = io.circe.generic.semiauto.deriveEncoder[FastTurnaroundInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class FixedExposureModeInput(val count: NonNegInt, val time: TimeSpanInput)
    object FixedExposureModeInput {
      val count: monocle.Lens[FixedExposureModeInput, NonNegInt] = monocle.macros.GenLens[FixedExposureModeInput](_.count)
      val time: monocle.Lens[FixedExposureModeInput, TimeSpanInput] = monocle.macros.GenLens[FixedExposureModeInput](_.time)
      implicit val eqFixedExposureModeInput: cats.Eq[FixedExposureModeInput] = cats.Eq.fromUniversalEquals
      implicit val showFixedExposureModeInput: cats.Show[FixedExposureModeInput] = cats.Show.fromToString
      implicit val jsonEncoderFixedExposureModeInput: io.circe.Encoder.AsObject[FixedExposureModeInput] = io.circe.generic.semiauto.deriveEncoder[FixedExposureModeInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class FluxDensity(val wavelength: WavelengthInput, val density: PosBigDecimal)
    object FluxDensity {
      val wavelength: monocle.Lens[FluxDensity, WavelengthInput] = monocle.macros.GenLens[FluxDensity](_.wavelength)
      val density: monocle.Lens[FluxDensity, PosBigDecimal] = monocle.macros.GenLens[FluxDensity](_.density)
      implicit val eqFluxDensity: cats.Eq[FluxDensity] = cats.Eq.fromUniversalEquals
      implicit val showFluxDensity: cats.Show[FluxDensity] = cats.Show.fromToString
      implicit val jsonEncoderFluxDensity: io.circe.Encoder.AsObject[FluxDensity] = io.circe.generic.semiauto.deriveEncoder[FluxDensity].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class FluxDensityContinuumIntegratedInput(val value: PosBigDecimal, val units: FluxDensityContinuumIntegratedUnits, val error: clue.data.Input[PosBigDecimal] = clue.data.Ignore)
    object FluxDensityContinuumIntegratedInput {
      val value: monocle.Lens[FluxDensityContinuumIntegratedInput, PosBigDecimal] = monocle.macros.GenLens[FluxDensityContinuumIntegratedInput](_.value)
      val units: monocle.Lens[FluxDensityContinuumIntegratedInput, FluxDensityContinuumIntegratedUnits] = monocle.macros.GenLens[FluxDensityContinuumIntegratedInput](_.units)
      val error: monocle.Lens[FluxDensityContinuumIntegratedInput, clue.data.Input[PosBigDecimal]] = monocle.macros.GenLens[FluxDensityContinuumIntegratedInput](_.error)
      implicit val eqFluxDensityContinuumIntegratedInput: cats.Eq[FluxDensityContinuumIntegratedInput] = cats.Eq.fromUniversalEquals
      implicit val showFluxDensityContinuumIntegratedInput: cats.Show[FluxDensityContinuumIntegratedInput] = cats.Show.fromToString
      implicit val jsonEncoderFluxDensityContinuumIntegratedInput: io.circe.Encoder.AsObject[FluxDensityContinuumIntegratedInput] = io.circe.generic.semiauto.deriveEncoder[FluxDensityContinuumIntegratedInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class FluxDensityContinuumSurfaceInput(val value: PosBigDecimal, val units: FluxDensityContinuumSurfaceUnits, val error: clue.data.Input[PosBigDecimal] = clue.data.Ignore)
    object FluxDensityContinuumSurfaceInput {
      val value: monocle.Lens[FluxDensityContinuumSurfaceInput, PosBigDecimal] = monocle.macros.GenLens[FluxDensityContinuumSurfaceInput](_.value)
      val units: monocle.Lens[FluxDensityContinuumSurfaceInput, FluxDensityContinuumSurfaceUnits] = monocle.macros.GenLens[FluxDensityContinuumSurfaceInput](_.units)
      val error: monocle.Lens[FluxDensityContinuumSurfaceInput, clue.data.Input[PosBigDecimal]] = monocle.macros.GenLens[FluxDensityContinuumSurfaceInput](_.error)
      implicit val eqFluxDensityContinuumSurfaceInput: cats.Eq[FluxDensityContinuumSurfaceInput] = cats.Eq.fromUniversalEquals
      implicit val showFluxDensityContinuumSurfaceInput: cats.Show[FluxDensityContinuumSurfaceInput] = cats.Show.fromToString
      implicit val jsonEncoderFluxDensityContinuumSurfaceInput: io.circe.Encoder.AsObject[FluxDensityContinuumSurfaceInput] = io.circe.generic.semiauto.deriveEncoder[FluxDensityContinuumSurfaceInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class GaussianInput(val fwhm: clue.data.Input[AngleInput] = clue.data.Ignore, val spectralDefinition: clue.data.Input[SpectralDefinitionIntegratedInput] = clue.data.Ignore)
    object GaussianInput {
      val fwhm: monocle.Lens[GaussianInput, clue.data.Input[AngleInput]] = monocle.macros.GenLens[GaussianInput](_.fwhm)
      val spectralDefinition: monocle.Lens[GaussianInput, clue.data.Input[SpectralDefinitionIntegratedInput]] = monocle.macros.GenLens[GaussianInput](_.spectralDefinition)
      implicit val eqGaussianInput: cats.Eq[GaussianInput] = cats.Eq.fromUniversalEquals
      implicit val showGaussianInput: cats.Show[GaussianInput] = cats.Show.fromToString
      implicit val jsonEncoderGaussianInput: io.circe.Encoder.AsObject[GaussianInput] = io.circe.generic.semiauto.deriveEncoder[GaussianInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class GmosCcdModeInput(val xBin: clue.data.Input[GmosXBinning] = clue.data.Ignore, val yBin: clue.data.Input[GmosYBinning] = clue.data.Ignore, val ampCount: clue.data.Input[GmosAmpCount] = clue.data.Ignore, val ampGain: clue.data.Input[GmosAmpGain] = clue.data.Ignore, val ampReadMode: clue.data.Input[GmosAmpReadMode] = clue.data.Ignore)
    object GmosCcdModeInput {
      val xBin: monocle.Lens[GmosCcdModeInput, clue.data.Input[GmosXBinning]] = monocle.macros.GenLens[GmosCcdModeInput](_.xBin)
      val yBin: monocle.Lens[GmosCcdModeInput, clue.data.Input[GmosYBinning]] = monocle.macros.GenLens[GmosCcdModeInput](_.yBin)
      val ampCount: monocle.Lens[GmosCcdModeInput, clue.data.Input[GmosAmpCount]] = monocle.macros.GenLens[GmosCcdModeInput](_.ampCount)
      val ampGain: monocle.Lens[GmosCcdModeInput, clue.data.Input[GmosAmpGain]] = monocle.macros.GenLens[GmosCcdModeInput](_.ampGain)
      val ampReadMode: monocle.Lens[GmosCcdModeInput, clue.data.Input[GmosAmpReadMode]] = monocle.macros.GenLens[GmosCcdModeInput](_.ampReadMode)
      implicit val eqGmosCcdModeInput: cats.Eq[GmosCcdModeInput] = cats.Eq.fromUniversalEquals
      implicit val showGmosCcdModeInput: cats.Show[GmosCcdModeInput] = cats.Show.fromToString
      implicit val jsonEncoderGmosCcdModeInput: io.circe.Encoder.AsObject[GmosCcdModeInput] = io.circe.generic.semiauto.deriveEncoder[GmosCcdModeInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class GmosCustomMaskInput(val filename: String, val slitWidth: GmosCustomSlitWidth)
    object GmosCustomMaskInput {
      val filename: monocle.Lens[GmosCustomMaskInput, String] = monocle.macros.GenLens[GmosCustomMaskInput](_.filename)
      val slitWidth: monocle.Lens[GmosCustomMaskInput, GmosCustomSlitWidth] = monocle.macros.GenLens[GmosCustomMaskInput](_.slitWidth)
      implicit val eqGmosCustomMaskInput: cats.Eq[GmosCustomMaskInput] = cats.Eq.fromUniversalEquals
      implicit val showGmosCustomMaskInput: cats.Show[GmosCustomMaskInput] = cats.Show.fromToString
      implicit val jsonEncoderGmosCustomMaskInput: io.circe.Encoder.AsObject[GmosCustomMaskInput] = io.circe.generic.semiauto.deriveEncoder[GmosCustomMaskInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class GmosNodAndShuffleInput(val posA: OffsetInput, val posB: OffsetInput, val eOffset: GmosEOffsetting, val shuffleOffset: PosInt, val shuffleCycles: PosInt)
    object GmosNodAndShuffleInput {
      val posA: monocle.Lens[GmosNodAndShuffleInput, OffsetInput] = monocle.macros.GenLens[GmosNodAndShuffleInput](_.posA)
      val posB: monocle.Lens[GmosNodAndShuffleInput, OffsetInput] = monocle.macros.GenLens[GmosNodAndShuffleInput](_.posB)
      val eOffset: monocle.Lens[GmosNodAndShuffleInput, GmosEOffsetting] = monocle.macros.GenLens[GmosNodAndShuffleInput](_.eOffset)
      val shuffleOffset: monocle.Lens[GmosNodAndShuffleInput, PosInt] = monocle.macros.GenLens[GmosNodAndShuffleInput](_.shuffleOffset)
      val shuffleCycles: monocle.Lens[GmosNodAndShuffleInput, PosInt] = monocle.macros.GenLens[GmosNodAndShuffleInput](_.shuffleCycles)
      implicit val eqGmosNodAndShuffleInput: cats.Eq[GmosNodAndShuffleInput] = cats.Eq.fromUniversalEquals
      implicit val showGmosNodAndShuffleInput: cats.Show[GmosNodAndShuffleInput] = cats.Show.fromToString
      implicit val jsonEncoderGmosNodAndShuffleInput: io.circe.Encoder.AsObject[GmosNodAndShuffleInput] = io.circe.generic.semiauto.deriveEncoder[GmosNodAndShuffleInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class GmosNorthDynamicInput(val exposure: TimeSpanInput, val readout: GmosCcdModeInput, val dtax: GmosDtax, val roi: GmosRoi, val gratingConfig: clue.data.Input[GmosNorthGratingConfigInput] = clue.data.Ignore, val filter: clue.data.Input[GmosNorthFilter] = clue.data.Ignore, val fpu: clue.data.Input[GmosNorthFpuInput] = clue.data.Ignore)
    object GmosNorthDynamicInput {
      val exposure: monocle.Lens[GmosNorthDynamicInput, TimeSpanInput] = monocle.macros.GenLens[GmosNorthDynamicInput](_.exposure)
      val readout: monocle.Lens[GmosNorthDynamicInput, GmosCcdModeInput] = monocle.macros.GenLens[GmosNorthDynamicInput](_.readout)
      val dtax: monocle.Lens[GmosNorthDynamicInput, GmosDtax] = monocle.macros.GenLens[GmosNorthDynamicInput](_.dtax)
      val roi: monocle.Lens[GmosNorthDynamicInput, GmosRoi] = monocle.macros.GenLens[GmosNorthDynamicInput](_.roi)
      val gratingConfig: monocle.Lens[GmosNorthDynamicInput, clue.data.Input[GmosNorthGratingConfigInput]] = monocle.macros.GenLens[GmosNorthDynamicInput](_.gratingConfig)
      val filter: monocle.Lens[GmosNorthDynamicInput, clue.data.Input[GmosNorthFilter]] = monocle.macros.GenLens[GmosNorthDynamicInput](_.filter)
      val fpu: monocle.Lens[GmosNorthDynamicInput, clue.data.Input[GmosNorthFpuInput]] = monocle.macros.GenLens[GmosNorthDynamicInput](_.fpu)
      implicit val eqGmosNorthDynamicInput: cats.Eq[GmosNorthDynamicInput] = cats.Eq.fromUniversalEquals
      implicit val showGmosNorthDynamicInput: cats.Show[GmosNorthDynamicInput] = cats.Show.fromToString
      implicit val jsonEncoderGmosNorthDynamicInput: io.circe.Encoder.AsObject[GmosNorthDynamicInput] = io.circe.generic.semiauto.deriveEncoder[GmosNorthDynamicInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class GmosNorthFpuInput(val customMask: clue.data.Input[GmosCustomMaskInput] = clue.data.Ignore, val builtin: clue.data.Input[GmosNorthBuiltinFpu] = clue.data.Ignore)
    object GmosNorthFpuInput {
      val customMask: monocle.Lens[GmosNorthFpuInput, clue.data.Input[GmosCustomMaskInput]] = monocle.macros.GenLens[GmosNorthFpuInput](_.customMask)
      val builtin: monocle.Lens[GmosNorthFpuInput, clue.data.Input[GmosNorthBuiltinFpu]] = monocle.macros.GenLens[GmosNorthFpuInput](_.builtin)
      implicit val eqGmosNorthFpuInput: cats.Eq[GmosNorthFpuInput] = cats.Eq.fromUniversalEquals
      implicit val showGmosNorthFpuInput: cats.Show[GmosNorthFpuInput] = cats.Show.fromToString
      implicit val jsonEncoderGmosNorthFpuInput: io.circe.Encoder.AsObject[GmosNorthFpuInput] = io.circe.generic.semiauto.deriveEncoder[GmosNorthFpuInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class GmosNorthGratingConfigInput(val grating: GmosNorthGrating, val order: GmosGratingOrder, val wavelength: WavelengthInput)
    object GmosNorthGratingConfigInput {
      val grating: monocle.Lens[GmosNorthGratingConfigInput, GmosNorthGrating] = monocle.macros.GenLens[GmosNorthGratingConfigInput](_.grating)
      val order: monocle.Lens[GmosNorthGratingConfigInput, GmosGratingOrder] = monocle.macros.GenLens[GmosNorthGratingConfigInput](_.order)
      val wavelength: monocle.Lens[GmosNorthGratingConfigInput, WavelengthInput] = monocle.macros.GenLens[GmosNorthGratingConfigInput](_.wavelength)
      implicit val eqGmosNorthGratingConfigInput: cats.Eq[GmosNorthGratingConfigInput] = cats.Eq.fromUniversalEquals
      implicit val showGmosNorthGratingConfigInput: cats.Show[GmosNorthGratingConfigInput] = cats.Show.fromToString
      implicit val jsonEncoderGmosNorthGratingConfigInput: io.circe.Encoder.AsObject[GmosNorthGratingConfigInput] = io.circe.generic.semiauto.deriveEncoder[GmosNorthGratingConfigInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class GmosNorthLongSlitInput(val grating: clue.data.Input[GmosNorthGrating] = clue.data.Ignore, val filter: clue.data.Input[GmosNorthFilter] = clue.data.Ignore, val fpu: clue.data.Input[GmosNorthBuiltinFpu] = clue.data.Ignore, val centralWavelength: clue.data.Input[WavelengthInput] = clue.data.Ignore, val explicitXBin: clue.data.Input[GmosXBinning] = clue.data.Ignore, val explicitYBin: clue.data.Input[GmosYBinning] = clue.data.Ignore, val explicitAmpReadMode: clue.data.Input[GmosAmpReadMode] = clue.data.Ignore, val explicitAmpGain: clue.data.Input[GmosAmpGain] = clue.data.Ignore, val explicitRoi: clue.data.Input[GmosRoi] = clue.data.Ignore, val explicitWavelengthDithers: clue.data.Input[List[WavelengthDitherInput]] = clue.data.Ignore, val explicitSpatialOffsets: clue.data.Input[List[OffsetComponentInput]] = clue.data.Ignore)
    object GmosNorthLongSlitInput {
      val grating: monocle.Lens[GmosNorthLongSlitInput, clue.data.Input[GmosNorthGrating]] = monocle.macros.GenLens[GmosNorthLongSlitInput](_.grating)
      val filter: monocle.Lens[GmosNorthLongSlitInput, clue.data.Input[GmosNorthFilter]] = monocle.macros.GenLens[GmosNorthLongSlitInput](_.filter)
      val fpu: monocle.Lens[GmosNorthLongSlitInput, clue.data.Input[GmosNorthBuiltinFpu]] = monocle.macros.GenLens[GmosNorthLongSlitInput](_.fpu)
      val centralWavelength: monocle.Lens[GmosNorthLongSlitInput, clue.data.Input[WavelengthInput]] = monocle.macros.GenLens[GmosNorthLongSlitInput](_.centralWavelength)
      val explicitXBin: monocle.Lens[GmosNorthLongSlitInput, clue.data.Input[GmosXBinning]] = monocle.macros.GenLens[GmosNorthLongSlitInput](_.explicitXBin)
      val explicitYBin: monocle.Lens[GmosNorthLongSlitInput, clue.data.Input[GmosYBinning]] = monocle.macros.GenLens[GmosNorthLongSlitInput](_.explicitYBin)
      val explicitAmpReadMode: monocle.Lens[GmosNorthLongSlitInput, clue.data.Input[GmosAmpReadMode]] = monocle.macros.GenLens[GmosNorthLongSlitInput](_.explicitAmpReadMode)
      val explicitAmpGain: monocle.Lens[GmosNorthLongSlitInput, clue.data.Input[GmosAmpGain]] = monocle.macros.GenLens[GmosNorthLongSlitInput](_.explicitAmpGain)
      val explicitRoi: monocle.Lens[GmosNorthLongSlitInput, clue.data.Input[GmosRoi]] = monocle.macros.GenLens[GmosNorthLongSlitInput](_.explicitRoi)
      val explicitWavelengthDithers: monocle.Lens[GmosNorthLongSlitInput, clue.data.Input[List[WavelengthDitherInput]]] = monocle.macros.GenLens[GmosNorthLongSlitInput](_.explicitWavelengthDithers)
      val explicitSpatialOffsets: monocle.Lens[GmosNorthLongSlitInput, clue.data.Input[List[OffsetComponentInput]]] = monocle.macros.GenLens[GmosNorthLongSlitInput](_.explicitSpatialOffsets)
      implicit val eqGmosNorthLongSlitInput: cats.Eq[GmosNorthLongSlitInput] = cats.Eq.fromUniversalEquals
      implicit val showGmosNorthLongSlitInput: cats.Show[GmosNorthLongSlitInput] = cats.Show.fromToString
      implicit val jsonEncoderGmosNorthLongSlitInput: io.circe.Encoder.AsObject[GmosNorthLongSlitInput] = io.circe.generic.semiauto.deriveEncoder[GmosNorthLongSlitInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class GmosNorthStaticInput(val stageMode: clue.data.Input[GmosNorthStageMode] = clue.data.Ignore, val detector: clue.data.Input[GmosNorthDetector] = clue.data.Ignore, val mosPreImaging: clue.data.Input[MosPreImaging] = clue.data.Ignore, val nodAndShuffle: clue.data.Input[GmosNodAndShuffleInput] = clue.data.Ignore)
    object GmosNorthStaticInput {
      val stageMode: monocle.Lens[GmosNorthStaticInput, clue.data.Input[GmosNorthStageMode]] = monocle.macros.GenLens[GmosNorthStaticInput](_.stageMode)
      val detector: monocle.Lens[GmosNorthStaticInput, clue.data.Input[GmosNorthDetector]] = monocle.macros.GenLens[GmosNorthStaticInput](_.detector)
      val mosPreImaging: monocle.Lens[GmosNorthStaticInput, clue.data.Input[MosPreImaging]] = monocle.macros.GenLens[GmosNorthStaticInput](_.mosPreImaging)
      val nodAndShuffle: monocle.Lens[GmosNorthStaticInput, clue.data.Input[GmosNodAndShuffleInput]] = monocle.macros.GenLens[GmosNorthStaticInput](_.nodAndShuffle)
      implicit val eqGmosNorthStaticInput: cats.Eq[GmosNorthStaticInput] = cats.Eq.fromUniversalEquals
      implicit val showGmosNorthStaticInput: cats.Show[GmosNorthStaticInput] = cats.Show.fromToString
      implicit val jsonEncoderGmosNorthStaticInput: io.circe.Encoder.AsObject[GmosNorthStaticInput] = io.circe.generic.semiauto.deriveEncoder[GmosNorthStaticInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class GmosSouthDynamicInput(val exposure: TimeSpanInput, val readout: GmosCcdModeInput, val dtax: GmosDtax, val roi: GmosRoi, val gratingConfig: clue.data.Input[GmosSouthGratingConfigInput] = clue.data.Ignore, val filter: clue.data.Input[GmosSouthFilter] = clue.data.Ignore, val fpu: clue.data.Input[GmosSouthFpuInput] = clue.data.Ignore)
    object GmosSouthDynamicInput {
      val exposure: monocle.Lens[GmosSouthDynamicInput, TimeSpanInput] = monocle.macros.GenLens[GmosSouthDynamicInput](_.exposure)
      val readout: monocle.Lens[GmosSouthDynamicInput, GmosCcdModeInput] = monocle.macros.GenLens[GmosSouthDynamicInput](_.readout)
      val dtax: monocle.Lens[GmosSouthDynamicInput, GmosDtax] = monocle.macros.GenLens[GmosSouthDynamicInput](_.dtax)
      val roi: monocle.Lens[GmosSouthDynamicInput, GmosRoi] = monocle.macros.GenLens[GmosSouthDynamicInput](_.roi)
      val gratingConfig: monocle.Lens[GmosSouthDynamicInput, clue.data.Input[GmosSouthGratingConfigInput]] = monocle.macros.GenLens[GmosSouthDynamicInput](_.gratingConfig)
      val filter: monocle.Lens[GmosSouthDynamicInput, clue.data.Input[GmosSouthFilter]] = monocle.macros.GenLens[GmosSouthDynamicInput](_.filter)
      val fpu: monocle.Lens[GmosSouthDynamicInput, clue.data.Input[GmosSouthFpuInput]] = monocle.macros.GenLens[GmosSouthDynamicInput](_.fpu)
      implicit val eqGmosSouthDynamicInput: cats.Eq[GmosSouthDynamicInput] = cats.Eq.fromUniversalEquals
      implicit val showGmosSouthDynamicInput: cats.Show[GmosSouthDynamicInput] = cats.Show.fromToString
      implicit val jsonEncoderGmosSouthDynamicInput: io.circe.Encoder.AsObject[GmosSouthDynamicInput] = io.circe.generic.semiauto.deriveEncoder[GmosSouthDynamicInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class GmosSouthFpuInput(val customMask: clue.data.Input[GmosCustomMaskInput] = clue.data.Ignore, val builtin: clue.data.Input[GmosSouthBuiltinFpu] = clue.data.Ignore)
    object GmosSouthFpuInput {
      val customMask: monocle.Lens[GmosSouthFpuInput, clue.data.Input[GmosCustomMaskInput]] = monocle.macros.GenLens[GmosSouthFpuInput](_.customMask)
      val builtin: monocle.Lens[GmosSouthFpuInput, clue.data.Input[GmosSouthBuiltinFpu]] = monocle.macros.GenLens[GmosSouthFpuInput](_.builtin)
      implicit val eqGmosSouthFpuInput: cats.Eq[GmosSouthFpuInput] = cats.Eq.fromUniversalEquals
      implicit val showGmosSouthFpuInput: cats.Show[GmosSouthFpuInput] = cats.Show.fromToString
      implicit val jsonEncoderGmosSouthFpuInput: io.circe.Encoder.AsObject[GmosSouthFpuInput] = io.circe.generic.semiauto.deriveEncoder[GmosSouthFpuInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class GmosSouthGratingConfigInput(val grating: GmosSouthGrating, val order: GmosGratingOrder, val wavelength: WavelengthInput)
    object GmosSouthGratingConfigInput {
      val grating: monocle.Lens[GmosSouthGratingConfigInput, GmosSouthGrating] = monocle.macros.GenLens[GmosSouthGratingConfigInput](_.grating)
      val order: monocle.Lens[GmosSouthGratingConfigInput, GmosGratingOrder] = monocle.macros.GenLens[GmosSouthGratingConfigInput](_.order)
      val wavelength: monocle.Lens[GmosSouthGratingConfigInput, WavelengthInput] = monocle.macros.GenLens[GmosSouthGratingConfigInput](_.wavelength)
      implicit val eqGmosSouthGratingConfigInput: cats.Eq[GmosSouthGratingConfigInput] = cats.Eq.fromUniversalEquals
      implicit val showGmosSouthGratingConfigInput: cats.Show[GmosSouthGratingConfigInput] = cats.Show.fromToString
      implicit val jsonEncoderGmosSouthGratingConfigInput: io.circe.Encoder.AsObject[GmosSouthGratingConfigInput] = io.circe.generic.semiauto.deriveEncoder[GmosSouthGratingConfigInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class GmosSouthLongSlitInput(val grating: clue.data.Input[GmosSouthGrating] = clue.data.Ignore, val filter: clue.data.Input[GmosSouthFilter] = clue.data.Ignore, val fpu: clue.data.Input[GmosSouthBuiltinFpu] = clue.data.Ignore, val centralWavelength: clue.data.Input[WavelengthInput] = clue.data.Ignore, val explicitXBin: clue.data.Input[GmosXBinning] = clue.data.Ignore, val explicitYBin: clue.data.Input[GmosYBinning] = clue.data.Ignore, val explicitAmpReadMode: clue.data.Input[GmosAmpReadMode] = clue.data.Ignore, val explicitAmpGain: clue.data.Input[GmosAmpGain] = clue.data.Ignore, val explicitRoi: clue.data.Input[GmosRoi] = clue.data.Ignore, val explicitWavelengthDithers: clue.data.Input[List[WavelengthDitherInput]] = clue.data.Ignore, val explicitSpatialOffsets: clue.data.Input[List[OffsetComponentInput]] = clue.data.Ignore)
    object GmosSouthLongSlitInput {
      val grating: monocle.Lens[GmosSouthLongSlitInput, clue.data.Input[GmosSouthGrating]] = monocle.macros.GenLens[GmosSouthLongSlitInput](_.grating)
      val filter: monocle.Lens[GmosSouthLongSlitInput, clue.data.Input[GmosSouthFilter]] = monocle.macros.GenLens[GmosSouthLongSlitInput](_.filter)
      val fpu: monocle.Lens[GmosSouthLongSlitInput, clue.data.Input[GmosSouthBuiltinFpu]] = monocle.macros.GenLens[GmosSouthLongSlitInput](_.fpu)
      val centralWavelength: monocle.Lens[GmosSouthLongSlitInput, clue.data.Input[WavelengthInput]] = monocle.macros.GenLens[GmosSouthLongSlitInput](_.centralWavelength)
      val explicitXBin: monocle.Lens[GmosSouthLongSlitInput, clue.data.Input[GmosXBinning]] = monocle.macros.GenLens[GmosSouthLongSlitInput](_.explicitXBin)
      val explicitYBin: monocle.Lens[GmosSouthLongSlitInput, clue.data.Input[GmosYBinning]] = monocle.macros.GenLens[GmosSouthLongSlitInput](_.explicitYBin)
      val explicitAmpReadMode: monocle.Lens[GmosSouthLongSlitInput, clue.data.Input[GmosAmpReadMode]] = monocle.macros.GenLens[GmosSouthLongSlitInput](_.explicitAmpReadMode)
      val explicitAmpGain: monocle.Lens[GmosSouthLongSlitInput, clue.data.Input[GmosAmpGain]] = monocle.macros.GenLens[GmosSouthLongSlitInput](_.explicitAmpGain)
      val explicitRoi: monocle.Lens[GmosSouthLongSlitInput, clue.data.Input[GmosRoi]] = monocle.macros.GenLens[GmosSouthLongSlitInput](_.explicitRoi)
      val explicitWavelengthDithers: monocle.Lens[GmosSouthLongSlitInput, clue.data.Input[List[WavelengthDitherInput]]] = monocle.macros.GenLens[GmosSouthLongSlitInput](_.explicitWavelengthDithers)
      val explicitSpatialOffsets: monocle.Lens[GmosSouthLongSlitInput, clue.data.Input[List[OffsetComponentInput]]] = monocle.macros.GenLens[GmosSouthLongSlitInput](_.explicitSpatialOffsets)
      implicit val eqGmosSouthLongSlitInput: cats.Eq[GmosSouthLongSlitInput] = cats.Eq.fromUniversalEquals
      implicit val showGmosSouthLongSlitInput: cats.Show[GmosSouthLongSlitInput] = cats.Show.fromToString
      implicit val jsonEncoderGmosSouthLongSlitInput: io.circe.Encoder.AsObject[GmosSouthLongSlitInput] = io.circe.generic.semiauto.deriveEncoder[GmosSouthLongSlitInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class GmosSouthStaticInput(val stageMode: clue.data.Input[GmosSouthStageMode] = clue.data.Ignore, val detector: clue.data.Input[GmosSouthDetector] = clue.data.Ignore, val mosPreImaging: clue.data.Input[MosPreImaging] = clue.data.Ignore, val nodAndShuffle: clue.data.Input[GmosNodAndShuffleInput] = clue.data.Ignore)
    object GmosSouthStaticInput {
      val stageMode: monocle.Lens[GmosSouthStaticInput, clue.data.Input[GmosSouthStageMode]] = monocle.macros.GenLens[GmosSouthStaticInput](_.stageMode)
      val detector: monocle.Lens[GmosSouthStaticInput, clue.data.Input[GmosSouthDetector]] = monocle.macros.GenLens[GmosSouthStaticInput](_.detector)
      val mosPreImaging: monocle.Lens[GmosSouthStaticInput, clue.data.Input[MosPreImaging]] = monocle.macros.GenLens[GmosSouthStaticInput](_.mosPreImaging)
      val nodAndShuffle: monocle.Lens[GmosSouthStaticInput, clue.data.Input[GmosNodAndShuffleInput]] = monocle.macros.GenLens[GmosSouthStaticInput](_.nodAndShuffle)
      implicit val eqGmosSouthStaticInput: cats.Eq[GmosSouthStaticInput] = cats.Eq.fromUniversalEquals
      implicit val showGmosSouthStaticInput: cats.Show[GmosSouthStaticInput] = cats.Show.fromToString
      implicit val jsonEncoderGmosSouthStaticInput: io.circe.Encoder.AsObject[GmosSouthStaticInput] = io.circe.generic.semiauto.deriveEncoder[GmosSouthStaticInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class HourAngleRangeInput(val minHours: clue.data.Input[BigDecimal] = clue.data.Ignore, val maxHours: clue.data.Input[BigDecimal] = clue.data.Ignore)
    object HourAngleRangeInput {
      val minHours: monocle.Lens[HourAngleRangeInput, clue.data.Input[BigDecimal]] = monocle.macros.GenLens[HourAngleRangeInput](_.minHours)
      val maxHours: monocle.Lens[HourAngleRangeInput, clue.data.Input[BigDecimal]] = monocle.macros.GenLens[HourAngleRangeInput](_.maxHours)
      implicit val eqHourAngleRangeInput: cats.Eq[HourAngleRangeInput] = cats.Eq.fromUniversalEquals
      implicit val showHourAngleRangeInput: cats.Show[HourAngleRangeInput] = cats.Show.fromToString
      implicit val jsonEncoderHourAngleRangeInput: io.circe.Encoder.AsObject[HourAngleRangeInput] = io.circe.generic.semiauto.deriveEncoder[HourAngleRangeInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class IntensiveInput(val minPercentTime: clue.data.Input[IntPercent] = clue.data.Ignore, val minPercentTotalTime: clue.data.Input[IntPercent] = clue.data.Ignore, val totalTime: clue.data.Input[TimeSpanInput] = clue.data.Ignore)
    object IntensiveInput {
      val minPercentTime: monocle.Lens[IntensiveInput, clue.data.Input[IntPercent]] = monocle.macros.GenLens[IntensiveInput](_.minPercentTime)
      val minPercentTotalTime: monocle.Lens[IntensiveInput, clue.data.Input[IntPercent]] = monocle.macros.GenLens[IntensiveInput](_.minPercentTotalTime)
      val totalTime: monocle.Lens[IntensiveInput, clue.data.Input[TimeSpanInput]] = monocle.macros.GenLens[IntensiveInput](_.totalTime)
      implicit val eqIntensiveInput: cats.Eq[IntensiveInput] = cats.Eq.fromUniversalEquals
      implicit val showIntensiveInput: cats.Show[IntensiveInput] = cats.Show.fromToString
      implicit val jsonEncoderIntensiveInput: io.circe.Encoder.AsObject[IntensiveInput] = io.circe.generic.semiauto.deriveEncoder[IntensiveInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class LargeProgramInput(val minPercentTime: clue.data.Input[IntPercent] = clue.data.Ignore, val minPercentTotalTime: clue.data.Input[IntPercent] = clue.data.Ignore, val totalTime: clue.data.Input[TimeSpanInput] = clue.data.Ignore)
    object LargeProgramInput {
      val minPercentTime: monocle.Lens[LargeProgramInput, clue.data.Input[IntPercent]] = monocle.macros.GenLens[LargeProgramInput](_.minPercentTime)
      val minPercentTotalTime: monocle.Lens[LargeProgramInput, clue.data.Input[IntPercent]] = monocle.macros.GenLens[LargeProgramInput](_.minPercentTotalTime)
      val totalTime: monocle.Lens[LargeProgramInput, clue.data.Input[TimeSpanInput]] = monocle.macros.GenLens[LargeProgramInput](_.totalTime)
      implicit val eqLargeProgramInput: cats.Eq[LargeProgramInput] = cats.Eq.fromUniversalEquals
      implicit val showLargeProgramInput: cats.Show[LargeProgramInput] = cats.Show.fromToString
      implicit val jsonEncoderLargeProgramInput: io.circe.Encoder.AsObject[LargeProgramInput] = io.circe.generic.semiauto.deriveEncoder[LargeProgramInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class LineFluxIntegratedInput(val value: PosBigDecimal, val units: LineFluxIntegratedUnits)
    object LineFluxIntegratedInput {
      val value: monocle.Lens[LineFluxIntegratedInput, PosBigDecimal] = monocle.macros.GenLens[LineFluxIntegratedInput](_.value)
      val units: monocle.Lens[LineFluxIntegratedInput, LineFluxIntegratedUnits] = monocle.macros.GenLens[LineFluxIntegratedInput](_.units)
      implicit val eqLineFluxIntegratedInput: cats.Eq[LineFluxIntegratedInput] = cats.Eq.fromUniversalEquals
      implicit val showLineFluxIntegratedInput: cats.Show[LineFluxIntegratedInput] = cats.Show.fromToString
      implicit val jsonEncoderLineFluxIntegratedInput: io.circe.Encoder.AsObject[LineFluxIntegratedInput] = io.circe.generic.semiauto.deriveEncoder[LineFluxIntegratedInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class LineFluxSurfaceInput(val value: PosBigDecimal, val units: LineFluxSurfaceUnits)
    object LineFluxSurfaceInput {
      val value: monocle.Lens[LineFluxSurfaceInput, PosBigDecimal] = monocle.macros.GenLens[LineFluxSurfaceInput](_.value)
      val units: monocle.Lens[LineFluxSurfaceInput, LineFluxSurfaceUnits] = monocle.macros.GenLens[LineFluxSurfaceInput](_.units)
      implicit val eqLineFluxSurfaceInput: cats.Eq[LineFluxSurfaceInput] = cats.Eq.fromUniversalEquals
      implicit val showLineFluxSurfaceInput: cats.Show[LineFluxSurfaceInput] = cats.Show.fromToString
      implicit val jsonEncoderLineFluxSurfaceInput: io.circe.Encoder.AsObject[LineFluxSurfaceInput] = io.circe.generic.semiauto.deriveEncoder[LineFluxSurfaceInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class LinkUserInput(val programId: ProgramId, val userId: UserId, val role: ProgramUserRole, val supportType: clue.data.Input[ProgramUserSupportRoleType] = clue.data.Ignore, val supportPartner: clue.data.Input[Partner] = clue.data.Ignore)
    object LinkUserInput {
      val programId: monocle.Lens[LinkUserInput, ProgramId] = monocle.macros.GenLens[LinkUserInput](_.programId)
      val userId: monocle.Lens[LinkUserInput, UserId] = monocle.macros.GenLens[LinkUserInput](_.userId)
      val role: monocle.Lens[LinkUserInput, ProgramUserRole] = monocle.macros.GenLens[LinkUserInput](_.role)
      val supportType: monocle.Lens[LinkUserInput, clue.data.Input[ProgramUserSupportRoleType]] = monocle.macros.GenLens[LinkUserInput](_.supportType)
      val supportPartner: monocle.Lens[LinkUserInput, clue.data.Input[Partner]] = monocle.macros.GenLens[LinkUserInput](_.supportPartner)
      implicit val eqLinkUserInput: cats.Eq[LinkUserInput] = cats.Eq.fromUniversalEquals
      implicit val showLinkUserInput: cats.Show[LinkUserInput] = cats.Show.fromToString
      implicit val jsonEncoderLinkUserInput: io.circe.Encoder.AsObject[LinkUserInput] = io.circe.generic.semiauto.deriveEncoder[LinkUserInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class CreateUserInvitationInput(val programId: ProgramId, val role: ProgramUserRole, val supportType: clue.data.Input[ProgramUserSupportRoleType] = clue.data.Ignore, val supportPartner: clue.data.Input[Partner] = clue.data.Ignore)
    object CreateUserInvitationInput {
      val programId: monocle.Lens[CreateUserInvitationInput, ProgramId] = monocle.macros.GenLens[CreateUserInvitationInput](_.programId)
      val role: monocle.Lens[CreateUserInvitationInput, ProgramUserRole] = monocle.macros.GenLens[CreateUserInvitationInput](_.role)
      val supportType: monocle.Lens[CreateUserInvitationInput, clue.data.Input[ProgramUserSupportRoleType]] = monocle.macros.GenLens[CreateUserInvitationInput](_.supportType)
      val supportPartner: monocle.Lens[CreateUserInvitationInput, clue.data.Input[Partner]] = monocle.macros.GenLens[CreateUserInvitationInput](_.supportPartner)
      implicit val eqCreateUserInvitationInput: cats.Eq[CreateUserInvitationInput] = cats.Eq.fromUniversalEquals
      implicit val showCreateUserInvitationInput: cats.Show[CreateUserInvitationInput] = cats.Show.fromToString
      implicit val jsonEncoderCreateUserInvitationInput: io.circe.Encoder.AsObject[CreateUserInvitationInput] = io.circe.generic.semiauto.deriveEncoder[CreateUserInvitationInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class RedeemUserInvitationInput(val key: UserInvitationKey, val accept: clue.data.Input[Boolean] = clue.data.Ignore)
    object RedeemUserInvitationInput {
      val key: monocle.Lens[RedeemUserInvitationInput, UserInvitationKey] = monocle.macros.GenLens[RedeemUserInvitationInput](_.key)
      val accept: monocle.Lens[RedeemUserInvitationInput, clue.data.Input[Boolean]] = monocle.macros.GenLens[RedeemUserInvitationInput](_.accept)
      implicit val eqRedeemUserInvitationInput: cats.Eq[RedeemUserInvitationInput] = cats.Eq.fromUniversalEquals
      implicit val showRedeemUserInvitationInput: cats.Show[RedeemUserInvitationInput] = cats.Show.fromToString
      implicit val jsonEncoderRedeemUserInvitationInput: io.circe.Encoder.AsObject[RedeemUserInvitationInput] = io.circe.generic.semiauto.deriveEncoder[RedeemUserInvitationInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class RevokeUserInvitationInput(val id: UserInvitationId)
    object RevokeUserInvitationInput {
      val id: monocle.Lens[RevokeUserInvitationInput, UserInvitationId] = monocle.macros.GenLens[RevokeUserInvitationInput](_.id)
      implicit val eqRevokeUserInvitationInput: cats.Eq[RevokeUserInvitationInput] = cats.Eq.fromUniversalEquals
      implicit val showRevokeUserInvitationInput: cats.Show[RevokeUserInvitationInput] = cats.Show.fromToString
      implicit val jsonEncoderRevokeUserInvitationInput: io.circe.Encoder.AsObject[RevokeUserInvitationInput] = io.circe.generic.semiauto.deriveEncoder[RevokeUserInvitationInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class NonsiderealInput(val keyType: clue.data.Input[EphemerisKeyType] = clue.data.Ignore, val des: clue.data.Input[NonEmptyString] = clue.data.Ignore, val key: clue.data.Input[NonEmptyString] = clue.data.Ignore)
    object NonsiderealInput {
      val keyType: monocle.Lens[NonsiderealInput, clue.data.Input[EphemerisKeyType]] = monocle.macros.GenLens[NonsiderealInput](_.keyType)
      val des: monocle.Lens[NonsiderealInput, clue.data.Input[NonEmptyString]] = monocle.macros.GenLens[NonsiderealInput](_.des)
      val key: monocle.Lens[NonsiderealInput, clue.data.Input[NonEmptyString]] = monocle.macros.GenLens[NonsiderealInput](_.key)
      implicit val eqNonsiderealInput: cats.Eq[NonsiderealInput] = cats.Eq.fromUniversalEquals
      implicit val showNonsiderealInput: cats.Show[NonsiderealInput] = cats.Show.fromToString
      implicit val jsonEncoderNonsiderealInput: io.circe.Encoder.AsObject[NonsiderealInput] = io.circe.generic.semiauto.deriveEncoder[NonsiderealInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class ObservationPropertiesInput(val subtitle: clue.data.Input[NonEmptyString] = clue.data.Ignore, val status: clue.data.Input[ObsStatus] = clue.data.Ignore, val activeStatus: clue.data.Input[ObsActiveStatus] = clue.data.Ignore, val visualizationTime: clue.data.Input[Timestamp] = clue.data.Ignore, val posAngleConstraint: clue.data.Input[PosAngleConstraintInput] = clue.data.Ignore, val targetEnvironment: clue.data.Input[TargetEnvironmentInput] = clue.data.Ignore, val constraintSet: clue.data.Input[ConstraintSetInput] = clue.data.Ignore, val timingWindows: clue.data.Input[List[TimingWindowInput]] = clue.data.Ignore, val obsAttachments: clue.data.Input[List[ObsAttachmentId]] = clue.data.Ignore, val scienceRequirements: clue.data.Input[ScienceRequirementsInput] = clue.data.Ignore, val observingMode: clue.data.Input[ObservingModeInput] = clue.data.Ignore, val existence: clue.data.Input[Existence] = clue.data.Ignore, val groupId: clue.data.Input[GroupId] = clue.data.Ignore, val groupIndex: clue.data.Input[NonNegShort] = clue.data.Ignore)
    object ObservationPropertiesInput {
      val subtitle: monocle.Lens[ObservationPropertiesInput, clue.data.Input[NonEmptyString]] = monocle.macros.GenLens[ObservationPropertiesInput](_.subtitle)
      val status: monocle.Lens[ObservationPropertiesInput, clue.data.Input[ObsStatus]] = monocle.macros.GenLens[ObservationPropertiesInput](_.status)
      val activeStatus: monocle.Lens[ObservationPropertiesInput, clue.data.Input[ObsActiveStatus]] = monocle.macros.GenLens[ObservationPropertiesInput](_.activeStatus)
      val visualizationTime: monocle.Lens[ObservationPropertiesInput, clue.data.Input[Timestamp]] = monocle.macros.GenLens[ObservationPropertiesInput](_.visualizationTime)
      val posAngleConstraint: monocle.Lens[ObservationPropertiesInput, clue.data.Input[PosAngleConstraintInput]] = monocle.macros.GenLens[ObservationPropertiesInput](_.posAngleConstraint)
      val targetEnvironment: monocle.Lens[ObservationPropertiesInput, clue.data.Input[TargetEnvironmentInput]] = monocle.macros.GenLens[ObservationPropertiesInput](_.targetEnvironment)
      val constraintSet: monocle.Lens[ObservationPropertiesInput, clue.data.Input[ConstraintSetInput]] = monocle.macros.GenLens[ObservationPropertiesInput](_.constraintSet)
      val timingWindows: monocle.Lens[ObservationPropertiesInput, clue.data.Input[List[TimingWindowInput]]] = monocle.macros.GenLens[ObservationPropertiesInput](_.timingWindows)
      val obsAttachments: monocle.Lens[ObservationPropertiesInput, clue.data.Input[List[ObsAttachmentId]]] = monocle.macros.GenLens[ObservationPropertiesInput](_.obsAttachments)
      val scienceRequirements: monocle.Lens[ObservationPropertiesInput, clue.data.Input[ScienceRequirementsInput]] = monocle.macros.GenLens[ObservationPropertiesInput](_.scienceRequirements)
      val observingMode: monocle.Lens[ObservationPropertiesInput, clue.data.Input[ObservingModeInput]] = monocle.macros.GenLens[ObservationPropertiesInput](_.observingMode)
      val existence: monocle.Lens[ObservationPropertiesInput, clue.data.Input[Existence]] = monocle.macros.GenLens[ObservationPropertiesInput](_.existence)
      val groupId: monocle.Lens[ObservationPropertiesInput, clue.data.Input[GroupId]] = monocle.macros.GenLens[ObservationPropertiesInput](_.groupId)
      val groupIndex: monocle.Lens[ObservationPropertiesInput, clue.data.Input[NonNegShort]] = monocle.macros.GenLens[ObservationPropertiesInput](_.groupIndex)
      implicit val eqObservationPropertiesInput: cats.Eq[ObservationPropertiesInput] = cats.Eq.fromUniversalEquals
      implicit val showObservationPropertiesInput: cats.Show[ObservationPropertiesInput] = cats.Show.fromToString
      implicit val jsonEncoderObservationPropertiesInput: io.circe.Encoder.AsObject[ObservationPropertiesInput] = io.circe.generic.semiauto.deriveEncoder[ObservationPropertiesInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class OffsetComponentInput(val microarcseconds: clue.data.Input[Long] = clue.data.Ignore, val milliarcseconds: clue.data.Input[BigDecimal] = clue.data.Ignore, val arcseconds: clue.data.Input[BigDecimal] = clue.data.Ignore)
    object OffsetComponentInput {
      val microarcseconds: monocle.Lens[OffsetComponentInput, clue.data.Input[Long]] = monocle.macros.GenLens[OffsetComponentInput](_.microarcseconds)
      val milliarcseconds: monocle.Lens[OffsetComponentInput, clue.data.Input[BigDecimal]] = monocle.macros.GenLens[OffsetComponentInput](_.milliarcseconds)
      val arcseconds: monocle.Lens[OffsetComponentInput, clue.data.Input[BigDecimal]] = monocle.macros.GenLens[OffsetComponentInput](_.arcseconds)
      implicit val eqOffsetComponentInput: cats.Eq[OffsetComponentInput] = cats.Eq.fromUniversalEquals
      implicit val showOffsetComponentInput: cats.Show[OffsetComponentInput] = cats.Show.fromToString
      implicit val jsonEncoderOffsetComponentInput: io.circe.Encoder.AsObject[OffsetComponentInput] = io.circe.generic.semiauto.deriveEncoder[OffsetComponentInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class OffsetInput(val p: OffsetComponentInput, val q: OffsetComponentInput)
    object OffsetInput {
      val p: monocle.Lens[OffsetInput, OffsetComponentInput] = monocle.macros.GenLens[OffsetInput](_.p)
      val q: monocle.Lens[OffsetInput, OffsetComponentInput] = monocle.macros.GenLens[OffsetInput](_.q)
      implicit val eqOffsetInput: cats.Eq[OffsetInput] = cats.Eq.fromUniversalEquals
      implicit val showOffsetInput: cats.Show[OffsetInput] = cats.Show.fromToString
      implicit val jsonEncoderOffsetInput: io.circe.Encoder.AsObject[OffsetInput] = io.circe.generic.semiauto.deriveEncoder[OffsetInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class ParallaxInput(val microarcseconds: clue.data.Input[Long] = clue.data.Ignore, val milliarcseconds: clue.data.Input[BigDecimal] = clue.data.Ignore)
    object ParallaxInput {
      val microarcseconds: monocle.Lens[ParallaxInput, clue.data.Input[Long]] = monocle.macros.GenLens[ParallaxInput](_.microarcseconds)
      val milliarcseconds: monocle.Lens[ParallaxInput, clue.data.Input[BigDecimal]] = monocle.macros.GenLens[ParallaxInput](_.milliarcseconds)
      implicit val eqParallaxInput: cats.Eq[ParallaxInput] = cats.Eq.fromUniversalEquals
      implicit val showParallaxInput: cats.Show[ParallaxInput] = cats.Show.fromToString
      implicit val jsonEncoderParallaxInput: io.circe.Encoder.AsObject[ParallaxInput] = io.circe.generic.semiauto.deriveEncoder[ParallaxInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class PartnerSplitInput(val partner: Partner, val percent: IntPercent)
    object PartnerSplitInput {
      val partner: monocle.Lens[PartnerSplitInput, Partner] = monocle.macros.GenLens[PartnerSplitInput](_.partner)
      val percent: monocle.Lens[PartnerSplitInput, IntPercent] = monocle.macros.GenLens[PartnerSplitInput](_.percent)
      implicit val eqPartnerSplitInput: cats.Eq[PartnerSplitInput] = cats.Eq.fromUniversalEquals
      implicit val showPartnerSplitInput: cats.Show[PartnerSplitInput] = cats.Show.fromToString
      implicit val jsonEncoderPartnerSplitInput: io.circe.Encoder.AsObject[PartnerSplitInput] = io.circe.generic.semiauto.deriveEncoder[PartnerSplitInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class PoorWeatherInput(val minPercentTime: clue.data.Input[IntPercent] = clue.data.Ignore)
    object PoorWeatherInput {
      val minPercentTime: monocle.Lens[PoorWeatherInput, clue.data.Input[IntPercent]] = monocle.macros.GenLens[PoorWeatherInput](_.minPercentTime)
      implicit val eqPoorWeatherInput: cats.Eq[PoorWeatherInput] = cats.Eq.fromUniversalEquals
      implicit val showPoorWeatherInput: cats.Show[PoorWeatherInput] = cats.Show.fromToString
      implicit val jsonEncoderPoorWeatherInput: io.circe.Encoder.AsObject[PoorWeatherInput] = io.circe.generic.semiauto.deriveEncoder[PoorWeatherInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class PosAngleConstraintInput(val mode: clue.data.Input[PosAngleConstraintMode] = clue.data.Ignore, val angle: clue.data.Input[AngleInput] = clue.data.Ignore)
    object PosAngleConstraintInput {
      val mode: monocle.Lens[PosAngleConstraintInput, clue.data.Input[PosAngleConstraintMode]] = monocle.macros.GenLens[PosAngleConstraintInput](_.mode)
      val angle: monocle.Lens[PosAngleConstraintInput, clue.data.Input[AngleInput]] = monocle.macros.GenLens[PosAngleConstraintInput](_.angle)
      implicit val eqPosAngleConstraintInput: cats.Eq[PosAngleConstraintInput] = cats.Eq.fromUniversalEquals
      implicit val showPosAngleConstraintInput: cats.Show[PosAngleConstraintInput] = cats.Show.fromToString
      implicit val jsonEncoderPosAngleConstraintInput: io.circe.Encoder.AsObject[PosAngleConstraintInput] = io.circe.generic.semiauto.deriveEncoder[PosAngleConstraintInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class ProgramPropertiesInput(val name: clue.data.Input[NonEmptyString] = clue.data.Ignore, val semester: clue.data.Input[Semester] = clue.data.Ignore, val proposalStatus: clue.data.Input[ProposalStatus] = clue.data.Ignore, val proposal: clue.data.Input[ProposalInput] = clue.data.Ignore, val existence: clue.data.Input[Existence] = clue.data.Ignore)
    object ProgramPropertiesInput {
      val name: monocle.Lens[ProgramPropertiesInput, clue.data.Input[NonEmptyString]] = monocle.macros.GenLens[ProgramPropertiesInput](_.name)
      val semester: monocle.Lens[ProgramPropertiesInput, clue.data.Input[Semester]] = monocle.macros.GenLens[ProgramPropertiesInput](_.semester)
      val proposalStatus: monocle.Lens[ProgramPropertiesInput, clue.data.Input[ProposalStatus]] = monocle.macros.GenLens[ProgramPropertiesInput](_.proposalStatus)
      val proposal: monocle.Lens[ProgramPropertiesInput, clue.data.Input[ProposalInput]] = monocle.macros.GenLens[ProgramPropertiesInput](_.proposal)
      val existence: monocle.Lens[ProgramPropertiesInput, clue.data.Input[Existence]] = monocle.macros.GenLens[ProgramPropertiesInput](_.existence)
      implicit val eqProgramPropertiesInput: cats.Eq[ProgramPropertiesInput] = cats.Eq.fromUniversalEquals
      implicit val showProgramPropertiesInput: cats.Show[ProgramPropertiesInput] = cats.Show.fromToString
      implicit val jsonEncoderProgramPropertiesInput: io.circe.Encoder.AsObject[ProgramPropertiesInput] = io.circe.generic.semiauto.deriveEncoder[ProgramPropertiesInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class ProperMotionComponentInput(val microarcsecondsPerYear: clue.data.Input[Long] = clue.data.Ignore, val milliarcsecondsPerYear: clue.data.Input[BigDecimal] = clue.data.Ignore)
    object ProperMotionComponentInput {
      val microarcsecondsPerYear: monocle.Lens[ProperMotionComponentInput, clue.data.Input[Long]] = monocle.macros.GenLens[ProperMotionComponentInput](_.microarcsecondsPerYear)
      val milliarcsecondsPerYear: monocle.Lens[ProperMotionComponentInput, clue.data.Input[BigDecimal]] = monocle.macros.GenLens[ProperMotionComponentInput](_.milliarcsecondsPerYear)
      implicit val eqProperMotionComponentInput: cats.Eq[ProperMotionComponentInput] = cats.Eq.fromUniversalEquals
      implicit val showProperMotionComponentInput: cats.Show[ProperMotionComponentInput] = cats.Show.fromToString
      implicit val jsonEncoderProperMotionComponentInput: io.circe.Encoder.AsObject[ProperMotionComponentInput] = io.circe.generic.semiauto.deriveEncoder[ProperMotionComponentInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class ProperMotionInput(val ra: ProperMotionComponentInput, val dec: ProperMotionComponentInput)
    object ProperMotionInput {
      val ra: monocle.Lens[ProperMotionInput, ProperMotionComponentInput] = monocle.macros.GenLens[ProperMotionInput](_.ra)
      val dec: monocle.Lens[ProperMotionInput, ProperMotionComponentInput] = monocle.macros.GenLens[ProperMotionInput](_.dec)
      implicit val eqProperMotionInput: cats.Eq[ProperMotionInput] = cats.Eq.fromUniversalEquals
      implicit val showProperMotionInput: cats.Show[ProperMotionInput] = cats.Show.fromToString
      implicit val jsonEncoderProperMotionInput: io.circe.Encoder.AsObject[ProperMotionInput] = io.circe.generic.semiauto.deriveEncoder[ProperMotionInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class ProposalClassInput(val classical: clue.data.Input[ClassicalInput] = clue.data.Ignore, val demoScience: clue.data.Input[DemoScienceInput] = clue.data.Ignore, val directorsTime: clue.data.Input[DirectorsTimeInput] = clue.data.Ignore, val exchange: clue.data.Input[ExchangeInput] = clue.data.Ignore, val fastTurnaround: clue.data.Input[FastTurnaroundInput] = clue.data.Ignore, val poorWeather: clue.data.Input[PoorWeatherInput] = clue.data.Ignore, val queue: clue.data.Input[QueueInput] = clue.data.Ignore, val systemVerification: clue.data.Input[SystemVerificationInput] = clue.data.Ignore, val largeProgram: clue.data.Input[LargeProgramInput] = clue.data.Ignore, val intensive: clue.data.Input[IntensiveInput] = clue.data.Ignore)
    object ProposalClassInput {
      val classical: monocle.Lens[ProposalClassInput, clue.data.Input[ClassicalInput]] = monocle.macros.GenLens[ProposalClassInput](_.classical)
      val demoScience: monocle.Lens[ProposalClassInput, clue.data.Input[DemoScienceInput]] = monocle.macros.GenLens[ProposalClassInput](_.demoScience)
      val directorsTime: monocle.Lens[ProposalClassInput, clue.data.Input[DirectorsTimeInput]] = monocle.macros.GenLens[ProposalClassInput](_.directorsTime)
      val exchange: monocle.Lens[ProposalClassInput, clue.data.Input[ExchangeInput]] = monocle.macros.GenLens[ProposalClassInput](_.exchange)
      val fastTurnaround: monocle.Lens[ProposalClassInput, clue.data.Input[FastTurnaroundInput]] = monocle.macros.GenLens[ProposalClassInput](_.fastTurnaround)
      val poorWeather: monocle.Lens[ProposalClassInput, clue.data.Input[PoorWeatherInput]] = monocle.macros.GenLens[ProposalClassInput](_.poorWeather)
      val queue: monocle.Lens[ProposalClassInput, clue.data.Input[QueueInput]] = monocle.macros.GenLens[ProposalClassInput](_.queue)
      val systemVerification: monocle.Lens[ProposalClassInput, clue.data.Input[SystemVerificationInput]] = monocle.macros.GenLens[ProposalClassInput](_.systemVerification)
      val largeProgram: monocle.Lens[ProposalClassInput, clue.data.Input[LargeProgramInput]] = monocle.macros.GenLens[ProposalClassInput](_.largeProgram)
      val intensive: monocle.Lens[ProposalClassInput, clue.data.Input[IntensiveInput]] = monocle.macros.GenLens[ProposalClassInput](_.intensive)
      implicit val eqProposalClassInput: cats.Eq[ProposalClassInput] = cats.Eq.fromUniversalEquals
      implicit val showProposalClassInput: cats.Show[ProposalClassInput] = cats.Show.fromToString
      implicit val jsonEncoderProposalClassInput: io.circe.Encoder.AsObject[ProposalClassInput] = io.circe.generic.semiauto.deriveEncoder[ProposalClassInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class ProposalInput(val title: clue.data.Input[NonEmptyString] = clue.data.Ignore, val proposalClass: clue.data.Input[ProposalClassInput] = clue.data.Ignore, val category: clue.data.Input[TacCategory] = clue.data.Ignore, val toOActivation: clue.data.Input[ToOActivation] = clue.data.Ignore, val `abstract`: clue.data.Input[NonEmptyString] = clue.data.Ignore, val partnerSplits: clue.data.Input[List[PartnerSplitInput]] = clue.data.Ignore)
    object ProposalInput {
      val title: monocle.Lens[ProposalInput, clue.data.Input[NonEmptyString]] = monocle.macros.GenLens[ProposalInput](_.title)
      val proposalClass: monocle.Lens[ProposalInput, clue.data.Input[ProposalClassInput]] = monocle.macros.GenLens[ProposalInput](_.proposalClass)
      val category: monocle.Lens[ProposalInput, clue.data.Input[TacCategory]] = monocle.macros.GenLens[ProposalInput](_.category)
      val toOActivation: monocle.Lens[ProposalInput, clue.data.Input[ToOActivation]] = monocle.macros.GenLens[ProposalInput](_.toOActivation)
      val `abstract`: monocle.Lens[ProposalInput, clue.data.Input[NonEmptyString]] = monocle.macros.GenLens[ProposalInput](_.`abstract`)
      val partnerSplits: monocle.Lens[ProposalInput, clue.data.Input[List[PartnerSplitInput]]] = monocle.macros.GenLens[ProposalInput](_.partnerSplits)
      implicit val eqProposalInput: cats.Eq[ProposalInput] = cats.Eq.fromUniversalEquals
      implicit val showProposalInput: cats.Show[ProposalInput] = cats.Show.fromToString
      implicit val jsonEncoderProposalInput: io.circe.Encoder.AsObject[ProposalInput] = io.circe.generic.semiauto.deriveEncoder[ProposalInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class QueueInput(val minPercentTime: clue.data.Input[IntPercent] = clue.data.Ignore)
    object QueueInput {
      val minPercentTime: monocle.Lens[QueueInput, clue.data.Input[IntPercent]] = monocle.macros.GenLens[QueueInput](_.minPercentTime)
      implicit val eqQueueInput: cats.Eq[QueueInput] = cats.Eq.fromUniversalEquals
      implicit val showQueueInput: cats.Show[QueueInput] = cats.Show.fromToString
      implicit val jsonEncoderQueueInput: io.circe.Encoder.AsObject[QueueInput] = io.circe.generic.semiauto.deriveEncoder[QueueInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class RadialVelocityInput(val centimetersPerSecond: clue.data.Input[Long] = clue.data.Ignore, val metersPerSecond: clue.data.Input[BigDecimal] = clue.data.Ignore, val kilometersPerSecond: clue.data.Input[BigDecimal] = clue.data.Ignore)
    object RadialVelocityInput {
      val centimetersPerSecond: monocle.Lens[RadialVelocityInput, clue.data.Input[Long]] = monocle.macros.GenLens[RadialVelocityInput](_.centimetersPerSecond)
      val metersPerSecond: monocle.Lens[RadialVelocityInput, clue.data.Input[BigDecimal]] = monocle.macros.GenLens[RadialVelocityInput](_.metersPerSecond)
      val kilometersPerSecond: monocle.Lens[RadialVelocityInput, clue.data.Input[BigDecimal]] = monocle.macros.GenLens[RadialVelocityInput](_.kilometersPerSecond)
      implicit val eqRadialVelocityInput: cats.Eq[RadialVelocityInput] = cats.Eq.fromUniversalEquals
      implicit val showRadialVelocityInput: cats.Show[RadialVelocityInput] = cats.Show.fromToString
      implicit val jsonEncoderRadialVelocityInput: io.circe.Encoder.AsObject[RadialVelocityInput] = io.circe.generic.semiauto.deriveEncoder[RadialVelocityInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class RecordAtomInput(val visitId: VisitId, val instrument: Instrument, val sequenceType: SequenceType, val stepCount: NonNegShort)
    object RecordAtomInput {
      val visitId: monocle.Lens[RecordAtomInput, VisitId] = monocle.macros.GenLens[RecordAtomInput](_.visitId)
      val instrument: monocle.Lens[RecordAtomInput, Instrument] = monocle.macros.GenLens[RecordAtomInput](_.instrument)
      val sequenceType: monocle.Lens[RecordAtomInput, SequenceType] = monocle.macros.GenLens[RecordAtomInput](_.sequenceType)
      val stepCount: monocle.Lens[RecordAtomInput, NonNegShort] = monocle.macros.GenLens[RecordAtomInput](_.stepCount)
      implicit val eqRecordAtomInput: cats.Eq[RecordAtomInput] = cats.Eq.fromUniversalEquals
      implicit val showRecordAtomInput: cats.Show[RecordAtomInput] = cats.Show.fromToString
      implicit val jsonEncoderRecordAtomInput: io.circe.Encoder.AsObject[RecordAtomInput] = io.circe.generic.semiauto.deriveEncoder[RecordAtomInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class RecordDatasetInput(val stepId: StepId, val filename: DatasetFilename, val qaState: clue.data.Input[DatasetQaState] = clue.data.Ignore)
    object RecordDatasetInput {
      val stepId: monocle.Lens[RecordDatasetInput, StepId] = monocle.macros.GenLens[RecordDatasetInput](_.stepId)
      val filename: monocle.Lens[RecordDatasetInput, DatasetFilename] = monocle.macros.GenLens[RecordDatasetInput](_.filename)
      val qaState: monocle.Lens[RecordDatasetInput, clue.data.Input[DatasetQaState]] = monocle.macros.GenLens[RecordDatasetInput](_.qaState)
      implicit val eqRecordDatasetInput: cats.Eq[RecordDatasetInput] = cats.Eq.fromUniversalEquals
      implicit val showRecordDatasetInput: cats.Show[RecordDatasetInput] = cats.Show.fromToString
      implicit val jsonEncoderRecordDatasetInput: io.circe.Encoder.AsObject[RecordDatasetInput] = io.circe.generic.semiauto.deriveEncoder[RecordDatasetInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class RecordGmosNorthStepInput(val atomId: AtomId, val gmosNorth: GmosNorthDynamicInput, val stepConfig: StepConfigInput, val observeClass: ObserveClass)
    object RecordGmosNorthStepInput {
      val atomId: monocle.Lens[RecordGmosNorthStepInput, AtomId] = monocle.macros.GenLens[RecordGmosNorthStepInput](_.atomId)
      val gmosNorth: monocle.Lens[RecordGmosNorthStepInput, GmosNorthDynamicInput] = monocle.macros.GenLens[RecordGmosNorthStepInput](_.gmosNorth)
      val stepConfig: monocle.Lens[RecordGmosNorthStepInput, StepConfigInput] = monocle.macros.GenLens[RecordGmosNorthStepInput](_.stepConfig)
      val observeClass: monocle.Lens[RecordGmosNorthStepInput, ObserveClass] = monocle.macros.GenLens[RecordGmosNorthStepInput](_.observeClass)
      implicit val eqRecordGmosNorthStepInput: cats.Eq[RecordGmosNorthStepInput] = cats.Eq.fromUniversalEquals
      implicit val showRecordGmosNorthStepInput: cats.Show[RecordGmosNorthStepInput] = cats.Show.fromToString
      implicit val jsonEncoderRecordGmosNorthStepInput: io.circe.Encoder.AsObject[RecordGmosNorthStepInput] = io.circe.generic.semiauto.deriveEncoder[RecordGmosNorthStepInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class RecordGmosNorthVisitInput(val observationId: ObservationId, val gmosNorth: GmosNorthStaticInput)
    object RecordGmosNorthVisitInput {
      val observationId: monocle.Lens[RecordGmosNorthVisitInput, ObservationId] = monocle.macros.GenLens[RecordGmosNorthVisitInput](_.observationId)
      val gmosNorth: monocle.Lens[RecordGmosNorthVisitInput, GmosNorthStaticInput] = monocle.macros.GenLens[RecordGmosNorthVisitInput](_.gmosNorth)
      implicit val eqRecordGmosNorthVisitInput: cats.Eq[RecordGmosNorthVisitInput] = cats.Eq.fromUniversalEquals
      implicit val showRecordGmosNorthVisitInput: cats.Show[RecordGmosNorthVisitInput] = cats.Show.fromToString
      implicit val jsonEncoderRecordGmosNorthVisitInput: io.circe.Encoder.AsObject[RecordGmosNorthVisitInput] = io.circe.generic.semiauto.deriveEncoder[RecordGmosNorthVisitInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class RecordGmosSouthStepInput(val atomId: AtomId, val gmosSouth: GmosSouthDynamicInput, val stepConfig: StepConfigInput, val observeClass: ObserveClass)
    object RecordGmosSouthStepInput {
      val atomId: monocle.Lens[RecordGmosSouthStepInput, AtomId] = monocle.macros.GenLens[RecordGmosSouthStepInput](_.atomId)
      val gmosSouth: monocle.Lens[RecordGmosSouthStepInput, GmosSouthDynamicInput] = monocle.macros.GenLens[RecordGmosSouthStepInput](_.gmosSouth)
      val stepConfig: monocle.Lens[RecordGmosSouthStepInput, StepConfigInput] = monocle.macros.GenLens[RecordGmosSouthStepInput](_.stepConfig)
      val observeClass: monocle.Lens[RecordGmosSouthStepInput, ObserveClass] = monocle.macros.GenLens[RecordGmosSouthStepInput](_.observeClass)
      implicit val eqRecordGmosSouthStepInput: cats.Eq[RecordGmosSouthStepInput] = cats.Eq.fromUniversalEquals
      implicit val showRecordGmosSouthStepInput: cats.Show[RecordGmosSouthStepInput] = cats.Show.fromToString
      implicit val jsonEncoderRecordGmosSouthStepInput: io.circe.Encoder.AsObject[RecordGmosSouthStepInput] = io.circe.generic.semiauto.deriveEncoder[RecordGmosSouthStepInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class RecordGmosSouthVisitInput(val observationId: ObservationId, val gmosSouth: GmosSouthStaticInput)
    object RecordGmosSouthVisitInput {
      val observationId: monocle.Lens[RecordGmosSouthVisitInput, ObservationId] = monocle.macros.GenLens[RecordGmosSouthVisitInput](_.observationId)
      val gmosSouth: monocle.Lens[RecordGmosSouthVisitInput, GmosSouthStaticInput] = monocle.macros.GenLens[RecordGmosSouthVisitInput](_.gmosSouth)
      implicit val eqRecordGmosSouthVisitInput: cats.Eq[RecordGmosSouthVisitInput] = cats.Eq.fromUniversalEquals
      implicit val showRecordGmosSouthVisitInput: cats.Show[RecordGmosSouthVisitInput] = cats.Show.fromToString
      implicit val jsonEncoderRecordGmosSouthVisitInput: io.circe.Encoder.AsObject[RecordGmosSouthVisitInput] = io.circe.generic.semiauto.deriveEncoder[RecordGmosSouthVisitInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class RightAscensionInput(val microarcseconds: clue.data.Input[Long] = clue.data.Ignore, val microseconds: clue.data.Input[Long] = clue.data.Ignore, val degrees: clue.data.Input[BigDecimal] = clue.data.Ignore, val hours: clue.data.Input[BigDecimal] = clue.data.Ignore, val hms: clue.data.Input[HmsString] = clue.data.Ignore)
    object RightAscensionInput {
      val microarcseconds: monocle.Lens[RightAscensionInput, clue.data.Input[Long]] = monocle.macros.GenLens[RightAscensionInput](_.microarcseconds)
      val microseconds: monocle.Lens[RightAscensionInput, clue.data.Input[Long]] = monocle.macros.GenLens[RightAscensionInput](_.microseconds)
      val degrees: monocle.Lens[RightAscensionInput, clue.data.Input[BigDecimal]] = monocle.macros.GenLens[RightAscensionInput](_.degrees)
      val hours: monocle.Lens[RightAscensionInput, clue.data.Input[BigDecimal]] = monocle.macros.GenLens[RightAscensionInput](_.hours)
      val hms: monocle.Lens[RightAscensionInput, clue.data.Input[HmsString]] = monocle.macros.GenLens[RightAscensionInput](_.hms)
      implicit val eqRightAscensionInput: cats.Eq[RightAscensionInput] = cats.Eq.fromUniversalEquals
      implicit val showRightAscensionInput: cats.Show[RightAscensionInput] = cats.Show.fromToString
      implicit val jsonEncoderRightAscensionInput: io.circe.Encoder.AsObject[RightAscensionInput] = io.circe.generic.semiauto.deriveEncoder[RightAscensionInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class ObservingModeInput(val gmosNorthLongSlit: clue.data.Input[GmosNorthLongSlitInput] = clue.data.Ignore, val gmosSouthLongSlit: clue.data.Input[GmosSouthLongSlitInput] = clue.data.Ignore)
    object ObservingModeInput {
      val gmosNorthLongSlit: monocle.Lens[ObservingModeInput, clue.data.Input[GmosNorthLongSlitInput]] = monocle.macros.GenLens[ObservingModeInput](_.gmosNorthLongSlit)
      val gmosSouthLongSlit: monocle.Lens[ObservingModeInput, clue.data.Input[GmosSouthLongSlitInput]] = monocle.macros.GenLens[ObservingModeInput](_.gmosSouthLongSlit)
      implicit val eqObservingModeInput: cats.Eq[ObservingModeInput] = cats.Eq.fromUniversalEquals
      implicit val showObservingModeInput: cats.Show[ObservingModeInput] = cats.Show.fromToString
      implicit val jsonEncoderObservingModeInput: io.circe.Encoder.AsObject[ObservingModeInput] = io.circe.generic.semiauto.deriveEncoder[ObservingModeInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class ScienceRequirementsInput(val mode: clue.data.Input[ScienceMode] = clue.data.Ignore, val spectroscopy: clue.data.Input[SpectroscopyScienceRequirementsInput] = clue.data.Ignore)
    object ScienceRequirementsInput {
      val mode: monocle.Lens[ScienceRequirementsInput, clue.data.Input[ScienceMode]] = monocle.macros.GenLens[ScienceRequirementsInput](_.mode)
      val spectroscopy: monocle.Lens[ScienceRequirementsInput, clue.data.Input[SpectroscopyScienceRequirementsInput]] = monocle.macros.GenLens[ScienceRequirementsInput](_.spectroscopy)
      implicit val eqScienceRequirementsInput: cats.Eq[ScienceRequirementsInput] = cats.Eq.fromUniversalEquals
      implicit val showScienceRequirementsInput: cats.Show[ScienceRequirementsInput] = cats.Show.fromToString
      implicit val jsonEncoderScienceRequirementsInput: io.circe.Encoder.AsObject[ScienceRequirementsInput] = io.circe.generic.semiauto.deriveEncoder[ScienceRequirementsInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class SetAllocationInput(val programId: ProgramId, val partner: Partner, val duration: TimeSpanInput)
    object SetAllocationInput {
      val programId: monocle.Lens[SetAllocationInput, ProgramId] = monocle.macros.GenLens[SetAllocationInput](_.programId)
      val partner: monocle.Lens[SetAllocationInput, Partner] = monocle.macros.GenLens[SetAllocationInput](_.partner)
      val duration: monocle.Lens[SetAllocationInput, TimeSpanInput] = monocle.macros.GenLens[SetAllocationInput](_.duration)
      implicit val eqSetAllocationInput: cats.Eq[SetAllocationInput] = cats.Eq.fromUniversalEquals
      implicit val showSetAllocationInput: cats.Show[SetAllocationInput] = cats.Show.fromToString
      implicit val jsonEncoderSetAllocationInput: io.circe.Encoder.AsObject[SetAllocationInput] = io.circe.generic.semiauto.deriveEncoder[SetAllocationInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class SetProgramReferenceInput(val programId: clue.data.Input[ProgramId] = clue.data.Ignore, val proposalReference: clue.data.Input[ProposalReferenceLabel] = clue.data.Ignore, val programReference: clue.data.Input[ProgramReferenceLabel] = clue.data.Ignore, val SET: ProgramReferencePropertiesInput)
    object SetProgramReferenceInput {
      val programId: monocle.Lens[SetProgramReferenceInput, clue.data.Input[ProgramId]] = monocle.macros.GenLens[SetProgramReferenceInput](_.programId)
      val proposalReference: monocle.Lens[SetProgramReferenceInput, clue.data.Input[ProposalReferenceLabel]] = monocle.macros.GenLens[SetProgramReferenceInput](_.proposalReference)
      val programReference: monocle.Lens[SetProgramReferenceInput, clue.data.Input[ProgramReferenceLabel]] = monocle.macros.GenLens[SetProgramReferenceInput](_.programReference)
      val SET: monocle.Lens[SetProgramReferenceInput, ProgramReferencePropertiesInput] = monocle.macros.GenLens[SetProgramReferenceInput](_.SET)
      implicit val eqSetProgramReferenceInput: cats.Eq[SetProgramReferenceInput] = cats.Eq.fromUniversalEquals
      implicit val showSetProgramReferenceInput: cats.Show[SetProgramReferenceInput] = cats.Show.fromToString
      implicit val jsonEncoderSetProgramReferenceInput: io.circe.Encoder.AsObject[SetProgramReferenceInput] = io.circe.generic.semiauto.deriveEncoder[SetProgramReferenceInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class ProgramReferencePropertiesInput(val calibration: clue.data.Input[ProgramReferencePropertiesCalibrationInput] = clue.data.Ignore, val engineering: clue.data.Input[ProgramReferencePropertiesEngineeringInput] = clue.data.Ignore, val example: clue.data.Input[ProgramReferencePropertiesExampleInput] = clue.data.Ignore, val library: clue.data.Input[ProgramReferencePropertiesLibraryInput] = clue.data.Ignore, val science: clue.data.Input[ProgramReferencePropertiesScienceInput] = clue.data.Ignore)
    object ProgramReferencePropertiesInput {
      val calibration: monocle.Lens[ProgramReferencePropertiesInput, clue.data.Input[ProgramReferencePropertiesCalibrationInput]] = monocle.macros.GenLens[ProgramReferencePropertiesInput](_.calibration)
      val engineering: monocle.Lens[ProgramReferencePropertiesInput, clue.data.Input[ProgramReferencePropertiesEngineeringInput]] = monocle.macros.GenLens[ProgramReferencePropertiesInput](_.engineering)
      val example: monocle.Lens[ProgramReferencePropertiesInput, clue.data.Input[ProgramReferencePropertiesExampleInput]] = monocle.macros.GenLens[ProgramReferencePropertiesInput](_.example)
      val library: monocle.Lens[ProgramReferencePropertiesInput, clue.data.Input[ProgramReferencePropertiesLibraryInput]] = monocle.macros.GenLens[ProgramReferencePropertiesInput](_.library)
      val science: monocle.Lens[ProgramReferencePropertiesInput, clue.data.Input[ProgramReferencePropertiesScienceInput]] = monocle.macros.GenLens[ProgramReferencePropertiesInput](_.science)
      implicit val eqProgramReferencePropertiesInput: cats.Eq[ProgramReferencePropertiesInput] = cats.Eq.fromUniversalEquals
      implicit val showProgramReferencePropertiesInput: cats.Show[ProgramReferencePropertiesInput] = cats.Show.fromToString
      implicit val jsonEncoderProgramReferencePropertiesInput: io.circe.Encoder.AsObject[ProgramReferencePropertiesInput] = io.circe.generic.semiauto.deriveEncoder[ProgramReferencePropertiesInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class ProgramReferencePropertiesCalibrationInput(val semester: Semester, val instrument: Instrument)
    object ProgramReferencePropertiesCalibrationInput {
      val semester: monocle.Lens[ProgramReferencePropertiesCalibrationInput, Semester] = monocle.macros.GenLens[ProgramReferencePropertiesCalibrationInput](_.semester)
      val instrument: monocle.Lens[ProgramReferencePropertiesCalibrationInput, Instrument] = monocle.macros.GenLens[ProgramReferencePropertiesCalibrationInput](_.instrument)
      implicit val eqProgramReferencePropertiesCalibrationInput: cats.Eq[ProgramReferencePropertiesCalibrationInput] = cats.Eq.fromUniversalEquals
      implicit val showProgramReferencePropertiesCalibrationInput: cats.Show[ProgramReferencePropertiesCalibrationInput] = cats.Show.fromToString
      implicit val jsonEncoderProgramReferencePropertiesCalibrationInput: io.circe.Encoder.AsObject[ProgramReferencePropertiesCalibrationInput] = io.circe.generic.semiauto.deriveEncoder[ProgramReferencePropertiesCalibrationInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class ProgramReferencePropertiesEngineeringInput(val semester: Semester, val instrument: Instrument)
    object ProgramReferencePropertiesEngineeringInput {
      val semester: monocle.Lens[ProgramReferencePropertiesEngineeringInput, Semester] = monocle.macros.GenLens[ProgramReferencePropertiesEngineeringInput](_.semester)
      val instrument: monocle.Lens[ProgramReferencePropertiesEngineeringInput, Instrument] = monocle.macros.GenLens[ProgramReferencePropertiesEngineeringInput](_.instrument)
      implicit val eqProgramReferencePropertiesEngineeringInput: cats.Eq[ProgramReferencePropertiesEngineeringInput] = cats.Eq.fromUniversalEquals
      implicit val showProgramReferencePropertiesEngineeringInput: cats.Show[ProgramReferencePropertiesEngineeringInput] = cats.Show.fromToString
      implicit val jsonEncoderProgramReferencePropertiesEngineeringInput: io.circe.Encoder.AsObject[ProgramReferencePropertiesEngineeringInput] = io.circe.generic.semiauto.deriveEncoder[ProgramReferencePropertiesEngineeringInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class ProgramReferencePropertiesExampleInput(val instrument: Instrument)
    object ProgramReferencePropertiesExampleInput {
      val instrument: monocle.Lens[ProgramReferencePropertiesExampleInput, Instrument] = monocle.macros.GenLens[ProgramReferencePropertiesExampleInput](_.instrument)
      implicit val eqProgramReferencePropertiesExampleInput: cats.Eq[ProgramReferencePropertiesExampleInput] = cats.Eq.fromUniversalEquals
      implicit val showProgramReferencePropertiesExampleInput: cats.Show[ProgramReferencePropertiesExampleInput] = cats.Show.fromToString
      implicit val jsonEncoderProgramReferencePropertiesExampleInput: io.circe.Encoder.AsObject[ProgramReferencePropertiesExampleInput] = io.circe.generic.semiauto.deriveEncoder[ProgramReferencePropertiesExampleInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class ProgramReferencePropertiesLibraryInput(val instrument: Instrument, val description: NonEmptyString)
    object ProgramReferencePropertiesLibraryInput {
      val instrument: monocle.Lens[ProgramReferencePropertiesLibraryInput, Instrument] = monocle.macros.GenLens[ProgramReferencePropertiesLibraryInput](_.instrument)
      val description: monocle.Lens[ProgramReferencePropertiesLibraryInput, NonEmptyString] = monocle.macros.GenLens[ProgramReferencePropertiesLibraryInput](_.description)
      implicit val eqProgramReferencePropertiesLibraryInput: cats.Eq[ProgramReferencePropertiesLibraryInput] = cats.Eq.fromUniversalEquals
      implicit val showProgramReferencePropertiesLibraryInput: cats.Show[ProgramReferencePropertiesLibraryInput] = cats.Show.fromToString
      implicit val jsonEncoderProgramReferencePropertiesLibraryInput: io.circe.Encoder.AsObject[ProgramReferencePropertiesLibraryInput] = io.circe.generic.semiauto.deriveEncoder[ProgramReferencePropertiesLibraryInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class ProgramReferencePropertiesScienceInput(val semester: Semester, val scienceSubtype: ScienceSubtype)
    object ProgramReferencePropertiesScienceInput {
      val semester: monocle.Lens[ProgramReferencePropertiesScienceInput, Semester] = monocle.macros.GenLens[ProgramReferencePropertiesScienceInput](_.semester)
      val scienceSubtype: monocle.Lens[ProgramReferencePropertiesScienceInput, ScienceSubtype] = monocle.macros.GenLens[ProgramReferencePropertiesScienceInput](_.scienceSubtype)
      implicit val eqProgramReferencePropertiesScienceInput: cats.Eq[ProgramReferencePropertiesScienceInput] = cats.Eq.fromUniversalEquals
      implicit val showProgramReferencePropertiesScienceInput: cats.Show[ProgramReferencePropertiesScienceInput] = cats.Show.fromToString
      implicit val jsonEncoderProgramReferencePropertiesScienceInput: io.circe.Encoder.AsObject[ProgramReferencePropertiesScienceInput] = io.circe.generic.semiauto.deriveEncoder[ProgramReferencePropertiesScienceInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class SiderealInput(val ra: clue.data.Input[RightAscensionInput] = clue.data.Ignore, val dec: clue.data.Input[DeclinationInput] = clue.data.Ignore, val epoch: clue.data.Input[EpochString] = clue.data.Ignore, val properMotion: clue.data.Input[ProperMotionInput] = clue.data.Ignore, val radialVelocity: clue.data.Input[RadialVelocityInput] = clue.data.Ignore, val parallax: clue.data.Input[ParallaxInput] = clue.data.Ignore, val catalogInfo: clue.data.Input[CatalogInfoInput] = clue.data.Ignore)
    object SiderealInput {
      val ra: monocle.Lens[SiderealInput, clue.data.Input[RightAscensionInput]] = monocle.macros.GenLens[SiderealInput](_.ra)
      val dec: monocle.Lens[SiderealInput, clue.data.Input[DeclinationInput]] = monocle.macros.GenLens[SiderealInput](_.dec)
      val epoch: monocle.Lens[SiderealInput, clue.data.Input[EpochString]] = monocle.macros.GenLens[SiderealInput](_.epoch)
      val properMotion: monocle.Lens[SiderealInput, clue.data.Input[ProperMotionInput]] = monocle.macros.GenLens[SiderealInput](_.properMotion)
      val radialVelocity: monocle.Lens[SiderealInput, clue.data.Input[RadialVelocityInput]] = monocle.macros.GenLens[SiderealInput](_.radialVelocity)
      val parallax: monocle.Lens[SiderealInput, clue.data.Input[ParallaxInput]] = monocle.macros.GenLens[SiderealInput](_.parallax)
      val catalogInfo: monocle.Lens[SiderealInput, clue.data.Input[CatalogInfoInput]] = monocle.macros.GenLens[SiderealInput](_.catalogInfo)
      implicit val eqSiderealInput: cats.Eq[SiderealInput] = cats.Eq.fromUniversalEquals
      implicit val showSiderealInput: cats.Show[SiderealInput] = cats.Show.fromToString
      implicit val jsonEncoderSiderealInput: io.circe.Encoder.AsObject[SiderealInput] = io.circe.generic.semiauto.deriveEncoder[SiderealInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class SignalToNoiseModeInput(val value: SignalToNoise)
    object SignalToNoiseModeInput {
      val value: monocle.Lens[SignalToNoiseModeInput, SignalToNoise] = monocle.macros.GenLens[SignalToNoiseModeInput](_.value)
      implicit val eqSignalToNoiseModeInput: cats.Eq[SignalToNoiseModeInput] = cats.Eq.fromUniversalEquals
      implicit val showSignalToNoiseModeInput: cats.Show[SignalToNoiseModeInput] = cats.Show.fromToString
      implicit val jsonEncoderSignalToNoiseModeInput: io.circe.Encoder.AsObject[SignalToNoiseModeInput] = io.circe.generic.semiauto.deriveEncoder[SignalToNoiseModeInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class SourceProfileInput(val point: clue.data.Input[SpectralDefinitionIntegratedInput] = clue.data.Ignore, val uniform: clue.data.Input[SpectralDefinitionSurfaceInput] = clue.data.Ignore, val gaussian: clue.data.Input[GaussianInput] = clue.data.Ignore)
    object SourceProfileInput {
      val point: monocle.Lens[SourceProfileInput, clue.data.Input[SpectralDefinitionIntegratedInput]] = monocle.macros.GenLens[SourceProfileInput](_.point)
      val uniform: monocle.Lens[SourceProfileInput, clue.data.Input[SpectralDefinitionSurfaceInput]] = monocle.macros.GenLens[SourceProfileInput](_.uniform)
      val gaussian: monocle.Lens[SourceProfileInput, clue.data.Input[GaussianInput]] = monocle.macros.GenLens[SourceProfileInput](_.gaussian)
      implicit val eqSourceProfileInput: cats.Eq[SourceProfileInput] = cats.Eq.fromUniversalEquals
      implicit val showSourceProfileInput: cats.Show[SourceProfileInput] = cats.Show.fromToString
      implicit val jsonEncoderSourceProfileInput: io.circe.Encoder.AsObject[SourceProfileInput] = io.circe.generic.semiauto.deriveEncoder[SourceProfileInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class SpectralDefinitionIntegratedInput(val bandNormalized: clue.data.Input[BandNormalizedIntegratedInput] = clue.data.Ignore, val emissionLines: clue.data.Input[EmissionLinesIntegratedInput] = clue.data.Ignore)
    object SpectralDefinitionIntegratedInput {
      val bandNormalized: monocle.Lens[SpectralDefinitionIntegratedInput, clue.data.Input[BandNormalizedIntegratedInput]] = monocle.macros.GenLens[SpectralDefinitionIntegratedInput](_.bandNormalized)
      val emissionLines: monocle.Lens[SpectralDefinitionIntegratedInput, clue.data.Input[EmissionLinesIntegratedInput]] = monocle.macros.GenLens[SpectralDefinitionIntegratedInput](_.emissionLines)
      implicit val eqSpectralDefinitionIntegratedInput: cats.Eq[SpectralDefinitionIntegratedInput] = cats.Eq.fromUniversalEquals
      implicit val showSpectralDefinitionIntegratedInput: cats.Show[SpectralDefinitionIntegratedInput] = cats.Show.fromToString
      implicit val jsonEncoderSpectralDefinitionIntegratedInput: io.circe.Encoder.AsObject[SpectralDefinitionIntegratedInput] = io.circe.generic.semiauto.deriveEncoder[SpectralDefinitionIntegratedInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class SpectralDefinitionSurfaceInput(val bandNormalized: clue.data.Input[BandNormalizedSurfaceInput] = clue.data.Ignore, val emissionLines: clue.data.Input[EmissionLinesSurfaceInput] = clue.data.Ignore)
    object SpectralDefinitionSurfaceInput {
      val bandNormalized: monocle.Lens[SpectralDefinitionSurfaceInput, clue.data.Input[BandNormalizedSurfaceInput]] = monocle.macros.GenLens[SpectralDefinitionSurfaceInput](_.bandNormalized)
      val emissionLines: monocle.Lens[SpectralDefinitionSurfaceInput, clue.data.Input[EmissionLinesSurfaceInput]] = monocle.macros.GenLens[SpectralDefinitionSurfaceInput](_.emissionLines)
      implicit val eqSpectralDefinitionSurfaceInput: cats.Eq[SpectralDefinitionSurfaceInput] = cats.Eq.fromUniversalEquals
      implicit val showSpectralDefinitionSurfaceInput: cats.Show[SpectralDefinitionSurfaceInput] = cats.Show.fromToString
      implicit val jsonEncoderSpectralDefinitionSurfaceInput: io.circe.Encoder.AsObject[SpectralDefinitionSurfaceInput] = io.circe.generic.semiauto.deriveEncoder[SpectralDefinitionSurfaceInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class SpectroscopyScienceRequirementsInput(val wavelength: clue.data.Input[WavelengthInput] = clue.data.Ignore, val resolution: clue.data.Input[PosInt] = clue.data.Ignore, val signalToNoise: clue.data.Input[SignalToNoise] = clue.data.Ignore, val signalToNoiseAt: clue.data.Input[WavelengthInput] = clue.data.Ignore, val wavelengthCoverage: clue.data.Input[WavelengthInput] = clue.data.Ignore, val focalPlane: clue.data.Input[FocalPlane] = clue.data.Ignore, val focalPlaneAngle: clue.data.Input[AngleInput] = clue.data.Ignore, val capability: clue.data.Input[SpectroscopyCapabilities] = clue.data.Ignore)
    object SpectroscopyScienceRequirementsInput {
      val wavelength: monocle.Lens[SpectroscopyScienceRequirementsInput, clue.data.Input[WavelengthInput]] = monocle.macros.GenLens[SpectroscopyScienceRequirementsInput](_.wavelength)
      val resolution: monocle.Lens[SpectroscopyScienceRequirementsInput, clue.data.Input[PosInt]] = monocle.macros.GenLens[SpectroscopyScienceRequirementsInput](_.resolution)
      val signalToNoise: monocle.Lens[SpectroscopyScienceRequirementsInput, clue.data.Input[SignalToNoise]] = monocle.macros.GenLens[SpectroscopyScienceRequirementsInput](_.signalToNoise)
      val signalToNoiseAt: monocle.Lens[SpectroscopyScienceRequirementsInput, clue.data.Input[WavelengthInput]] = monocle.macros.GenLens[SpectroscopyScienceRequirementsInput](_.signalToNoiseAt)
      val wavelengthCoverage: monocle.Lens[SpectroscopyScienceRequirementsInput, clue.data.Input[WavelengthInput]] = monocle.macros.GenLens[SpectroscopyScienceRequirementsInput](_.wavelengthCoverage)
      val focalPlane: monocle.Lens[SpectroscopyScienceRequirementsInput, clue.data.Input[FocalPlane]] = monocle.macros.GenLens[SpectroscopyScienceRequirementsInput](_.focalPlane)
      val focalPlaneAngle: monocle.Lens[SpectroscopyScienceRequirementsInput, clue.data.Input[AngleInput]] = monocle.macros.GenLens[SpectroscopyScienceRequirementsInput](_.focalPlaneAngle)
      val capability: monocle.Lens[SpectroscopyScienceRequirementsInput, clue.data.Input[SpectroscopyCapabilities]] = monocle.macros.GenLens[SpectroscopyScienceRequirementsInput](_.capability)
      implicit val eqSpectroscopyScienceRequirementsInput: cats.Eq[SpectroscopyScienceRequirementsInput] = cats.Eq.fromUniversalEquals
      implicit val showSpectroscopyScienceRequirementsInput: cats.Show[SpectroscopyScienceRequirementsInput] = cats.Show.fromToString
      implicit val jsonEncoderSpectroscopyScienceRequirementsInput: io.circe.Encoder.AsObject[SpectroscopyScienceRequirementsInput] = io.circe.generic.semiauto.deriveEncoder[SpectroscopyScienceRequirementsInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class StepConfigInput(val bias: clue.data.Input[Boolean] = clue.data.Ignore, val dark: clue.data.Input[Boolean] = clue.data.Ignore, val gcal: clue.data.Input[StepConfigGcalInput] = clue.data.Ignore, val science: clue.data.Input[StepConfigScienceInput] = clue.data.Ignore, val smartGcal: clue.data.Input[StepConfigSmartGcalInput] = clue.data.Ignore)
    object StepConfigInput {
      val bias: monocle.Lens[StepConfigInput, clue.data.Input[Boolean]] = monocle.macros.GenLens[StepConfigInput](_.bias)
      val dark: monocle.Lens[StepConfigInput, clue.data.Input[Boolean]] = monocle.macros.GenLens[StepConfigInput](_.dark)
      val gcal: monocle.Lens[StepConfigInput, clue.data.Input[StepConfigGcalInput]] = monocle.macros.GenLens[StepConfigInput](_.gcal)
      val science: monocle.Lens[StepConfigInput, clue.data.Input[StepConfigScienceInput]] = monocle.macros.GenLens[StepConfigInput](_.science)
      val smartGcal: monocle.Lens[StepConfigInput, clue.data.Input[StepConfigSmartGcalInput]] = monocle.macros.GenLens[StepConfigInput](_.smartGcal)
      implicit val eqStepConfigInput: cats.Eq[StepConfigInput] = cats.Eq.fromUniversalEquals
      implicit val showStepConfigInput: cats.Show[StepConfigInput] = cats.Show.fromToString
      implicit val jsonEncoderStepConfigInput: io.circe.Encoder.AsObject[StepConfigInput] = io.circe.generic.semiauto.deriveEncoder[StepConfigInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class StepConfigGcalInput(val arcs: clue.data.Input[List[GcalArc]] = clue.data.Ignore, val continuum: clue.data.Input[GcalContinuum] = clue.data.Ignore, val diffuser: GcalDiffuser, val filter: GcalFilter, val shutter: GcalShutter)
    object StepConfigGcalInput {
      val arcs: monocle.Lens[StepConfigGcalInput, clue.data.Input[List[GcalArc]]] = monocle.macros.GenLens[StepConfigGcalInput](_.arcs)
      val continuum: monocle.Lens[StepConfigGcalInput, clue.data.Input[GcalContinuum]] = monocle.macros.GenLens[StepConfigGcalInput](_.continuum)
      val diffuser: monocle.Lens[StepConfigGcalInput, GcalDiffuser] = monocle.macros.GenLens[StepConfigGcalInput](_.diffuser)
      val filter: monocle.Lens[StepConfigGcalInput, GcalFilter] = monocle.macros.GenLens[StepConfigGcalInput](_.filter)
      val shutter: monocle.Lens[StepConfigGcalInput, GcalShutter] = monocle.macros.GenLens[StepConfigGcalInput](_.shutter)
      implicit val eqStepConfigGcalInput: cats.Eq[StepConfigGcalInput] = cats.Eq.fromUniversalEquals
      implicit val showStepConfigGcalInput: cats.Show[StepConfigGcalInput] = cats.Show.fromToString
      implicit val jsonEncoderStepConfigGcalInput: io.circe.Encoder.AsObject[StepConfigGcalInput] = io.circe.generic.semiauto.deriveEncoder[StepConfigGcalInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class StepConfigScienceInput(val offset: OffsetInput, val guiding: clue.data.Input[GuideState] = clue.data.Ignore)
    object StepConfigScienceInput {
      val offset: monocle.Lens[StepConfigScienceInput, OffsetInput] = monocle.macros.GenLens[StepConfigScienceInput](_.offset)
      val guiding: monocle.Lens[StepConfigScienceInput, clue.data.Input[GuideState]] = monocle.macros.GenLens[StepConfigScienceInput](_.guiding)
      implicit val eqStepConfigScienceInput: cats.Eq[StepConfigScienceInput] = cats.Eq.fromUniversalEquals
      implicit val showStepConfigScienceInput: cats.Show[StepConfigScienceInput] = cats.Show.fromToString
      implicit val jsonEncoderStepConfigScienceInput: io.circe.Encoder.AsObject[StepConfigScienceInput] = io.circe.generic.semiauto.deriveEncoder[StepConfigScienceInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class StepConfigSmartGcalInput(val smartGcalType: SmartGcalType)
    object StepConfigSmartGcalInput {
      val smartGcalType: monocle.Lens[StepConfigSmartGcalInput, SmartGcalType] = monocle.macros.GenLens[StepConfigSmartGcalInput](_.smartGcalType)
      implicit val eqStepConfigSmartGcalInput: cats.Eq[StepConfigSmartGcalInput] = cats.Eq.fromUniversalEquals
      implicit val showStepConfigSmartGcalInput: cats.Show[StepConfigSmartGcalInput] = cats.Show.fromToString
      implicit val jsonEncoderStepConfigSmartGcalInput: io.circe.Encoder.AsObject[StepConfigSmartGcalInput] = io.circe.generic.semiauto.deriveEncoder[StepConfigSmartGcalInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class TargetEditInput(val targetId: clue.data.Input[TargetId] = clue.data.Ignore, val programId: clue.data.Input[ProgramId] = clue.data.Ignore)
    object TargetEditInput {
      val targetId: monocle.Lens[TargetEditInput, clue.data.Input[TargetId]] = monocle.macros.GenLens[TargetEditInput](_.targetId)
      val programId: monocle.Lens[TargetEditInput, clue.data.Input[ProgramId]] = monocle.macros.GenLens[TargetEditInput](_.programId)
      implicit val eqTargetEditInput: cats.Eq[TargetEditInput] = cats.Eq.fromUniversalEquals
      implicit val showTargetEditInput: cats.Show[TargetEditInput] = cats.Show.fromToString
      implicit val jsonEncoderTargetEditInput: io.circe.Encoder.AsObject[TargetEditInput] = io.circe.generic.semiauto.deriveEncoder[TargetEditInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class GroupEditInput(val groupId: clue.data.Input[GroupId] = clue.data.Ignore, val programId: clue.data.Input[ProgramId] = clue.data.Ignore)
    object GroupEditInput {
      val groupId: monocle.Lens[GroupEditInput, clue.data.Input[GroupId]] = monocle.macros.GenLens[GroupEditInput](_.groupId)
      val programId: monocle.Lens[GroupEditInput, clue.data.Input[ProgramId]] = monocle.macros.GenLens[GroupEditInput](_.programId)
      implicit val eqGroupEditInput: cats.Eq[GroupEditInput] = cats.Eq.fromUniversalEquals
      implicit val showGroupEditInput: cats.Show[GroupEditInput] = cats.Show.fromToString
      implicit val jsonEncoderGroupEditInput: io.circe.Encoder.AsObject[GroupEditInput] = io.circe.generic.semiauto.deriveEncoder[GroupEditInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class ObservationEditInput(val observationId: clue.data.Input[ObservationId] = clue.data.Ignore, val programId: clue.data.Input[ProgramId] = clue.data.Ignore)
    object ObservationEditInput {
      val observationId: monocle.Lens[ObservationEditInput, clue.data.Input[ObservationId]] = monocle.macros.GenLens[ObservationEditInput](_.observationId)
      val programId: monocle.Lens[ObservationEditInput, clue.data.Input[ProgramId]] = monocle.macros.GenLens[ObservationEditInput](_.programId)
      implicit val eqObservationEditInput: cats.Eq[ObservationEditInput] = cats.Eq.fromUniversalEquals
      implicit val showObservationEditInput: cats.Show[ObservationEditInput] = cats.Show.fromToString
      implicit val jsonEncoderObservationEditInput: io.circe.Encoder.AsObject[ObservationEditInput] = io.circe.generic.semiauto.deriveEncoder[ObservationEditInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class ProgramEditInput(val programId: clue.data.Input[ProgramId] = clue.data.Ignore)
    object ProgramEditInput {
      val programId: monocle.Lens[ProgramEditInput, clue.data.Input[ProgramId]] = monocle.macros.GenLens[ProgramEditInput](_.programId)
      implicit val eqProgramEditInput: cats.Eq[ProgramEditInput] = cats.Eq.fromUniversalEquals
      implicit val showProgramEditInput: cats.Show[ProgramEditInput] = cats.Show.fromToString
      implicit val jsonEncoderProgramEditInput: io.circe.Encoder.AsObject[ProgramEditInput] = io.circe.generic.semiauto.deriveEncoder[ProgramEditInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class SystemVerificationInput(val minPercentTime: clue.data.Input[IntPercent] = clue.data.Ignore)
    object SystemVerificationInput {
      val minPercentTime: monocle.Lens[SystemVerificationInput, clue.data.Input[IntPercent]] = monocle.macros.GenLens[SystemVerificationInput](_.minPercentTime)
      implicit val eqSystemVerificationInput: cats.Eq[SystemVerificationInput] = cats.Eq.fromUniversalEquals
      implicit val showSystemVerificationInput: cats.Show[SystemVerificationInput] = cats.Show.fromToString
      implicit val jsonEncoderSystemVerificationInput: io.circe.Encoder.AsObject[SystemVerificationInput] = io.circe.generic.semiauto.deriveEncoder[SystemVerificationInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class TargetEnvironmentInput(val explicitBase: clue.data.Input[CoordinatesInput] = clue.data.Ignore, val asterism: clue.data.Input[List[TargetId]] = clue.data.Ignore)
    object TargetEnvironmentInput {
      val explicitBase: monocle.Lens[TargetEnvironmentInput, clue.data.Input[CoordinatesInput]] = monocle.macros.GenLens[TargetEnvironmentInput](_.explicitBase)
      val asterism: monocle.Lens[TargetEnvironmentInput, clue.data.Input[List[TargetId]]] = monocle.macros.GenLens[TargetEnvironmentInput](_.asterism)
      implicit val eqTargetEnvironmentInput: cats.Eq[TargetEnvironmentInput] = cats.Eq.fromUniversalEquals
      implicit val showTargetEnvironmentInput: cats.Show[TargetEnvironmentInput] = cats.Show.fromToString
      implicit val jsonEncoderTargetEnvironmentInput: io.circe.Encoder.AsObject[TargetEnvironmentInput] = io.circe.generic.semiauto.deriveEncoder[TargetEnvironmentInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class TargetPropertiesInput(val name: clue.data.Input[NonEmptyString] = clue.data.Ignore, val sidereal: clue.data.Input[SiderealInput] = clue.data.Ignore, val nonsidereal: clue.data.Input[NonsiderealInput] = clue.data.Ignore, val sourceProfile: clue.data.Input[SourceProfileInput] = clue.data.Ignore, val existence: clue.data.Input[Existence] = clue.data.Ignore)
    object TargetPropertiesInput {
      val name: monocle.Lens[TargetPropertiesInput, clue.data.Input[NonEmptyString]] = monocle.macros.GenLens[TargetPropertiesInput](_.name)
      val sidereal: monocle.Lens[TargetPropertiesInput, clue.data.Input[SiderealInput]] = monocle.macros.GenLens[TargetPropertiesInput](_.sidereal)
      val nonsidereal: monocle.Lens[TargetPropertiesInput, clue.data.Input[NonsiderealInput]] = monocle.macros.GenLens[TargetPropertiesInput](_.nonsidereal)
      val sourceProfile: monocle.Lens[TargetPropertiesInput, clue.data.Input[SourceProfileInput]] = monocle.macros.GenLens[TargetPropertiesInput](_.sourceProfile)
      val existence: monocle.Lens[TargetPropertiesInput, clue.data.Input[Existence]] = monocle.macros.GenLens[TargetPropertiesInput](_.existence)
      implicit val eqTargetPropertiesInput: cats.Eq[TargetPropertiesInput] = cats.Eq.fromUniversalEquals
      implicit val showTargetPropertiesInput: cats.Show[TargetPropertiesInput] = cats.Show.fromToString
      implicit val jsonEncoderTargetPropertiesInput: io.circe.Encoder.AsObject[TargetPropertiesInput] = io.circe.generic.semiauto.deriveEncoder[TargetPropertiesInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class TimingWindowRepeatInput(val period: TimeSpanInput, val times: clue.data.Input[PosInt] = clue.data.Ignore)
    object TimingWindowRepeatInput {
      val period: monocle.Lens[TimingWindowRepeatInput, TimeSpanInput] = monocle.macros.GenLens[TimingWindowRepeatInput](_.period)
      val times: monocle.Lens[TimingWindowRepeatInput, clue.data.Input[PosInt]] = monocle.macros.GenLens[TimingWindowRepeatInput](_.times)
      implicit val eqTimingWindowRepeatInput: cats.Eq[TimingWindowRepeatInput] = cats.Eq.fromUniversalEquals
      implicit val showTimingWindowRepeatInput: cats.Show[TimingWindowRepeatInput] = cats.Show.fromToString
      implicit val jsonEncoderTimingWindowRepeatInput: io.circe.Encoder.AsObject[TimingWindowRepeatInput] = io.circe.generic.semiauto.deriveEncoder[TimingWindowRepeatInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class TimingWindowEndInput(val atUtc: clue.data.Input[Timestamp] = clue.data.Ignore, val after: clue.data.Input[TimeSpanInput] = clue.data.Ignore, val repeat: clue.data.Input[TimingWindowRepeatInput] = clue.data.Ignore)
    object TimingWindowEndInput {
      val atUtc: monocle.Lens[TimingWindowEndInput, clue.data.Input[Timestamp]] = monocle.macros.GenLens[TimingWindowEndInput](_.atUtc)
      val after: monocle.Lens[TimingWindowEndInput, clue.data.Input[TimeSpanInput]] = monocle.macros.GenLens[TimingWindowEndInput](_.after)
      val repeat: monocle.Lens[TimingWindowEndInput, clue.data.Input[TimingWindowRepeatInput]] = monocle.macros.GenLens[TimingWindowEndInput](_.repeat)
      implicit val eqTimingWindowEndInput: cats.Eq[TimingWindowEndInput] = cats.Eq.fromUniversalEquals
      implicit val showTimingWindowEndInput: cats.Show[TimingWindowEndInput] = cats.Show.fromToString
      implicit val jsonEncoderTimingWindowEndInput: io.circe.Encoder.AsObject[TimingWindowEndInput] = io.circe.generic.semiauto.deriveEncoder[TimingWindowEndInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class TimingWindowInput(val inclusion: TimingWindowInclusion, val startUtc: Timestamp, val end: clue.data.Input[TimingWindowEndInput] = clue.data.Ignore)
    object TimingWindowInput {
      val inclusion: monocle.Lens[TimingWindowInput, TimingWindowInclusion] = monocle.macros.GenLens[TimingWindowInput](_.inclusion)
      val startUtc: monocle.Lens[TimingWindowInput, Timestamp] = monocle.macros.GenLens[TimingWindowInput](_.startUtc)
      val end: monocle.Lens[TimingWindowInput, clue.data.Input[TimingWindowEndInput]] = monocle.macros.GenLens[TimingWindowInput](_.end)
      implicit val eqTimingWindowInput: cats.Eq[TimingWindowInput] = cats.Eq.fromUniversalEquals
      implicit val showTimingWindowInput: cats.Show[TimingWindowInput] = cats.Show.fromToString
      implicit val jsonEncoderTimingWindowInput: io.circe.Encoder.AsObject[TimingWindowInput] = io.circe.generic.semiauto.deriveEncoder[TimingWindowInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class UnnormalizedSedInput(val stellarLibrary: clue.data.Input[StellarLibrarySpectrum] = clue.data.Ignore, val coolStar: clue.data.Input[CoolStarTemperature] = clue.data.Ignore, val galaxy: clue.data.Input[GalaxySpectrum] = clue.data.Ignore, val planet: clue.data.Input[PlanetSpectrum] = clue.data.Ignore, val quasar: clue.data.Input[QuasarSpectrum] = clue.data.Ignore, val hiiRegion: clue.data.Input[HiiRegionSpectrum] = clue.data.Ignore, val planetaryNebula: clue.data.Input[PlanetaryNebulaSpectrum] = clue.data.Ignore, val powerLaw: clue.data.Input[BigDecimal] = clue.data.Ignore, val blackBodyTempK: clue.data.Input[PosInt] = clue.data.Ignore, val fluxDensities: clue.data.Input[List[FluxDensity]] = clue.data.Ignore)
    object UnnormalizedSedInput {
      val stellarLibrary: monocle.Lens[UnnormalizedSedInput, clue.data.Input[StellarLibrarySpectrum]] = monocle.macros.GenLens[UnnormalizedSedInput](_.stellarLibrary)
      val coolStar: monocle.Lens[UnnormalizedSedInput, clue.data.Input[CoolStarTemperature]] = monocle.macros.GenLens[UnnormalizedSedInput](_.coolStar)
      val galaxy: monocle.Lens[UnnormalizedSedInput, clue.data.Input[GalaxySpectrum]] = monocle.macros.GenLens[UnnormalizedSedInput](_.galaxy)
      val planet: monocle.Lens[UnnormalizedSedInput, clue.data.Input[PlanetSpectrum]] = monocle.macros.GenLens[UnnormalizedSedInput](_.planet)
      val quasar: monocle.Lens[UnnormalizedSedInput, clue.data.Input[QuasarSpectrum]] = monocle.macros.GenLens[UnnormalizedSedInput](_.quasar)
      val hiiRegion: monocle.Lens[UnnormalizedSedInput, clue.data.Input[HiiRegionSpectrum]] = monocle.macros.GenLens[UnnormalizedSedInput](_.hiiRegion)
      val planetaryNebula: monocle.Lens[UnnormalizedSedInput, clue.data.Input[PlanetaryNebulaSpectrum]] = monocle.macros.GenLens[UnnormalizedSedInput](_.planetaryNebula)
      val powerLaw: monocle.Lens[UnnormalizedSedInput, clue.data.Input[BigDecimal]] = monocle.macros.GenLens[UnnormalizedSedInput](_.powerLaw)
      val blackBodyTempK: monocle.Lens[UnnormalizedSedInput, clue.data.Input[PosInt]] = monocle.macros.GenLens[UnnormalizedSedInput](_.blackBodyTempK)
      val fluxDensities: monocle.Lens[UnnormalizedSedInput, clue.data.Input[List[FluxDensity]]] = monocle.macros.GenLens[UnnormalizedSedInput](_.fluxDensities)
      implicit val eqUnnormalizedSedInput: cats.Eq[UnnormalizedSedInput] = cats.Eq.fromUniversalEquals
      implicit val showUnnormalizedSedInput: cats.Show[UnnormalizedSedInput] = cats.Show.fromToString
      implicit val jsonEncoderUnnormalizedSedInput: io.circe.Encoder.AsObject[UnnormalizedSedInput] = io.circe.generic.semiauto.deriveEncoder[UnnormalizedSedInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class UpdateAsterismsInput(val SET: EditAsterismsPatchInput, val WHERE: clue.data.Input[WhereObservation] = clue.data.Ignore, val LIMIT: clue.data.Input[NonNegInt] = clue.data.Ignore, val includeDeleted: clue.data.Input[Boolean] = clue.data.Ignore)
    object UpdateAsterismsInput {
      val SET: monocle.Lens[UpdateAsterismsInput, EditAsterismsPatchInput] = monocle.macros.GenLens[UpdateAsterismsInput](_.SET)
      val WHERE: monocle.Lens[UpdateAsterismsInput, clue.data.Input[WhereObservation]] = monocle.macros.GenLens[UpdateAsterismsInput](_.WHERE)
      val LIMIT: monocle.Lens[UpdateAsterismsInput, clue.data.Input[NonNegInt]] = monocle.macros.GenLens[UpdateAsterismsInput](_.LIMIT)
      val includeDeleted: monocle.Lens[UpdateAsterismsInput, clue.data.Input[Boolean]] = monocle.macros.GenLens[UpdateAsterismsInput](_.includeDeleted)
      implicit val eqUpdateAsterismsInput: cats.Eq[UpdateAsterismsInput] = cats.Eq.fromUniversalEquals
      implicit val showUpdateAsterismsInput: cats.Show[UpdateAsterismsInput] = cats.Show.fromToString
      implicit val jsonEncoderUpdateAsterismsInput: io.circe.Encoder.AsObject[UpdateAsterismsInput] = io.circe.generic.semiauto.deriveEncoder[UpdateAsterismsInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class UpdateObsAttachmentsInput(val SET: ObsAttachmentPropertiesInput, val WHERE: clue.data.Input[WhereObsAttachment] = clue.data.Ignore, val LIMIT: clue.data.Input[NonNegInt] = clue.data.Ignore)
    object UpdateObsAttachmentsInput {
      val SET: monocle.Lens[UpdateObsAttachmentsInput, ObsAttachmentPropertiesInput] = monocle.macros.GenLens[UpdateObsAttachmentsInput](_.SET)
      val WHERE: monocle.Lens[UpdateObsAttachmentsInput, clue.data.Input[WhereObsAttachment]] = monocle.macros.GenLens[UpdateObsAttachmentsInput](_.WHERE)
      val LIMIT: monocle.Lens[UpdateObsAttachmentsInput, clue.data.Input[NonNegInt]] = monocle.macros.GenLens[UpdateObsAttachmentsInput](_.LIMIT)
      implicit val eqUpdateObsAttachmentsInput: cats.Eq[UpdateObsAttachmentsInput] = cats.Eq.fromUniversalEquals
      implicit val showUpdateObsAttachmentsInput: cats.Show[UpdateObsAttachmentsInput] = cats.Show.fromToString
      implicit val jsonEncoderUpdateObsAttachmentsInput: io.circe.Encoder.AsObject[UpdateObsAttachmentsInput] = io.circe.generic.semiauto.deriveEncoder[UpdateObsAttachmentsInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class UpdateDatasetsInput(val SET: DatasetPropertiesInput, val WHERE: clue.data.Input[WhereDataset] = clue.data.Ignore, val LIMIT: clue.data.Input[NonNegInt] = clue.data.Ignore)
    object UpdateDatasetsInput {
      val SET: monocle.Lens[UpdateDatasetsInput, DatasetPropertiesInput] = monocle.macros.GenLens[UpdateDatasetsInput](_.SET)
      val WHERE: monocle.Lens[UpdateDatasetsInput, clue.data.Input[WhereDataset]] = monocle.macros.GenLens[UpdateDatasetsInput](_.WHERE)
      val LIMIT: monocle.Lens[UpdateDatasetsInput, clue.data.Input[NonNegInt]] = monocle.macros.GenLens[UpdateDatasetsInput](_.LIMIT)
      implicit val eqUpdateDatasetsInput: cats.Eq[UpdateDatasetsInput] = cats.Eq.fromUniversalEquals
      implicit val showUpdateDatasetsInput: cats.Show[UpdateDatasetsInput] = cats.Show.fromToString
      implicit val jsonEncoderUpdateDatasetsInput: io.circe.Encoder.AsObject[UpdateDatasetsInput] = io.circe.generic.semiauto.deriveEncoder[UpdateDatasetsInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class UpdateGroupsInput(val SET: GroupPropertiesInput, val WHERE: clue.data.Input[WhereGroup] = clue.data.Ignore, val LIMIT: clue.data.Input[NonNegInt] = clue.data.Ignore)
    object UpdateGroupsInput {
      val SET: monocle.Lens[UpdateGroupsInput, GroupPropertiesInput] = monocle.macros.GenLens[UpdateGroupsInput](_.SET)
      val WHERE: monocle.Lens[UpdateGroupsInput, clue.data.Input[WhereGroup]] = monocle.macros.GenLens[UpdateGroupsInput](_.WHERE)
      val LIMIT: monocle.Lens[UpdateGroupsInput, clue.data.Input[NonNegInt]] = monocle.macros.GenLens[UpdateGroupsInput](_.LIMIT)
      implicit val eqUpdateGroupsInput: cats.Eq[UpdateGroupsInput] = cats.Eq.fromUniversalEquals
      implicit val showUpdateGroupsInput: cats.Show[UpdateGroupsInput] = cats.Show.fromToString
      implicit val jsonEncoderUpdateGroupsInput: io.circe.Encoder.AsObject[UpdateGroupsInput] = io.circe.generic.semiauto.deriveEncoder[UpdateGroupsInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class UpdateObservationsInput(val SET: ObservationPropertiesInput, val WHERE: clue.data.Input[WhereObservation] = clue.data.Ignore, val LIMIT: clue.data.Input[NonNegInt] = clue.data.Ignore, val includeDeleted: clue.data.Input[Boolean] = clue.data.Ignore)
    object UpdateObservationsInput {
      val SET: monocle.Lens[UpdateObservationsInput, ObservationPropertiesInput] = monocle.macros.GenLens[UpdateObservationsInput](_.SET)
      val WHERE: monocle.Lens[UpdateObservationsInput, clue.data.Input[WhereObservation]] = monocle.macros.GenLens[UpdateObservationsInput](_.WHERE)
      val LIMIT: monocle.Lens[UpdateObservationsInput, clue.data.Input[NonNegInt]] = monocle.macros.GenLens[UpdateObservationsInput](_.LIMIT)
      val includeDeleted: monocle.Lens[UpdateObservationsInput, clue.data.Input[Boolean]] = monocle.macros.GenLens[UpdateObservationsInput](_.includeDeleted)
      implicit val eqUpdateObservationsInput: cats.Eq[UpdateObservationsInput] = cats.Eq.fromUniversalEquals
      implicit val showUpdateObservationsInput: cats.Show[UpdateObservationsInput] = cats.Show.fromToString
      implicit val jsonEncoderUpdateObservationsInput: io.circe.Encoder.AsObject[UpdateObservationsInput] = io.circe.generic.semiauto.deriveEncoder[UpdateObservationsInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class UpdateProgramsInput(val SET: ProgramPropertiesInput, val WHERE: clue.data.Input[WhereProgram] = clue.data.Ignore, val LIMIT: clue.data.Input[NonNegInt] = clue.data.Ignore, val includeDeleted: clue.data.Input[Boolean] = clue.data.Ignore)
    object UpdateProgramsInput {
      val SET: monocle.Lens[UpdateProgramsInput, ProgramPropertiesInput] = monocle.macros.GenLens[UpdateProgramsInput](_.SET)
      val WHERE: monocle.Lens[UpdateProgramsInput, clue.data.Input[WhereProgram]] = monocle.macros.GenLens[UpdateProgramsInput](_.WHERE)
      val LIMIT: monocle.Lens[UpdateProgramsInput, clue.data.Input[NonNegInt]] = monocle.macros.GenLens[UpdateProgramsInput](_.LIMIT)
      val includeDeleted: monocle.Lens[UpdateProgramsInput, clue.data.Input[Boolean]] = monocle.macros.GenLens[UpdateProgramsInput](_.includeDeleted)
      implicit val eqUpdateProgramsInput: cats.Eq[UpdateProgramsInput] = cats.Eq.fromUniversalEquals
      implicit val showUpdateProgramsInput: cats.Show[UpdateProgramsInput] = cats.Show.fromToString
      implicit val jsonEncoderUpdateProgramsInput: io.circe.Encoder.AsObject[UpdateProgramsInput] = io.circe.generic.semiauto.deriveEncoder[UpdateProgramsInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class UpdateTargetsInput(val SET: TargetPropertiesInput, val WHERE: clue.data.Input[WhereTarget] = clue.data.Ignore, val LIMIT: clue.data.Input[NonNegInt] = clue.data.Ignore, val includeDeleted: clue.data.Input[Boolean] = clue.data.Ignore)
    object UpdateTargetsInput {
      val SET: monocle.Lens[UpdateTargetsInput, TargetPropertiesInput] = monocle.macros.GenLens[UpdateTargetsInput](_.SET)
      val WHERE: monocle.Lens[UpdateTargetsInput, clue.data.Input[WhereTarget]] = monocle.macros.GenLens[UpdateTargetsInput](_.WHERE)
      val LIMIT: monocle.Lens[UpdateTargetsInput, clue.data.Input[NonNegInt]] = monocle.macros.GenLens[UpdateTargetsInput](_.LIMIT)
      val includeDeleted: monocle.Lens[UpdateTargetsInput, clue.data.Input[Boolean]] = monocle.macros.GenLens[UpdateTargetsInput](_.includeDeleted)
      implicit val eqUpdateTargetsInput: cats.Eq[UpdateTargetsInput] = cats.Eq.fromUniversalEquals
      implicit val showUpdateTargetsInput: cats.Show[UpdateTargetsInput] = cats.Show.fromToString
      implicit val jsonEncoderUpdateTargetsInput: io.circe.Encoder.AsObject[UpdateTargetsInput] = io.circe.generic.semiauto.deriveEncoder[UpdateTargetsInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class WavelengthInput(val picometers: clue.data.Input[PosInt] = clue.data.Ignore, val angstroms: clue.data.Input[PosBigDecimal] = clue.data.Ignore, val nanometers: clue.data.Input[PosBigDecimal] = clue.data.Ignore, val micrometers: clue.data.Input[PosBigDecimal] = clue.data.Ignore)
    object WavelengthInput {
      val picometers: monocle.Lens[WavelengthInput, clue.data.Input[PosInt]] = monocle.macros.GenLens[WavelengthInput](_.picometers)
      val angstroms: monocle.Lens[WavelengthInput, clue.data.Input[PosBigDecimal]] = monocle.macros.GenLens[WavelengthInput](_.angstroms)
      val nanometers: monocle.Lens[WavelengthInput, clue.data.Input[PosBigDecimal]] = monocle.macros.GenLens[WavelengthInput](_.nanometers)
      val micrometers: monocle.Lens[WavelengthInput, clue.data.Input[PosBigDecimal]] = monocle.macros.GenLens[WavelengthInput](_.micrometers)
      implicit val eqWavelengthInput: cats.Eq[WavelengthInput] = cats.Eq.fromUniversalEquals
      implicit val showWavelengthInput: cats.Show[WavelengthInput] = cats.Show.fromToString
      implicit val jsonEncoderWavelengthInput: io.circe.Encoder.AsObject[WavelengthInput] = io.circe.generic.semiauto.deriveEncoder[WavelengthInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class WavelengthDitherInput(val picometers: clue.data.Input[Int] = clue.data.Ignore, val angstroms: clue.data.Input[BigDecimal] = clue.data.Ignore, val nanometers: clue.data.Input[BigDecimal] = clue.data.Ignore, val micrometers: clue.data.Input[BigDecimal] = clue.data.Ignore)
    object WavelengthDitherInput {
      val picometers: monocle.Lens[WavelengthDitherInput, clue.data.Input[Int]] = monocle.macros.GenLens[WavelengthDitherInput](_.picometers)
      val angstroms: monocle.Lens[WavelengthDitherInput, clue.data.Input[BigDecimal]] = monocle.macros.GenLens[WavelengthDitherInput](_.angstroms)
      val nanometers: monocle.Lens[WavelengthDitherInput, clue.data.Input[BigDecimal]] = monocle.macros.GenLens[WavelengthDitherInput](_.nanometers)
      val micrometers: monocle.Lens[WavelengthDitherInput, clue.data.Input[BigDecimal]] = monocle.macros.GenLens[WavelengthDitherInput](_.micrometers)
      implicit val eqWavelengthDitherInput: cats.Eq[WavelengthDitherInput] = cats.Eq.fromUniversalEquals
      implicit val showWavelengthDitherInput: cats.Show[WavelengthDitherInput] = cats.Show.fromToString
      implicit val jsonEncoderWavelengthDitherInput: io.circe.Encoder.AsObject[WavelengthDitherInput] = io.circe.generic.semiauto.deriveEncoder[WavelengthDitherInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class ObsAttachmentPropertiesInput(val description: clue.data.Input[NonEmptyString] = clue.data.Ignore, val checked: clue.data.Input[Boolean] = clue.data.Ignore)
    object ObsAttachmentPropertiesInput {
      val description: monocle.Lens[ObsAttachmentPropertiesInput, clue.data.Input[NonEmptyString]] = monocle.macros.GenLens[ObsAttachmentPropertiesInput](_.description)
      val checked: monocle.Lens[ObsAttachmentPropertiesInput, clue.data.Input[Boolean]] = monocle.macros.GenLens[ObsAttachmentPropertiesInput](_.checked)
      implicit val eqObsAttachmentPropertiesInput: cats.Eq[ObsAttachmentPropertiesInput] = cats.Eq.fromUniversalEquals
      implicit val showObsAttachmentPropertiesInput: cats.Show[ObsAttachmentPropertiesInput] = cats.Show.fromToString
      implicit val jsonEncoderObsAttachmentPropertiesInput: io.circe.Encoder.AsObject[ObsAttachmentPropertiesInput] = io.circe.generic.semiauto.deriveEncoder[ObsAttachmentPropertiesInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class GroupPropertiesInput(val name: clue.data.Input[NonEmptyString] = clue.data.Ignore, val description: clue.data.Input[NonEmptyString] = clue.data.Ignore, val minimumRequired: clue.data.Input[NonNegShort] = clue.data.Ignore, val ordered: clue.data.Input[Boolean] = clue.data.Ignore, val minimumInterval: clue.data.Input[TimeSpanInput] = clue.data.Ignore, val maximumInterval: clue.data.Input[TimeSpanInput] = clue.data.Ignore, val parentGroup: clue.data.Input[GroupId] = clue.data.Ignore, val parentGroupIndex: clue.data.Input[NonNegShort] = clue.data.Ignore)
    object GroupPropertiesInput {
      val name: monocle.Lens[GroupPropertiesInput, clue.data.Input[NonEmptyString]] = monocle.macros.GenLens[GroupPropertiesInput](_.name)
      val description: monocle.Lens[GroupPropertiesInput, clue.data.Input[NonEmptyString]] = monocle.macros.GenLens[GroupPropertiesInput](_.description)
      val minimumRequired: monocle.Lens[GroupPropertiesInput, clue.data.Input[NonNegShort]] = monocle.macros.GenLens[GroupPropertiesInput](_.minimumRequired)
      val ordered: monocle.Lens[GroupPropertiesInput, clue.data.Input[Boolean]] = monocle.macros.GenLens[GroupPropertiesInput](_.ordered)
      val minimumInterval: monocle.Lens[GroupPropertiesInput, clue.data.Input[TimeSpanInput]] = monocle.macros.GenLens[GroupPropertiesInput](_.minimumInterval)
      val maximumInterval: monocle.Lens[GroupPropertiesInput, clue.data.Input[TimeSpanInput]] = monocle.macros.GenLens[GroupPropertiesInput](_.maximumInterval)
      val parentGroup: monocle.Lens[GroupPropertiesInput, clue.data.Input[GroupId]] = monocle.macros.GenLens[GroupPropertiesInput](_.parentGroup)
      val parentGroupIndex: monocle.Lens[GroupPropertiesInput, clue.data.Input[NonNegShort]] = monocle.macros.GenLens[GroupPropertiesInput](_.parentGroupIndex)
      implicit val eqGroupPropertiesInput: cats.Eq[GroupPropertiesInput] = cats.Eq.fromUniversalEquals
      implicit val showGroupPropertiesInput: cats.Show[GroupPropertiesInput] = cats.Show.fromToString
      implicit val jsonEncoderGroupPropertiesInput: io.circe.Encoder.AsObject[GroupPropertiesInput] = io.circe.generic.semiauto.deriveEncoder[GroupPropertiesInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class CreateGroupInput(val programId: clue.data.Input[ProgramId] = clue.data.Ignore, val proposalReference: clue.data.Input[ProposalReferenceLabel] = clue.data.Ignore, val programReference: clue.data.Input[ProgramReferenceLabel] = clue.data.Ignore, val SET: clue.data.Input[GroupPropertiesInput] = clue.data.Ignore)
    object CreateGroupInput {
      val programId: monocle.Lens[CreateGroupInput, clue.data.Input[ProgramId]] = monocle.macros.GenLens[CreateGroupInput](_.programId)
      val proposalReference: monocle.Lens[CreateGroupInput, clue.data.Input[ProposalReferenceLabel]] = monocle.macros.GenLens[CreateGroupInput](_.proposalReference)
      val programReference: monocle.Lens[CreateGroupInput, clue.data.Input[ProgramReferenceLabel]] = monocle.macros.GenLens[CreateGroupInput](_.programReference)
      val SET: monocle.Lens[CreateGroupInput, clue.data.Input[GroupPropertiesInput]] = monocle.macros.GenLens[CreateGroupInput](_.SET)
      implicit val eqCreateGroupInput: cats.Eq[CreateGroupInput] = cats.Eq.fromUniversalEquals
      implicit val showCreateGroupInput: cats.Show[CreateGroupInput] = cats.Show.fromToString
      implicit val jsonEncoderCreateGroupInput: io.circe.Encoder.AsObject[CreateGroupInput] = io.circe.generic.semiauto.deriveEncoder[CreateGroupInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class TimeChargeCorrectionInput(val chargeClass: ChargeClass, val op: TimeChargeCorrectionOp, val amount: TimeSpanInput, val comment: clue.data.Input[String] = clue.data.Ignore)
    object TimeChargeCorrectionInput {
      val chargeClass: monocle.Lens[TimeChargeCorrectionInput, ChargeClass] = monocle.macros.GenLens[TimeChargeCorrectionInput](_.chargeClass)
      val op: monocle.Lens[TimeChargeCorrectionInput, TimeChargeCorrectionOp] = monocle.macros.GenLens[TimeChargeCorrectionInput](_.op)
      val amount: monocle.Lens[TimeChargeCorrectionInput, TimeSpanInput] = monocle.macros.GenLens[TimeChargeCorrectionInput](_.amount)
      val comment: monocle.Lens[TimeChargeCorrectionInput, clue.data.Input[String]] = monocle.macros.GenLens[TimeChargeCorrectionInput](_.comment)
      implicit val eqTimeChargeCorrectionInput: cats.Eq[TimeChargeCorrectionInput] = cats.Eq.fromUniversalEquals
      implicit val showTimeChargeCorrectionInput: cats.Show[TimeChargeCorrectionInput] = cats.Show.fromToString
      implicit val jsonEncoderTimeChargeCorrectionInput: io.circe.Encoder.AsObject[TimeChargeCorrectionInput] = io.circe.generic.semiauto.deriveEncoder[TimeChargeCorrectionInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class TimeSpanInput(val microseconds: clue.data.Input[Long] = clue.data.Ignore, val milliseconds: clue.data.Input[BigDecimal] = clue.data.Ignore, val seconds: clue.data.Input[BigDecimal] = clue.data.Ignore, val minutes: clue.data.Input[BigDecimal] = clue.data.Ignore, val hours: clue.data.Input[BigDecimal] = clue.data.Ignore, val iso: clue.data.Input[String] = clue.data.Ignore)
    object TimeSpanInput {
      val microseconds: monocle.Lens[TimeSpanInput, clue.data.Input[Long]] = monocle.macros.GenLens[TimeSpanInput](_.microseconds)
      val milliseconds: monocle.Lens[TimeSpanInput, clue.data.Input[BigDecimal]] = monocle.macros.GenLens[TimeSpanInput](_.milliseconds)
      val seconds: monocle.Lens[TimeSpanInput, clue.data.Input[BigDecimal]] = monocle.macros.GenLens[TimeSpanInput](_.seconds)
      val minutes: monocle.Lens[TimeSpanInput, clue.data.Input[BigDecimal]] = monocle.macros.GenLens[TimeSpanInput](_.minutes)
      val hours: monocle.Lens[TimeSpanInput, clue.data.Input[BigDecimal]] = monocle.macros.GenLens[TimeSpanInput](_.hours)
      val iso: monocle.Lens[TimeSpanInput, clue.data.Input[String]] = monocle.macros.GenLens[TimeSpanInput](_.iso)
      implicit val eqTimeSpanInput: cats.Eq[TimeSpanInput] = cats.Eq.fromUniversalEquals
      implicit val showTimeSpanInput: cats.Show[TimeSpanInput] = cats.Show.fromToString
      implicit val jsonEncoderTimeSpanInput: io.circe.Encoder.AsObject[TimeSpanInput] = io.circe.generic.semiauto.deriveEncoder[TimeSpanInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class UnlinkUserInput(val programId: ProgramId, val userId: UserId)
    object UnlinkUserInput {
      val programId: monocle.Lens[UnlinkUserInput, ProgramId] = monocle.macros.GenLens[UnlinkUserInput](_.programId)
      val userId: monocle.Lens[UnlinkUserInput, UserId] = monocle.macros.GenLens[UnlinkUserInput](_.userId)
      implicit val eqUnlinkUserInput: cats.Eq[UnlinkUserInput] = cats.Eq.fromUniversalEquals
      implicit val showUnlinkUserInput: cats.Show[UnlinkUserInput] = cats.Show.fromToString
      implicit val jsonEncoderUnlinkUserInput: io.circe.Encoder.AsObject[UnlinkUserInput] = io.circe.generic.semiauto.deriveEncoder[UnlinkUserInput].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class WhereObsAttachment(val AND: clue.data.Input[List[WhereObsAttachment]] = clue.data.Ignore, val OR: clue.data.Input[List[WhereObsAttachment]] = clue.data.Ignore, val NOT: clue.data.Input[WhereObsAttachment] = clue.data.Ignore, val id: clue.data.Input[WhereOrderObsAttachmentId] = clue.data.Ignore, val fileName: clue.data.Input[WhereString] = clue.data.Ignore, val description: clue.data.Input[WhereOptionString] = clue.data.Ignore, val attachmentType: clue.data.Input[WhereObsAttachmentType] = clue.data.Ignore, val checked: clue.data.Input[Boolean] = clue.data.Ignore, val program: clue.data.Input[WhereProgram] = clue.data.Ignore)
    object WhereObsAttachment {
      val AND: monocle.Lens[WhereObsAttachment, clue.data.Input[List[WhereObsAttachment]]] = monocle.macros.GenLens[WhereObsAttachment](_.AND)
      val OR: monocle.Lens[WhereObsAttachment, clue.data.Input[List[WhereObsAttachment]]] = monocle.macros.GenLens[WhereObsAttachment](_.OR)
      val NOT: monocle.Lens[WhereObsAttachment, clue.data.Input[WhereObsAttachment]] = monocle.macros.GenLens[WhereObsAttachment](_.NOT)
      val id: monocle.Lens[WhereObsAttachment, clue.data.Input[WhereOrderObsAttachmentId]] = monocle.macros.GenLens[WhereObsAttachment](_.id)
      val fileName: monocle.Lens[WhereObsAttachment, clue.data.Input[WhereString]] = monocle.macros.GenLens[WhereObsAttachment](_.fileName)
      val description: monocle.Lens[WhereObsAttachment, clue.data.Input[WhereOptionString]] = monocle.macros.GenLens[WhereObsAttachment](_.description)
      val attachmentType: monocle.Lens[WhereObsAttachment, clue.data.Input[WhereObsAttachmentType]] = monocle.macros.GenLens[WhereObsAttachment](_.attachmentType)
      val checked: monocle.Lens[WhereObsAttachment, clue.data.Input[Boolean]] = monocle.macros.GenLens[WhereObsAttachment](_.checked)
      val program: monocle.Lens[WhereObsAttachment, clue.data.Input[WhereProgram]] = monocle.macros.GenLens[WhereObsAttachment](_.program)
      implicit val eqWhereObsAttachment: cats.Eq[WhereObsAttachment] = cats.Eq.fromUniversalEquals
      implicit val showWhereObsAttachment: cats.Show[WhereObsAttachment] = cats.Show.fromToString
      implicit val jsonEncoderWhereObsAttachment: io.circe.Encoder.AsObject[WhereObsAttachment] = io.circe.generic.semiauto.deriveEncoder[WhereObsAttachment].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class WhereObsAttachmentType(val EQ: clue.data.Input[ObsAttachmentType] = clue.data.Ignore, val NEQ: clue.data.Input[ObsAttachmentType] = clue.data.Ignore, val IN: clue.data.Input[List[ObsAttachmentType]] = clue.data.Ignore, val NIN: clue.data.Input[List[ObsAttachmentType]] = clue.data.Ignore)
    object WhereObsAttachmentType {
      val EQ: monocle.Lens[WhereObsAttachmentType, clue.data.Input[ObsAttachmentType]] = monocle.macros.GenLens[WhereObsAttachmentType](_.EQ)
      val NEQ: monocle.Lens[WhereObsAttachmentType, clue.data.Input[ObsAttachmentType]] = monocle.macros.GenLens[WhereObsAttachmentType](_.NEQ)
      val IN: monocle.Lens[WhereObsAttachmentType, clue.data.Input[List[ObsAttachmentType]]] = monocle.macros.GenLens[WhereObsAttachmentType](_.IN)
      val NIN: monocle.Lens[WhereObsAttachmentType, clue.data.Input[List[ObsAttachmentType]]] = monocle.macros.GenLens[WhereObsAttachmentType](_.NIN)
      implicit val eqWhereObsAttachmentType: cats.Eq[WhereObsAttachmentType] = cats.Eq.fromUniversalEquals
      implicit val showWhereObsAttachmentType: cats.Show[WhereObsAttachmentType] = cats.Show.fromToString
      implicit val jsonEncoderWhereObsAttachmentType: io.circe.Encoder.AsObject[WhereObsAttachmentType] = io.circe.generic.semiauto.deriveEncoder[WhereObsAttachmentType].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class WhereOrderObsAttachmentId(val EQ: clue.data.Input[ObsAttachmentId] = clue.data.Ignore, val NEQ: clue.data.Input[ObsAttachmentId] = clue.data.Ignore, val IN: clue.data.Input[List[ObsAttachmentId]] = clue.data.Ignore, val NIN: clue.data.Input[List[ObsAttachmentId]] = clue.data.Ignore, val GT: clue.data.Input[ObsAttachmentId] = clue.data.Ignore, val LT: clue.data.Input[ObsAttachmentId] = clue.data.Ignore, val GTE: clue.data.Input[ObsAttachmentId] = clue.data.Ignore, val LTE: clue.data.Input[ObsAttachmentId] = clue.data.Ignore)
    object WhereOrderObsAttachmentId {
      val EQ: monocle.Lens[WhereOrderObsAttachmentId, clue.data.Input[ObsAttachmentId]] = monocle.macros.GenLens[WhereOrderObsAttachmentId](_.EQ)
      val NEQ: monocle.Lens[WhereOrderObsAttachmentId, clue.data.Input[ObsAttachmentId]] = monocle.macros.GenLens[WhereOrderObsAttachmentId](_.NEQ)
      val IN: monocle.Lens[WhereOrderObsAttachmentId, clue.data.Input[List[ObsAttachmentId]]] = monocle.macros.GenLens[WhereOrderObsAttachmentId](_.IN)
      val NIN: monocle.Lens[WhereOrderObsAttachmentId, clue.data.Input[List[ObsAttachmentId]]] = monocle.macros.GenLens[WhereOrderObsAttachmentId](_.NIN)
      val GT: monocle.Lens[WhereOrderObsAttachmentId, clue.data.Input[ObsAttachmentId]] = monocle.macros.GenLens[WhereOrderObsAttachmentId](_.GT)
      val LT: monocle.Lens[WhereOrderObsAttachmentId, clue.data.Input[ObsAttachmentId]] = monocle.macros.GenLens[WhereOrderObsAttachmentId](_.LT)
      val GTE: monocle.Lens[WhereOrderObsAttachmentId, clue.data.Input[ObsAttachmentId]] = monocle.macros.GenLens[WhereOrderObsAttachmentId](_.GTE)
      val LTE: monocle.Lens[WhereOrderObsAttachmentId, clue.data.Input[ObsAttachmentId]] = monocle.macros.GenLens[WhereOrderObsAttachmentId](_.LTE)
      implicit val eqWhereOrderObsAttachmentId: cats.Eq[WhereOrderObsAttachmentId] = cats.Eq.fromUniversalEquals
      implicit val showWhereOrderObsAttachmentId: cats.Show[WhereOrderObsAttachmentId] = cats.Show.fromToString
      implicit val jsonEncoderWhereOrderObsAttachmentId: io.circe.Encoder.AsObject[WhereOrderObsAttachmentId] = io.circe.generic.semiauto.deriveEncoder[WhereOrderObsAttachmentId].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class WhereDataset(val AND: clue.data.Input[List[WhereDataset]] = clue.data.Ignore, val OR: clue.data.Input[List[WhereDataset]] = clue.data.Ignore, val NOT: clue.data.Input[WhereDataset] = clue.data.Ignore, val id: clue.data.Input[WhereOrderDatasetId] = clue.data.Ignore, val observation: clue.data.Input[WhereObservation] = clue.data.Ignore, val stepId: clue.data.Input[WhereEqStepId] = clue.data.Ignore, val index: clue.data.Input[WhereOrderPosShort] = clue.data.Ignore, val filename: clue.data.Input[WhereString] = clue.data.Ignore, val qaState: clue.data.Input[WhereOptionEqQaState] = clue.data.Ignore)
    object WhereDataset {
      val AND: monocle.Lens[WhereDataset, clue.data.Input[List[WhereDataset]]] = monocle.macros.GenLens[WhereDataset](_.AND)
      val OR: monocle.Lens[WhereDataset, clue.data.Input[List[WhereDataset]]] = monocle.macros.GenLens[WhereDataset](_.OR)
      val NOT: monocle.Lens[WhereDataset, clue.data.Input[WhereDataset]] = monocle.macros.GenLens[WhereDataset](_.NOT)
      val id: monocle.Lens[WhereDataset, clue.data.Input[WhereOrderDatasetId]] = monocle.macros.GenLens[WhereDataset](_.id)
      val observation: monocle.Lens[WhereDataset, clue.data.Input[WhereObservation]] = monocle.macros.GenLens[WhereDataset](_.observation)
      val stepId: monocle.Lens[WhereDataset, clue.data.Input[WhereEqStepId]] = monocle.macros.GenLens[WhereDataset](_.stepId)
      val index: monocle.Lens[WhereDataset, clue.data.Input[WhereOrderPosShort]] = monocle.macros.GenLens[WhereDataset](_.index)
      val filename: monocle.Lens[WhereDataset, clue.data.Input[WhereString]] = monocle.macros.GenLens[WhereDataset](_.filename)
      val qaState: monocle.Lens[WhereDataset, clue.data.Input[WhereOptionEqQaState]] = monocle.macros.GenLens[WhereDataset](_.qaState)
      implicit val eqWhereDataset: cats.Eq[WhereDataset] = cats.Eq.fromUniversalEquals
      implicit val showWhereDataset: cats.Show[WhereDataset] = cats.Show.fromToString
      implicit val jsonEncoderWhereDataset: io.circe.Encoder.AsObject[WhereDataset] = io.circe.generic.semiauto.deriveEncoder[WhereDataset].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class WhereEqExecutionEventType(val EQ: clue.data.Input[ExecutionEventType] = clue.data.Ignore, val NEQ: clue.data.Input[ExecutionEventType] = clue.data.Ignore, val IN: clue.data.Input[List[ExecutionEventType]] = clue.data.Ignore, val NIN: clue.data.Input[List[ExecutionEventType]] = clue.data.Ignore)
    object WhereEqExecutionEventType {
      val EQ: monocle.Lens[WhereEqExecutionEventType, clue.data.Input[ExecutionEventType]] = monocle.macros.GenLens[WhereEqExecutionEventType](_.EQ)
      val NEQ: monocle.Lens[WhereEqExecutionEventType, clue.data.Input[ExecutionEventType]] = monocle.macros.GenLens[WhereEqExecutionEventType](_.NEQ)
      val IN: monocle.Lens[WhereEqExecutionEventType, clue.data.Input[List[ExecutionEventType]]] = monocle.macros.GenLens[WhereEqExecutionEventType](_.IN)
      val NIN: monocle.Lens[WhereEqExecutionEventType, clue.data.Input[List[ExecutionEventType]]] = monocle.macros.GenLens[WhereEqExecutionEventType](_.NIN)
      implicit val eqWhereEqExecutionEventType: cats.Eq[WhereEqExecutionEventType] = cats.Eq.fromUniversalEquals
      implicit val showWhereEqExecutionEventType: cats.Show[WhereEqExecutionEventType] = cats.Show.fromToString
      implicit val jsonEncoderWhereEqExecutionEventType: io.circe.Encoder.AsObject[WhereEqExecutionEventType] = io.circe.generic.semiauto.deriveEncoder[WhereEqExecutionEventType].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class WhereEqPartner(val EQ: clue.data.Input[Partner] = clue.data.Ignore, val NEQ: clue.data.Input[Partner] = clue.data.Ignore, val IN: clue.data.Input[List[Partner]] = clue.data.Ignore, val NIN: clue.data.Input[List[Partner]] = clue.data.Ignore)
    object WhereEqPartner {
      val EQ: monocle.Lens[WhereEqPartner, clue.data.Input[Partner]] = monocle.macros.GenLens[WhereEqPartner](_.EQ)
      val NEQ: monocle.Lens[WhereEqPartner, clue.data.Input[Partner]] = monocle.macros.GenLens[WhereEqPartner](_.NEQ)
      val IN: monocle.Lens[WhereEqPartner, clue.data.Input[List[Partner]]] = monocle.macros.GenLens[WhereEqPartner](_.IN)
      val NIN: monocle.Lens[WhereEqPartner, clue.data.Input[List[Partner]]] = monocle.macros.GenLens[WhereEqPartner](_.NIN)
      implicit val eqWhereEqPartner: cats.Eq[WhereEqPartner] = cats.Eq.fromUniversalEquals
      implicit val showWhereEqPartner: cats.Show[WhereEqPartner] = cats.Show.fromToString
      implicit val jsonEncoderWhereEqPartner: io.circe.Encoder.AsObject[WhereEqPartner] = io.circe.generic.semiauto.deriveEncoder[WhereEqPartner].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class WhereEqProgramType(val EQ: clue.data.Input[ProgramType] = clue.data.Ignore, val NEQ: clue.data.Input[ProgramType] = clue.data.Ignore, val IN: clue.data.Input[List[ProgramType]] = clue.data.Ignore, val NIN: clue.data.Input[List[ProgramType]] = clue.data.Ignore)
    object WhereEqProgramType {
      val EQ: monocle.Lens[WhereEqProgramType, clue.data.Input[ProgramType]] = monocle.macros.GenLens[WhereEqProgramType](_.EQ)
      val NEQ: monocle.Lens[WhereEqProgramType, clue.data.Input[ProgramType]] = monocle.macros.GenLens[WhereEqProgramType](_.NEQ)
      val IN: monocle.Lens[WhereEqProgramType, clue.data.Input[List[ProgramType]]] = monocle.macros.GenLens[WhereEqProgramType](_.IN)
      val NIN: monocle.Lens[WhereEqProgramType, clue.data.Input[List[ProgramType]]] = monocle.macros.GenLens[WhereEqProgramType](_.NIN)
      implicit val eqWhereEqProgramType: cats.Eq[WhereEqProgramType] = cats.Eq.fromUniversalEquals
      implicit val showWhereEqProgramType: cats.Show[WhereEqProgramType] = cats.Show.fromToString
      implicit val jsonEncoderWhereEqProgramType: io.circe.Encoder.AsObject[WhereEqProgramType] = io.circe.generic.semiauto.deriveEncoder[WhereEqProgramType].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class WhereEqProposalClassType(val EQ: clue.data.Input[ProposalClassEnum] = clue.data.Ignore, val NEQ: clue.data.Input[ProposalClassEnum] = clue.data.Ignore, val IN: clue.data.Input[List[ProposalClassEnum]] = clue.data.Ignore, val NIN: clue.data.Input[List[ProposalClassEnum]] = clue.data.Ignore)
    object WhereEqProposalClassType {
      val EQ: monocle.Lens[WhereEqProposalClassType, clue.data.Input[ProposalClassEnum]] = monocle.macros.GenLens[WhereEqProposalClassType](_.EQ)
      val NEQ: monocle.Lens[WhereEqProposalClassType, clue.data.Input[ProposalClassEnum]] = monocle.macros.GenLens[WhereEqProposalClassType](_.NEQ)
      val IN: monocle.Lens[WhereEqProposalClassType, clue.data.Input[List[ProposalClassEnum]]] = monocle.macros.GenLens[WhereEqProposalClassType](_.IN)
      val NIN: monocle.Lens[WhereEqProposalClassType, clue.data.Input[List[ProposalClassEnum]]] = monocle.macros.GenLens[WhereEqProposalClassType](_.NIN)
      implicit val eqWhereEqProposalClassType: cats.Eq[WhereEqProposalClassType] = cats.Eq.fromUniversalEquals
      implicit val showWhereEqProposalClassType: cats.Show[WhereEqProposalClassType] = cats.Show.fromToString
      implicit val jsonEncoderWhereEqProposalClassType: io.circe.Encoder.AsObject[WhereEqProposalClassType] = io.circe.generic.semiauto.deriveEncoder[WhereEqProposalClassType].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class WhereEqProposalStatus(val EQ: clue.data.Input[ProposalStatus] = clue.data.Ignore, val NEQ: clue.data.Input[ProposalStatus] = clue.data.Ignore, val IN: clue.data.Input[List[ProposalStatus]] = clue.data.Ignore, val NIN: clue.data.Input[List[ProposalStatus]] = clue.data.Ignore)
    object WhereEqProposalStatus {
      val EQ: monocle.Lens[WhereEqProposalStatus, clue.data.Input[ProposalStatus]] = monocle.macros.GenLens[WhereEqProposalStatus](_.EQ)
      val NEQ: monocle.Lens[WhereEqProposalStatus, clue.data.Input[ProposalStatus]] = monocle.macros.GenLens[WhereEqProposalStatus](_.NEQ)
      val IN: monocle.Lens[WhereEqProposalStatus, clue.data.Input[List[ProposalStatus]]] = monocle.macros.GenLens[WhereEqProposalStatus](_.IN)
      val NIN: monocle.Lens[WhereEqProposalStatus, clue.data.Input[List[ProposalStatus]]] = monocle.macros.GenLens[WhereEqProposalStatus](_.NIN)
      implicit val eqWhereEqProposalStatus: cats.Eq[WhereEqProposalStatus] = cats.Eq.fromUniversalEquals
      implicit val showWhereEqProposalStatus: cats.Show[WhereEqProposalStatus] = cats.Show.fromToString
      implicit val jsonEncoderWhereEqProposalStatus: io.circe.Encoder.AsObject[WhereEqProposalStatus] = io.circe.generic.semiauto.deriveEncoder[WhereEqProposalStatus].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class WhereEqStepId(val EQ: clue.data.Input[StepId] = clue.data.Ignore, val NEQ: clue.data.Input[StepId] = clue.data.Ignore, val IN: clue.data.Input[List[StepId]] = clue.data.Ignore, val NIN: clue.data.Input[List[StepId]] = clue.data.Ignore)
    object WhereEqStepId {
      val EQ: monocle.Lens[WhereEqStepId, clue.data.Input[StepId]] = monocle.macros.GenLens[WhereEqStepId](_.EQ)
      val NEQ: monocle.Lens[WhereEqStepId, clue.data.Input[StepId]] = monocle.macros.GenLens[WhereEqStepId](_.NEQ)
      val IN: monocle.Lens[WhereEqStepId, clue.data.Input[List[StepId]]] = monocle.macros.GenLens[WhereEqStepId](_.IN)
      val NIN: monocle.Lens[WhereEqStepId, clue.data.Input[List[StepId]]] = monocle.macros.GenLens[WhereEqStepId](_.NIN)
      implicit val eqWhereEqStepId: cats.Eq[WhereEqStepId] = cats.Eq.fromUniversalEquals
      implicit val showWhereEqStepId: cats.Show[WhereEqStepId] = cats.Show.fromToString
      implicit val jsonEncoderWhereEqStepId: io.circe.Encoder.AsObject[WhereEqStepId] = io.circe.generic.semiauto.deriveEncoder[WhereEqStepId].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class WhereEqToOActivation(val EQ: clue.data.Input[ToOActivation] = clue.data.Ignore, val NEQ: clue.data.Input[ToOActivation] = clue.data.Ignore, val IN: clue.data.Input[List[ToOActivation]] = clue.data.Ignore, val NIN: clue.data.Input[List[ToOActivation]] = clue.data.Ignore)
    object WhereEqToOActivation {
      val EQ: monocle.Lens[WhereEqToOActivation, clue.data.Input[ToOActivation]] = monocle.macros.GenLens[WhereEqToOActivation](_.EQ)
      val NEQ: monocle.Lens[WhereEqToOActivation, clue.data.Input[ToOActivation]] = monocle.macros.GenLens[WhereEqToOActivation](_.NEQ)
      val IN: monocle.Lens[WhereEqToOActivation, clue.data.Input[List[ToOActivation]]] = monocle.macros.GenLens[WhereEqToOActivation](_.IN)
      val NIN: monocle.Lens[WhereEqToOActivation, clue.data.Input[List[ToOActivation]]] = monocle.macros.GenLens[WhereEqToOActivation](_.NIN)
      implicit val eqWhereEqToOActivation: cats.Eq[WhereEqToOActivation] = cats.Eq.fromUniversalEquals
      implicit val showWhereEqToOActivation: cats.Show[WhereEqToOActivation] = cats.Show.fromToString
      implicit val jsonEncoderWhereEqToOActivation: io.circe.Encoder.AsObject[WhereEqToOActivation] = io.circe.generic.semiauto.deriveEncoder[WhereEqToOActivation].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class WhereEqVisitId(val EQ: clue.data.Input[VisitId] = clue.data.Ignore, val NEQ: clue.data.Input[VisitId] = clue.data.Ignore, val IN: clue.data.Input[List[VisitId]] = clue.data.Ignore, val NIN: clue.data.Input[List[VisitId]] = clue.data.Ignore)
    object WhereEqVisitId {
      val EQ: monocle.Lens[WhereEqVisitId, clue.data.Input[VisitId]] = monocle.macros.GenLens[WhereEqVisitId](_.EQ)
      val NEQ: monocle.Lens[WhereEqVisitId, clue.data.Input[VisitId]] = monocle.macros.GenLens[WhereEqVisitId](_.NEQ)
      val IN: monocle.Lens[WhereEqVisitId, clue.data.Input[List[VisitId]]] = monocle.macros.GenLens[WhereEqVisitId](_.IN)
      val NIN: monocle.Lens[WhereEqVisitId, clue.data.Input[List[VisitId]]] = monocle.macros.GenLens[WhereEqVisitId](_.NIN)
      implicit val eqWhereEqVisitId: cats.Eq[WhereEqVisitId] = cats.Eq.fromUniversalEquals
      implicit val showWhereEqVisitId: cats.Show[WhereEqVisitId] = cats.Show.fromToString
      implicit val jsonEncoderWhereEqVisitId: io.circe.Encoder.AsObject[WhereEqVisitId] = io.circe.generic.semiauto.deriveEncoder[WhereEqVisitId].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class WhereExecutionEvent(val AND: clue.data.Input[List[WhereExecutionEvent]] = clue.data.Ignore, val OR: clue.data.Input[List[WhereExecutionEvent]] = clue.data.Ignore, val NOT: clue.data.Input[WhereExecutionEvent] = clue.data.Ignore, val id: clue.data.Input[WhereOrderExecutionEventId] = clue.data.Ignore, val visitId: clue.data.Input[WhereEqVisitId] = clue.data.Ignore, val observationId: clue.data.Input[WhereOrderObservationId] = clue.data.Ignore, val received: clue.data.Input[WhereOrderTimestamp] = clue.data.Ignore, val eventType: clue.data.Input[WhereEqExecutionEventType] = clue.data.Ignore, val sequenceCommand: clue.data.Input[WhereOrderSequenceCommand] = clue.data.Ignore, val stepId: clue.data.Input[WhereEqStepId] = clue.data.Ignore, val stepStage: clue.data.Input[WhereOrderStepStage] = clue.data.Ignore, val datasetId: clue.data.Input[WhereOrderDatasetId] = clue.data.Ignore, val datasetStage: clue.data.Input[WhereOrderDatasetStage] = clue.data.Ignore)
    object WhereExecutionEvent {
      val AND: monocle.Lens[WhereExecutionEvent, clue.data.Input[List[WhereExecutionEvent]]] = monocle.macros.GenLens[WhereExecutionEvent](_.AND)
      val OR: monocle.Lens[WhereExecutionEvent, clue.data.Input[List[WhereExecutionEvent]]] = monocle.macros.GenLens[WhereExecutionEvent](_.OR)
      val NOT: monocle.Lens[WhereExecutionEvent, clue.data.Input[WhereExecutionEvent]] = monocle.macros.GenLens[WhereExecutionEvent](_.NOT)
      val id: monocle.Lens[WhereExecutionEvent, clue.data.Input[WhereOrderExecutionEventId]] = monocle.macros.GenLens[WhereExecutionEvent](_.id)
      val visitId: monocle.Lens[WhereExecutionEvent, clue.data.Input[WhereEqVisitId]] = monocle.macros.GenLens[WhereExecutionEvent](_.visitId)
      val observationId: monocle.Lens[WhereExecutionEvent, clue.data.Input[WhereOrderObservationId]] = monocle.macros.GenLens[WhereExecutionEvent](_.observationId)
      val received: monocle.Lens[WhereExecutionEvent, clue.data.Input[WhereOrderTimestamp]] = monocle.macros.GenLens[WhereExecutionEvent](_.received)
      val eventType: monocle.Lens[WhereExecutionEvent, clue.data.Input[WhereEqExecutionEventType]] = monocle.macros.GenLens[WhereExecutionEvent](_.eventType)
      val sequenceCommand: monocle.Lens[WhereExecutionEvent, clue.data.Input[WhereOrderSequenceCommand]] = monocle.macros.GenLens[WhereExecutionEvent](_.sequenceCommand)
      val stepId: monocle.Lens[WhereExecutionEvent, clue.data.Input[WhereEqStepId]] = monocle.macros.GenLens[WhereExecutionEvent](_.stepId)
      val stepStage: monocle.Lens[WhereExecutionEvent, clue.data.Input[WhereOrderStepStage]] = monocle.macros.GenLens[WhereExecutionEvent](_.stepStage)
      val datasetId: monocle.Lens[WhereExecutionEvent, clue.data.Input[WhereOrderDatasetId]] = monocle.macros.GenLens[WhereExecutionEvent](_.datasetId)
      val datasetStage: monocle.Lens[WhereExecutionEvent, clue.data.Input[WhereOrderDatasetStage]] = monocle.macros.GenLens[WhereExecutionEvent](_.datasetStage)
      implicit val eqWhereExecutionEvent: cats.Eq[WhereExecutionEvent] = cats.Eq.fromUniversalEquals
      implicit val showWhereExecutionEvent: cats.Show[WhereExecutionEvent] = cats.Show.fromToString
      implicit val jsonEncoderWhereExecutionEvent: io.circe.Encoder.AsObject[WhereExecutionEvent] = io.circe.generic.semiauto.deriveEncoder[WhereExecutionEvent].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class WhereObservation(val AND: clue.data.Input[List[WhereObservation]] = clue.data.Ignore, val OR: clue.data.Input[List[WhereObservation]] = clue.data.Ignore, val NOT: clue.data.Input[WhereObservation] = clue.data.Ignore, val id: clue.data.Input[WhereOrderObservationId] = clue.data.Ignore, val program: clue.data.Input[WhereProgram] = clue.data.Ignore, val subtitle: clue.data.Input[WhereOptionString] = clue.data.Ignore, val status: clue.data.Input[WhereOrderObsStatus] = clue.data.Ignore, val activeStatus: clue.data.Input[WhereOrderObsActiveStatus] = clue.data.Ignore)
    object WhereObservation {
      val AND: monocle.Lens[WhereObservation, clue.data.Input[List[WhereObservation]]] = monocle.macros.GenLens[WhereObservation](_.AND)
      val OR: monocle.Lens[WhereObservation, clue.data.Input[List[WhereObservation]]] = monocle.macros.GenLens[WhereObservation](_.OR)
      val NOT: monocle.Lens[WhereObservation, clue.data.Input[WhereObservation]] = monocle.macros.GenLens[WhereObservation](_.NOT)
      val id: monocle.Lens[WhereObservation, clue.data.Input[WhereOrderObservationId]] = monocle.macros.GenLens[WhereObservation](_.id)
      val program: monocle.Lens[WhereObservation, clue.data.Input[WhereProgram]] = monocle.macros.GenLens[WhereObservation](_.program)
      val subtitle: monocle.Lens[WhereObservation, clue.data.Input[WhereOptionString]] = monocle.macros.GenLens[WhereObservation](_.subtitle)
      val status: monocle.Lens[WhereObservation, clue.data.Input[WhereOrderObsStatus]] = monocle.macros.GenLens[WhereObservation](_.status)
      val activeStatus: monocle.Lens[WhereObservation, clue.data.Input[WhereOrderObsActiveStatus]] = monocle.macros.GenLens[WhereObservation](_.activeStatus)
      implicit val eqWhereObservation: cats.Eq[WhereObservation] = cats.Eq.fromUniversalEquals
      implicit val showWhereObservation: cats.Show[WhereObservation] = cats.Show.fromToString
      implicit val jsonEncoderWhereObservation: io.circe.Encoder.AsObject[WhereObservation] = io.circe.generic.semiauto.deriveEncoder[WhereObservation].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class WhereGroup(val AND: clue.data.Input[List[WhereGroup]] = clue.data.Ignore, val OR: clue.data.Input[List[WhereGroup]] = clue.data.Ignore, val NOT: clue.data.Input[WhereGroup] = clue.data.Ignore, val id: clue.data.Input[WhereOrderGroupId] = clue.data.Ignore, val name: clue.data.Input[WhereOptionString] = clue.data.Ignore, val description: clue.data.Input[WhereOptionString] = clue.data.Ignore)
    object WhereGroup {
      val AND: monocle.Lens[WhereGroup, clue.data.Input[List[WhereGroup]]] = monocle.macros.GenLens[WhereGroup](_.AND)
      val OR: monocle.Lens[WhereGroup, clue.data.Input[List[WhereGroup]]] = monocle.macros.GenLens[WhereGroup](_.OR)
      val NOT: monocle.Lens[WhereGroup, clue.data.Input[WhereGroup]] = monocle.macros.GenLens[WhereGroup](_.NOT)
      val id: monocle.Lens[WhereGroup, clue.data.Input[WhereOrderGroupId]] = monocle.macros.GenLens[WhereGroup](_.id)
      val name: monocle.Lens[WhereGroup, clue.data.Input[WhereOptionString]] = monocle.macros.GenLens[WhereGroup](_.name)
      val description: monocle.Lens[WhereGroup, clue.data.Input[WhereOptionString]] = monocle.macros.GenLens[WhereGroup](_.description)
      implicit val eqWhereGroup: cats.Eq[WhereGroup] = cats.Eq.fromUniversalEquals
      implicit val showWhereGroup: cats.Show[WhereGroup] = cats.Show.fromToString
      implicit val jsonEncoderWhereGroup: io.circe.Encoder.AsObject[WhereGroup] = io.circe.generic.semiauto.deriveEncoder[WhereGroup].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class WhereOrderGroupId(val EQ: clue.data.Input[GroupId] = clue.data.Ignore, val NEQ: clue.data.Input[GroupId] = clue.data.Ignore, val IN: clue.data.Input[List[GroupId]] = clue.data.Ignore, val NIN: clue.data.Input[List[GroupId]] = clue.data.Ignore, val GT: clue.data.Input[GroupId] = clue.data.Ignore, val LT: clue.data.Input[GroupId] = clue.data.Ignore, val GTE: clue.data.Input[GroupId] = clue.data.Ignore, val LTE: clue.data.Input[GroupId] = clue.data.Ignore)
    object WhereOrderGroupId {
      val EQ: monocle.Lens[WhereOrderGroupId, clue.data.Input[GroupId]] = monocle.macros.GenLens[WhereOrderGroupId](_.EQ)
      val NEQ: monocle.Lens[WhereOrderGroupId, clue.data.Input[GroupId]] = monocle.macros.GenLens[WhereOrderGroupId](_.NEQ)
      val IN: monocle.Lens[WhereOrderGroupId, clue.data.Input[List[GroupId]]] = monocle.macros.GenLens[WhereOrderGroupId](_.IN)
      val NIN: monocle.Lens[WhereOrderGroupId, clue.data.Input[List[GroupId]]] = monocle.macros.GenLens[WhereOrderGroupId](_.NIN)
      val GT: monocle.Lens[WhereOrderGroupId, clue.data.Input[GroupId]] = monocle.macros.GenLens[WhereOrderGroupId](_.GT)
      val LT: monocle.Lens[WhereOrderGroupId, clue.data.Input[GroupId]] = monocle.macros.GenLens[WhereOrderGroupId](_.LT)
      val GTE: monocle.Lens[WhereOrderGroupId, clue.data.Input[GroupId]] = monocle.macros.GenLens[WhereOrderGroupId](_.GTE)
      val LTE: monocle.Lens[WhereOrderGroupId, clue.data.Input[GroupId]] = monocle.macros.GenLens[WhereOrderGroupId](_.LTE)
      implicit val eqWhereOrderGroupId: cats.Eq[WhereOrderGroupId] = cats.Eq.fromUniversalEquals
      implicit val showWhereOrderGroupId: cats.Show[WhereOrderGroupId] = cats.Show.fromToString
      implicit val jsonEncoderWhereOrderGroupId: io.circe.Encoder.AsObject[WhereOrderGroupId] = io.circe.generic.semiauto.deriveEncoder[WhereOrderGroupId].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class WhereEqInstrument(val EQ: clue.data.Input[Instrument] = clue.data.Ignore, val NEQ: clue.data.Input[Instrument] = clue.data.Ignore, val IN: clue.data.Input[List[Instrument]] = clue.data.Ignore, val NIN: clue.data.Input[List[Instrument]] = clue.data.Ignore)
    object WhereEqInstrument {
      val EQ: monocle.Lens[WhereEqInstrument, clue.data.Input[Instrument]] = monocle.macros.GenLens[WhereEqInstrument](_.EQ)
      val NEQ: monocle.Lens[WhereEqInstrument, clue.data.Input[Instrument]] = monocle.macros.GenLens[WhereEqInstrument](_.NEQ)
      val IN: monocle.Lens[WhereEqInstrument, clue.data.Input[List[Instrument]]] = monocle.macros.GenLens[WhereEqInstrument](_.IN)
      val NIN: monocle.Lens[WhereEqInstrument, clue.data.Input[List[Instrument]]] = monocle.macros.GenLens[WhereEqInstrument](_.NIN)
      implicit val eqWhereEqInstrument: cats.Eq[WhereEqInstrument] = cats.Eq.fromUniversalEquals
      implicit val showWhereEqInstrument: cats.Show[WhereEqInstrument] = cats.Show.fromToString
      implicit val jsonEncoderWhereEqInstrument: io.circe.Encoder.AsObject[WhereEqInstrument] = io.circe.generic.semiauto.deriveEncoder[WhereEqInstrument].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class WhereOptionEqQaState(val IS_NULL: clue.data.Input[Boolean] = clue.data.Ignore, val EQ: clue.data.Input[DatasetQaState] = clue.data.Ignore, val NEQ: clue.data.Input[DatasetQaState] = clue.data.Ignore, val IN: clue.data.Input[List[DatasetQaState]] = clue.data.Ignore, val NIN: clue.data.Input[List[DatasetQaState]] = clue.data.Ignore)
    object WhereOptionEqQaState {
      val IS_NULL: monocle.Lens[WhereOptionEqQaState, clue.data.Input[Boolean]] = monocle.macros.GenLens[WhereOptionEqQaState](_.IS_NULL)
      val EQ: monocle.Lens[WhereOptionEqQaState, clue.data.Input[DatasetQaState]] = monocle.macros.GenLens[WhereOptionEqQaState](_.EQ)
      val NEQ: monocle.Lens[WhereOptionEqQaState, clue.data.Input[DatasetQaState]] = monocle.macros.GenLens[WhereOptionEqQaState](_.NEQ)
      val IN: monocle.Lens[WhereOptionEqQaState, clue.data.Input[List[DatasetQaState]]] = monocle.macros.GenLens[WhereOptionEqQaState](_.IN)
      val NIN: monocle.Lens[WhereOptionEqQaState, clue.data.Input[List[DatasetQaState]]] = monocle.macros.GenLens[WhereOptionEqQaState](_.NIN)
      implicit val eqWhereOptionEqQaState: cats.Eq[WhereOptionEqQaState] = cats.Eq.fromUniversalEquals
      implicit val showWhereOptionEqQaState: cats.Show[WhereOptionEqQaState] = cats.Show.fromToString
      implicit val jsonEncoderWhereOptionEqQaState: io.circe.Encoder.AsObject[WhereOptionEqQaState] = io.circe.generic.semiauto.deriveEncoder[WhereOptionEqQaState].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class WhereEqScienceSubtype(val EQ: clue.data.Input[ScienceSubtype] = clue.data.Ignore, val NEQ: clue.data.Input[ScienceSubtype] = clue.data.Ignore, val IN: clue.data.Input[List[ScienceSubtype]] = clue.data.Ignore, val NIN: clue.data.Input[List[ScienceSubtype]] = clue.data.Ignore)
    object WhereEqScienceSubtype {
      val EQ: monocle.Lens[WhereEqScienceSubtype, clue.data.Input[ScienceSubtype]] = monocle.macros.GenLens[WhereEqScienceSubtype](_.EQ)
      val NEQ: monocle.Lens[WhereEqScienceSubtype, clue.data.Input[ScienceSubtype]] = monocle.macros.GenLens[WhereEqScienceSubtype](_.NEQ)
      val IN: monocle.Lens[WhereEqScienceSubtype, clue.data.Input[List[ScienceSubtype]]] = monocle.macros.GenLens[WhereEqScienceSubtype](_.IN)
      val NIN: monocle.Lens[WhereEqScienceSubtype, clue.data.Input[List[ScienceSubtype]]] = monocle.macros.GenLens[WhereEqScienceSubtype](_.NIN)
      implicit val eqWhereEqScienceSubtype: cats.Eq[WhereEqScienceSubtype] = cats.Eq.fromUniversalEquals
      implicit val showWhereEqScienceSubtype: cats.Show[WhereEqScienceSubtype] = cats.Show.fromToString
      implicit val jsonEncoderWhereEqScienceSubtype: io.circe.Encoder.AsObject[WhereEqScienceSubtype] = io.circe.generic.semiauto.deriveEncoder[WhereEqScienceSubtype].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class WhereOptionEqTacCategory(val IS_NULL: clue.data.Input[Boolean] = clue.data.Ignore, val EQ: clue.data.Input[TacCategory] = clue.data.Ignore, val NEQ: clue.data.Input[TacCategory] = clue.data.Ignore, val IN: clue.data.Input[List[TacCategory]] = clue.data.Ignore, val NIN: clue.data.Input[List[TacCategory]] = clue.data.Ignore)
    object WhereOptionEqTacCategory {
      val IS_NULL: monocle.Lens[WhereOptionEqTacCategory, clue.data.Input[Boolean]] = monocle.macros.GenLens[WhereOptionEqTacCategory](_.IS_NULL)
      val EQ: monocle.Lens[WhereOptionEqTacCategory, clue.data.Input[TacCategory]] = monocle.macros.GenLens[WhereOptionEqTacCategory](_.EQ)
      val NEQ: monocle.Lens[WhereOptionEqTacCategory, clue.data.Input[TacCategory]] = monocle.macros.GenLens[WhereOptionEqTacCategory](_.NEQ)
      val IN: monocle.Lens[WhereOptionEqTacCategory, clue.data.Input[List[TacCategory]]] = monocle.macros.GenLens[WhereOptionEqTacCategory](_.IN)
      val NIN: monocle.Lens[WhereOptionEqTacCategory, clue.data.Input[List[TacCategory]]] = monocle.macros.GenLens[WhereOptionEqTacCategory](_.NIN)
      implicit val eqWhereOptionEqTacCategory: cats.Eq[WhereOptionEqTacCategory] = cats.Eq.fromUniversalEquals
      implicit val showWhereOptionEqTacCategory: cats.Show[WhereOptionEqTacCategory] = cats.Show.fromToString
      implicit val jsonEncoderWhereOptionEqTacCategory: io.circe.Encoder.AsObject[WhereOptionEqTacCategory] = io.circe.generic.semiauto.deriveEncoder[WhereOptionEqTacCategory].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class WhereOptionString(val IS_NULL: clue.data.Input[Boolean] = clue.data.Ignore, val EQ: clue.data.Input[NonEmptyString] = clue.data.Ignore, val NEQ: clue.data.Input[NonEmptyString] = clue.data.Ignore, val IN: clue.data.Input[List[NonEmptyString]] = clue.data.Ignore, val NIN: clue.data.Input[List[NonEmptyString]] = clue.data.Ignore, val LIKE: clue.data.Input[NonEmptyString] = clue.data.Ignore, val NLIKE: clue.data.Input[NonEmptyString] = clue.data.Ignore, val MATCH_CASE: clue.data.Input[Boolean] = clue.data.Ignore)
    object WhereOptionString {
      val IS_NULL: monocle.Lens[WhereOptionString, clue.data.Input[Boolean]] = monocle.macros.GenLens[WhereOptionString](_.IS_NULL)
      val EQ: monocle.Lens[WhereOptionString, clue.data.Input[NonEmptyString]] = monocle.macros.GenLens[WhereOptionString](_.EQ)
      val NEQ: monocle.Lens[WhereOptionString, clue.data.Input[NonEmptyString]] = monocle.macros.GenLens[WhereOptionString](_.NEQ)
      val IN: monocle.Lens[WhereOptionString, clue.data.Input[List[NonEmptyString]]] = monocle.macros.GenLens[WhereOptionString](_.IN)
      val NIN: monocle.Lens[WhereOptionString, clue.data.Input[List[NonEmptyString]]] = monocle.macros.GenLens[WhereOptionString](_.NIN)
      val LIKE: monocle.Lens[WhereOptionString, clue.data.Input[NonEmptyString]] = monocle.macros.GenLens[WhereOptionString](_.LIKE)
      val NLIKE: monocle.Lens[WhereOptionString, clue.data.Input[NonEmptyString]] = monocle.macros.GenLens[WhereOptionString](_.NLIKE)
      val MATCH_CASE: monocle.Lens[WhereOptionString, clue.data.Input[Boolean]] = monocle.macros.GenLens[WhereOptionString](_.MATCH_CASE)
      implicit val eqWhereOptionString: cats.Eq[WhereOptionString] = cats.Eq.fromUniversalEquals
      implicit val showWhereOptionString: cats.Show[WhereOptionString] = cats.Show.fromToString
      implicit val jsonEncoderWhereOptionString: io.circe.Encoder.AsObject[WhereOptionString] = io.circe.generic.semiauto.deriveEncoder[WhereOptionString].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class WhereOrderPosShort(val EQ: clue.data.Input[PosShort] = clue.data.Ignore, val NEQ: clue.data.Input[PosShort] = clue.data.Ignore, val IN: clue.data.Input[List[PosShort]] = clue.data.Ignore, val NIN: clue.data.Input[List[PosShort]] = clue.data.Ignore, val GT: clue.data.Input[PosShort] = clue.data.Ignore, val LT: clue.data.Input[PosShort] = clue.data.Ignore, val GTE: clue.data.Input[PosShort] = clue.data.Ignore, val LTE: clue.data.Input[PosShort] = clue.data.Ignore)
    object WhereOrderPosShort {
      val EQ: monocle.Lens[WhereOrderPosShort, clue.data.Input[PosShort]] = monocle.macros.GenLens[WhereOrderPosShort](_.EQ)
      val NEQ: monocle.Lens[WhereOrderPosShort, clue.data.Input[PosShort]] = monocle.macros.GenLens[WhereOrderPosShort](_.NEQ)
      val IN: monocle.Lens[WhereOrderPosShort, clue.data.Input[List[PosShort]]] = monocle.macros.GenLens[WhereOrderPosShort](_.IN)
      val NIN: monocle.Lens[WhereOrderPosShort, clue.data.Input[List[PosShort]]] = monocle.macros.GenLens[WhereOrderPosShort](_.NIN)
      val GT: monocle.Lens[WhereOrderPosShort, clue.data.Input[PosShort]] = monocle.macros.GenLens[WhereOrderPosShort](_.GT)
      val LT: monocle.Lens[WhereOrderPosShort, clue.data.Input[PosShort]] = monocle.macros.GenLens[WhereOrderPosShort](_.LT)
      val GTE: monocle.Lens[WhereOrderPosShort, clue.data.Input[PosShort]] = monocle.macros.GenLens[WhereOrderPosShort](_.GTE)
      val LTE: monocle.Lens[WhereOrderPosShort, clue.data.Input[PosShort]] = monocle.macros.GenLens[WhereOrderPosShort](_.LTE)
      implicit val eqWhereOrderPosShort: cats.Eq[WhereOrderPosShort] = cats.Eq.fromUniversalEquals
      implicit val showWhereOrderPosShort: cats.Show[WhereOrderPosShort] = cats.Show.fromToString
      implicit val jsonEncoderWhereOrderPosShort: io.circe.Encoder.AsObject[WhereOrderPosShort] = io.circe.generic.semiauto.deriveEncoder[WhereOrderPosShort].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class WhereOrderDatasetId(val EQ: clue.data.Input[DatasetId] = clue.data.Ignore, val NEQ: clue.data.Input[DatasetId] = clue.data.Ignore, val IN: clue.data.Input[List[DatasetId]] = clue.data.Ignore, val NIN: clue.data.Input[List[DatasetId]] = clue.data.Ignore, val GT: clue.data.Input[DatasetId] = clue.data.Ignore, val LT: clue.data.Input[DatasetId] = clue.data.Ignore, val GTE: clue.data.Input[DatasetId] = clue.data.Ignore, val LTE: clue.data.Input[DatasetId] = clue.data.Ignore)
    object WhereOrderDatasetId {
      val EQ: monocle.Lens[WhereOrderDatasetId, clue.data.Input[DatasetId]] = monocle.macros.GenLens[WhereOrderDatasetId](_.EQ)
      val NEQ: monocle.Lens[WhereOrderDatasetId, clue.data.Input[DatasetId]] = monocle.macros.GenLens[WhereOrderDatasetId](_.NEQ)
      val IN: monocle.Lens[WhereOrderDatasetId, clue.data.Input[List[DatasetId]]] = monocle.macros.GenLens[WhereOrderDatasetId](_.IN)
      val NIN: monocle.Lens[WhereOrderDatasetId, clue.data.Input[List[DatasetId]]] = monocle.macros.GenLens[WhereOrderDatasetId](_.NIN)
      val GT: monocle.Lens[WhereOrderDatasetId, clue.data.Input[DatasetId]] = monocle.macros.GenLens[WhereOrderDatasetId](_.GT)
      val LT: monocle.Lens[WhereOrderDatasetId, clue.data.Input[DatasetId]] = monocle.macros.GenLens[WhereOrderDatasetId](_.LT)
      val GTE: monocle.Lens[WhereOrderDatasetId, clue.data.Input[DatasetId]] = monocle.macros.GenLens[WhereOrderDatasetId](_.GTE)
      val LTE: monocle.Lens[WhereOrderDatasetId, clue.data.Input[DatasetId]] = monocle.macros.GenLens[WhereOrderDatasetId](_.LTE)
      implicit val eqWhereOrderDatasetId: cats.Eq[WhereOrderDatasetId] = cats.Eq.fromUniversalEquals
      implicit val showWhereOrderDatasetId: cats.Show[WhereOrderDatasetId] = cats.Show.fromToString
      implicit val jsonEncoderWhereOrderDatasetId: io.circe.Encoder.AsObject[WhereOrderDatasetId] = io.circe.generic.semiauto.deriveEncoder[WhereOrderDatasetId].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class WhereOrderDatasetStage(val EQ: clue.data.Input[DatasetStage] = clue.data.Ignore, val NEQ: clue.data.Input[DatasetStage] = clue.data.Ignore, val IN: clue.data.Input[List[DatasetStage]] = clue.data.Ignore, val NIN: clue.data.Input[List[DatasetStage]] = clue.data.Ignore, val GT: clue.data.Input[DatasetStage] = clue.data.Ignore, val LT: clue.data.Input[DatasetStage] = clue.data.Ignore, val GTE: clue.data.Input[DatasetStage] = clue.data.Ignore, val LTE: clue.data.Input[DatasetStage] = clue.data.Ignore)
    object WhereOrderDatasetStage {
      val EQ: monocle.Lens[WhereOrderDatasetStage, clue.data.Input[DatasetStage]] = monocle.macros.GenLens[WhereOrderDatasetStage](_.EQ)
      val NEQ: monocle.Lens[WhereOrderDatasetStage, clue.data.Input[DatasetStage]] = monocle.macros.GenLens[WhereOrderDatasetStage](_.NEQ)
      val IN: monocle.Lens[WhereOrderDatasetStage, clue.data.Input[List[DatasetStage]]] = monocle.macros.GenLens[WhereOrderDatasetStage](_.IN)
      val NIN: monocle.Lens[WhereOrderDatasetStage, clue.data.Input[List[DatasetStage]]] = monocle.macros.GenLens[WhereOrderDatasetStage](_.NIN)
      val GT: monocle.Lens[WhereOrderDatasetStage, clue.data.Input[DatasetStage]] = monocle.macros.GenLens[WhereOrderDatasetStage](_.GT)
      val LT: monocle.Lens[WhereOrderDatasetStage, clue.data.Input[DatasetStage]] = monocle.macros.GenLens[WhereOrderDatasetStage](_.LT)
      val GTE: monocle.Lens[WhereOrderDatasetStage, clue.data.Input[DatasetStage]] = monocle.macros.GenLens[WhereOrderDatasetStage](_.GTE)
      val LTE: monocle.Lens[WhereOrderDatasetStage, clue.data.Input[DatasetStage]] = monocle.macros.GenLens[WhereOrderDatasetStage](_.LTE)
      implicit val eqWhereOrderDatasetStage: cats.Eq[WhereOrderDatasetStage] = cats.Eq.fromUniversalEquals
      implicit val showWhereOrderDatasetStage: cats.Show[WhereOrderDatasetStage] = cats.Show.fromToString
      implicit val jsonEncoderWhereOrderDatasetStage: io.circe.Encoder.AsObject[WhereOrderDatasetStage] = io.circe.generic.semiauto.deriveEncoder[WhereOrderDatasetStage].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class WhereOrderExecutionEventId(val EQ: clue.data.Input[ExecutionEventId] = clue.data.Ignore, val NEQ: clue.data.Input[ExecutionEventId] = clue.data.Ignore, val IN: clue.data.Input[List[ExecutionEventId]] = clue.data.Ignore, val NIN: clue.data.Input[List[ExecutionEventId]] = clue.data.Ignore, val GT: clue.data.Input[ExecutionEventId] = clue.data.Ignore, val LT: clue.data.Input[ExecutionEventId] = clue.data.Ignore, val GTE: clue.data.Input[ExecutionEventId] = clue.data.Ignore, val LTE: clue.data.Input[ExecutionEventId] = clue.data.Ignore)
    object WhereOrderExecutionEventId {
      val EQ: monocle.Lens[WhereOrderExecutionEventId, clue.data.Input[ExecutionEventId]] = monocle.macros.GenLens[WhereOrderExecutionEventId](_.EQ)
      val NEQ: monocle.Lens[WhereOrderExecutionEventId, clue.data.Input[ExecutionEventId]] = monocle.macros.GenLens[WhereOrderExecutionEventId](_.NEQ)
      val IN: monocle.Lens[WhereOrderExecutionEventId, clue.data.Input[List[ExecutionEventId]]] = monocle.macros.GenLens[WhereOrderExecutionEventId](_.IN)
      val NIN: monocle.Lens[WhereOrderExecutionEventId, clue.data.Input[List[ExecutionEventId]]] = monocle.macros.GenLens[WhereOrderExecutionEventId](_.NIN)
      val GT: monocle.Lens[WhereOrderExecutionEventId, clue.data.Input[ExecutionEventId]] = monocle.macros.GenLens[WhereOrderExecutionEventId](_.GT)
      val LT: monocle.Lens[WhereOrderExecutionEventId, clue.data.Input[ExecutionEventId]] = monocle.macros.GenLens[WhereOrderExecutionEventId](_.LT)
      val GTE: monocle.Lens[WhereOrderExecutionEventId, clue.data.Input[ExecutionEventId]] = monocle.macros.GenLens[WhereOrderExecutionEventId](_.GTE)
      val LTE: monocle.Lens[WhereOrderExecutionEventId, clue.data.Input[ExecutionEventId]] = monocle.macros.GenLens[WhereOrderExecutionEventId](_.LTE)
      implicit val eqWhereOrderExecutionEventId: cats.Eq[WhereOrderExecutionEventId] = cats.Eq.fromUniversalEquals
      implicit val showWhereOrderExecutionEventId: cats.Show[WhereOrderExecutionEventId] = cats.Show.fromToString
      implicit val jsonEncoderWhereOrderExecutionEventId: io.circe.Encoder.AsObject[WhereOrderExecutionEventId] = io.circe.generic.semiauto.deriveEncoder[WhereOrderExecutionEventId].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class WhereOrderTimestamp(val EQ: clue.data.Input[Timestamp] = clue.data.Ignore, val NEQ: clue.data.Input[Timestamp] = clue.data.Ignore, val IN: clue.data.Input[List[Timestamp]] = clue.data.Ignore, val NIN: clue.data.Input[List[Timestamp]] = clue.data.Ignore, val GT: clue.data.Input[Timestamp] = clue.data.Ignore, val LT: clue.data.Input[Timestamp] = clue.data.Ignore, val GTE: clue.data.Input[Timestamp] = clue.data.Ignore, val LTE: clue.data.Input[Timestamp] = clue.data.Ignore)
    object WhereOrderTimestamp {
      val EQ: monocle.Lens[WhereOrderTimestamp, clue.data.Input[Timestamp]] = monocle.macros.GenLens[WhereOrderTimestamp](_.EQ)
      val NEQ: monocle.Lens[WhereOrderTimestamp, clue.data.Input[Timestamp]] = monocle.macros.GenLens[WhereOrderTimestamp](_.NEQ)
      val IN: monocle.Lens[WhereOrderTimestamp, clue.data.Input[List[Timestamp]]] = monocle.macros.GenLens[WhereOrderTimestamp](_.IN)
      val NIN: monocle.Lens[WhereOrderTimestamp, clue.data.Input[List[Timestamp]]] = monocle.macros.GenLens[WhereOrderTimestamp](_.NIN)
      val GT: monocle.Lens[WhereOrderTimestamp, clue.data.Input[Timestamp]] = monocle.macros.GenLens[WhereOrderTimestamp](_.GT)
      val LT: monocle.Lens[WhereOrderTimestamp, clue.data.Input[Timestamp]] = monocle.macros.GenLens[WhereOrderTimestamp](_.LT)
      val GTE: monocle.Lens[WhereOrderTimestamp, clue.data.Input[Timestamp]] = monocle.macros.GenLens[WhereOrderTimestamp](_.GTE)
      val LTE: monocle.Lens[WhereOrderTimestamp, clue.data.Input[Timestamp]] = monocle.macros.GenLens[WhereOrderTimestamp](_.LTE)
      implicit val eqWhereOrderTimestamp: cats.Eq[WhereOrderTimestamp] = cats.Eq.fromUniversalEquals
      implicit val showWhereOrderTimestamp: cats.Show[WhereOrderTimestamp] = cats.Show.fromToString
      implicit val jsonEncoderWhereOrderTimestamp: io.circe.Encoder.AsObject[WhereOrderTimestamp] = io.circe.generic.semiauto.deriveEncoder[WhereOrderTimestamp].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class WhereOrderInt(val EQ: clue.data.Input[Int] = clue.data.Ignore, val NEQ: clue.data.Input[Int] = clue.data.Ignore, val IN: clue.data.Input[List[Int]] = clue.data.Ignore, val NIN: clue.data.Input[List[Int]] = clue.data.Ignore, val GT: clue.data.Input[Int] = clue.data.Ignore, val LT: clue.data.Input[Int] = clue.data.Ignore, val GTE: clue.data.Input[Int] = clue.data.Ignore, val LTE: clue.data.Input[Int] = clue.data.Ignore)
    object WhereOrderInt {
      val EQ: monocle.Lens[WhereOrderInt, clue.data.Input[Int]] = monocle.macros.GenLens[WhereOrderInt](_.EQ)
      val NEQ: monocle.Lens[WhereOrderInt, clue.data.Input[Int]] = monocle.macros.GenLens[WhereOrderInt](_.NEQ)
      val IN: monocle.Lens[WhereOrderInt, clue.data.Input[List[Int]]] = monocle.macros.GenLens[WhereOrderInt](_.IN)
      val NIN: monocle.Lens[WhereOrderInt, clue.data.Input[List[Int]]] = monocle.macros.GenLens[WhereOrderInt](_.NIN)
      val GT: monocle.Lens[WhereOrderInt, clue.data.Input[Int]] = monocle.macros.GenLens[WhereOrderInt](_.GT)
      val LT: monocle.Lens[WhereOrderInt, clue.data.Input[Int]] = monocle.macros.GenLens[WhereOrderInt](_.LT)
      val GTE: monocle.Lens[WhereOrderInt, clue.data.Input[Int]] = monocle.macros.GenLens[WhereOrderInt](_.GTE)
      val LTE: monocle.Lens[WhereOrderInt, clue.data.Input[Int]] = monocle.macros.GenLens[WhereOrderInt](_.LTE)
      implicit val eqWhereOrderInt: cats.Eq[WhereOrderInt] = cats.Eq.fromUniversalEquals
      implicit val showWhereOrderInt: cats.Show[WhereOrderInt] = cats.Show.fromToString
      implicit val jsonEncoderWhereOrderInt: io.circe.Encoder.AsObject[WhereOrderInt] = io.circe.generic.semiauto.deriveEncoder[WhereOrderInt].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class WhereOrderObsActiveStatus(val EQ: clue.data.Input[ObsActiveStatus] = clue.data.Ignore, val NEQ: clue.data.Input[ObsActiveStatus] = clue.data.Ignore, val IN: clue.data.Input[List[ObsActiveStatus]] = clue.data.Ignore, val NIN: clue.data.Input[List[ObsActiveStatus]] = clue.data.Ignore, val GT: clue.data.Input[ObsActiveStatus] = clue.data.Ignore, val LT: clue.data.Input[ObsActiveStatus] = clue.data.Ignore, val GTE: clue.data.Input[ObsActiveStatus] = clue.data.Ignore, val LTE: clue.data.Input[ObsActiveStatus] = clue.data.Ignore)
    object WhereOrderObsActiveStatus {
      val EQ: monocle.Lens[WhereOrderObsActiveStatus, clue.data.Input[ObsActiveStatus]] = monocle.macros.GenLens[WhereOrderObsActiveStatus](_.EQ)
      val NEQ: monocle.Lens[WhereOrderObsActiveStatus, clue.data.Input[ObsActiveStatus]] = monocle.macros.GenLens[WhereOrderObsActiveStatus](_.NEQ)
      val IN: monocle.Lens[WhereOrderObsActiveStatus, clue.data.Input[List[ObsActiveStatus]]] = monocle.macros.GenLens[WhereOrderObsActiveStatus](_.IN)
      val NIN: monocle.Lens[WhereOrderObsActiveStatus, clue.data.Input[List[ObsActiveStatus]]] = monocle.macros.GenLens[WhereOrderObsActiveStatus](_.NIN)
      val GT: monocle.Lens[WhereOrderObsActiveStatus, clue.data.Input[ObsActiveStatus]] = monocle.macros.GenLens[WhereOrderObsActiveStatus](_.GT)
      val LT: monocle.Lens[WhereOrderObsActiveStatus, clue.data.Input[ObsActiveStatus]] = monocle.macros.GenLens[WhereOrderObsActiveStatus](_.LT)
      val GTE: monocle.Lens[WhereOrderObsActiveStatus, clue.data.Input[ObsActiveStatus]] = monocle.macros.GenLens[WhereOrderObsActiveStatus](_.GTE)
      val LTE: monocle.Lens[WhereOrderObsActiveStatus, clue.data.Input[ObsActiveStatus]] = monocle.macros.GenLens[WhereOrderObsActiveStatus](_.LTE)
      implicit val eqWhereOrderObsActiveStatus: cats.Eq[WhereOrderObsActiveStatus] = cats.Eq.fromUniversalEquals
      implicit val showWhereOrderObsActiveStatus: cats.Show[WhereOrderObsActiveStatus] = cats.Show.fromToString
      implicit val jsonEncoderWhereOrderObsActiveStatus: io.circe.Encoder.AsObject[WhereOrderObsActiveStatus] = io.circe.generic.semiauto.deriveEncoder[WhereOrderObsActiveStatus].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class WhereOrderObsStatus(val EQ: clue.data.Input[ObsStatus] = clue.data.Ignore, val NEQ: clue.data.Input[ObsStatus] = clue.data.Ignore, val IN: clue.data.Input[List[ObsStatus]] = clue.data.Ignore, val NIN: clue.data.Input[List[ObsStatus]] = clue.data.Ignore, val GT: clue.data.Input[ObsStatus] = clue.data.Ignore, val LT: clue.data.Input[ObsStatus] = clue.data.Ignore, val GTE: clue.data.Input[ObsStatus] = clue.data.Ignore, val LTE: clue.data.Input[ObsStatus] = clue.data.Ignore)
    object WhereOrderObsStatus {
      val EQ: monocle.Lens[WhereOrderObsStatus, clue.data.Input[ObsStatus]] = monocle.macros.GenLens[WhereOrderObsStatus](_.EQ)
      val NEQ: monocle.Lens[WhereOrderObsStatus, clue.data.Input[ObsStatus]] = monocle.macros.GenLens[WhereOrderObsStatus](_.NEQ)
      val IN: monocle.Lens[WhereOrderObsStatus, clue.data.Input[List[ObsStatus]]] = monocle.macros.GenLens[WhereOrderObsStatus](_.IN)
      val NIN: monocle.Lens[WhereOrderObsStatus, clue.data.Input[List[ObsStatus]]] = monocle.macros.GenLens[WhereOrderObsStatus](_.NIN)
      val GT: monocle.Lens[WhereOrderObsStatus, clue.data.Input[ObsStatus]] = monocle.macros.GenLens[WhereOrderObsStatus](_.GT)
      val LT: monocle.Lens[WhereOrderObsStatus, clue.data.Input[ObsStatus]] = monocle.macros.GenLens[WhereOrderObsStatus](_.LT)
      val GTE: monocle.Lens[WhereOrderObsStatus, clue.data.Input[ObsStatus]] = monocle.macros.GenLens[WhereOrderObsStatus](_.GTE)
      val LTE: monocle.Lens[WhereOrderObsStatus, clue.data.Input[ObsStatus]] = monocle.macros.GenLens[WhereOrderObsStatus](_.LTE)
      implicit val eqWhereOrderObsStatus: cats.Eq[WhereOrderObsStatus] = cats.Eq.fromUniversalEquals
      implicit val showWhereOrderObsStatus: cats.Show[WhereOrderObsStatus] = cats.Show.fromToString
      implicit val jsonEncoderWhereOrderObsStatus: io.circe.Encoder.AsObject[WhereOrderObsStatus] = io.circe.generic.semiauto.deriveEncoder[WhereOrderObsStatus].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class WhereOrderObservationId(val EQ: clue.data.Input[ObservationId] = clue.data.Ignore, val NEQ: clue.data.Input[ObservationId] = clue.data.Ignore, val IN: clue.data.Input[List[ObservationId]] = clue.data.Ignore, val NIN: clue.data.Input[List[ObservationId]] = clue.data.Ignore, val GT: clue.data.Input[ObservationId] = clue.data.Ignore, val LT: clue.data.Input[ObservationId] = clue.data.Ignore, val GTE: clue.data.Input[ObservationId] = clue.data.Ignore, val LTE: clue.data.Input[ObservationId] = clue.data.Ignore)
    object WhereOrderObservationId {
      val EQ: monocle.Lens[WhereOrderObservationId, clue.data.Input[ObservationId]] = monocle.macros.GenLens[WhereOrderObservationId](_.EQ)
      val NEQ: monocle.Lens[WhereOrderObservationId, clue.data.Input[ObservationId]] = monocle.macros.GenLens[WhereOrderObservationId](_.NEQ)
      val IN: monocle.Lens[WhereOrderObservationId, clue.data.Input[List[ObservationId]]] = monocle.macros.GenLens[WhereOrderObservationId](_.IN)
      val NIN: monocle.Lens[WhereOrderObservationId, clue.data.Input[List[ObservationId]]] = monocle.macros.GenLens[WhereOrderObservationId](_.NIN)
      val GT: monocle.Lens[WhereOrderObservationId, clue.data.Input[ObservationId]] = monocle.macros.GenLens[WhereOrderObservationId](_.GT)
      val LT: monocle.Lens[WhereOrderObservationId, clue.data.Input[ObservationId]] = monocle.macros.GenLens[WhereOrderObservationId](_.LT)
      val GTE: monocle.Lens[WhereOrderObservationId, clue.data.Input[ObservationId]] = monocle.macros.GenLens[WhereOrderObservationId](_.GTE)
      val LTE: monocle.Lens[WhereOrderObservationId, clue.data.Input[ObservationId]] = monocle.macros.GenLens[WhereOrderObservationId](_.LTE)
      implicit val eqWhereOrderObservationId: cats.Eq[WhereOrderObservationId] = cats.Eq.fromUniversalEquals
      implicit val showWhereOrderObservationId: cats.Show[WhereOrderObservationId] = cats.Show.fromToString
      implicit val jsonEncoderWhereOrderObservationId: io.circe.Encoder.AsObject[WhereOrderObservationId] = io.circe.generic.semiauto.deriveEncoder[WhereOrderObservationId].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class WhereOrderProgramId(val EQ: clue.data.Input[ProgramId] = clue.data.Ignore, val NEQ: clue.data.Input[ProgramId] = clue.data.Ignore, val IN: clue.data.Input[List[ProgramId]] = clue.data.Ignore, val NIN: clue.data.Input[List[ProgramId]] = clue.data.Ignore, val GT: clue.data.Input[ProgramId] = clue.data.Ignore, val LT: clue.data.Input[ProgramId] = clue.data.Ignore, val GTE: clue.data.Input[ProgramId] = clue.data.Ignore, val LTE: clue.data.Input[ProgramId] = clue.data.Ignore)
    object WhereOrderProgramId {
      val EQ: monocle.Lens[WhereOrderProgramId, clue.data.Input[ProgramId]] = monocle.macros.GenLens[WhereOrderProgramId](_.EQ)
      val NEQ: monocle.Lens[WhereOrderProgramId, clue.data.Input[ProgramId]] = monocle.macros.GenLens[WhereOrderProgramId](_.NEQ)
      val IN: monocle.Lens[WhereOrderProgramId, clue.data.Input[List[ProgramId]]] = monocle.macros.GenLens[WhereOrderProgramId](_.IN)
      val NIN: monocle.Lens[WhereOrderProgramId, clue.data.Input[List[ProgramId]]] = monocle.macros.GenLens[WhereOrderProgramId](_.NIN)
      val GT: monocle.Lens[WhereOrderProgramId, clue.data.Input[ProgramId]] = monocle.macros.GenLens[WhereOrderProgramId](_.GT)
      val LT: monocle.Lens[WhereOrderProgramId, clue.data.Input[ProgramId]] = monocle.macros.GenLens[WhereOrderProgramId](_.LT)
      val GTE: monocle.Lens[WhereOrderProgramId, clue.data.Input[ProgramId]] = monocle.macros.GenLens[WhereOrderProgramId](_.GTE)
      val LTE: monocle.Lens[WhereOrderProgramId, clue.data.Input[ProgramId]] = monocle.macros.GenLens[WhereOrderProgramId](_.LTE)
      implicit val eqWhereOrderProgramId: cats.Eq[WhereOrderProgramId] = cats.Eq.fromUniversalEquals
      implicit val showWhereOrderProgramId: cats.Show[WhereOrderProgramId] = cats.Show.fromToString
      implicit val jsonEncoderWhereOrderProgramId: io.circe.Encoder.AsObject[WhereOrderProgramId] = io.circe.generic.semiauto.deriveEncoder[WhereOrderProgramId].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class WhereOrderPosInt(val EQ: clue.data.Input[PosInt] = clue.data.Ignore, val NEQ: clue.data.Input[PosInt] = clue.data.Ignore, val IN: clue.data.Input[List[PosInt]] = clue.data.Ignore, val NIN: clue.data.Input[List[PosInt]] = clue.data.Ignore, val GT: clue.data.Input[PosInt] = clue.data.Ignore, val LT: clue.data.Input[PosInt] = clue.data.Ignore, val GTE: clue.data.Input[PosInt] = clue.data.Ignore, val LTE: clue.data.Input[PosInt] = clue.data.Ignore)
    object WhereOrderPosInt {
      val EQ: monocle.Lens[WhereOrderPosInt, clue.data.Input[PosInt]] = monocle.macros.GenLens[WhereOrderPosInt](_.EQ)
      val NEQ: monocle.Lens[WhereOrderPosInt, clue.data.Input[PosInt]] = monocle.macros.GenLens[WhereOrderPosInt](_.NEQ)
      val IN: monocle.Lens[WhereOrderPosInt, clue.data.Input[List[PosInt]]] = monocle.macros.GenLens[WhereOrderPosInt](_.IN)
      val NIN: monocle.Lens[WhereOrderPosInt, clue.data.Input[List[PosInt]]] = monocle.macros.GenLens[WhereOrderPosInt](_.NIN)
      val GT: monocle.Lens[WhereOrderPosInt, clue.data.Input[PosInt]] = monocle.macros.GenLens[WhereOrderPosInt](_.GT)
      val LT: monocle.Lens[WhereOrderPosInt, clue.data.Input[PosInt]] = monocle.macros.GenLens[WhereOrderPosInt](_.LT)
      val GTE: monocle.Lens[WhereOrderPosInt, clue.data.Input[PosInt]] = monocle.macros.GenLens[WhereOrderPosInt](_.GTE)
      val LTE: monocle.Lens[WhereOrderPosInt, clue.data.Input[PosInt]] = monocle.macros.GenLens[WhereOrderPosInt](_.LTE)
      implicit val eqWhereOrderPosInt: cats.Eq[WhereOrderPosInt] = cats.Eq.fromUniversalEquals
      implicit val showWhereOrderPosInt: cats.Show[WhereOrderPosInt] = cats.Show.fromToString
      implicit val jsonEncoderWhereOrderPosInt: io.circe.Encoder.AsObject[WhereOrderPosInt] = io.circe.generic.semiauto.deriveEncoder[WhereOrderPosInt].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class WhereProposalReference(val IS_NULL: clue.data.Input[Boolean] = clue.data.Ignore, val label: clue.data.Input[WhereString] = clue.data.Ignore, val semester: clue.data.Input[WhereOrderSemester] = clue.data.Ignore, val semesterIndex: clue.data.Input[WhereOrderPosInt] = clue.data.Ignore)
    object WhereProposalReference {
      val IS_NULL: monocle.Lens[WhereProposalReference, clue.data.Input[Boolean]] = monocle.macros.GenLens[WhereProposalReference](_.IS_NULL)
      val label: monocle.Lens[WhereProposalReference, clue.data.Input[WhereString]] = monocle.macros.GenLens[WhereProposalReference](_.label)
      val semester: monocle.Lens[WhereProposalReference, clue.data.Input[WhereOrderSemester]] = monocle.macros.GenLens[WhereProposalReference](_.semester)
      val semesterIndex: monocle.Lens[WhereProposalReference, clue.data.Input[WhereOrderPosInt]] = monocle.macros.GenLens[WhereProposalReference](_.semesterIndex)
      implicit val eqWhereProposalReference: cats.Eq[WhereProposalReference] = cats.Eq.fromUniversalEquals
      implicit val showWhereProposalReference: cats.Show[WhereProposalReference] = cats.Show.fromToString
      implicit val jsonEncoderWhereProposalReference: io.circe.Encoder.AsObject[WhereProposalReference] = io.circe.generic.semiauto.deriveEncoder[WhereProposalReference].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class WhereOrderSemester(val EQ: clue.data.Input[Semester] = clue.data.Ignore, val NEQ: clue.data.Input[Semester] = clue.data.Ignore, val IN: clue.data.Input[List[Semester]] = clue.data.Ignore, val NIN: clue.data.Input[List[Semester]] = clue.data.Ignore, val GT: clue.data.Input[Semester] = clue.data.Ignore, val LT: clue.data.Input[Semester] = clue.data.Ignore, val GTE: clue.data.Input[Semester] = clue.data.Ignore, val LTE: clue.data.Input[Semester] = clue.data.Ignore)
    object WhereOrderSemester {
      val EQ: monocle.Lens[WhereOrderSemester, clue.data.Input[Semester]] = monocle.macros.GenLens[WhereOrderSemester](_.EQ)
      val NEQ: monocle.Lens[WhereOrderSemester, clue.data.Input[Semester]] = monocle.macros.GenLens[WhereOrderSemester](_.NEQ)
      val IN: monocle.Lens[WhereOrderSemester, clue.data.Input[List[Semester]]] = monocle.macros.GenLens[WhereOrderSemester](_.IN)
      val NIN: monocle.Lens[WhereOrderSemester, clue.data.Input[List[Semester]]] = monocle.macros.GenLens[WhereOrderSemester](_.NIN)
      val GT: monocle.Lens[WhereOrderSemester, clue.data.Input[Semester]] = monocle.macros.GenLens[WhereOrderSemester](_.GT)
      val LT: monocle.Lens[WhereOrderSemester, clue.data.Input[Semester]] = monocle.macros.GenLens[WhereOrderSemester](_.LT)
      val GTE: monocle.Lens[WhereOrderSemester, clue.data.Input[Semester]] = monocle.macros.GenLens[WhereOrderSemester](_.GTE)
      val LTE: monocle.Lens[WhereOrderSemester, clue.data.Input[Semester]] = monocle.macros.GenLens[WhereOrderSemester](_.LTE)
      implicit val eqWhereOrderSemester: cats.Eq[WhereOrderSemester] = cats.Eq.fromUniversalEquals
      implicit val showWhereOrderSemester: cats.Show[WhereOrderSemester] = cats.Show.fromToString
      implicit val jsonEncoderWhereOrderSemester: io.circe.Encoder.AsObject[WhereOrderSemester] = io.circe.generic.semiauto.deriveEncoder[WhereOrderSemester].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class WhereOrderSequenceCommand(val EQ: clue.data.Input[SequenceCommand] = clue.data.Ignore, val NEQ: clue.data.Input[SequenceCommand] = clue.data.Ignore, val IN: clue.data.Input[List[SequenceCommand]] = clue.data.Ignore, val NIN: clue.data.Input[List[SequenceCommand]] = clue.data.Ignore, val GT: clue.data.Input[SequenceCommand] = clue.data.Ignore, val LT: clue.data.Input[SequenceCommand] = clue.data.Ignore, val GTE: clue.data.Input[SequenceCommand] = clue.data.Ignore, val LTE: clue.data.Input[SequenceCommand] = clue.data.Ignore)
    object WhereOrderSequenceCommand {
      val EQ: monocle.Lens[WhereOrderSequenceCommand, clue.data.Input[SequenceCommand]] = monocle.macros.GenLens[WhereOrderSequenceCommand](_.EQ)
      val NEQ: monocle.Lens[WhereOrderSequenceCommand, clue.data.Input[SequenceCommand]] = monocle.macros.GenLens[WhereOrderSequenceCommand](_.NEQ)
      val IN: monocle.Lens[WhereOrderSequenceCommand, clue.data.Input[List[SequenceCommand]]] = monocle.macros.GenLens[WhereOrderSequenceCommand](_.IN)
      val NIN: monocle.Lens[WhereOrderSequenceCommand, clue.data.Input[List[SequenceCommand]]] = monocle.macros.GenLens[WhereOrderSequenceCommand](_.NIN)
      val GT: monocle.Lens[WhereOrderSequenceCommand, clue.data.Input[SequenceCommand]] = monocle.macros.GenLens[WhereOrderSequenceCommand](_.GT)
      val LT: monocle.Lens[WhereOrderSequenceCommand, clue.data.Input[SequenceCommand]] = monocle.macros.GenLens[WhereOrderSequenceCommand](_.LT)
      val GTE: monocle.Lens[WhereOrderSequenceCommand, clue.data.Input[SequenceCommand]] = monocle.macros.GenLens[WhereOrderSequenceCommand](_.GTE)
      val LTE: monocle.Lens[WhereOrderSequenceCommand, clue.data.Input[SequenceCommand]] = monocle.macros.GenLens[WhereOrderSequenceCommand](_.LTE)
      implicit val eqWhereOrderSequenceCommand: cats.Eq[WhereOrderSequenceCommand] = cats.Eq.fromUniversalEquals
      implicit val showWhereOrderSequenceCommand: cats.Show[WhereOrderSequenceCommand] = cats.Show.fromToString
      implicit val jsonEncoderWhereOrderSequenceCommand: io.circe.Encoder.AsObject[WhereOrderSequenceCommand] = io.circe.generic.semiauto.deriveEncoder[WhereOrderSequenceCommand].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class WhereOrderSequenceType(val EQ: clue.data.Input[SequenceType] = clue.data.Ignore, val NEQ: clue.data.Input[SequenceType] = clue.data.Ignore, val IN: clue.data.Input[List[SequenceType]] = clue.data.Ignore, val NIN: clue.data.Input[List[SequenceType]] = clue.data.Ignore, val GT: clue.data.Input[SequenceType] = clue.data.Ignore, val LT: clue.data.Input[SequenceType] = clue.data.Ignore, val GTE: clue.data.Input[SequenceType] = clue.data.Ignore, val LTE: clue.data.Input[SequenceType] = clue.data.Ignore)
    object WhereOrderSequenceType {
      val EQ: monocle.Lens[WhereOrderSequenceType, clue.data.Input[SequenceType]] = monocle.macros.GenLens[WhereOrderSequenceType](_.EQ)
      val NEQ: monocle.Lens[WhereOrderSequenceType, clue.data.Input[SequenceType]] = monocle.macros.GenLens[WhereOrderSequenceType](_.NEQ)
      val IN: monocle.Lens[WhereOrderSequenceType, clue.data.Input[List[SequenceType]]] = monocle.macros.GenLens[WhereOrderSequenceType](_.IN)
      val NIN: monocle.Lens[WhereOrderSequenceType, clue.data.Input[List[SequenceType]]] = monocle.macros.GenLens[WhereOrderSequenceType](_.NIN)
      val GT: monocle.Lens[WhereOrderSequenceType, clue.data.Input[SequenceType]] = monocle.macros.GenLens[WhereOrderSequenceType](_.GT)
      val LT: monocle.Lens[WhereOrderSequenceType, clue.data.Input[SequenceType]] = monocle.macros.GenLens[WhereOrderSequenceType](_.LT)
      val GTE: monocle.Lens[WhereOrderSequenceType, clue.data.Input[SequenceType]] = monocle.macros.GenLens[WhereOrderSequenceType](_.GTE)
      val LTE: monocle.Lens[WhereOrderSequenceType, clue.data.Input[SequenceType]] = monocle.macros.GenLens[WhereOrderSequenceType](_.LTE)
      implicit val eqWhereOrderSequenceType: cats.Eq[WhereOrderSequenceType] = cats.Eq.fromUniversalEquals
      implicit val showWhereOrderSequenceType: cats.Show[WhereOrderSequenceType] = cats.Show.fromToString
      implicit val jsonEncoderWhereOrderSequenceType: io.circe.Encoder.AsObject[WhereOrderSequenceType] = io.circe.generic.semiauto.deriveEncoder[WhereOrderSequenceType].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class WhereOrderStepStage(val EQ: clue.data.Input[StepStage] = clue.data.Ignore, val NEQ: clue.data.Input[StepStage] = clue.data.Ignore, val IN: clue.data.Input[List[StepStage]] = clue.data.Ignore, val NIN: clue.data.Input[List[StepStage]] = clue.data.Ignore, val GT: clue.data.Input[StepStage] = clue.data.Ignore, val LT: clue.data.Input[StepStage] = clue.data.Ignore, val GTE: clue.data.Input[StepStage] = clue.data.Ignore, val LTE: clue.data.Input[StepStage] = clue.data.Ignore)
    object WhereOrderStepStage {
      val EQ: monocle.Lens[WhereOrderStepStage, clue.data.Input[StepStage]] = monocle.macros.GenLens[WhereOrderStepStage](_.EQ)
      val NEQ: monocle.Lens[WhereOrderStepStage, clue.data.Input[StepStage]] = monocle.macros.GenLens[WhereOrderStepStage](_.NEQ)
      val IN: monocle.Lens[WhereOrderStepStage, clue.data.Input[List[StepStage]]] = monocle.macros.GenLens[WhereOrderStepStage](_.IN)
      val NIN: monocle.Lens[WhereOrderStepStage, clue.data.Input[List[StepStage]]] = monocle.macros.GenLens[WhereOrderStepStage](_.NIN)
      val GT: monocle.Lens[WhereOrderStepStage, clue.data.Input[StepStage]] = monocle.macros.GenLens[WhereOrderStepStage](_.GT)
      val LT: monocle.Lens[WhereOrderStepStage, clue.data.Input[StepStage]] = monocle.macros.GenLens[WhereOrderStepStage](_.LT)
      val GTE: monocle.Lens[WhereOrderStepStage, clue.data.Input[StepStage]] = monocle.macros.GenLens[WhereOrderStepStage](_.GTE)
      val LTE: monocle.Lens[WhereOrderStepStage, clue.data.Input[StepStage]] = monocle.macros.GenLens[WhereOrderStepStage](_.LTE)
      implicit val eqWhereOrderStepStage: cats.Eq[WhereOrderStepStage] = cats.Eq.fromUniversalEquals
      implicit val showWhereOrderStepStage: cats.Show[WhereOrderStepStage] = cats.Show.fromToString
      implicit val jsonEncoderWhereOrderStepStage: io.circe.Encoder.AsObject[WhereOrderStepStage] = io.circe.generic.semiauto.deriveEncoder[WhereOrderStepStage].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class WhereOrderTargetId(val EQ: clue.data.Input[TargetId] = clue.data.Ignore, val NEQ: clue.data.Input[TargetId] = clue.data.Ignore, val IN: clue.data.Input[List[TargetId]] = clue.data.Ignore, val NIN: clue.data.Input[List[TargetId]] = clue.data.Ignore, val GT: clue.data.Input[TargetId] = clue.data.Ignore, val LT: clue.data.Input[TargetId] = clue.data.Ignore, val GTE: clue.data.Input[TargetId] = clue.data.Ignore, val LTE: clue.data.Input[TargetId] = clue.data.Ignore)
    object WhereOrderTargetId {
      val EQ: monocle.Lens[WhereOrderTargetId, clue.data.Input[TargetId]] = monocle.macros.GenLens[WhereOrderTargetId](_.EQ)
      val NEQ: monocle.Lens[WhereOrderTargetId, clue.data.Input[TargetId]] = monocle.macros.GenLens[WhereOrderTargetId](_.NEQ)
      val IN: monocle.Lens[WhereOrderTargetId, clue.data.Input[List[TargetId]]] = monocle.macros.GenLens[WhereOrderTargetId](_.IN)
      val NIN: monocle.Lens[WhereOrderTargetId, clue.data.Input[List[TargetId]]] = monocle.macros.GenLens[WhereOrderTargetId](_.NIN)
      val GT: monocle.Lens[WhereOrderTargetId, clue.data.Input[TargetId]] = monocle.macros.GenLens[WhereOrderTargetId](_.GT)
      val LT: monocle.Lens[WhereOrderTargetId, clue.data.Input[TargetId]] = monocle.macros.GenLens[WhereOrderTargetId](_.LT)
      val GTE: monocle.Lens[WhereOrderTargetId, clue.data.Input[TargetId]] = monocle.macros.GenLens[WhereOrderTargetId](_.GTE)
      val LTE: monocle.Lens[WhereOrderTargetId, clue.data.Input[TargetId]] = monocle.macros.GenLens[WhereOrderTargetId](_.LTE)
      implicit val eqWhereOrderTargetId: cats.Eq[WhereOrderTargetId] = cats.Eq.fromUniversalEquals
      implicit val showWhereOrderTargetId: cats.Show[WhereOrderTargetId] = cats.Show.fromToString
      implicit val jsonEncoderWhereOrderTargetId: io.circe.Encoder.AsObject[WhereOrderTargetId] = io.circe.generic.semiauto.deriveEncoder[WhereOrderTargetId].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class WhereProgram(val AND: clue.data.Input[List[WhereProgram]] = clue.data.Ignore, val OR: clue.data.Input[List[WhereProgram]] = clue.data.Ignore, val NOT: clue.data.Input[WhereProgram] = clue.data.Ignore, val id: clue.data.Input[WhereOrderProgramId] = clue.data.Ignore, val name: clue.data.Input[WhereOptionString] = clue.data.Ignore, val `type`: clue.data.Input[WhereEqProgramType] = clue.data.Ignore, val reference: clue.data.Input[WhereProgramReference] = clue.data.Ignore, val proposalStatus: clue.data.Input[WhereEqProposalStatus] = clue.data.Ignore, val proposal: clue.data.Input[WhereProposal] = clue.data.Ignore)
    object WhereProgram {
      val AND: monocle.Lens[WhereProgram, clue.data.Input[List[WhereProgram]]] = monocle.macros.GenLens[WhereProgram](_.AND)
      val OR: monocle.Lens[WhereProgram, clue.data.Input[List[WhereProgram]]] = monocle.macros.GenLens[WhereProgram](_.OR)
      val NOT: monocle.Lens[WhereProgram, clue.data.Input[WhereProgram]] = monocle.macros.GenLens[WhereProgram](_.NOT)
      val id: monocle.Lens[WhereProgram, clue.data.Input[WhereOrderProgramId]] = monocle.macros.GenLens[WhereProgram](_.id)
      val name: monocle.Lens[WhereProgram, clue.data.Input[WhereOptionString]] = monocle.macros.GenLens[WhereProgram](_.name)
      val `type`: monocle.Lens[WhereProgram, clue.data.Input[WhereEqProgramType]] = monocle.macros.GenLens[WhereProgram](_.`type`)
      val reference: monocle.Lens[WhereProgram, clue.data.Input[WhereProgramReference]] = monocle.macros.GenLens[WhereProgram](_.reference)
      val proposalStatus: monocle.Lens[WhereProgram, clue.data.Input[WhereEqProposalStatus]] = monocle.macros.GenLens[WhereProgram](_.proposalStatus)
      val proposal: monocle.Lens[WhereProgram, clue.data.Input[WhereProposal]] = monocle.macros.GenLens[WhereProgram](_.proposal)
      implicit val eqWhereProgram: cats.Eq[WhereProgram] = cats.Eq.fromUniversalEquals
      implicit val showWhereProgram: cats.Show[WhereProgram] = cats.Show.fromToString
      implicit val jsonEncoderWhereProgram: io.circe.Encoder.AsObject[WhereProgram] = io.circe.generic.semiauto.deriveEncoder[WhereProgram].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class WhereProgramReference(val IS_NULL: clue.data.Input[Boolean] = clue.data.Ignore, val label: clue.data.Input[WhereString] = clue.data.Ignore, val semester: clue.data.Input[WhereOrderSemester] = clue.data.Ignore, val semesterIndex: clue.data.Input[WhereOrderPosInt] = clue.data.Ignore, val instrument: clue.data.Input[WhereEqInstrument] = clue.data.Ignore, val description: clue.data.Input[WhereString] = clue.data.Ignore, val scienceSubtype: clue.data.Input[WhereEqScienceSubtype] = clue.data.Ignore)
    object WhereProgramReference {
      val IS_NULL: monocle.Lens[WhereProgramReference, clue.data.Input[Boolean]] = monocle.macros.GenLens[WhereProgramReference](_.IS_NULL)
      val label: monocle.Lens[WhereProgramReference, clue.data.Input[WhereString]] = monocle.macros.GenLens[WhereProgramReference](_.label)
      val semester: monocle.Lens[WhereProgramReference, clue.data.Input[WhereOrderSemester]] = monocle.macros.GenLens[WhereProgramReference](_.semester)
      val semesterIndex: monocle.Lens[WhereProgramReference, clue.data.Input[WhereOrderPosInt]] = monocle.macros.GenLens[WhereProgramReference](_.semesterIndex)
      val instrument: monocle.Lens[WhereProgramReference, clue.data.Input[WhereEqInstrument]] = monocle.macros.GenLens[WhereProgramReference](_.instrument)
      val description: monocle.Lens[WhereProgramReference, clue.data.Input[WhereString]] = monocle.macros.GenLens[WhereProgramReference](_.description)
      val scienceSubtype: monocle.Lens[WhereProgramReference, clue.data.Input[WhereEqScienceSubtype]] = monocle.macros.GenLens[WhereProgramReference](_.scienceSubtype)
      implicit val eqWhereProgramReference: cats.Eq[WhereProgramReference] = cats.Eq.fromUniversalEquals
      implicit val showWhereProgramReference: cats.Show[WhereProgramReference] = cats.Show.fromToString
      implicit val jsonEncoderWhereProgramReference: io.circe.Encoder.AsObject[WhereProgramReference] = io.circe.generic.semiauto.deriveEncoder[WhereProgramReference].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class WhereProposal(val IS_NULL: clue.data.Input[Boolean] = clue.data.Ignore, val AND: clue.data.Input[List[WhereProposal]] = clue.data.Ignore, val OR: clue.data.Input[List[WhereProposal]] = clue.data.Ignore, val NOT: clue.data.Input[WhereProposal] = clue.data.Ignore, val title: clue.data.Input[WhereOptionString] = clue.data.Ignore, val reference: clue.data.Input[WhereProposalReference] = clue.data.Ignore)
    object WhereProposal {
      val IS_NULL: monocle.Lens[WhereProposal, clue.data.Input[Boolean]] = monocle.macros.GenLens[WhereProposal](_.IS_NULL)
      val AND: monocle.Lens[WhereProposal, clue.data.Input[List[WhereProposal]]] = monocle.macros.GenLens[WhereProposal](_.AND)
      val OR: monocle.Lens[WhereProposal, clue.data.Input[List[WhereProposal]]] = monocle.macros.GenLens[WhereProposal](_.OR)
      val NOT: monocle.Lens[WhereProposal, clue.data.Input[WhereProposal]] = monocle.macros.GenLens[WhereProposal](_.NOT)
      val title: monocle.Lens[WhereProposal, clue.data.Input[WhereOptionString]] = monocle.macros.GenLens[WhereProposal](_.title)
      val reference: monocle.Lens[WhereProposal, clue.data.Input[WhereProposalReference]] = monocle.macros.GenLens[WhereProposal](_.reference)
      implicit val eqWhereProposal: cats.Eq[WhereProposal] = cats.Eq.fromUniversalEquals
      implicit val showWhereProposal: cats.Show[WhereProposal] = cats.Show.fromToString
      implicit val jsonEncoderWhereProposal: io.circe.Encoder.AsObject[WhereProposal] = io.circe.generic.semiauto.deriveEncoder[WhereProposal].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class WhereProposalClass(val `type`: clue.data.Input[WhereEqProposalClassType] = clue.data.Ignore, val minPercent: clue.data.Input[WhereOrderInt] = clue.data.Ignore)
    object WhereProposalClass {
      val `type`: monocle.Lens[WhereProposalClass, clue.data.Input[WhereEqProposalClassType]] = monocle.macros.GenLens[WhereProposalClass](_.`type`)
      val minPercent: monocle.Lens[WhereProposalClass, clue.data.Input[WhereOrderInt]] = monocle.macros.GenLens[WhereProposalClass](_.minPercent)
      implicit val eqWhereProposalClass: cats.Eq[WhereProposalClass] = cats.Eq.fromUniversalEquals
      implicit val showWhereProposalClass: cats.Show[WhereProposalClass] = cats.Show.fromToString
      implicit val jsonEncoderWhereProposalClass: io.circe.Encoder.AsObject[WhereProposalClass] = io.circe.generic.semiauto.deriveEncoder[WhereProposalClass].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class WhereProposalPartnerEntry(val AND: clue.data.Input[List[WhereProposalPartnerEntry]] = clue.data.Ignore, val OR: clue.data.Input[List[WhereProposalPartnerEntry]] = clue.data.Ignore, val NOT: clue.data.Input[WhereProposalPartnerEntry] = clue.data.Ignore, val partner: clue.data.Input[WhereEqPartner] = clue.data.Ignore, val percent: clue.data.Input[WhereOrderInt] = clue.data.Ignore)
    object WhereProposalPartnerEntry {
      val AND: monocle.Lens[WhereProposalPartnerEntry, clue.data.Input[List[WhereProposalPartnerEntry]]] = monocle.macros.GenLens[WhereProposalPartnerEntry](_.AND)
      val OR: monocle.Lens[WhereProposalPartnerEntry, clue.data.Input[List[WhereProposalPartnerEntry]]] = monocle.macros.GenLens[WhereProposalPartnerEntry](_.OR)
      val NOT: monocle.Lens[WhereProposalPartnerEntry, clue.data.Input[WhereProposalPartnerEntry]] = monocle.macros.GenLens[WhereProposalPartnerEntry](_.NOT)
      val partner: monocle.Lens[WhereProposalPartnerEntry, clue.data.Input[WhereEqPartner]] = monocle.macros.GenLens[WhereProposalPartnerEntry](_.partner)
      val percent: monocle.Lens[WhereProposalPartnerEntry, clue.data.Input[WhereOrderInt]] = monocle.macros.GenLens[WhereProposalPartnerEntry](_.percent)
      implicit val eqWhereProposalPartnerEntry: cats.Eq[WhereProposalPartnerEntry] = cats.Eq.fromUniversalEquals
      implicit val showWhereProposalPartnerEntry: cats.Show[WhereProposalPartnerEntry] = cats.Show.fromToString
      implicit val jsonEncoderWhereProposalPartnerEntry: io.circe.Encoder.AsObject[WhereProposalPartnerEntry] = io.circe.generic.semiauto.deriveEncoder[WhereProposalPartnerEntry].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class WhereProposalPartners(val MATCH: clue.data.Input[WhereProposalPartnerEntry] = clue.data.Ignore, val EQ: clue.data.Input[List[Partner]] = clue.data.Ignore, val isJoint: clue.data.Input[Boolean] = clue.data.Ignore)
    object WhereProposalPartners {
      val MATCH: monocle.Lens[WhereProposalPartners, clue.data.Input[WhereProposalPartnerEntry]] = monocle.macros.GenLens[WhereProposalPartners](_.MATCH)
      val EQ: monocle.Lens[WhereProposalPartners, clue.data.Input[List[Partner]]] = monocle.macros.GenLens[WhereProposalPartners](_.EQ)
      val isJoint: monocle.Lens[WhereProposalPartners, clue.data.Input[Boolean]] = monocle.macros.GenLens[WhereProposalPartners](_.isJoint)
      implicit val eqWhereProposalPartners: cats.Eq[WhereProposalPartners] = cats.Eq.fromUniversalEquals
      implicit val showWhereProposalPartners: cats.Show[WhereProposalPartners] = cats.Show.fromToString
      implicit val jsonEncoderWhereProposalPartners: io.circe.Encoder.AsObject[WhereProposalPartners] = io.circe.generic.semiauto.deriveEncoder[WhereProposalPartners].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class WhereString(val EQ: clue.data.Input[NonEmptyString] = clue.data.Ignore, val NEQ: clue.data.Input[NonEmptyString] = clue.data.Ignore, val IN: clue.data.Input[List[NonEmptyString]] = clue.data.Ignore, val NIN: clue.data.Input[List[NonEmptyString]] = clue.data.Ignore, val LIKE: clue.data.Input[NonEmptyString] = clue.data.Ignore, val NLIKE: clue.data.Input[NonEmptyString] = clue.data.Ignore, val MATCH_CASE: clue.data.Input[Boolean] = clue.data.Ignore)
    object WhereString {
      val EQ: monocle.Lens[WhereString, clue.data.Input[NonEmptyString]] = monocle.macros.GenLens[WhereString](_.EQ)
      val NEQ: monocle.Lens[WhereString, clue.data.Input[NonEmptyString]] = monocle.macros.GenLens[WhereString](_.NEQ)
      val IN: monocle.Lens[WhereString, clue.data.Input[List[NonEmptyString]]] = monocle.macros.GenLens[WhereString](_.IN)
      val NIN: monocle.Lens[WhereString, clue.data.Input[List[NonEmptyString]]] = monocle.macros.GenLens[WhereString](_.NIN)
      val LIKE: monocle.Lens[WhereString, clue.data.Input[NonEmptyString]] = monocle.macros.GenLens[WhereString](_.LIKE)
      val NLIKE: monocle.Lens[WhereString, clue.data.Input[NonEmptyString]] = monocle.macros.GenLens[WhereString](_.NLIKE)
      val MATCH_CASE: monocle.Lens[WhereString, clue.data.Input[Boolean]] = monocle.macros.GenLens[WhereString](_.MATCH_CASE)
      implicit val eqWhereString: cats.Eq[WhereString] = cats.Eq.fromUniversalEquals
      implicit val showWhereString: cats.Show[WhereString] = cats.Show.fromToString
      implicit val jsonEncoderWhereString: io.circe.Encoder.AsObject[WhereString] = io.circe.generic.semiauto.deriveEncoder[WhereString].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class WhereTarget(val AND: clue.data.Input[List[WhereTarget]] = clue.data.Ignore, val OR: clue.data.Input[List[WhereTarget]] = clue.data.Ignore, val NOT: clue.data.Input[WhereTarget] = clue.data.Ignore, val id: clue.data.Input[WhereOrderTargetId] = clue.data.Ignore, val program: clue.data.Input[WhereProgram] = clue.data.Ignore, val name: clue.data.Input[WhereString] = clue.data.Ignore)
    object WhereTarget {
      val AND: monocle.Lens[WhereTarget, clue.data.Input[List[WhereTarget]]] = monocle.macros.GenLens[WhereTarget](_.AND)
      val OR: monocle.Lens[WhereTarget, clue.data.Input[List[WhereTarget]]] = monocle.macros.GenLens[WhereTarget](_.OR)
      val NOT: monocle.Lens[WhereTarget, clue.data.Input[WhereTarget]] = monocle.macros.GenLens[WhereTarget](_.NOT)
      val id: monocle.Lens[WhereTarget, clue.data.Input[WhereOrderTargetId]] = monocle.macros.GenLens[WhereTarget](_.id)
      val program: monocle.Lens[WhereTarget, clue.data.Input[WhereProgram]] = monocle.macros.GenLens[WhereTarget](_.program)
      val name: monocle.Lens[WhereTarget, clue.data.Input[WhereString]] = monocle.macros.GenLens[WhereTarget](_.name)
      implicit val eqWhereTarget: cats.Eq[WhereTarget] = cats.Eq.fromUniversalEquals
      implicit val showWhereTarget: cats.Show[WhereTarget] = cats.Show.fromToString
      implicit val jsonEncoderWhereTarget: io.circe.Encoder.AsObject[WhereTarget] = io.circe.generic.semiauto.deriveEncoder[WhereTarget].mapJsonObject(clue.data.Input.dropIgnores)
    }
  }
}
