// Copyright (c) 2016-2023 Association of Universities for Research in Astronomy, Inc. (AURA)
// For license information see LICENSE or https://opensource.org/licenses/BSD-3-Clause

package lucuma.schemas.odb

import clue.GraphQLOperation
import lucuma.core.model.sequence.*
import lucuma.schemas.ObservationDB
import lucuma.odb.json.sequence.given

object SequenceQueriesGQL:
  
  object SequenceQuery extends GraphQLOperation[ObservationDB] {
    import ObservationDB.Scalars._
    ignoreUnusedImportScalars()
    import ObservationDB.Enums._
    ignoreUnusedImportEnums()
    import ObservationDB.Types._
    ignoreUnusedImportTypes()
    val document = s"""
          query($$obsId: ObservationId!) {
            observation(observationId: $$obsId) {
              execution {
                config(futureLimit: 100) {
                  instrument
                  gmosNorth {
                    static {
                      stageMode
                      detector
                      mosPreImaging
                      nodAndShuffle {
                        ...nodAndShuffleFields
                      }
                    }
                    acquisition {
                      ...gmosNorthSequenceFields
                    }
                    science {
                      ...gmosNorthSequenceFields
                    }
                  }
                  gmosSouth {
                    static {
                      stageMode
                      detector
                      mosPreImaging
                      nodAndShuffle {
                        ...nodAndShuffleFields
                      }
                    }
                    acquisition {
                      ...gmosSouthSequenceFields
                    }
                    science {
                      ...gmosSouthSequenceFields
                    }
                  }
                }
              }
            }
          }
  
          fragment nodAndShuffleFields on GmosNodAndShuffle {
            posA { ...offsetFields }
            posB { ...offsetFields }
            eOffset
            shuffleOffset
            shuffleCycles
          }
  
          fragment stepConfigFields on StepConfig {
            stepType
            ... on Gcal {
              continuum
              arcs
              filter
              diffuser
              shutter
            }
            ... on Science {
              offset { ...offsetFields }
              guiding
            }
            ... on SmartGcal {
              smartGcalType
            }
          }
  
          fragment stepEstimateFields on StepEstimate {
            configChange {
              all {
                name
                description
                estimate { microseconds }
              }
              index
            }
            detector {
              all {
                name
                description
                dataset {
                  exposure { microseconds }
                  readout { microseconds }
                  write { microseconds }
                }
                count
              }
              index
            }
          }
  
          fragment gmosNorthAtomFields on GmosNorthAtom {
            id
            description
            steps {
              id
              instrumentConfig {
                exposure { microseconds }
                readout {
                  xBin
                  yBin
                  ampCount
                  ampGain
                  ampReadMode
                }
                dtax
                roi
                gratingConfig {
                  grating
                  order
                  wavelength { picometers }
                }
                filter
                fpu {
                  builtin
                }
              }
              stepConfig {
                ...stepConfigFields
              }
              estimate {
                ...stepEstimateFields
              }
              observeClass
              breakpoint
            }
          }
  
          fragment gmosNorthSequenceFields on GmosNorthExecutionSequence {
            nextAtom {
              ...gmosNorthAtomFields
            }
            possibleFuture {
              ...gmosNorthAtomFields
            }
            hasMore
          }
  
          fragment gmosSouthAtomFields on GmosSouthAtom {
            id
            description
            steps {
              id
              instrumentConfig {
                exposure { microseconds }
                readout {
                  xBin
                  yBin
                  ampCount
                  ampGain
                  ampReadMode
                }
                dtax
                roi
                gratingConfig {
                  grating
                  order
                  wavelength { picometers }
                }
                filter
                fpu {
                  builtin
                }
              }
              stepConfig {
                ...stepConfigFields
              }
              estimate {
                ...stepEstimateFields
              }
              observeClass
              breakpoint
            }
          }
  
          fragment gmosSouthSequenceFields on GmosSouthExecutionSequence {
            nextAtom {
              ...gmosSouthAtomFields
            }
            possibleFuture {
              ...gmosSouthAtomFields
            }
            hasMore
          }
  
          fragment offsetFields on Offset {
            p { microarcseconds }
            q { microarcseconds }
          }
        """
    object Data {
      object Observation {
        object Execution {
          type Config = InstrumentExecutionConfig
          val config: monocle.Lens[Data.Observation.Execution, Option[Data.Observation.Execution.Config]] = monocle.macros.GenLens[Data.Observation.Execution](_.config)
          implicit val eqExecution: cats.Eq[Data.Observation.Execution] = cats.Eq.fromUniversalEquals
          implicit val showExecution: cats.Show[Data.Observation.Execution] = cats.Show.fromToString
          implicit val jsonDecoderExecution: io.circe.Decoder[Data.Observation.Execution] = io.circe.generic.semiauto.deriveDecoder[Data.Observation.Execution]
        }
        case class Execution(val config: Option[Data.Observation.Execution.Config] = None)
        val execution: monocle.Lens[Data.Observation, Data.Observation.Execution] = monocle.macros.GenLens[Data.Observation](_.execution)
        implicit val eqObservation: cats.Eq[Data.Observation] = cats.Eq.fromUniversalEquals
        implicit val showObservation: cats.Show[Data.Observation] = cats.Show.fromToString
        implicit val jsonDecoderObservation: io.circe.Decoder[Data.Observation] = io.circe.generic.semiauto.deriveDecoder[Data.Observation]
      }
      case class Observation(val execution: Data.Observation.Execution)
      val observation: monocle.Lens[Data, Option[Data.Observation]] = monocle.macros.GenLens[Data](_.observation)
      implicit val eqData: cats.Eq[Data] = cats.Eq.fromUniversalEquals
      implicit val showData: cats.Show[Data] = cats.Show.fromToString
      implicit val jsonDecoderData: io.circe.Decoder[Data] = io.circe.generic.semiauto.deriveDecoder[Data]
    }
    case class Variables(val obsId: ObservationId)
    object Variables {
      val obsId: monocle.Lens[Variables, ObservationId] = monocle.macros.GenLens[Variables](_.obsId)
      implicit val eqVariables: cats.Eq[Variables] = cats.Eq.fromUniversalEquals
      implicit val showVariables: cats.Show[Variables] = cats.Show.fromToString
      implicit val jsonEncoderVariables: io.circe.Encoder.AsObject[Variables] = io.circe.generic.semiauto.deriveEncoder[Variables].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class Data(val observation: Option[Data.Observation] = None)
    val varEncoder: io.circe.Encoder.AsObject[Variables] = Variables.jsonEncoderVariables
    val dataDecoder: io.circe.Decoder[Data] = Data.jsonDecoderData
    def apply[F[_]]: clue.ClientAppliedF[F, ObservationDB, ClientAppliedFP] = new clue.ClientAppliedF[F, ObservationDB, ClientAppliedFP] { def applyP[P](client: clue.FetchClientWithPars[F, P, ObservationDB]) = new ClientAppliedFP(client) }
    class ClientAppliedFP[F[_], P](val client: clue.FetchClientWithPars[F, P, ObservationDB]) { def query(obsId: ObservationId, modParams: P => P = identity)(implicit errorPolicy: clue.ErrorPolicy) = client.request(SequenceQuery).withInput(Variables(obsId), modParams) }
  }

  
  object DigestQuery extends GraphQLOperation[ObservationDB] {
    import ObservationDB.Scalars._
    ignoreUnusedImportScalars()
    import ObservationDB.Enums._
    ignoreUnusedImportEnums()
    import ObservationDB.Types._
    ignoreUnusedImportTypes()
    val document = s"""
          query($$obsId: ObservationId!) {
            observation(observationId: $$obsId) {
              execution {
                digest {
                  setup {
                    ...setupTimeFields
                  }
                  acquisition {
                    ...sequenceDigestFields
                  }
                  science {
                    ...sequenceDigestFields
                  }
                }
              }
            }
          }
  
          fragment setupTimeFields on SetupTime {
            full { microseconds }
            reacquisition { microseconds }
          }
  
          fragment sequenceDigestFields on SequenceDigest {
            observeClass
            timeEstimate {
              program { microseconds }
              partner { microseconds }
              nonCharged { microseconds }
            }
            offsets { ...offsetFields }
            atomCount
          }
  
          fragment offsetFields on Offset {
            p { microarcseconds }
            q { microarcseconds }
          }
        """
    object Data {
      object Observation {
        object Execution {
          type Digest = ExecutionDigest
          val digest: monocle.Lens[Data.Observation.Execution, Option[Data.Observation.Execution.Digest]] = monocle.macros.GenLens[Data.Observation.Execution](_.digest)
          implicit val eqExecution: cats.Eq[Data.Observation.Execution] = cats.Eq.fromUniversalEquals
          implicit val showExecution: cats.Show[Data.Observation.Execution] = cats.Show.fromToString
          implicit val jsonDecoderExecution: io.circe.Decoder[Data.Observation.Execution] = io.circe.generic.semiauto.deriveDecoder[Data.Observation.Execution]
        }
        case class Execution(val digest: Option[Data.Observation.Execution.Digest] = None)
        val execution: monocle.Lens[Data.Observation, Data.Observation.Execution] = monocle.macros.GenLens[Data.Observation](_.execution)
        implicit val eqObservation: cats.Eq[Data.Observation] = cats.Eq.fromUniversalEquals
        implicit val showObservation: cats.Show[Data.Observation] = cats.Show.fromToString
        implicit val jsonDecoderObservation: io.circe.Decoder[Data.Observation] = io.circe.generic.semiauto.deriveDecoder[Data.Observation]
      }
      case class Observation(val execution: Data.Observation.Execution)
      val observation: monocle.Lens[Data, Option[Data.Observation]] = monocle.macros.GenLens[Data](_.observation)
      implicit val eqData: cats.Eq[Data] = cats.Eq.fromUniversalEquals
      implicit val showData: cats.Show[Data] = cats.Show.fromToString
      implicit val jsonDecoderData: io.circe.Decoder[Data] = io.circe.generic.semiauto.deriveDecoder[Data]
    }
    case class Variables(val obsId: ObservationId)
    object Variables {
      val obsId: monocle.Lens[Variables, ObservationId] = monocle.macros.GenLens[Variables](_.obsId)
      implicit val eqVariables: cats.Eq[Variables] = cats.Eq.fromUniversalEquals
      implicit val showVariables: cats.Show[Variables] = cats.Show.fromToString
      implicit val jsonEncoderVariables: io.circe.Encoder.AsObject[Variables] = io.circe.generic.semiauto.deriveEncoder[Variables].mapJsonObject(clue.data.Input.dropIgnores)
    }
    case class Data(val observation: Option[Data.Observation] = None)
    val varEncoder: io.circe.Encoder.AsObject[Variables] = Variables.jsonEncoderVariables
    val dataDecoder: io.circe.Decoder[Data] = Data.jsonDecoderData
    def apply[F[_]]: clue.ClientAppliedF[F, ObservationDB, ClientAppliedFP] = new clue.ClientAppliedF[F, ObservationDB, ClientAppliedFP] { def applyP[P](client: clue.FetchClientWithPars[F, P, ObservationDB]) = new ClientAppliedFP(client) }
    class ClientAppliedFP[F[_], P](val client: clue.FetchClientWithPars[F, P, ObservationDB]) { def query(obsId: ObservationId, modParams: P => P = identity)(implicit errorPolicy: clue.ErrorPolicy) = client.request(DigestQuery).withInput(Variables(obsId), modParams) }
  }
