/*
 * Decompiled with CFR 0.152.
 */
package edu.kit.datamanager.security.filter;

import com.nimbusds.jose.proc.BadJOSEException;
import edu.kit.datamanager.entities.RepoUserRole;
import edu.kit.datamanager.exceptions.InvalidAuthenticationException;
import edu.kit.datamanager.security.filter.JwtAuthenticationToken;
import edu.kit.datamanager.security.filter.JwtServiceToken;
import edu.kit.datamanager.security.filter.KeycloakTokenValidator;
import edu.kit.datamanager.util.NetworkUtils;
import io.jsonwebtoken.Claims;
import io.jsonwebtoken.ExpiredJwtException;
import io.jsonwebtoken.Jws;
import io.jsonwebtoken.MalformedJwtException;
import jakarta.servlet.FilterChain;
import jakarta.servlet.ServletException;
import jakarta.servlet.ServletRequest;
import jakarta.servlet.ServletResponse;
import jakarta.servlet.http.HttpServletRequest;
import jakarta.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Set;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.security.web.authentication.WebAuthenticationDetailsSource;
import org.springframework.web.filter.OncePerRequestFilter;

public class KeycloakTokenFilter
extends OncePerRequestFilter {
    private static final Logger LOG = LoggerFactory.getLogger(KeycloakTokenFilter.class);
    private static final String BEARER = "Bearer ";
    public static final String AUTHORIZATION_HEADER = "Authorization";
    public static final String BASIC_AUTH_PREFIX = "BASIC";
    private final KeycloakTokenValidator tokenValidator;
    private JwtAuthenticationToken jwToken;

    public KeycloakTokenFilter(KeycloakTokenValidator tokenValidator) {
        this.tokenValidator = tokenValidator;
    }

    protected void doFilterInternal(HttpServletRequest request, HttpServletResponse response, FilterChain filterChain) throws ServletException, IOException {
        boolean contextSet;
        block10: {
            String token = request.getHeader(AUTHORIZATION_HEADER);
            contextSet = false;
            if (token != null && !token.toUpperCase().startsWith(BASIC_AUTH_PREFIX) && token.startsWith(BEARER)) {
                LOG.trace("Starting JWT filtering.");
                try {
                    if (token.length() > BEARER.length()) {
                        token = token.substring(BEARER.length());
                    }
                    boolean localAuthenticationSucceeded = false;
                    if (this.tokenValidator.supportsLocalJwt() && this.attemptLocalAuthentication(request, response, token)) {
                        LOG.trace("Authenticated using local JWT secret.");
                        localAuthenticationSucceeded = true;
                        contextSet = true;
                    }
                    if (localAuthenticationSucceeded) break block10;
                    LOG.trace("Validating received JWT.");
                    this.jwToken = this.tokenValidator.validate(token);
                    LOG.trace("JWT validation finished. Checking result.");
                    if (this.jwToken != null && SecurityContextHolder.getContext().getAuthentication() == null) {
                        LOG.trace("Authenticated username: {}", this.jwToken.getPrincipal());
                        if (this.jwToken.getAuthorities().stream().filter(a -> a.getAuthority().equals(RepoUserRole.INACTIVE.toString())).count() > 0L) {
                            LOG.debug("User roles contain ROLE_INACTIVE. Access denied for user.");
                            response.sendError(403, "Unauthorized: User is marked inactive.");
                            return;
                        }
                        this.setContext(request, this.jwToken);
                        contextSet = true;
                        break block10;
                    }
                    LOG.info("Invalid Request: Token is expired or tampered");
                    response.sendError(401, "Unauthorized: Token is expired or tampered");
                    return;
                }
                catch (BadJOSEException | MalformedJwtException | IOException e) {
                    LOG.error("Failed to validate JWT.", e);
                }
            } else {
                LOG.info("Bearer token not being sent in Headers.");
            }
        }
        if (!contextSet) {
            LOG.trace("Continue with filterChain as no valid JWT authentication was found.");
        } else {
            LOG.trace("Valid authentication context set from JWT. Continue with filterChain.");
        }
        filterChain.doFilter((ServletRequest)request, (ServletResponse)response);
    }

    private boolean attemptLocalAuthentication(HttpServletRequest request, HttpServletResponse response, String token) {
        boolean authenticated = false;
        try {
            Jws<Claims> claimsJws = this.tokenValidator.getJwsClaims(token);
            Set claims = ((Claims)claimsJws.getBody()).entrySet();
            HashMap<String, Object> claimMap = new HashMap<String, Object>();
            claims.forEach(entry -> claimMap.put((String)entry.getKey(), entry.getValue()));
            this.jwToken = JwtAuthenticationToken.factoryToken(token, claimMap);
            if (this.jwToken instanceof JwtServiceToken && ((JwtServiceToken)this.jwToken).getSources() != null) {
                JwtServiceToken serviceToken = (JwtServiceToken)this.jwToken;
                LOG.debug("Performing source check for JWToken for service {} and sources {}.", serviceToken.getPrincipal(), Arrays.asList(serviceToken.getSources()));
                String remoteAddr = request.getRemoteAddr();
                LOG.debug("Trying to match remote address {} with at least one allowed source.", (Object)remoteAddr);
                boolean matchFound = false;
                for (String source : serviceToken.getSources()) {
                    if (!NetworkUtils.matches(remoteAddr, source)) continue;
                    matchFound = true;
                    break;
                }
                if (!matchFound) {
                    LOG.warn("Invalid request from remote address {} to service {} found. Request denied.", (Object)remoteAddr, serviceToken.getPrincipal());
                    throw new InvalidAuthenticationException("You are not allowed to authenticate using the provided token from your current location.");
                }
            }
            LOG.trace("JWT validation finished. Checking result.");
            if (this.jwToken != null && SecurityContextHolder.getContext().getAuthentication() == null) {
                LOG.info("Authenticated username: {}", this.jwToken.getPrincipal());
                this.setContext(request, this.jwToken);
            } else {
                try {
                    LOG.info("Invalid Request: Token is expired or tampered");
                    response.sendError(401, "Unauthorized: Token is expired or tampered");
                }
                catch (IOException ex) {
                    LOG.error("Failed to send Unauthorized response back to client.", (Throwable)ex);
                }
            }
            authenticated = true;
        }
        catch (ExpiredJwtException ex) {
            LOG.debug("Provided token has expired. Refresh of login required.", (Throwable)ex);
            throw new InvalidAuthenticationException("Your token has expired. Please refresh your login.");
        }
        catch (MalformedJwtException ex) {
            LOG.debug("Provided token is malformed.", (Throwable)ex);
            throw new MalformedJwtException("Your token is malformed.");
        }
        catch (IllegalArgumentException ex) {
            LOG.debug("Illegal argument exception while local authentication attempt.", (Throwable)ex);
        }
        return authenticated;
    }

    private void setContext(HttpServletRequest request, JwtAuthenticationToken token) {
        LOG.trace("Setting authentication context.");
        token.setDetails(new WebAuthenticationDetailsSource().buildDetails(request));
        SecurityContextHolder.getContext().setAuthentication((Authentication)token);
        LOG.trace("Setting request attribute 'username' to {}.", token.getPrincipal());
        request.setAttribute("username", token.getPrincipal());
    }

    public boolean isValid() {
        return this.tokenValidator.isValid();
    }
}

