package edu.uiuc.ncsa.myproxy.oa4mp.oauth2.servlet;

import edu.uiuc.ncsa.myproxy.oa4mp.oauth2.OA2ServiceTransaction;
import edu.uiuc.ncsa.myproxy.oa4mp.server.servlet.AbstractInitServlet;
import edu.uiuc.ncsa.security.delegation.server.ServiceTransaction;
import edu.uiuc.ncsa.security.delegation.server.request.AGResponse;
import edu.uiuc.ncsa.security.delegation.server.request.IssuerResponse;
import edu.uiuc.ncsa.security.oauth_2_0.*;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.UnsupportedEncodingException;
import java.net.URI;
import java.util.ArrayList;
import java.util.Map;
import java.util.StringTokenizer;

import static edu.uiuc.ncsa.security.oauth_2_0.OA2Constants.*;

/**
 * <p>Created by Jeff Gaynor<br>
 * on 10/3/13 at  2:01 PM
 */
public class OA2AuthorizedServlet extends AbstractInitServlet {

    @Override
    protected void doIt(HttpServletRequest httpServletRequest, HttpServletResponse httpServletResponse) throws Throwable {
        String callback = httpServletRequest.getParameter(OA2Constants.REDIRECT_URI);
        if (httpServletRequest.getParameterMap().containsKey(OA2Constants.REQUEST_URI)) {
            throw new OA2Error(OA2Errors.REQUEST_URI_NOT_SUPPORTED,
                    "Request uri not supported by this server",
                    httpServletRequest.getParameter(OA2Constants.STATE),
                    callback);
        }
        if (httpServletRequest.getParameterMap().containsKey(OA2Constants.REQUEST)) {
            throw new OA2Error(OA2Errors.REQUEST_NOT_SUPPORTED,
                    "Request not supported by this server",
                    httpServletRequest.getParameter(OA2Constants.STATE),
                    callback);
        }

        if (!httpServletRequest.getParameterMap().containsKey(OA2Constants.RESPONSE_TYPE)) {
            throw new OA2Error(OA2Errors.INVALID_REQUEST,
                    "no response type",
                    httpServletRequest.getParameter(OA2Constants.STATE),
                    callback);
        }
        super.doIt(httpServletRequest, httpServletResponse);
    }

    @Override
    public ServiceTransaction verifyAndGet(IssuerResponse iResponse) throws UnsupportedEncodingException {
        AGResponse agResponse = (AGResponse) iResponse;
        Map<String, String> params = agResponse.getParameters();
        // Since the state (if present) has to be returned with any error message, we have to see if there is one
        // there first.
        String state = null;
        if (params.containsKey(STATE)) {
            state = params.get(STATE);
        }
        //Spec says that the redirect must match one of the ones stored and if not, the request is rejected.
        String givenRedirect = params.get(REDIRECT_URI);
        OA2ClientCheck.check(agResponse.getClient(), givenRedirect);

        // by this point it has been verified that the redirect uri is valid.

        String rawSecret = params.get(CLIENT_SECRET);
        if (rawSecret != null) {
            info("Client is sending secret in initial request. Though not forbidden by the protocol this is discouraged.");
        }
        String nonce = params.get(NONCE);
        // FIX for OAUTH-180. Server must support clients that do not use a nonce. Just log it and rock on.
        if(nonce == null || nonce.length()==0){
             info("No nonce in initial request for " + ((AGResponse) iResponse).getClient().getIdentifierString() );
        }else {
            NonceHerder.checkNonce(nonce);
        }
        if (params.containsKey(DISPLAY)) {
            if (!params.get(DISPLAY).equals(DISPLAY_PAGE)) {
                throw new OA2Error(OA2Errors.INVALID_REQUEST, "Only " + DISPLAY + "=" + DISPLAY_PAGE + " is supported", state, givenRedirect);
            }
        }


        String rawScopes = params.get(SCOPE);
        if (rawScopes == null || rawScopes.length() == 0) {
            throw new OA2Error(OA2Errors.INVALID_SCOPE, "Missing scopes parameter.", state, givenRedirect);
        }

        StringTokenizer stringTokenizer = new StringTokenizer(rawScopes);
        ArrayList<String> scopes = new ArrayList<>();
        boolean hasOpenIDScope = false;
        while (stringTokenizer.hasMoreTokens()) {
            String x = stringTokenizer.nextToken();
            if (!OA2Scopes.ScopeUtil.isScopeValid(x)) {
                throw new OA2Error(OA2Errors.INVALID_SCOPE, "Unrecognized scope", state, givenRedirect);
            }
            if (x.equals(OA2Scopes.SCOPE_OPENID)) hasOpenIDScope = true;
            scopes.add(x);
        }


        if (!hasOpenIDScope)
            throw new OA2Error(OA2Errors.INVALID_REQUEST, "Scopes must contain openid", state, givenRedirect);

        /*if (params.containsKey(OA2Constants.ID_TOKEN_HINT)) {
            JSONObject json = IDTokenUtil.readIDToken(params.get(OA2Constants.ID_TOKEN_HINT));
            json.getString(OA2Claims.)
        }*/

        OA2ServiceTransaction st = new OA2ServiceTransaction(agResponse.getGrant());
        st.setScopes(scopes);
        st.setAuthGrantValid(false);
        st.setAccessTokenValid(false);
        st.setCallback(URI.create(params.get(REDIRECT_URI)));
        // fine if the nonce is null or empty, just set what they sent.
        st.setNonce(nonce);
        // Store the callback the user needs to use for this request, since the spec allows for many.
        // and now check for a bunch of stuff that might fail.

        checkPrompts(params);
        if (params.containsKey(REQUEST)) {
            throw new OA2Error(OA2Errors.REQUEST_NOT_SUPPORTED, "The \"request\" parameter is not supported on this server", state, givenRedirect);
        }
        if (params.containsKey(REQUEST_URI)) {
            throw new OA2Error(OA2Errors.REQUEST_URI_NOT_SUPPORTED, "The \"request_uri\" parameter is not supported on this server", state, givenRedirect);
        }

        return st;
    }

    /**
     * Basically, if the prompt parameter is there, we only support the login option.
     *
     * @param map
     */
    protected void checkPrompts(Map<String, String> map) {
        if (!map.containsKey(PROMPT)) return;  //nix to do
        String prompts = map.get(PROMPT);
        // now we have tos ee what is in it.
        StringTokenizer st = new StringTokenizer(prompts);
        ArrayList<String> prompt = new ArrayList<>();

        while (st.hasMoreElements()) {
            prompt.add(st.nextToken());
        }
        // CIL-91 if prompt = none is passed in, return an error with login_required as the message.
        if (!prompt.contains(PROMPT_NONE) && prompt.size() == 0) {
            throw new OA2Error(OA2Errors.LOGIN_REQUIRED, "A login is required on this server", map.get(OA2Constants.STATE));
        }
        if (prompt.contains(PROMPT_NONE) && 1 < prompt.size()) {
            throw new OA2Error(OA2Errors.INVALID_REQUEST, "You cannot specify \"none\" for the prompt and any other option", map.get(OA2Constants.STATE));
        }

        if (prompt.contains(PROMPT_LOGIN)) return;

        // At this point there is neither a "none" or a "login" and we don's support anything else.

        throw new OA2Error(OA2Errors.LOGIN_REQUIRED, "You must specify \"login\" as an option", map.get(OA2Constants.STATE));


    }
}
