/*
 * Copyright (c) 2017 by Gerrit Grunwald
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package eu.hansolo.tilesfx;

import eu.hansolo.tilesfx.Tile.SkinType;
import eu.hansolo.tilesfx.Tile.TileColor;
import eu.hansolo.tilesfx.chart.ChartData;
import eu.hansolo.tilesfx.chart.ChartDataBuilder;
import eu.hansolo.tilesfx.colors.Bright;
import eu.hansolo.tilesfx.colors.Dark;
import eu.hansolo.tilesfx.tools.GradientLookup;
import eu.hansolo.tilesfx.tools.Helper;
import javafx.animation.AnimationTimer;
import javafx.application.Application;
import javafx.beans.property.DoubleProperty;
import javafx.beans.property.SimpleDoubleProperty;
import javafx.collections.ObservableList;
import javafx.geometry.Insets;
import javafx.scene.Node;
import javafx.scene.Parent;
import javafx.scene.Scene;
import javafx.scene.layout.StackPane;
import javafx.scene.paint.Color;
import javafx.scene.paint.Stop;
import javafx.scene.text.TextAlignment;
import javafx.stage.Stage;

import java.time.Instant;
import java.util.Arrays;
import java.util.Random;
import java.util.concurrent.TimeUnit;


/**
 * User: hansolo
 * Date: 13.10.17
 * Time: 14:52
 */
public class Test extends Application {
    private static final Random         RND       = new Random();
    private static       int            noOfNodes = 0;
    private              Tile           tile1;
    private              DoubleProperty value;
    private              long           lastTimerCall;

    private Tile           tile2;
    private GradientLookup gradientLookup = new GradientLookup(Arrays.asList(new Stop(0.0, Bright.GREEN),
                                                                             new Stop(0.4, Bright.YELLOW),
                                                                             new Stop(0.7, Bright.RED)));
    private ChartData      cpuItem        = ChartDataBuilder.create().name("CPU").animated(true).formatString("%.0f%%").textColor(Color.rgb(238, 238, 238)).gradientLookup(gradientLookup).build();
    private ChartData      memItem        = ChartDataBuilder.create().name("MEM").animated(true).formatString("%.0f%%").textColor(Color.rgb(238, 238, 238)).gradientLookup(gradientLookup).build();
    private ChartData      tempItem       = ChartDataBuilder.create().name("TEMP").animated(true).formatString("%.0f\u00b0C").textColor(Color.rgb(238, 238, 238)).maxValue(100).gradientLookup(gradientLookup).build();

    private long           lastCpuTimerCall;
    private long           lastMemTimerCall;
    private long           lastTempTimerCall;

    private              AnimationTimer timer;


    @Override public void init() {
        value = new SimpleDoubleProperty();
        tile1 = TileBuilder.create()
                           .skinType(SkinType.TIMELINE)
                           .prefSize(600, 600)
                           .title("Blood Sugar")
                           .unit("mg/dl")
                           .minValue(0)
                           .maxValue(350)
                           .smoothing(false)
                           .lowerThreshold(70)
                           .lowerThresholdColor(TileColor.RED.color)
                           .threshold(240)
                           .thresholdColor(TileColor.RED.color)
                           .thresholdVisible(true)
                           .tickLabelColor(Helper.getColorWithOpacity(Tile.FOREGROUND, 0.5))
                           .sections(new Section(0, 70, "Low", Helper.getColorWithOpacity(Dark.RED, 0.1)),
                                     new Section(70, 140, "Ok", Helper.getColorWithOpacity(Bright.GREEN, 0.15)),
                                     new Section(140, 350, "High", Helper.getColorWithOpacity(Dark.RED, 0.1)))
                           .sectionsVisible(true)
                           .textAlignment(TextAlignment.CENTER)
                           .timePeriod(java.time.Duration.ofSeconds(60))
                           .maxTimePeriod(java.time.Duration.ofSeconds(60))
                           .timePeriodResolution(TimeUnit.SECONDS)
                           .numberOfValuesForTrendCalculation(5)
                           .trendVisible(true)
                           .maxTimePeriod(java.time.Duration.ofSeconds(60))
                           .gradientStops(new Stop(0, Dark.RED),
                                          new Stop(0.15, Dark.RED),
                                          new Stop(0.2, Bright.YELLOW_ORANGE),
                                          new Stop(0.25, Bright.GREEN),
                                          new Stop(0.3, Bright.GREEN),
                                          new Stop(0.35, Bright.GREEN),
                                          new Stop(0.45, Bright.YELLOW_ORANGE),
                                          new Stop(0.5, Bright.ORANGE),
                                          new Stop(0.685, Dark.RED),
                                          new Stop(1.0, Dark.RED))
                           .strokeWithGradient(true)
                           .averagingEnabled(true)
                           .averageVisible(true)
                           .build();


        lastCpuTimerCall  = System.nanoTime();
        lastMemTimerCall  = System.nanoTime();
        lastTempTimerCall = System.nanoTime();
        tile2 = TileBuilder.create()
                           .skinType(SkinType.CLUSTER_MONITOR)
                           .prefSize(100, 100)
                           .unit("\u0025")
                           .chartData(memItem, cpuItem, tempItem)
                           .animated(false)
                           .title("Node")
                           .textVisible(true)
                           .text("Last Sync")
                           .fillWithGradient(false)
                           .build();

        lastTimerCall = System.nanoTime();
        timer = new AnimationTimer() {
            @Override public void handle(final long now) {
                if (now > lastTimerCall + 2_000_000_000l) {
                    tile1.addChartData(new ChartData("", RND.nextDouble() * 300 + 50, Instant.now()));
                    //tile1.setValue(RND.nextDouble() * 300 + 50);
                    lastTimerCall = now;
                }
                if (now > lastCpuTimerCall + 1_000_000_000l) {
                    double cpuValue = RND.nextDouble() * 100;
                    cpuItem.setFillColor(gradientLookup.getColorAt(cpuValue / 100.0));
                    cpuItem.setValue(cpuValue);
                    lastCpuTimerCall = now;
                }
                if (now > lastMemTimerCall + 1_250_000_000l) {
                    double memValue = RND.nextDouble() * 100;
                    memItem.setFillColor(gradientLookup.getColorAt(memValue / 100.0));
                    memItem.setValue(memValue);
                    lastMemTimerCall = now;
                }
                if (now > lastTempTimerCall + 1_300_000_000l) {
                    double tempValue = RND.nextDouble() * tempItem.getMaxValue();
                    tempItem.setFillColor(gradientLookup.getColorAt(tempValue / tempItem.getMaxValue()));
                    tempItem.setValue(tempValue);
                    lastTempTimerCall = now;
                }
            }
        };
    }

    @Override public void start(Stage stage) {
        StackPane pane = new StackPane(tile2);
        pane.setPadding(new Insets(10));

        Scene scene = new Scene(pane);

        stage.setTitle("TimelineTileSkin");
        stage.setScene(scene);
        stage.show();

        // Calculate number of nodes
        calcNoOfNodes(pane);
        System.out.println(noOfNodes + " Nodes in SceneGraph");

        timer.start();
    }

    @Override public void stop() {
        System.exit(0);
    }


    private void setData() {
        tile1.addChartData(new ChartData("", 92, Instant.ofEpochSecond(1567751401316l)));
        tile1.addChartData(new ChartData("", 89, Instant.ofEpochSecond(1567750894782l)));
        tile1.addChartData(new ChartData("", 88, Instant.ofEpochSecond(1567750602007l)));
        tile1.addChartData(new ChartData("", 87, Instant.ofEpochSecond(1567750295207l)));
        tile1.addChartData(new ChartData("", 87, Instant.ofEpochSecond(1567749994876l)));
        tile1.addChartData(new ChartData("", 90, Instant.ofEpochSecond(1567749694972l)));
        tile1.addChartData(new ChartData("", 89, Instant.ofEpochSecond(1567749394859l)));
        tile1.addChartData(new ChartData("", 94, Instant.ofEpochSecond(1567749094905l)));
        tile1.addChartData(new ChartData("", 105, Instant.ofEpochSecond(1567748794970l)));
        tile1.addChartData(new ChartData("", 111, Instant.ofEpochSecond(1567748494904l)));
    }

    private static void calcNoOfNodes(Node node) {
        if (node instanceof Parent) {
            if (((Parent) node).getChildrenUnmodifiable().size() != 0) {
                ObservableList<Node> tempChildren = ((Parent) node).getChildrenUnmodifiable();
                noOfNodes += tempChildren.size();
                for (Node n : tempChildren) { calcNoOfNodes(n); }
            }
        }
    }


    public static void main(String[] args) {
        launch(args);
    }
}
