package eu.xenit;

/**
 * Collection of property names in order to control host name/host name resolution.
 */
public class RuntimeContainerProperties {
    /**
     * Set this system property to a simple hostname to use a fixed host name.
     */
    public static final String PROPERTY_LOGSTASH_JSON_HOSTNAME = "xenit-json.hostname";

    /**
     * Set this system property to a fully qualified hostname to use a fixed host name.
     */
    public static final String PROPERTY_LOGSTASH_JSON_FQDN_HOSTNAME = "xenit-json.fqdn.hostname";

    /**
     * Set this system property to <code>true</code> to skip hosname resolution. The string <code>unknown</code> will be used as
     * hostname.
     */
    public static final String PROPERTY_LOGSTASH_JSON_SKIP_HOSTNAME_RESOLUTION = "xenit-json.skipHostnameResolution";

    /**
     * Set this propery to {@link #RESOLUTION_ORDER_LOCALHOST_NETWORK_FALLBACK} or
     * {@link #RESOLUTION_ORDER_NETWORK_LOCALHOST_FALLBACK} to control the hostname resolution order.
     */
    public static final String PROPERTY_LOGSTASH_JSON_HOSTNAME_RESOLUTION_ORDER = "xenit-json.resolutionOrder";

    /**
     * Resolution order: First inspect the local host name, then try to get the host name from network devices.
     */
    public static final String RESOLUTION_ORDER_LOCALHOST_NETWORK_FALLBACK = "localhost,network";

    /**
     * Resolution order: First inspect the network devices to retrieve a host name, then try to get the host name from the local
     * host.
     */
    public static final String RESOLUTION_ORDER_NETWORK_LOCALHOST_FALLBACK = "network,localhost";

    private RuntimeContainerProperties() {

    }

    /**
     * Lookup property from ({@link System#getenv(String)} and {@link System#getProperty(String)} as fallback.
     *
     * @param key          the property key
     * @param defaultValue the default value
     * @return the property value.
     */
    public static String getProperty(String key, String defaultValue) {
        String env = System.getenv(key);
        if (env != null && !"".equals(env)) {
            return env;
        }
        return System.getProperty(key, defaultValue);
    }
}
