package eu.xenit.json;

import eu.xenit.json.intern.JsonMessage;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.regex.Pattern;

/**
 * Builder to create a JsonMessage.
 */
public class JsonMessageBuilder {

    protected String version = JsonMessage.JSON_VERSION;
    protected String host;
    protected String shortMessage;
    protected String fullMessage;
    protected long javaTimestamp;
    protected String level;
    protected String facility = JsonMessage.DEFAULT_FACILITY;
    protected int maximumMessageSize = JsonMessage.DEFAULT_MESSAGE_SIZE;
    protected Map<String, String> additionalFields = new HashMap<>();
    protected Map<String, String> additionalFieldTypes = new HashMap<>();
    protected Map<Pattern, String> dynamicMdcFieldTypes = Collections.emptyMap();

    protected JsonMessageBuilder() {
    }

    /**
     * Creates a new instance of the JsonMessageBuilder.
     *
     * @return JsonMessageBuilder
     */
    public static JsonMessageBuilder newInstance() {
        return new JsonMessageBuilder();
    }

    /**
     * Set the version.
     *
     * @param version the version
     * @return JsonMessageBuilder
     */
    public JsonMessageBuilder withVersion(String version) {
        this.version = version;
        return this;
    }

    /**
     * Set the host.
     *
     * @param host the host
     * @return JsonMessageBuilder
     */
    public JsonMessageBuilder withHost(String host) {
        this.host = host;
        return this;
    }

    /**
     * Set the shortMessage.
     *
     * @param shortMessage the shortMessage
     * @return JsonMessageBuilder
     */
    public JsonMessageBuilder withShortMessage(String shortMessage) {
        this.shortMessage = shortMessage;
        return this;
    }

    /**
     * Set the fullMessage.
     *
     * @param fullMessage the fullMessage
     * @return JsonMessageBuilder
     */
    public JsonMessageBuilder withFullMessage(String fullMessage) {
        this.fullMessage = fullMessage;
        return this;
    }

    /**
     * Set the level (severity).
     *
     * @param level the level
     * @return JsonMessageBuilder
     */
    public JsonMessageBuilder withLevel(String level) {
        this.level = level;
        return this;
    }

    /**
     * Set the facility.
     *
     * @param facility the facility
     * @return JsonMessageBuilder
     */
    public JsonMessageBuilder withFacility(String facility) {
        this.facility = facility;
        return this;
    }

    /**
     * Set the max message size.
     *
     * @param maximumMessageSize the maximumMessageSize
     * @return JsonMessageBuilder
     */
    public JsonMessageBuilder withMaximumMessageSize(int maximumMessageSize) {
        this.maximumMessageSize = maximumMessageSize;
        return this;
    }

    /**
     * Set the java timestamp (millis).
     *
     * @param javaTimestamp the javaTimestamp
     * @return JsonMessageBuilder
     */
    public JsonMessageBuilder withJavaTimestamp(long javaTimestamp) {
        this.javaTimestamp = javaTimestamp;
        return this;
    }

    /**
     * Add additional fields.
     *
     * @param additionalFields the additionalFields
     * @return JsonMessageBuilder
     */
    public JsonMessageBuilder withFields(Map<String, String> additionalFields) {
        this.additionalFields.putAll(additionalFields);
        return this;
    }

    /**
     * Add an additional field.
     *
     * @param key   the key
     * @param value the value
     * @return JsonMessageBuilder
     */
    public JsonMessageBuilder withField(String key, String value) {
        this.additionalFields.put(key, value);
        return this;
    }

    /**
     * Set additional field types
     *
     * @param additionalFieldTypes the type map
     * @return JsonMessageBuilder
     */
    public JsonMessageBuilder withAdditionalFieldTypes(Map<String, String> additionalFieldTypes) {
        this.additionalFieldTypes.putAll(additionalFieldTypes);
        return this;
    }


    /**
     * Set dynamic mdc field types
     *
     * @param dynamicMdcFieldTypes the type map
     * @return JsonMessageBuilder
     */
    public JsonMessageBuilder withDynamicMdcFieldTypes(Map<Pattern, String> dynamicMdcFieldTypes) {
        this.dynamicMdcFieldTypes = dynamicMdcFieldTypes;
        return this;
    }

    /**
     * Build a new Json message based on the builder settings.
     *
     * @return JsonMessage
     */
    public JsonMessage build() {

        JsonMessage jsonMessage = new JsonMessage(shortMessage, fullMessage, javaTimestamp, level);
        jsonMessage.addFields(additionalFields);
        jsonMessage.setMaximumMessageSize(maximumMessageSize);
        jsonMessage.setVersion(version);
        jsonMessage.setHost(host);
        jsonMessage.setJavaTimestamp(javaTimestamp);
        jsonMessage.setFacility(facility);
        jsonMessage.setAdditionalFieldTypes(additionalFieldTypes);
        jsonMessage.setDynamicMdcFieldTypes(dynamicMdcFieldTypes);

        return jsonMessage;
    }

}
