package eu.xenit.json;

import eu.xenit.json.intern.JsonMessage;

import java.util.Set;

/**
 * Message-Assembler using MDC.
 */
public class MdcJsonMessageAssembler extends JsonMessageAssembler {

    public static final String PROPERTY_MDC_PROFILING = "mdcProfiling";
    public static final String PROPERTY_INCLUDE_FULL_MDC = "includeFullMdc";
    public static final String PROPERTY_MDC_FIELD = "mdcField.";
    public static final String PROPERTY_DYNAMIC_MDC_FIELD = "dynamicMdcFields.";

    private boolean mdcProfiling;
    private boolean includeFullMdc;

    @Override
    public void initialize(PropertyProvider propertyProvider) {

        super.initialize(propertyProvider);
        mdcProfiling = "true".equalsIgnoreCase(propertyProvider.getProperty(PROPERTY_MDC_PROFILING));
        includeFullMdc = "true".equalsIgnoreCase(propertyProvider.getProperty(PROPERTY_INCLUDE_FULL_MDC));

    }

    @Override
    public JsonMessage createJsonMessage(LogEvent logEvent) {

        JsonMessage jsonMessage = super.createJsonMessage(logEvent);
        if (mdcProfiling) {
            XenitJsonUtil.addMdcProfiling(logEvent, jsonMessage);
        }

        if (includeFullMdc) {
            Set<String> mdcNames = logEvent.getMdcNames();
            for (String mdcName : mdcNames) {

                if (mdcName == null) {
                    continue;
                }

                String mdcValue = logEvent.getMdcValue(mdcName);
                if (mdcValue != null) {
                    jsonMessage.addField(mdcName, mdcValue);
                }
            }
        }

        return jsonMessage;
    }

    public boolean isMdcProfiling() {
        return mdcProfiling;
    }

    public void setMdcProfiling(boolean mdcProfiling) {
        this.mdcProfiling = mdcProfiling;
    }

    public boolean isIncludeFullMdc() {
        return includeFullMdc;
    }

    public void setIncludeFullMdc(boolean includeFullMdc) {
        this.includeFullMdc = includeFullMdc;
    }
}
