package eu.xenit.json;

import eu.xenit.json.intern.JsonMessage;

import java.math.BigDecimal;
import java.text.SimpleDateFormat;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneOffset;
import java.time.format.DateTimeFormatter;
import java.util.Date;
import java.util.HashSet;
import java.util.Set;


public class XenitJsonUtil {

    /**
     * Profiling Start in MDC (msec).
     */
    public static final String MDC_REQUEST_START_MS = "profiling.requestStart.millis";
    /**
     * Profiling End in MDC.
     */
    public static final String MDC_REQUEST_END = "profiling.requestEnd";

    /**
     * Profiling Duration in MDC.
     */
    public static final String MDC_REQUEST_DURATION = "profiling.requestDuration";

    public static final String MDC_REQUEST_DURATION_MILLIS = "profiling.requestDuration.millis";

    private static final String DATE_PATTERN = "yyyy-MM-dd HH:mm:ss,SSS";
    private static final DateTimeFormatter DATE_TIME_FORMATTER = DateTimeFormatter.ofPattern(DATE_PATTERN);

    private XenitJsonUtil() {

    }

    public static double getTimestamp(long timestamp) {
        return timestamp/1000.0;
    }

    public static String formatLocalDateTime(LocalDateTime localDateTime) {
        return localDateTime.atOffset(ZoneOffset.UTC).format(DATE_TIME_FORMATTER);
    }

    public static String formatDate(Date date) {
        return new SimpleDateFormat(DATE_PATTERN).format(date);
    }

    public static void addMdcProfiling(LogEvent logEvent, JsonMessage jsonMessage) {

        String requestStartMs = logEvent.getMdcValue(MDC_REQUEST_START_MS);
        long timestamp;

        if (requestStartMs != null && !requestStartMs.isEmpty()) {
            timestamp = Long.parseLong(requestStartMs);
        } else {
            return;
        }

        if (timestamp > 0) {
            long now = System.currentTimeMillis();
            long durationMs = now - timestamp;

            String durationText;

            if (durationMs > 10000) {
                long durationSec = durationMs / 1000;
                durationText = durationSec + "sec";
            } else {
                durationText = durationMs + "ms";
            }
            jsonMessage.addField(MDC_REQUEST_DURATION, durationText);
            jsonMessage.addField(MDC_REQUEST_DURATION_MILLIS, "" + durationMs);
            jsonMessage.addField(MDC_REQUEST_END, new Date(now).toString());
        }
    }

    public static String getSimpleClassName(String className) {

        if (className == null) {
            return null;
        }

        int index = className.lastIndexOf('.');
        if (index != -1) {
            return className.substring(index + 1);
        }
        return className;
    }

    public static Set<String> getMatchingMdcNames(DynamicMdcMessageField field, Set<String> mdcNames) {
        Set<String> matchingMdcNames = new HashSet<>();

        for (String mdcName : mdcNames) {
            if (field.getPattern().matcher(mdcName).matches()) {
                matchingMdcNames.add(mdcName);
            }
        }
        return matchingMdcNames;
    }
}
