package eu.xenit.json.intern;

import eu.xenit.json.intern.sender.DefaultJsonSenderProvider;
import eu.xenit.json.intern.sender.KafkaJsonSenderProvider;
import eu.xenit.json.intern.sender.RedisJsonSenderProvider;

import java.io.IOException;
import java.net.SocketException;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.ServiceLoader;

/**
 * Factory to create a {@link JsonSender} based on the host and protocol details. This factory uses Java's {@link ServiceLoader}
 * mechanism to discover classes implementing {@link JsonSenderProvider}.
 */
public final class JsonSenderFactory {

    private JsonSenderFactory() {
        // no instance allowed
    }

    /**
     * Create a JsonSender based on the configuration.
     *
     * @param hostAndPortProvider          the host and port
     * @param errorReporter                the error reporter
     * @param senderSpecificConfigurations configuration map
     * @return a new {@link JsonSender} instance
     */
    public static JsonSender createSender(final HostAndPortProvider hostAndPortProvider, final ErrorReporter errorReporter,
                                          final Map<String, Object> senderSpecificConfigurations) {
        JsonSenderConfiguration senderConfiguration = new JsonSenderConfiguration() {

            @Override
            public int getPort() {
                return hostAndPortProvider.getPort();
            }

            @Override
            public String getHost() {
                return hostAndPortProvider.getHost();
            }

            @Override
            public ErrorReporter getErrorReporter() {
                return errorReporter;
            }

            @Override
            public Map<String, Object> getSpecificConfigurations() {
                return senderSpecificConfigurations;
            }

        };

        return createSender(senderConfiguration);
    }

    /**
     * Create a JsonSender based on the configuration.
     *
     * @param senderConfiguration the configuration
     * @return a new {@link JsonSender} instance
     */
    public static JsonSender createSender(JsonSenderConfiguration senderConfiguration) {

        ErrorReporter errorReporter = senderConfiguration.getErrorReporter();
        if (senderConfiguration.getHost() == null) {
            senderConfiguration.getErrorReporter().reportError("Json server hostname is empty!", null);
        } else {

            try {
                for (JsonSenderProvider provider : SenderProviderHolder.getSenderProvider()) {
                    if (provider.supports(senderConfiguration.getHost())) {
                        return provider.create(senderConfiguration);
                    }
                }
                senderConfiguration.getErrorReporter().reportError("No sender found for host " + senderConfiguration.getHost(),
                        null);
                return null;
            } catch (UnknownHostException e) {
                errorReporter.reportError("Unknown Json server hostname:" + senderConfiguration.getHost(), e);
            } catch (SocketException e) {
                errorReporter.reportError("Socket exception: " + e.getMessage(), e);
            } catch (IOException e) {
                errorReporter.reportError("IO exception: " + e.getMessage(), e);
            }
        }

        return null;
    }

    public static void addJsonSenderProvider(JsonSenderProvider provider) {
        SenderProviderHolder.addSenderProvider(provider);
    }

    public static void removeJsonSenderProvider(JsonSenderProvider provider) {
        SenderProviderHolder.removeSenderProvider(provider);
    }

    public static void removeAllAddedSenderProviders() {
        SenderProviderHolder.removeAllAddedSenderProviders();
    }

    // For thread safe lazy intialization of provider list
    private static class SenderProviderHolder {

        private static ServiceLoader<JsonSenderProvider> jsonSenderProvider = ServiceLoader.load(JsonSenderProvider.class);
        private static List<JsonSenderProvider> providerList = new ArrayList<>();
        private static List<JsonSenderProvider> addedProviders = new ArrayList<>();

        static {
            Iterator<JsonSenderProvider> iter = jsonSenderProvider.iterator();
            while (iter.hasNext()) {
                providerList.add(iter.next());
            }
            providerList.add(new RedisJsonSenderProvider());
            providerList.add(new KafkaJsonSenderProvider());
            providerList.add(new DefaultJsonSenderProvider());
        }

        static List<JsonSenderProvider> getSenderProvider() {
            return providerList;
        }

        static void addSenderProvider(JsonSenderProvider provider) {
            synchronized (providerList) {
                addedProviders.add(provider);
                if (!providerList.contains(provider)) {
                    providerList.add(0, provider); // To take precedence over built-in providers
                }
            }
        }

        static void removeAllAddedSenderProviders() {
            synchronized (providerList) {
                providerList.removeAll(addedProviders);
                addedProviders.clear();
            }
        }

        static void removeSenderProvider(JsonSenderProvider provider) {
            synchronized (providerList) {
                addedProviders.remove(provider);
                providerList.remove(provider);
            }
        }
    }
}
