package eu.xenit.json.intern;

import eu.xenit.RuntimeContainerProperties;
import eu.xenit.json.JsonMessageBuilder;

public class PoolingJsonMessageBuilder extends JsonMessageBuilder {

    /**
     * Can be
     * <ul>
     * <li>{@literal static} (default value) for static held pools</li>
     * <li>{@literal true} for using instance-based held pools</li>
     * <li>{@literal false} to disable pooling</li>
     * </ul>
     */
    public static final String PROPERTY_USE_POOLING = "JsonMessageBuilder.message.pooling";

    private static final String USE_POOLING_VAL = RuntimeContainerProperties.getProperty(PROPERTY_USE_POOLING, "static");
    private static final boolean STATIC_POOLING = USE_POOLING_VAL.equalsIgnoreCase("static");
    private static final PoolHolder STATIC_POOL_HOLDER = STATIC_POOLING ? PoolHolder.threadLocal() : PoolHolder.noop();

    private final PoolHolder poolHolder;

    private PoolingJsonMessageBuilder(PoolHolder poolHolder) {
        this.poolHolder = poolHolder;
    }

    /**
     * Creates a new instance of the JsonMessageBuilder.
     *
     * @return JsonMessageBuilder
     */
    public static PoolingJsonMessageBuilder newInstance() {
        return new PoolingJsonMessageBuilder(STATIC_POOLING ? STATIC_POOL_HOLDER : PoolHolder.threadLocal());
    }

    /**
     * @return {@literal true} if pooling (static/instance-held pools) is enabled.
     */
    public static boolean usePooling() {
        return STATIC_POOLING || USE_POOLING_VAL.equalsIgnoreCase("true");
    }

    /**
     * Recycle this {@link JsonMessageBuilder} to a default state.
     *
     * @return {@code this} {@link JsonMessageBuilder}
     */
    public JsonMessageBuilder recycle() {

        version = JsonMessage.JSON_VERSION;
        host = null;
        shortMessage = null;
        fullMessage = null;
        javaTimestamp = 0;
        level = null;
        facility = JsonMessage.DEFAULT_FACILITY;
        maximumMessageSize = JsonMessage.DEFAULT_MESSAGE_SIZE;

        additionalFields.clear();
        additionalFieldTypes.clear();

        return this;
    }

    /**
     * Build a new Json message based on the builder settings.
     *
     * @return JsonMessage
     */
    @Override
    public JsonMessage build() {

        JsonMessage jsonMessage = new PoolingJsonMessage(shortMessage, fullMessage, javaTimestamp, level, poolHolder);
        jsonMessage.addFields(additionalFields);
        jsonMessage.setMaximumMessageSize(maximumMessageSize);
        jsonMessage.setVersion(version);
        jsonMessage.setHost(host);
        jsonMessage.setJavaTimestamp(javaTimestamp);
        jsonMessage.setFacility(facility);
        jsonMessage.setAdditionalFieldTypes(additionalFieldTypes);
        jsonMessage.setDynamicMdcFieldTypes(dynamicMdcFieldTypes);

        return jsonMessage;
    }
}
