package eu.xenit.json.intern.sender;

import eu.xenit.json.intern.JsonSender;
import eu.xenit.json.intern.JsonSenderConfiguration;
import eu.xenit.json.intern.JsonSenderProvider;

import javax.net.ssl.SSLContext;
import java.io.IOException;
import java.net.URI;
import java.net.URL;
import java.security.NoSuchAlgorithmException;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.TimeUnit;

/**
 * Default provider for {@link JsonSender} that creates UDP, TCP and HTTP senders.
 */
public class DefaultJsonSenderProvider implements JsonSenderProvider {

    /**
     * Default Json port.
     */
    public static final int DEFAULT_PORT = 12201;

    private static final Map<String, JsonSenderProducer> factories;

    private static final JsonSenderProducer tcpSenderFactory = new JsonSenderProducer() {

        @Override
        public JsonSender create(JsonSenderConfiguration configuration, String host, int port) throws IOException {

            int defaultTimeoutMs = (int) TimeUnit.MILLISECONDS.convert(2, TimeUnit.SECONDS);

            URI uri = URI.create(host);

            Map<String, String> params = QueryStringParser.parse(uri);
            int connectionTimeMs = (int) QueryStringParser.getTimeAsMs(params, JsonTCPSender.CONNECTION_TIMEOUT, defaultTimeoutMs);
            int readTimeMs = (int) QueryStringParser.getTimeAsMs(params, JsonTCPSender.READ_TIMEOUT, defaultTimeoutMs);
            int deliveryAttempts = QueryStringParser.getInt(params, JsonTCPSender.RETRIES, 1);
            boolean keepAlive = QueryStringParser.getString(params, JsonTCPSender.KEEPALIVE, false);

            int writeBackoffTimeMs = (int) QueryStringParser.getTimeAsMs(params, JsonTCPSender.WRITE_BACKOFF_TIME, 50);
            int writeBackoffThreshold = QueryStringParser.getInt(params, JsonTCPSender.WRITE_BACKOFF_THRESHOLD, 10);
            int maxWriteBackoffTimeMs = (int) QueryStringParser.getTimeAsMs(params, JsonTCPSender.MAX_WRITE_BACKOFF_TIME, connectionTimeMs);

            String rHost = QueryStringParser.getHost(uri);

            return new JsonTCPSender(rHost, port, connectionTimeMs, readTimeMs, deliveryAttempts, keepAlive,
                    new BoundedBackOff(new ConstantBackOff(writeBackoffTimeMs, TimeUnit.MILLISECONDS), maxWriteBackoffTimeMs,
                            TimeUnit.MILLISECONDS),
                    writeBackoffThreshold,
                    configuration.getErrorReporter());
        }
    };

    private static final JsonSenderProducer tcpSslSenderFactory = new JsonSenderProducer() {

        @Override
        public JsonSender create(JsonSenderConfiguration configuration, String host, int port) throws IOException {

            int defaultTimeoutMs = (int) TimeUnit.MILLISECONDS.convert(2, TimeUnit.SECONDS);

            URI uri = URI.create(host);

            Map<String, String> params = QueryStringParser.parse(uri);
            int connectionTimeMs = (int) QueryStringParser.getTimeAsMs(params, JsonTCPSender.CONNECTION_TIMEOUT, defaultTimeoutMs);
            int readTimeMs = (int) QueryStringParser.getTimeAsMs(params, JsonTCPSender.READ_TIMEOUT, defaultTimeoutMs);
            int deliveryAttempts = QueryStringParser.getInt(params, JsonTCPSender.RETRIES, 1);
            boolean keepAlive = QueryStringParser.getString(params, JsonTCPSender.KEEPALIVE, false);

            int writeBackoffTimeMs = (int) QueryStringParser.getTimeAsMs(params, JsonTCPSender.WRITE_BACKOFF_TIME, 50);
            int writeBackoffThreshold = QueryStringParser.getInt(params, JsonTCPSender.WRITE_BACKOFF_THRESHOLD, 10);
            int maxWriteBackoffTimeMs = (int) QueryStringParser.getTimeAsMs(params, JsonTCPSender.MAX_WRITE_BACKOFF_TIME,
                    connectionTimeMs);

            String tcpHost = QueryStringParser.getHost(uri);

            try {
                return new JsonTCPSSLSender(tcpHost, port, connectionTimeMs, readTimeMs, deliveryAttempts, keepAlive,
                        new BoundedBackOff(new ConstantBackOff(writeBackoffTimeMs, TimeUnit.MILLISECONDS),
                                maxWriteBackoffTimeMs, TimeUnit.MILLISECONDS),
                        writeBackoffThreshold,
                        configuration.getErrorReporter(), SSLContext.getDefault());
            } catch (NoSuchAlgorithmException e) {
                throw new IllegalStateException(e);
            }
        }
    };

    private static final JsonSenderProducer udpSenderFactory = new JsonSenderProducer() {

        @Override
        public JsonSender create(JsonSenderConfiguration configuration, String host, int port) throws IOException {

            URI uri = URI.create(host);
            String rHost = QueryStringParser.getHost(uri);
            return new JsonUDPSender(rHost, port, configuration.getErrorReporter());
        }
    };

    private static final JsonSenderProducer defaultSenderFactory = new JsonSenderProducer() {

        @Override
        public JsonSender create(JsonSenderConfiguration configuration, String host, int port) throws IOException {
            return new JsonUDPSender(host, port, configuration.getErrorReporter());
        }
    };

    private static final JsonSenderProducer httpSenderFactory = new JsonSenderProducer() {

        @Override
        public JsonSender create(JsonSenderConfiguration configuration, String host, int port) throws IOException {

            int defaultTimeoutMs = (int) TimeUnit.MILLISECONDS.convert(2, TimeUnit.SECONDS);
            URL url = new URL(host);

            return new JsonHTTPSender(url, defaultTimeoutMs, defaultTimeoutMs, configuration.getErrorReporter());
        }
    };

    static {

        Map<String, JsonSenderProducer> prefixToFactory = new HashMap<>();
        prefixToFactory.put("tcp:", tcpSenderFactory);
        prefixToFactory.put("ssl:", tcpSslSenderFactory);
        prefixToFactory.put("udp:", udpSenderFactory);
        prefixToFactory.put("http", httpSenderFactory);

        factories = Collections.unmodifiableMap(prefixToFactory);

    }

    @Override
    public boolean supports(String host) {
        return host != null;
    }

    @Override
    public JsonSender create(JsonSenderConfiguration configuration) throws IOException {

        String host = configuration.getHost();

        int port = configuration.getPort();
        if (port == 0) {
            port = DEFAULT_PORT;
        }

        for (Map.Entry<String, JsonSenderProducer> entry : factories.entrySet()) {
            if (host.startsWith(entry.getKey())) {
                return entry.getValue().create(configuration, host, port);
            }
        }

        return defaultSenderFactory.create(configuration, host, port);

    }

    private interface JsonSenderProducer {

        /**
         * Produce a {@link JsonSender} using {@code configuration}, {@code host} and {@code port},
         *
         * @param configuration
         * @param host
         * @param port
         * @return
         * @throws IOException
         */
        JsonSender create(JsonSenderConfiguration configuration, String host, int port) throws IOException;
    }
}
