package eu.xenit.json.intern.sender;

import eu.xenit.json.intern.JsonMessage;

import java.nio.BufferOverflowException;
import java.nio.ByteBuffer;

/**
 * Utility to create TCP/UDP buffers from a {@link JsonMessage} by working with {@link ByteBuffer}. Buffers are provided by
 * {@link ThreadLocal} and enlarged if the buffer size is exceeded. Enlarged buffers are returned to their originating
 * {@link ThreadLocal}.
 */
class JsonBuffers {

    private JsonBuffers() {
        // no instance allowed
    }

    /**
     * Create UDP buffers and apply auto-buffer-enlarging, if necessary.
     *
     * @param message
     * @param writeBuffers
     * @param tempBuffers
     * @return
     */
    protected static ByteBuffer[] toUDPBuffers(JsonMessage message, ThreadLocal<ByteBuffer> writeBuffers,
                                               ThreadLocal<ByteBuffer> tempBuffers) {

        while (true) {

            try {
                return message.toUDPBuffers(getBuffer(writeBuffers), getBuffer(tempBuffers));
            } catch (BufferOverflowException e) {
                enlargeBuffer(writeBuffers);
                enlargeBuffer(tempBuffers);
            }
        }
    }

    /**
     * Create TCP buffer and apply auto-buffer-enlarging, if necessary.
     *
     * @param message
     * @param writeBuffers
     * @return
     */
    protected static ByteBuffer toTCPBuffer(JsonMessage message, ThreadLocal<ByteBuffer> writeBuffers) {

        while (true) {

            try {
                return message.toTCPBuffer(getBuffer(writeBuffers));
            } catch (BufferOverflowException e) {
                enlargeBuffer(writeBuffers);
            }
        }
    }

    private static void enlargeBuffer(ThreadLocal<ByteBuffer> buffers) {

        ByteBuffer newBuffer = ByteBuffer.allocateDirect(calculateNewBufferSize(buffers.get().capacity()));
        buffers.set(newBuffer);
    }

    private static ByteBuffer getBuffer(ThreadLocal<ByteBuffer> buffers) {
        return buffers.get().clear();
    }

    private static int calculateNewBufferSize(int capacity) {
        return (int) (capacity * 1.5);
    }
}
