package eu.xenit.json.intern.sender;

import eu.xenit.json.intern.ErrorReporter;
import eu.xenit.json.intern.JsonMessage;
import eu.xenit.json.intern.JsonSender;
import redis.clients.jedis.Jedis;
import redis.clients.jedis.util.Pool;

import java.io.IOException;
import java.util.Collections;
import java.util.Set;
import java.util.WeakHashMap;

public class JsonREDISSender implements JsonSender {

    private final Pool<Jedis> jedisPool;
    private final ErrorReporter errorReporter;
    private final String redisKey;
    private final Set<Thread> callers = Collections.newSetFromMap(new WeakHashMap<Thread, Boolean>());

    public JsonREDISSender(Pool<Jedis> jedisPool, String redisKey, ErrorReporter errorReporter) {
        this.jedisPool = jedisPool;
        this.errorReporter = errorReporter;
        this.redisKey = redisKey;
    }

    public boolean sendMessage(JsonMessage message) {

        // prevent recursive self calls caused by the Redis driver since it
        if (!callers.add(Thread.currentThread())) {
            return false;
        }

        try {
            return sendMessage0(message);
        } finally {
            callers.remove(Thread.currentThread());
        }
    }

    protected boolean sendMessage0(JsonMessage message) {
        try (Jedis jedisClient = jedisPool.getResource()) {
            jedisClient.lpush(redisKey, message.toJson(""));
            return true;
        } catch (Exception e) {
            errorReporter.reportError(e.getMessage(), new IOException("Cannot send REDIS data with key URI " + redisKey, e));
            return false;
        }
    }

    public void close() {
        callers.clear();
        jedisPool.destroy();
    }
}
