package eu.xenit.json.jul;

import eu.xenit.json.LogMessageField;
import eu.xenit.json.MdcJsonMessageAssembler;
import eu.xenit.json.intern.ConfigurationSupport;
import eu.xenit.json.intern.JsonMessage;

import java.util.Collections;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.Set;
import java.util.logging.Formatter;
import java.util.logging.LogManager;
import java.util.logging.LogRecord;

/**
 * Log-Formatter for JSON using fields specified within Json. This formatter will produce a JSON object for each log event.
 * Example:
 *
 * <code>
 * {
 * "timestamp": "1439319236.722",
 * "SourceClassName": "eu.xenit.json.wildfly.WildFlyJsonLogFormatterTest",
 * "SourceMethodName": "testDefaults",
 * "level": "6",
 * "SourceSimpleClassName": "WildFlyJsonLogFormatterTest",
 * "facility": "xenit-json",
 * "fullMessage": "foo bar test log message",
 * "shortMessage": "foo bar test log message",
 * "MySeverity": "INFO",
 * "LoggerName": "eu.xenit.json.wildfly.WildFlyJsonLogFormatterTest",
 * "Thread": "main",
 * "Time": "2015-08-11 20:53:56,0722"
 * }
 * </code>
 * <p>
 * Following parameters are supported/needed:
 * <ul>
 * <li>lineBreak (Optional): End of line, platform dependent default value, see {@code System.getProperty("line.separator")}</li>
 * <li>fields (Optional): Comma-separated list of log event fields that should be included in the JSON. Defaults to
 * {@code Time, Severity, ThreadName, sourceClassName, SourceMethodName, SourceSimpleClassName, LoggerName}</li>
 * <li>originHost (Optional): Originating Hostname, default FQDN Hostname</li>
 * <li>extractStackTrace (Optional): Post Stack-Trace to StackTrace field (true/false/throwable reference [0 = throwable, 1 =
 * throwable.cause, -1 = root cause]), default false</li>
 * <li>filterStackTrace (Optional): Perform Stack-Trace filtering (true/false), default false</li>
 * <li>includeLocation (Optional): Include source code location, default true</li>
 * <li>includeLogMessageParameters (Optional): Include message parameters from the log event (see
 * {@link LogRecord#getParameters()}, default true</li>
 * <li>facility (Optional): Name of the Facility, default json-java</li>
 * <li>additionalFields(number) (Optional): Post additional fields. Eg. fieldName=Value,field2=value2</li>
 * <li>additionalFieldTypes (Optional): Type specification for additional and MDC fields. Supported types: String, long, Long,
 * double, Double and discover (default if not specified, discover field type on parseability). Eg. field=String,field2=double</li>
 * </ul>
 */
public class JsonFormatter extends Formatter {

    public static final String MULTI_VALUE_DELIMITTER = ",";
    public static final Set<LogMessageField.NamedLogField> SUPPORTED_FIELDS;
    public static final Set<LogMessageField.NamedLogField> DEFAULT_FIELDS;

    static {
        Set<LogMessageField.NamedLogField> defaultFields = new LinkedHashSet<>();
        defaultFields.add(LogMessageField.NamedLogField.Time);
        defaultFields.add(LogMessageField.NamedLogField.Severity);
        defaultFields.add(LogMessageField.NamedLogField.LoggerName);
        defaultFields.add(LogMessageField.NamedLogField.ThreadName);
        DEFAULT_FIELDS = Collections.unmodifiableSet(defaultFields);

        Set<LogMessageField.NamedLogField> supportedFields = new LinkedHashSet<>();
        supportedFields.add(LogMessageField.NamedLogField.Time);
        supportedFields.add(LogMessageField.NamedLogField.Severity);
        supportedFields.add(LogMessageField.NamedLogField.ThreadName);
        supportedFields.add(LogMessageField.NamedLogField.SourceClassName);
        supportedFields.add(LogMessageField.NamedLogField.SourceMethodName);
        supportedFields.add(LogMessageField.NamedLogField.SourceSimpleClassName);
        supportedFields.add(LogMessageField.NamedLogField.LoggerName);
        supportedFields.add(LogMessageField.NamedLogField.Server);
        SUPPORTED_FIELDS = Collections.unmodifiableSet(supportedFields);
    }

    private final MdcJsonMessageAssembler jsonMessageAssembler = new MdcJsonMessageAssembler();
    private String lineBreak = System.getProperty("line.separator");
    private boolean wasSetFieldsCalled = false;

    public JsonFormatter() {
        super();
        configure();
    }

    private void configure() {
        String cname = getClass().getName();
        LogManager manager = LogManager.getLogManager();

        String val = manager.getProperty(cname + ".fields");
        if (val != null) {
            setFields(val);
        }

        val = manager.getProperty(cname + ".version");
        if (val != null) {
            setVersion(val);
        }

        val = manager.getProperty(cname + ".facility");
        if (val != null) {
            setFacility(val);
        }

        val = manager.getProperty(cname + ".extractStackTrace");
        if (val != null) {
            setExtractStackTrace(val);
        }

        val = manager.getProperty(cname + ".filterStackTrace");
        if (val != null) {
            setFilterStackTrace(Boolean.valueOf(val));
        }

        val = manager.getProperty(cname + ".includeLogMessageParameters");
        if (val != null) {
            setIncludeLogMessageParameters(Boolean.valueOf(val));
        }

        val = manager.getProperty(cname + ".includeLocation");
        if (val != null) {
            setIncludeLocation(Boolean.valueOf(val));
        }

        val = manager.getProperty(cname + ".timestampPattern");
        if (val != null) {
            setTimestampPattern(val);
        }

        val = manager.getProperty(cname + ".additionalFields");
        if (val != null) {
            setAdditionalFields(val);
        }
        val = manager.getProperty(cname + ".type");
        if (val != null) {
            setType(val);
        }
        val = manager.getProperty(cname + ".component");
        if (val != null) {
            setComponent(val);
        }
        val = manager.getProperty(cname + ".additionalFieldTypes");
        if (val != null) {
            setAdditionalFieldTypes(val);
        }

        val = manager.getProperty(cname + ".originHost");
        if (val != null) {
            setOriginHost(val);
        }

        val = manager.getProperty(cname + ".linebreak");
        if (val != null) {
            setLineBreak(val);
        }
    }

    @Override
    public String format(final LogRecord logRecord) {
        if (!wasSetFieldsCalled) {
            addFields(new HashSet<>((DEFAULT_FIELDS)));
        }
        JsonMessage jsonMessage = jsonMessageAssembler.createJsonMessage(new JulLogEvent(logRecord));
        return jsonMessage.toJson("") + lineBreak;
    }

    public void setFields(String fieldSpec) {

        String[] properties = fieldSpec.split(MULTI_VALUE_DELIMITTER);
        Set<LogMessageField.NamedLogField> fields = new HashSet<>();
        for (String field : properties) {

            LogMessageField.NamedLogField namedLogField = LogMessageField.NamedLogField.byName(field.trim());
            if (namedLogField == null) {
                throw new IllegalArgumentException("Cannot resolve field name '" + field
                        + "' to a field. Supported field names are: " + SUPPORTED_FIELDS);
            }

            if (!SUPPORTED_FIELDS.contains(namedLogField)) {
                throw new IllegalArgumentException("Field '" + field + "' is not supported. Supported field names are: "
                        + SUPPORTED_FIELDS);
            }

            fields.add(namedLogField);
        }

        addFields(fields);
    }

    private void addFields(Set<LogMessageField.NamedLogField> fields) {
        fields.addAll(DEFAULT_FIELDS);
        jsonMessageAssembler.addFields(LogMessageField.getDefaultMapping(fields
                .toArray(new LogMessageField.NamedLogField[0])));

        wasSetFieldsCalled = true;
    }

    public void setType(String type) {
        ConfigurationSupport.setType(type, jsonMessageAssembler);
    }

    public void setComponent(String component) {
        ConfigurationSupport.setComponent(component, jsonMessageAssembler);
    }

    public void setAdditionalFields(String spec) {
        ConfigurationSupport.setAdditionalFields(spec, jsonMessageAssembler);
    }

    public void setAdditionalFieldTypes(String spec) {
        ConfigurationSupport.setAdditionalFieldTypes(spec, jsonMessageAssembler);
    }

    public String getOriginHost() {
        return jsonMessageAssembler.getOriginHost();
    }

    public void setOriginHost(String originHost) {
        jsonMessageAssembler.setOriginHost(originHost);
    }

    public String getFacility() {
        return jsonMessageAssembler.getFacility();
    }

    public void setFacility(String facility) {
        jsonMessageAssembler.setFacility(facility);
    }

    public String getExtractStackTrace() {
        return jsonMessageAssembler.getExtractStackTrace();
    }

    public void setExtractStackTrace(String extractStacktrace) {
        jsonMessageAssembler.setExtractStackTrace(extractStacktrace);
    }

    public boolean isFilterStackTrace() {
        return jsonMessageAssembler.isFilterStackTrace();
    }

    public void setFilterStackTrace(boolean filterStackTrace) {
        jsonMessageAssembler.setFilterStackTrace(filterStackTrace);
    }

    public boolean isIncludeLogMessageParameters() {
        return jsonMessageAssembler.isIncludeLogMessageParameters();
    }

    public void setIncludeLogMessageParameters(boolean includeLogMessageParameters) {
        jsonMessageAssembler.setIncludeLogMessageParameters(includeLogMessageParameters);
    }

    public boolean isIncludeLocation() {
        return jsonMessageAssembler.isIncludeLocation();
    }

    public void setIncludeLocation(boolean includeLocation) {
        jsonMessageAssembler.setIncludeLocation(includeLocation);
    }

    public String getTimestampPattern() {
        return jsonMessageAssembler.getTimestampPattern();
    }

    public void setTimestampPattern(String timestampPattern) {
        jsonMessageAssembler.setTimestampPattern(timestampPattern);
    }

    public String getVersion() {
        return jsonMessageAssembler.getVersion();
    }

    public void setVersion(String version) {
        jsonMessageAssembler.setVersion(version);
    }

    public String getLineBreak() {
        return lineBreak;
    }

    public void setLineBreak(String lineBreak) {
        this.lineBreak = lineBreak;
    }
}
