package eu.xenit.json.jul;

import eu.xenit.RuntimeContainer;
import eu.xenit.json.JsonMessageAssembler;
import eu.xenit.json.LogMessageField;
import eu.xenit.json.PropertyProvider;
import eu.xenit.json.intern.Closer;
import eu.xenit.json.intern.ConfigurationSupport;
import eu.xenit.json.intern.ErrorReporter;
import eu.xenit.json.intern.JsonMessage;
import eu.xenit.json.intern.JsonSender;
import eu.xenit.json.intern.JsonSenderFactory;
import eu.xenit.json.intern.MessagePostprocessingErrorReporter;

import java.util.Collections;
import java.util.logging.ErrorManager;
import java.util.logging.Filter;
import java.util.logging.Handler;
import java.util.logging.Level;
import java.util.logging.LogRecord;

/**
 * Logging-Handler for Json . This Java-Util-Logging Handler creates Json Messages and posts
 * them using UDP (default) or TCP. Following parameters are supported/needed:
 * <ul>
 * <li>host (Mandatory): Hostname/IP-Address of the Logstash Host
 * <ul>
 * <li>(the host) for UDP, e.g. 127.0.0.1 or some.host.com</li>
 * <li>See docs for more details</li>
 * </ul>
 * </li>
 * <li>port (Optional): Port, default 12201</li>
 * <li>version (Optional): Json Version 1.0 or 1.1, default 1.0</li>
 * <li>originHost (Optional): Originating Hostname, default FQDN Hostname</li>
 * <li>extractStackTrace (Optional): Post Stack-Trace to StackTrace field (true/false/throwable reference [0 = throwable, 1 =
 * throwable.cause, -1 = root cause]), default false</li>
 * <li>filterStackTrace (Optional): Perform Stack-Trace filtering (true/false), default false</li>
 * <li>includeLogMessageParameters (Optional): Include message parameters from the log event (see
 * {@link LogRecord#getParameters()}, default true</li>
 * <li>includeLocation (Optional): Include source code location, default true</li>
 * <li>mdcProfiling (Optional): Perform Profiling (Call-Duration) based on MDC Data. See <a href="#mdcProfiling">MDC
 * Profiling</a>, default false</li>
 * <li>facility (Optional): Name of the Facility, default json-java</li>
 * <li>level (Optional): Log-Level, default INFO</li>
 * <li>filter (Optional): Class-Name of a Log-Filter, default none</li>
 * <li>additionalField.(number) (Optional): Post additional fields. Eg. .jsonLogHandler.additionalField.0=fieldName=Value</li>
 * </ul>
 * <p>
 * The {@link #publish(LogRecord)} method is thread-safe and may be called by different threads at any time.
 */
public class JsonLogHandler extends Handler implements ErrorReporter {

    private final ErrorReporter errorReporter = new MessagePostprocessingErrorReporter(this);
    protected volatile JsonSender jsonSender;
    protected JsonMessageAssembler jsonMessageAssembler;

    public JsonLogHandler() {
        super();

        RuntimeContainer.initialize(errorReporter);
        jsonMessageAssembler = createJsonMessageAssembler();

        initializeDefaultFields();

        JulPropertyProvider propertyProvider = new JulPropertyProvider(JsonLogHandler.class);
        jsonMessageAssembler.initialize(propertyProvider);

        String level = propertyProvider.getProperty(PropertyProvider.PROPERTY_LEVEL);
        if (null != level) {
            setLevel(Level.parse(level.trim()));
        } else {
            setLevel(Level.INFO);
        }

        String additionalFields = propertyProvider.getProperty(PropertyProvider.PROPERTY_ADDITIONAL_FIELDS);
        if (null != additionalFields) {
            setAdditionalFields(additionalFields);
        }

        String additionalFieldTypes = propertyProvider.getProperty(PropertyProvider.PROPERTY_ADDITIONAL_FIELD_TYPES);
        if (null != additionalFieldTypes) {
            setAdditionalFieldTypes(additionalFieldTypes);
        }

        String includeLocation = propertyProvider.getProperty(PropertyProvider.PROPERTY_INCLUDE_LOCATION);
        if (null != includeLocation) {
            setIncludeLocation(Boolean.valueOf(includeLocation));
        }

        String filter = propertyProvider.getProperty(PropertyProvider.PROPERTY_FILTER);
        try {
            if (null != filter) {
                final Class clazz = ClassLoader.getSystemClassLoader().loadClass(filter);
                setFilter((Filter) clazz.getDeclaredConstructor().newInstance());
            }
        } catch (final Exception e) {
            // ignore
        }
    }

    protected void initializeDefaultFields() {
        jsonMessageAssembler.addFields(LogMessageField.getDefaultMapping(LogMessageField.NamedLogField.Time, LogMessageField.NamedLogField.Severity, LogMessageField.NamedLogField.ThreadName, LogMessageField.NamedLogField.SourceClassName,
                LogMessageField.NamedLogField.SourceMethodName, LogMessageField.NamedLogField.SourceSimpleClassName, LogMessageField.NamedLogField.LoggerName));
    }

    protected JsonMessageAssembler createJsonMessageAssembler() {
        return new JsonMessageAssembler();
    }

    @Override
    public void flush() {
        // nothing to do
    }

    @Override
    public void publish(final LogRecord logRecord) {
        if (!isLoggable(logRecord)) {
            return;
        }

        try {
            if (null == jsonSender) {
                synchronized (this) {
                    if (null == jsonSender) {
                        jsonSender = createJsonSender();
                    }
                }
            }
        } catch (Exception e) {
            reportError("Could not send Json message: " + e.getMessage(), e, ErrorManager.OPEN_FAILURE);
            return;
        }

        try {
            JsonMessage message = createJsonMessage(logRecord);
            if (!message.isValid()) {
                reportError("Json Message is invalid: " + message.toJson(), null, ErrorManager.WRITE_FAILURE);
                return;
            }

            if (null == jsonSender || !jsonSender.sendMessage(message)) {
                reportError("Could not send Json message", null, ErrorManager.WRITE_FAILURE);
            }
        } catch (Exception e) {
            reportError("Could not send Json message: " + e.getMessage(), e, ErrorManager.FORMAT_FAILURE);
        }
    }

    protected JsonSender createJsonSender() {
        return JsonSenderFactory.createSender(jsonMessageAssembler, errorReporter, Collections.<String, Object>emptyMap());
    }

    @Override
    public void reportError(String message, Exception e) {
        reportError(message, e, ErrorManager.GENERIC_FAILURE);
    }

    @Override
    public void close() {
        if (null != jsonSender) {
            Closer.close(jsonSender);
            jsonSender = null;
        }
    }

    protected JsonMessage createJsonMessage(final LogRecord logRecord) {
        return jsonMessageAssembler.createJsonMessage(new JulLogEvent(logRecord));
    }

    public void setAdditionalFields(String spec) {
        ConfigurationSupport.setAdditionalFields(spec, jsonMessageAssembler);
    }

    public void setType(String type) {
        ConfigurationSupport.setType(type, jsonMessageAssembler);
    }

    public void setComponent(String component) {
        ConfigurationSupport.setComponent(component, jsonMessageAssembler);
    }

    public void setAdditionalFieldTypes(String spec) {
        ConfigurationSupport.setAdditionalFieldTypes(spec, jsonMessageAssembler);
    }

    public void setMdcFields(String spec) {
        ConfigurationSupport.setMdcFields(spec, jsonMessageAssembler);
    }

    public void setDynamicMdcFields(String spec) {
        ConfigurationSupport.setDynamicMdcFields(spec, jsonMessageAssembler);
    }

    public void setDynamicMdcFieldTypes(String spec) {
        ConfigurationSupport.setDynamicMdcFieldTypes(spec, jsonMessageAssembler);
    }


    public String getOriginHost() {
        return jsonMessageAssembler.getOriginHost();
    }

    public void setOriginHost(String originHost) {
        jsonMessageAssembler.setOriginHost(originHost);
    }

    public String getHost() {
        return jsonMessageAssembler.getHost();
    }

    public void setHost(String host) {
        jsonMessageAssembler.setHost(host);
    }

    public int getPort() {
        return jsonMessageAssembler.getPort();
    }

    public void setPort(int port) {
        jsonMessageAssembler.setPort(port);
    }


    public String getFacility() {
        return jsonMessageAssembler.getFacility();
    }

    public void setFacility(String facility) {
        jsonMessageAssembler.setFacility(facility);
    }

    public String getExtractStackTrace() {
        return jsonMessageAssembler.getExtractStackTrace();
    }

    public void setExtractStackTrace(String extractStacktrace) {
        jsonMessageAssembler.setExtractStackTrace(extractStacktrace);
    }

    public boolean isFilterStackTrace() {
        return jsonMessageAssembler.isFilterStackTrace();
    }

    public void setFilterStackTrace(boolean filterStackTrace) {
        jsonMessageAssembler.setFilterStackTrace(filterStackTrace);
    }

    public boolean isIncludeLogMessageParameters() {
        return jsonMessageAssembler.isIncludeLogMessageParameters();
    }

    public void setIncludeLogMessageParameters(boolean includeLogMessageParameters) {
        jsonMessageAssembler.setIncludeLogMessageParameters(includeLogMessageParameters);
    }

    public boolean isIncludeLocation() {
        return jsonMessageAssembler.isIncludeLocation();
    }

    public void setIncludeLocation(boolean includeLocation) {
        jsonMessageAssembler.setIncludeLocation(includeLocation);
    }

    public String getTimestampPattern() {
        return jsonMessageAssembler.getTimestampPattern();
    }

    public void setTimestampPattern(String timestampPattern) {
        jsonMessageAssembler.setTimestampPattern(timestampPattern);
    }

    public int getMaximumMessageSize() {
        return jsonMessageAssembler.getMaximumMessageSize();
    }

    public void setMaximumMessageSize(int maximumMessageSize) {
        jsonMessageAssembler.setMaximumMessageSize(maximumMessageSize);
    }

    public String getVersion() {
        return jsonMessageAssembler.getVersion();
    }

    public void setVersion(String version) {
        jsonMessageAssembler.setVersion(version);
    }
}
