package eu.xenit.json.log4j;

import eu.xenit.json.LogMessageField;
import eu.xenit.json.MdcJsonMessageAssembler;
import eu.xenit.json.intern.ConfigurationSupport;
import eu.xenit.json.intern.JsonMessage;
import org.apache.log4j.Layout;
import org.apache.log4j.spi.LoggingEvent;

import java.util.Collections;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.Set;

/**
 * Log-Formatter for JSON using fields specified within Json. This formatter will produce a JSON object for each log event.
 * Example:
 *
 * <code>
 * {
 * "NDC": "ndc message",
 * "timestamp": "1439319236.722",
 * "SourceClassName": "eu.xenit.json.wildfly.WildFlyJsonLogFormatterTest",
 * "SourceMethodName": "testDefaults",
 * "level": "6",
 * "SourceSimpleClassName": "WildFlyJsonLogFormatterTest",
 * "facility": "xenit-json",
 * "fullMessage": "foo bar test log message",
 * "shortMessage": "foo bar test log message",
 * "MySeverity": "INFO",
 * "LoggerName": "eu.xenit.json.wildfly.WildFlyJsonLogFormatterTest",
 * "Thread": "main",
 * "MyTime": "2015-08-11 20:53:56,0722"
 * }
 * </code>
 * <p>
 * Following parameters are supported/needed:
 * <ul>
 * <li>lineBreak (Optional): End of line, platform dependent default value, see {@code System.getProperty("line.separator")}</li>
 * <li>fields (Optional): Comma-separated list of log event fields that should be included in the JSON. Defaults to
 * {@code Time, Severity, ThreadName, SourceClassName, SourceMethodName, SourceSimpleClassName, LoggerName, NDC}</li>
 * <li>originHost (Optional): Originating Hostname, default FQDN Hostname</li>
 * <li>extractStackTrace (Optional): Post Stack-Trace to StackTrace field (true/false/throwable reference [0 = throwable, 1 =
 * throwable.cause, -1 = root cause]), default false</li>
 * <li>filterStackTrace (Optional): Perform Stack-Trace filtering (true/false), default false</li>
 * <li>includeLocation (Optional): Include source code location, default true</li>
 * <li>mdcProfiling (Optional): Perform Profiling (Call-Duration) based on MDC Data. See <a href="#mdcProfiling">MDC
 * Profiling</a>, default false</li>
 * <li>facility (Optional): Name of the Facility, default json-java</li>
 * <li>additionalFields(number) (Optional): Post additional fields. Eg. fieldName=Value,field2=value2</li>
 * <li>additionalFieldTypes (Optional): Type specification for additional and MDC fields. Supported types: String, long, Long,
 * double, Double and discover (default if not specified, discover field type on parseability). Eg. field=String,field2=double</li>
 * <li>mdcFields (Optional): Post additional fields, pull Values from MDC. Name of the Fields are comma-separated
 * Application,Version,SomeOtherFieldName</li>
 * <li>dynamicMdcFields (Optional): Dynamic MDC Fields allows you to extract MDC values based on one or more regular
 * expressions. Multiple regex are comma-separated. The name of the MDC entry is used as Json field name. mdc.*,[mdc|MDC]fields</li>
 * <li>dynamicMdcFieldTypes (Optional): Pattern-based type specification for additional and MDC fields. Key-value pairs are
 * comma-separated. Supported types: String, long, Long, double, Double. Eg. my_field.*=String,business\..*\.field=double</li>
 * <li>includeFullMdc (Optional): Include all fields from the MDC, default false</li>
 * </ul>
 * <a name="mdcProfiling"></a> <h2>MDC Profiling</h2>
 * <p>
 * MDC Profiling allows to calculate the runtime from request start up to the time until the log message was generated. You must
 * set one value in the MDC:
 * <ul>
 * <li>profiling.requestStart.millis: Time Millis of the Request-Start (Long or String)</li>
 * </ul>
 * <p>
 * Two values are set by the formatter:
 * </p>
 * <ul>
 * <li>profiling.requestEnd: End-Time of the Request-End in Date.toString-representation</li>
 * <li>profiling.requestDuration: Duration of the request (e.g. 205ms, 16sec)</li>
 * </ul>
 */
public class JsonLayout extends Layout {

    public static final String MULTI_VALUE_DELIMITTER = ",";
    public static final Set<LogMessageField.NamedLogField> SUPPORTED_FIELDS;
    public static final Set<LogMessageField.NamedLogField> DEFAULT_FIELDS;

    static {
        Set<LogMessageField.NamedLogField> supportedFields = new LinkedHashSet<>();

        Set<LogMessageField.NamedLogField> defaultFields = new LinkedHashSet<>();
        defaultFields.add(LogMessageField.NamedLogField.Time);
        defaultFields.add(LogMessageField.NamedLogField.Severity);
        defaultFields.add(LogMessageField.NamedLogField.LoggerName);
        defaultFields.add(LogMessageField.NamedLogField.ThreadName);
        DEFAULT_FIELDS = Collections.unmodifiableSet(defaultFields);

        supportedFields.add(LogMessageField.NamedLogField.Time);
        supportedFields.add(LogMessageField.NamedLogField.Severity);
        supportedFields.add(LogMessageField.NamedLogField.ThreadName);
        supportedFields.add(LogMessageField.NamedLogField.SourceClassName);
        supportedFields.add(LogMessageField.NamedLogField.SourceMethodName);
        supportedFields.add(LogMessageField.NamedLogField.SourceLineNumber);
        supportedFields.add(LogMessageField.NamedLogField.SourceSimpleClassName);
        supportedFields.add(LogMessageField.NamedLogField.LoggerName);
        supportedFields.add(LogMessageField.NamedLogField.NDC);
        supportedFields.add(LogMessageField.NamedLogField.Server);

        SUPPORTED_FIELDS = Collections.unmodifiableSet(supportedFields);
    }

    private final MdcJsonMessageAssembler jsonMessageAssembler = new MdcJsonMessageAssembler();
    private String lineBreak = Layout.LINE_SEP;
    private boolean wasSetFieldsCalled = false;

    public JsonLayout() {
        super();
    }

    @Override
    public String format(LoggingEvent loggingEvent) {
        if (!wasSetFieldsCalled) {
            addFields(new HashSet<>((DEFAULT_FIELDS)));
        }
        JsonMessage jsonMessage = jsonMessageAssembler.createJsonMessage(new Log4jLogEvent(loggingEvent));
        return jsonMessage.toJson("") + lineBreak;
    }

    @Override
    public boolean ignoresThrowable() {
        return false;
    }

    @Override
    public void activateOptions() {
        // nothing to do
    }

    public void setFields(String fieldSpec) {

        String[] properties = fieldSpec.split(MULTI_VALUE_DELIMITTER);
        Set<LogMessageField.NamedLogField> fields = new HashSet<>();
        for (String field : properties) {

            LogMessageField.NamedLogField namedLogField = LogMessageField.NamedLogField.byName(field.trim());
            if (namedLogField == null) {
                throw new IllegalArgumentException("Cannot resolve field name '" + field
                        + "' to a field. Supported field names are: " + SUPPORTED_FIELDS);
            }

            if (!SUPPORTED_FIELDS.contains(namedLogField)) {
                throw new IllegalArgumentException("Field '" + field + "' is not supported. Supported field names are: "
                        + SUPPORTED_FIELDS);
            }

            fields.add(namedLogField);
        }

        addFields(fields);
    }

    private void addFields(Set<LogMessageField.NamedLogField> fields) {
        fields.addAll(DEFAULT_FIELDS);
        jsonMessageAssembler.addFields(LogMessageField.getDefaultMapping(fields
                .toArray(new LogMessageField.NamedLogField[0])));

        wasSetFieldsCalled = true;
    }

    public void setAdditionalFields(String spec) {
        ConfigurationSupport.setAdditionalFields(spec, jsonMessageAssembler);
    }

    public void setAdditionalFieldTypes(String spec) {
        ConfigurationSupport.setAdditionalFieldTypes(spec, jsonMessageAssembler);
    }

    public void setType(String type) {
        ConfigurationSupport.setType(type, jsonMessageAssembler);
    }

    public void setComponent(String component) {
        ConfigurationSupport.setComponent(component, jsonMessageAssembler);
    }

    public void setMdcFields(String spec) {
        ConfigurationSupport.setMdcFields(spec, jsonMessageAssembler);
    }

    public void setDynamicMdcFields(String spec) {
        ConfigurationSupport.setDynamicMdcFields(spec, jsonMessageAssembler);
    }

    public void setDynamicMdcFieldTypes(String spec) {
        ConfigurationSupport.setDynamicMdcFieldTypes(spec, jsonMessageAssembler);
    }

    public boolean isMdcProfiling() {
        return jsonMessageAssembler.isMdcProfiling();
    }

    public void setMdcProfiling(boolean mdcProfiling) {
        jsonMessageAssembler.setMdcProfiling(mdcProfiling);
    }

    public boolean isIncludeFullMdc() {
        return jsonMessageAssembler.isIncludeFullMdc();
    }

    public void setIncludeFullMdc(boolean includeFullMdc) {
        jsonMessageAssembler.setIncludeFullMdc(includeFullMdc);
    }

    public String getOriginHost() {
        return jsonMessageAssembler.getOriginHost();
    }

    public void setOriginHost(String originHost) {
        jsonMessageAssembler.setOriginHost(originHost);
    }

    public String getFacility() {
        return jsonMessageAssembler.getFacility();
    }

    public void setFacility(String facility) {
        jsonMessageAssembler.setFacility(facility);
    }

    public String getExtractStackTrace() {
        return jsonMessageAssembler.getExtractStackTrace();
    }

    public void setExtractStackTrace(String extractStacktrace) {
        jsonMessageAssembler.setExtractStackTrace(extractStacktrace);
    }

    public boolean isFilterStackTrace() {
        return jsonMessageAssembler.isFilterStackTrace();
    }

    public void setFilterStackTrace(boolean filterStackTrace) {
        jsonMessageAssembler.setFilterStackTrace(filterStackTrace);
    }

    public boolean isIncludeLocation() {
        return jsonMessageAssembler.isIncludeLocation();
    }

    public void setIncludeLocation(boolean includeLocation) {
        jsonMessageAssembler.setIncludeLocation(includeLocation);
    }

    public String getTimestampPattern() {
        return jsonMessageAssembler.getTimestampPattern();
    }

    public void setTimestampPattern(String timestampPattern) {
        jsonMessageAssembler.setTimestampPattern(timestampPattern);
    }

    public String getVersion() {
        return jsonMessageAssembler.getVersion();
    }

    public void setVersion(String version) {
        jsonMessageAssembler.setVersion(version);
    }

    public String getLineBreak() {
        return lineBreak;
    }

    public void setLineBreak(String lineBreak) {
        this.lineBreak = lineBreak;
    }
}
